<?php
/**
 * Verifizierungsstatus-Seite für User
 * Zeigt je nach Status unterschiedliche Inhalte:
 * - pending: Wartend - bitte Dokument hochladen
 * - approved: Erfolgreich verifiziert - Weiterleitung
 * - rejected: Abgelehnt - Grund anzeigen
 */

session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$user = getCurrentUser();

if (!$user) {
    header('Location: login.php');
    exit;
}

// Unternehmensdaten laden
$db = new Database();
$conn = $db->getConnection();

$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/BE-Logo-FINAL_Schwarz-1.png';
$appName = $companySettings['app_name'] ?? 'B&E Gruppe';

// Verifizierungsstatus
$verification_status = $user['verification_status'] ?? 'pending';

// Wenn approved, zur Startseite weiterleiten
if ($verification_status === 'approved') {
    header('Location: index.php');
    exit;
}

// Status-spezifische Informationen
$status_config = [
    'pending' => [
        'icon' => 'fas fa-hourglass-half',
        'title' => 'Verifizierung läuft',
        'message' => 'Ihr Konto wird von einem Administrator geprüft.',
        'submessage' => 'Sie erhalten eine E-Mail, sobald Ihre Verifizierung abgeschlossen ist.',
        'bg_color' => '#fff3cd',
        'icon_color' => '#ffc107',
        'show_upload' => true,
        'upload_message' => 'Falls noch nicht geschehen, laden Sie bitte ein Identitätsdokument hoch.'
    ],
    'rejected' => [
        'icon' => 'fas fa-times-circle',
        'title' => 'Verifizierung abgelehnt',
        'message' => 'Ihr Antrag wurde leider nicht genehmigt.',
        'submessage' => 'Grund: ' . htmlspecialchars($user['verification_reason'] ?? 'Nicht angegeben'),
        'bg_color' => '#f8d7da',
        'icon_color' => '#dc3545',
        'show_upload' => true,
        'upload_message' => 'Sie können ein neues Dokument hochladen und erneut einen Antrag stellen.'
    ]
];

$current_status = $status_config[$verification_status] ?? $status_config['pending'];
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($current_status['title']); ?> - <?php echo htmlspecialchars($appName); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        }
        
        .verification-container {
            max-width: 700px;
            margin: 4rem auto;
            padding: 2rem;
        }
        
        .verification-card {
            background: white;
            border-radius: 24px;
            padding: 3rem;
            box-shadow: var(--card-shadow);
            text-align: center;
        }
        
        .status-icon {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: linear-gradient(135deg, <?php echo $current_status['icon_color']; ?> 0%, <?php echo $current_status['bg_color']; ?> 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 2rem;
            font-size: 3rem;
            color: white;
            box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
        }
        
        .verification-title {
            font-size: 2rem;
            font-weight: 700;
            color: #003057;
            margin-bottom: 1rem;
        }
        
        .verification-message {
            font-size: 1.1rem;
            color: #495057;
            margin-bottom: 0.5rem;
            line-height: 1.6;
        }
        
        .verification-submessage {
            font-size: 0.95rem;
            color: #6c757d;
            margin-bottom: 2rem;
        }
        
        .upload-section {
            background: #f8f9fa;
            border-radius: 16px;
            padding: 2.5rem;
            margin-bottom: 2rem;
            border: 2px dashed #dee2e6;
            transition: all 0.3s;
        }
        
        .upload-section.dragover {
            border-color: #0056b3;
            background: #e3f2fd;
        }
        
        .upload-icon {
            font-size: 3.5rem;
            color: #0056b3;
            margin-bottom: 1rem;
        }
        
        .upload-text {
            color: #495057;
            font-weight: 600;
            margin-bottom: 1.5rem;
            font-size: 1.05rem;
        }
        
        .file-input-wrapper {
            position: relative;
            display: inline-block;
        }
        
        .file-input {
            display: none;
        }
        
        .file-input-label {
            display: inline-block;
            padding: 1rem 2.5rem;
            background: var(--primary-gradient);
            color: white;
            border-radius: 12px;
            cursor: pointer;
            font-weight: 600;
            font-size: 1.05rem;
            transition: all 0.3s;
            box-shadow: 0 4px 12px rgba(0, 48, 87, 0.3);
        }
        
        .file-input-label:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 48, 87, 0.4);
        }
        
        .alert {
            padding: 1.25rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            display: none;
        }
        
        .alert.show {
            display: block;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .btn-group {
            display: flex;
            gap: 1rem;
            justify-content: center;
            margin-top: 2rem;
        }
        
        .btn {
            padding: 0.875rem 2rem;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background: var(--primary-gradient);
            color: white;
            box-shadow: 0 4px 12px rgba(0, 48, 87, 0.3);
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 48, 87, 0.4);
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .info-box {
            background: #e7f3ff;
            border-left: 4px solid #2196F3;
            border-radius: 8px;
            padding: 1.25rem;
            margin-bottom: 2rem;
            text-align: left;
        }
        
        .info-box i {
            color: #2196F3;
            margin-right: 0.5rem;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar" style="position: fixed; top: 0; left: 0; right: 0; z-index: 1000; background: white; backdrop-filter: blur(20px); box-shadow: 0 2px 12px rgba(0,0,0,0.08);">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo" style="height: 50px;">
                    <div class="brand-text">
                        <h1 style="font-size: 1.5rem; color: #003057;"><?php echo htmlspecialchars($appName); ?></h1>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="api/logout.php" class="nav-link logout-link" style="color: #003057;">
                    <i class="fas fa-sign-out-alt"></i>
                    <span>Abmelden</span>
                </a>
            </div>
        </div>
    </nav>

    <div style="margin-top: 80px;"></div>

    <!-- Verification Container -->
    <div class="verification-container">
        <div class="verification-card">
            <div class="status-icon">
                <i class="<?php echo htmlspecialchars($current_status['icon']); ?>"></i>
            </div>
            
            <h1 class="verification-title"><?php echo htmlspecialchars($current_status['title']); ?></h1>
            
            <p class="verification-message">
                <?php echo htmlspecialchars($current_status['message']); ?>
            </p>
            
            <p class="verification-submessage">
                <?php echo htmlspecialchars($current_status['submessage']); ?>
            </p>
            
            <?php if ($user['verification_status'] === 'pending' && !empty($user['verification_document'])): ?>
            <div class="info-box">
                <i class="fas fa-check-circle"></i>
                <strong>Dokument bereits hochgeladen</strong><br>
                Wir bearbeiten Ihr Konto so schnell wie möglich.
            </div>
            <?php endif; ?>
            
            <!-- Alerts -->
            <div id="alert-message" class="alert"></div>
            
            <!-- Upload Section - nur zeigen wenn erlaubt -->
            <?php if ($current_status['show_upload']): ?>
            <div class="upload-section" id="upload-section">
                <div class="upload-icon">
                    <i class="fas fa-cloud-upload-alt"></i>
                </div>
                <div class="upload-text">
                    <?php echo htmlspecialchars($current_status['upload_message']); ?>
                </div>
                <div class="file-input-wrapper">
                    <input type="file" id="verification_document" name="verification_document" 
                           class="file-input" 
                           accept="image/jpeg,image/png,image/jpg,application/pdf">
                    <label for="verification_document" class="file-input-label">
                        <i class="fas fa-file-upload"></i> Datei auswählen
                    </label>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="btn-group">
                <a href="index.php" class="btn btn-primary">
                    <i class="fas fa-home"></i>
                    Zur Startseite
                </a>
                <a href="profile.php" class="btn btn-secondary">
                    <i class="fas fa-user"></i>
                    Zum Profil
                </a>
            </div>
        </div>
    </div>
    
    <script>
        const fileInput = document.getElementById('verification_document');
        const uploadSection = document.getElementById('upload-section');
        const alertMessage = document.getElementById('alert-message');
        
        <?php if ($current_status['show_upload']): ?>
        // Drag & Drop
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            uploadSection.addEventListener(eventName, preventDefaults, false);
            document.body.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        ['dragenter', 'dragover'].forEach(eventName => {
            uploadSection.addEventListener(eventName, () => {
                uploadSection.classList.add('dragover');
            }, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            uploadSection.addEventListener(eventName, () => {
                uploadSection.classList.remove('dragover');
            }, false);
        });
        
        uploadSection.addEventListener('drop', (e) => {
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                fileInput.files = files;
                handleFileUpload(files[0]);
            }
        }, false);
        
        // File Input Handler
        fileInput.addEventListener('change', function(e) {
            if (this.files.length > 0) {
                handleFileUpload(this.files[0]);
            }
        });
        
        async function handleFileUpload(file) {
            showAlert('Datei wird hochgeladen...', 'success');
            
            const formData = new FormData();
            formData.append('verification_document', file);
            
            try {
                const response = await fetch('api/upload_verification_document.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showAlert('Dokument erfolgreich hochgeladen! Ihr Konto wird nun geprüft.', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    showAlert(data.message || 'Fehler beim Hochladen', 'error');
                }
            } catch (error) {
                showAlert('Verbindungsfehler', 'error');
            }
        }
        <?php endif; ?>
        
        function showAlert(message, type) {
            alertMessage.textContent = message;
            alertMessage.className = `alert alert-${type} show`;
        }
    </script>
</body>
</html>
