<?php
/**
 * Profil-Seite für Benutzer - Modern Design
 */

session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$user = getCurrentUser();

if (!$user) {
    header('Location: login.php');
    exit;
}

// Fahrzeuge des Users laden
$db = new Database();
$conn = $db->getConnection();

$stmt = $conn->prepare("
    SELECT * FROM user_vehicles 
    WHERE user_id = ? AND is_active = 1 
    ORDER BY is_default DESC, created_at DESC
");
$stmt->execute([$user['id']]);
$vehicles = $stmt->fetchAll();

// Termine des Users laden
$stmt = $conn->prepare("
    SELECT 
        a.*,
        e.name as event_name,
        e.event_date,
        COALESCE(e.event_time, e.start_time) as event_time,
        e.description as event_description,
        ah.name as autohaus_name,
        ah.address as autohaus_address,
        ah.phone as autohaus_phone
    FROM appointments a
    LEFT JOIN events e ON a.event_id = e.id
    LEFT JOIN autohaus ah ON e.autohaus_id = ah.id
    WHERE a.user_id = ? AND a.is_active = 1
    ORDER BY e.event_date DESC, COALESCE(e.event_time, e.start_time) DESC
");
$stmt->execute([$user['id']]);
$appointments = $stmt->fetchAll();

// Statistiken berechnen
$total_appointments = count($appointments);
$upcoming_appointments = array_filter($appointments, function($apt) {
    return strtotime($apt['event_date'] . ' ' . $apt['event_time']) >= time();
});
$upcoming_count = count($upcoming_appointments);
$past_appointments = array_filter($appointments, function($apt) {
    return strtotime($apt['event_date'] . ' ' . $apt['event_time']) < time();
});
$past_count = count($past_appointments);

// Verifizierungsdokument laden falls vorhanden
$verification_document = !empty($user['verification_document']) ? $user['verification_document'] : null;

// Activity Log laden
$stmt = $conn->prepare("
    SELECT * FROM user_activity_log 
    WHERE user_id = ? 
    ORDER BY created_at DESC 
    LIMIT 20
");
$stmt->execute([$user['id']]);
$activity_log = $stmt->fetchAll();

// Benachrichtigungen laden
$stmt = $conn->prepare("
    SELECT * FROM user_notifications 
    WHERE user_id = ? 
    AND is_read = 0
    ORDER BY created_at DESC 
    LIMIT 10
");
$stmt->execute([$user['id']]);
$unread_notifications = $stmt->fetchAll();

$notifications_count = count($unread_notifications);

// Unternehmensdaten laden
$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('company_name', 'company_logo', 'app_name')");
$stmt->execute();
$companySettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$companyName = $companySettings['company_name'] ?? 'B&E Gruppe';
$companyLogo = $companySettings['company_logo'] ?? 'assets/images/volvo-logo.png';
$appName = $companySettings['app_name'] ?? 'B&E Gruppe';

// Verifizierungsstatus bestimmen
$verification_status = $user['verification_status'] ?? 'pending';
$status_info = [
    'pending' => ['icon' => 'fas fa-hourglass-half', 'text' => 'Wartend auf Verifizierung', 'color' => '#ffc107', 'bg' => '#fff3cd'],
    'approved' => ['icon' => 'fas fa-check-circle', 'text' => 'Verifiziert und freigeschaltet', 'color' => '#28a745', 'bg' => '#d4edda'],
    'rejected' => ['icon' => 'fas fa-times-circle', 'text' => 'Abgelehnt', 'color' => '#dc3545', 'bg' => '#f8d7da']
];

$current_status = $status_info[$verification_status] ?? $status_info['pending'];

// Prüfe ob alle Pflichtfelder ausgefüllt sind
$required_fields = ['phone_number', 'address', 'postal_code', 'city'];
$missing_fields = [];

foreach ($required_fields as $field) {
    if (empty($user[$field])) {
        $missing_fields[] = $field;
    }
}

$has_missing_fields = count($missing_fields) > 0;
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mein Profil - <?php echo htmlspecialchars($appName); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        :root {
            --primary-gradient: linear-gradient(135deg, #003057 0%, #0056b3 100%);
            --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            --card-shadow-hover: 0 8px 30px rgba(0, 0, 0, 0.12);
        }
        
        .profile-container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 2rem;
        }
        
        /* Status Banner */
        .status-banner {
            background: <?php echo $current_status['bg']; ?>;
            border-left: 4px solid <?php echo $current_status['color']; ?>;
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 1rem;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }
        
        .status-banner i {
            font-size: 2rem;
            color: <?php echo $current_status['color']; ?>;
        }
        
        .status-banner-content {
            flex: 1;
        }
        
        .status-banner-title {
            font-weight: 700;
            font-size: 1.1rem;
            color: #495057;
            margin-bottom: 0.5rem;
        }
        
        .status-banner-text {
            color: #6c757d;
            font-size: 0.95rem;
        }
        
        <?php if ($verification_status === 'pending' || $verification_status === 'rejected'): ?>
        .status-action {
            display: flex;
            gap: 0.75rem;
        }
        <?php endif; ?>
        
        /* Profile Header - Modern Card */
        .profile-header-card {
            background: var(--primary-gradient);
            border-radius: 24px;
            padding: 3rem;
            margin-bottom: 2rem;
            color: white;
            box-shadow: var(--card-shadow);
            position: relative;
            overflow: hidden;
        }
        
        .profile-header-card::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 100%;
            height: 100%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
            animation: pulse 3s ease-in-out infinite;
        }
        
        .profile-header-content {
            display: grid;
            grid-template-columns: auto 1fr auto;
            gap: 2rem;
            align-items: center;
            position: relative;
            z-index: 1;
        }
        
        .profile-avatar {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 3rem;
            font-weight: 700;
            border: 4px solid rgba(255, 255, 255, 0.3);
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.2);
        }
        
        .profile-info {
            color: white;
        }
        
        .profile-name {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        
        .profile-email {
            opacity: 0.9;
            font-size: 1.1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .profile-stats {
            display: flex;
            flex-direction: column;
            gap: 1rem;
        }
        
        .stat-item {
            background: rgba(255, 255, 255, 0.15);
            backdrop-filter: blur(10px);
            padding: 1rem 1.5rem;
            border-radius: 12px;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 700;
        }
        
        .stat-label {
            font-size: 0.9rem;
            opacity: 0.9;
            color: white;
        }
        
        /* Section Cards */
        .section-card {
            background: white;
            border-radius: 20px;
            padding: 2.5rem;
            margin-bottom: 2rem;
            box-shadow: var(--card-shadow);
            transition: all 0.3s;
        }
        
        .section-card:hover {
            box-shadow: var(--card-shadow-hover);
            transform: translateY(-2px);
        }
        
        .section-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #003057;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .section-title i {
            color: #0056b3;
        }
        
        /* Vehicle Cards - Enhanced */
        .vehicles-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 1.5rem;
        }
        
        .vehicle-card {
            border: 2px solid #e9ecef;
            border-radius: 16px;
            padding: 1.75rem;
            transition: all 0.3s;
            position: relative;
            background: white;
        }
        
        .vehicle-card:hover {
            border-color: #0056b3;
            transform: translateY(-4px);
            box-shadow: 0 8px 24px rgba(0, 86, 179, 0.15);
        }
        
        .vehicle-card.default {
            border-color: #28a745;
            background: linear-gradient(135deg, #d4edda 0%, #e8f5e9 100%);
            box-shadow: 0 4px 16px rgba(40, 167, 69, 0.2);
        }
        
        .vehicle-badge {
            position: absolute;
            top: 1rem;
            right: 1rem;
            background: #28a745;
            color: white;
            padding: 0.35rem 0.85rem;
            border-radius: 20px;
            font-size: 0.75rem;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            box-shadow: 0 2px 8px rgba(40, 167, 69, 0.3);
        }
        
        .vehicle-license {
            font-size: 1.75rem;
            font-weight: 700;
            color: #003057;
            margin-bottom: 0.75rem;
        }
        
        .vehicle-info {
            color: #6c757d;
            font-size: 0.95rem;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .vehicle-info i {
            width: 20px;
            color: #0056b3;
        }
        
        .vehicle-actions {
            display: flex;
            gap: 0.5rem;
            margin-top: 1.5rem;
            padding-top: 1.5rem;
            border-top: 1px solid #e9ecef;
        }
        
        .btn-action {
            flex: 1;
            padding: 0.75rem 1rem;
            border: none;
            border-radius: 10px;
            font-size: 0.95rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }
        
        .btn-action-edit {
            background: #0056b3;
            color: white;
        }
        
        .btn-action-edit:hover {
            background: #003057;
            transform: translateY(-2px);
        }
        
        .btn-action-delete {
            background: #dc3545;
            color: white;
        }
        
        .btn-action-delete:hover {
            background: #c82333;
            transform: translateY(-2px);
        }
        
        .add-vehicle-card {
            border: 2px dashed #dee2e6;
            border-radius: 16px;
            padding: 2rem;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s;
            min-height: 250px;
            background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
        }
        
        .add-vehicle-card:hover {
            border-color: #0056b3;
            background: linear-gradient(135deg, #e3f2fd 0%, #f8f9fa 100%);
            transform: translateY(-4px);
            box-shadow: 0 8px 24px rgba(0, 86, 179, 0.15);
        }
        
        .add-vehicle-icon {
            font-size: 4rem;
            color: #0056b3;
            margin-bottom: 1rem;
        }
        
        .add-vehicle-text {
            color: #6c757d;
            font-weight: 600;
            font-size: 1.1rem;
        }
        
        /* Modal */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.6);
            backdrop-filter: blur(4px);
            animation: fadeIn 0.3s;
        }
        
        .modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .modal-content {
            background: white;
            border-radius: 24px;
            padding: 2.5rem;
            max-width: 550px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            animation: slideUp 0.3s;
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(50px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .modal-title {
            font-size: 1.75rem;
            font-weight: 700;
            color: #003057;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            transition: all 0.3s;
        }
        
        .close-modal:hover {
            color: #dc3545;
            transform: rotate(90deg);
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            color: #495057;
            font-weight: 600;
            font-size: 0.95rem;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.875rem 1.25rem;
            border: 2px solid #dee2e6;
            border-radius: 12px;
            font-size: 1rem;
            transition: all 0.3s;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #0056b3;
            box-shadow: 0 0 0 4px rgba(0, 86, 179, 0.1);
        }
        
        .form-actions {
            display: flex;
            gap: 1rem;
        }
        
        .btn {
            flex: 1;
            padding: 1rem;
            border: none;
            border-radius: 12px;
            font-size: 1.05rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background: var(--primary-gradient);
            color: white;
            box-shadow: 0 4px 12px rgba(0, 48, 87, 0.3);
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 48, 87, 0.4);
        }
        
        .btn-secondary {
            background: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background: #5a6268;
        }
        
        .alert {
            padding: 1rem 1.25rem;
            border-radius: 12px;
            margin-bottom: 1.5rem;
            display: none;
            font-weight: 500;
        }
        
        .alert.show {
            display: block;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        /* Notification Banner */
        .notification-banner {
            position: fixed;
            top: 80px;
            left: 50%;
            transform: translateX(-50%);
            background: white;
            padding: 1.25rem 2rem;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.15);
            z-index: 9999;
            display: none;
            align-items: center;
            gap: 1rem;
            animation: slideDown 0.3s ease-out;
            max-width: 90%;
        }
        
        .notification-banner.show {
            display: flex;
        }
        
        .notification-banner i {
            font-size: 1.5rem;
        }
        
        .notification-banner.success {
            border-left: 4px solid #28a745;
        }
        
        .notification-banner.success i {
            color: #28a745;
        }
        
        .notification-banner.error {
            border-left: 4px solid #dc3545;
        }
        
        .notification-banner.error i {
            color: #dc3545;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateX(-50%) translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateX(-50%) translateY(0);
            }
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .profile-header-content {
                grid-template-columns: 1fr;
                text-align: center;
            }
            
            .vehicles-grid {
                grid-template-columns: 1fr;
            }
            
            .stat-item {
                text-align: center;
            }
            
            .notification-banner {
                top: 60px;
                padding: 1rem 1.5rem;
            }
        }
        
        /* Verbessertes Navigation Design - Volvo Style */
        .nav-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            border: 1px solid #e9ecef;
        }
        
        .nav-link i {
            font-size: 1rem;
        }
        
        .back-link {
            background: #6c757d;
            color: white;
            border-color: #6c757d;
        }
        
        .back-link:hover {
            background: #5a6268;
            border-color: #5a6268;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(108, 117, 125, 0.2);
        }
        
        .logout-link {
            background: #dc3545;
            color: white;
            border-color: #dc3545;
        }
        
        .logout-link:hover {
            background: #c82333;
            border-color: #c82333;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(220, 53, 69, 0.2);
        }
        
        @media (max-width: 768px) {
            .nav-menu {
                flex-wrap: wrap;
                gap: 0.5rem;
            }
            
            .nav-link {
                padding: 0.625rem 1rem;
                font-size: 0.875rem;
            }
            
            .nav-link span {
                display: none;
            }
            
            .nav-link {
                width: 44px;
                height: 44px;
                justify-content: center;
                padding: 0;
            }
        }
        
        /* Profile Details View */
        .profile-details-view {
            display: grid;
            gap: 1.5rem;
        }
        
        .profile-detail-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.25rem;
            background: #f8f9fa;
            border-radius: 12px;
            border: 1px solid #e9ecef;
            transition: all 0.3s;
        }
        
        .profile-detail-item:hover {
            background: #e9ecef;
            border-color: #dee2e6;
        }
        
        .profile-detail-label {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-weight: 500;
            color: #495057;
        }
        
        .profile-detail-label i {
            color: #0056b3;
            font-size: 1.1rem;
        }
        
        .profile-detail-value {
            font-weight: 600;
            color: #003057;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #495057;
        }
        
        .form-group input {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #0056b3;
            box-shadow: 0 0 0 3px rgba(0, 86, 179, 0.1);
        }
        
        /* Profile Edit Button */
        .profile-edit-btn {
            margin-left: 0.75rem;
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
            border-radius: 50%;
            width: 36px;
            height: 36px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .profile-edit-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            border-color: rgba(255, 255, 255, 0.5);
            transform: scale(1.1);
        }
        
        .profile-name {
            display: flex;
            align-items: center;
        }
        
        /* Profile Modal */
        #profile-modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            align-items: center;
            justify-content: center;
        }
        
        #profile-modal.active {
            display: flex;
        }
        
        #profile-modal .modal-content {
            background: white;
            border-radius: 20px;
            max-width: 600px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1.5rem 2rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .modal-header h3 {
            margin: 0;
            color: #003057;
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #6c757d;
            transition: color 0.3s;
        }
        
        .close-modal:hover {
            color: #003057;
        }
        
        .profile-image-upload {
            display: flex;
            flex-direction: column;
            align-items: center;
            margin-bottom: 2rem;
        }
        
        .profile-image-preview {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            object-fit: cover;
            border: 4px solid #e9ecef;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="<?php echo htmlspecialchars($companyLogo); ?>" alt="<?php echo htmlspecialchars($companyName); ?>" class="nav-logo">
                    <div class="brand-text">
                        <h1><?php echo htmlspecialchars($appName); ?></h1>
                        <span class="brand-subtitle">B&E Gruppe - Mein Profil</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="index.php" class="nav-link back-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Zurück</span>
                </a>
                <a href="api/logout.php" class="nav-link logout-link">
                    <i class="fas fa-sign-out-alt"></i>
                    <span>Abmelden</span>
                </a>
            </div>
        </div>
    </nav>

    <!-- Profile Container -->
    <div class="profile-container">
        <!-- Status Banner -->
        <div class="status-banner">
            <i class="<?php echo $current_status['icon']; ?>"></i>
            <div class="status-banner-content">
                <div class="status-banner-title"><?php echo $current_status['text']; ?></div>
                <div class="status-banner-text">
                    <?php if ($verification_status === 'pending'): ?>
                        Bitte laden Sie ein Identitätsdokument hoch, um Ihr Konto zu verifizieren.
                    <?php elseif ($verification_status === 'rejected'): ?>
                        Grund: <?php echo htmlspecialchars($user['verification_reason'] ?? 'Nicht angegeben'); ?>
                    <?php else: ?>
                        Ihr Konto ist verifiziert und aktiv. Sie können alle Funktionen nutzen.
                    <?php endif; ?>
                </div>
            </div>
            <?php if ($verification_status === 'pending' || $verification_status === 'rejected'): ?>
            <div class="status-action">
                <a href="verification_pending.php" class="btn btn-primary" style="white-space: nowrap;">
                    <i class="fas fa-file-upload"></i>
                    <?php echo $verification_status === 'rejected' ? 'Neues Dokument hochladen' : 'Dokument hochladen'; ?>
                </a>
            </div>
            <?php endif; ?>
        </div>
        
        <?php if ($has_missing_fields): ?>
        <div class="status-banner" style="background: #fff3cd; border-left-color: #ffc107;">
            <i class="fas fa-exclamation-triangle" style="color: #ffc107;"></i>
            <div class="status-banner-content">
                <div class="status-banner-title">Profil unvollständig</div>
                <div class="status-banner-text">
                    Bitte vervollständigen Sie Ihr Profil. Fehlende Pflichtangaben: 
                    <strong><?php 
                        $field_labels = [
                            'phone_number' => 'Telefonnummer',
                            'address' => 'Adresse',
                            'postal_code' => 'Postleitzahl',
                            'city' => 'Stadt'
                        ];
                        $missing_labels = array_map(function($field) use ($field_labels) {
                            return $field_labels[$field] ?? $field;
                        }, $missing_fields);
                        echo htmlspecialchars(implode(', ', $missing_labels));
                    ?></strong>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Profile Header -->
        <div class="profile-header-card">
            <div class="profile-header-content">
                <div class="profile-avatar">
                    <?php if (!empty($user['profile_image'])): ?>
                        <img src="<?php echo htmlspecialchars($user['profile_image']); ?>" alt="Profilbild" style="width: 100%; height: 100%; object-fit: cover; border-radius: 50%;">
                    <?php else: ?>
                        <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                    <?php endif; ?>
                </div>
                <div class="profile-info">
                    <div class="profile-name">
                        <?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>
                        <button class="profile-edit-btn" onclick="openProfileModal()" title="Profil bearbeiten">
                            <i class="fas fa-edit"></i>
                        </button>
                    </div>
                    <div class="profile-email">
                        <i class="fas fa-envelope"></i>
                        <?php echo htmlspecialchars($user['email']); ?>
                    </div>
                </div>
                <div class="profile-stats">
                    <div class="stat-item">
                        <div class="stat-value"><?php echo count($vehicles); ?></div>
                        <div class="stat-label">Fahrzeuge</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value"><?php echo $total_appointments; ?></div>
                        <div class="stat-label">Termine</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-value"><?php echo $upcoming_count; ?></div>
                        <div class="stat-label">Bevorstehend</div>
                    </div>
                </div>
            </div>
        </div>


        <!-- Alerts -->
        <div id="alert-message" class="alert"></div>

        <!-- Profile Tabs -->
        <div class="profile-tabs-container">
            <div class="profile-tabs">
                <button class="tab-btn active" onclick="showProfileTab('vehicles')" data-tab="vehicles">
                    <i class="fas fa-car"></i>
                    <span>Fahrzeuge</span>
                    <small><?php echo count($vehicles); ?></small>
                </button>
                <button class="tab-btn" onclick="showProfileTab('appointments')" data-tab="appointments">
                    <i class="fas fa-calendar-alt"></i>
                    <span>Termine</span>
                    <small><?php echo $total_appointments; ?></small>
                </button>
                <button class="tab-btn" onclick="showProfileTab('notifications')" data-tab="notifications">
                    <i class="fas fa-bell"></i>
                    <span>Benachrichtigungen</span>
                    <?php if ($notifications_count > 0): ?>
                    <small style="background: #dc3545; color: #fff;"><?php echo $notifications_count; ?></small>
                    <?php endif; ?>
                </button>
                <button class="tab-btn" onclick="showProfileTab('activity')" data-tab="activity">
                    <i class="fas fa-history"></i>
                    <span>Aktivität</span>
                </button>
                <?php if ($verification_document): ?>
                <button class="tab-btn" onclick="showProfileTab('documents')" data-tab="documents">
                    <i class="fas fa-file-alt"></i>
                    <span>Dokumente</span>
                </button>
                <?php endif; ?>
                <button class="tab-btn" onclick="showProfileTab('security')" data-tab="security">
                    <i class="fas fa-shield-alt"></i>
                    <span>Sicherheit</span>
                </button>
            </div>
        </div>

        <!-- Fahrzeuge Tab Content -->
        <div id="vehicles-tab" class="tab-content active">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-car"></i>
                Meine Fahrzeuge
            </h2>
            
            <div class="vehicles-grid">
                <?php if (empty($vehicles)): ?>
                    <div class="add-vehicle-card" onclick="openVehicleModal()">
                        <div class="add-vehicle-icon">
                            <i class="fas fa-plus-circle"></i>
                        </div>
                        <div class="add-vehicle-text">Fahrzeug hinzufügen</div>
                    </div>
                <?php else: ?>
                    <?php foreach ($vehicles as $vehicle): ?>
                        <div class="vehicle-card <?php echo $vehicle['is_default'] ? 'default' : ''; ?>">
                            <?php if ($vehicle['is_default']): ?>
                                <div class="vehicle-badge">
                                    <i class="fas fa-star"></i> Standard
                                </div>
                            <?php endif; ?>
                            <div class="vehicle-license">
                                <?php echo htmlspecialchars($vehicle['license_plate']); ?>
                            </div>
                            <?php if ($vehicle['vehicle_brand']): ?>
                                <div class="vehicle-info">
                                    <i class="fas fa-tag"></i>
                                    <span><?php echo htmlspecialchars($vehicle['vehicle_brand']); ?>
                                    <?php if ($vehicle['vehicle_model']): ?>
                                        <?php echo htmlspecialchars($vehicle['vehicle_model']); ?>
                                    <?php endif; ?></span>
                                </div>
                            <?php endif; ?>
                            <?php if ($vehicle['vehicle_year']): ?>
                                <div class="vehicle-info">
                                    <i class="fas fa-calendar"></i>
                                    <span>Baujahr <?php echo htmlspecialchars($vehicle['vehicle_year']); ?></span>
                                </div>
                            <?php endif; ?>
                            <?php if ($vehicle['notes']): ?>
                                <div class="vehicle-info">
                                    <i class="fas fa-sticky-note"></i>
                                    <span><?php echo htmlspecialchars($vehicle['notes']); ?></span>
                                </div>
                            <?php endif; ?>
                            <div class="vehicle-actions">
                                <button class="btn-action btn-action-edit" onclick="openVehicleModal(<?php echo $vehicle['id']; ?>)">
                                    <i class="fas fa-edit"></i>
                                    <span>Bearbeiten</span>
                                </button>
                                <?php if (!$vehicle['is_default']): ?>
                                    <button class="btn-action btn-action-delete" onclick="deleteVehicle(<?php echo $vehicle['id']; ?>)">
                                        <i class="fas fa-trash"></i>
                                        <span>Löschen</span>
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    <div class="add-vehicle-card" onclick="openVehicleModal()">
                        <div class="add-vehicle-icon">
                            <i class="fas fa-plus-circle"></i>
                        </div>
                        <div class="add-vehicle-text">Weiteres Fahrzeug hinzufügen</div>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Appointments Tab Content -->
        <div id="appointments-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-calendar-alt"></i>
                Meine Termine
            </h2>
            
            <div class="appointments-list">
                <?php if (empty($appointments)): ?>
                    <div class="empty-state">
                        <i class="fas fa-calendar-times"></i>
                        <p>Sie haben noch keine Termine gebucht.</p>
                        <a href="index.php" class="btn btn-primary">
                            <i class="fas fa-plus"></i>
                            Termin buchen
                        </a>
                    </div>
                <?php else: ?>
                    <div class="appointments-tabs" style="display: flex; gap: 1rem; margin-bottom: 1.5rem; border-bottom: 2px solid #e9ecef;">
                        <button class="tab-btn active" onclick="showAppointmentTab('all')" data-tab="all">
                            <i class="fas fa-list"></i> Alle (<?php echo $total_appointments; ?>)
                        </button>
                        <button class="tab-btn" onclick="showAppointmentTab('upcoming')" data-tab="upcoming">
                            <i class="fas fa-clock"></i> Bevorstehend (<?php echo $upcoming_count; ?>)
                        </button>
                        <button class="tab-btn" onclick="showAppointmentTab('past')" data-tab="past">
                            <i class="fas fa-history"></i> Vergangene (<?php echo $past_count; ?>)
                        </button>
                    </div>
                    
                    <div id="appointments-content">
                        <?php foreach ($appointments as $apt): ?>
                            <?php 
                            $is_past = strtotime($apt['event_date'] . ' ' . $apt['event_time']) < time();
                            $date_formatted = date('d.m.Y', strtotime($apt['event_date']));
                            $time_formatted = date('H:i', strtotime($apt['event_time']));
                            ?>
                            <div class="appointment-card <?php echo $is_past ? 'past' : ''; ?>" data-date="<?php echo $apt['event_date']; ?>">
                                <div class="appointment-status">
                                    <?php if ($is_past): ?>
                                        <i class="fas fa-check-circle" style="color: #6c757d;"></i>
                                        <span>Abgeschlossen</span>
                                    <?php else: ?>
                                        <i class="fas fa-clock" style="color: #28a745;"></i>
                                        <span>Bevorstehend</span>
                                    <?php endif; ?>
                                </div>
                                <div class="appointment-main">
                                    <h3><?php echo htmlspecialchars($apt['event_name']); ?></h3>
                                    <div class="appointment-details">
                                        <div class="detail-item">
                                            <i class="fas fa-calendar"></i>
                                            <span><?php echo $date_formatted; ?></span>
                                        </div>
                                        <div class="detail-item">
                                            <i class="fas fa-clock"></i>
                                            <span><?php echo $time_formatted; ?> Uhr</span>
                                        </div>
                                        <div class="detail-item">
                                            <i class="fas fa-building"></i>
                                            <span><?php echo htmlspecialchars($apt['autohaus_name']); ?></span>
                                        </div>
                                        <?php if ($apt['autohaus_phone']): ?>
                                        <div class="detail-item">
                                            <i class="fas fa-phone"></i>
                                            <span><?php echo htmlspecialchars($apt['autohaus_phone']); ?></span>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ($apt['notes']): ?>
                                        <div class="appointment-notes">
                                            <i class="fas fa-sticky-note"></i>
                                            <?php echo htmlspecialchars($apt['notes']); ?>
                                        </div>
                                    <?php endif; ?>
                                    <?php if (!$is_past): ?>
                                        <button class="appointment-cancel-btn" onclick="cancelAppointment(<?php echo $apt['id']; ?>)">
                                            <i class="fas fa-times"></i>
                                            Termin stornieren
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Documents Tab Content -->
        <div id="documents-tab" class="tab-content">
        <?php if ($verification_document): ?>
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-file-alt"></i>
                Meine Dokumente
            </h2>
            
            <div class="document-list">
                <div class="document-card">
                    <div class="document-icon">
                        <i class="fas fa-id-card"></i>
                    </div>
                    <div class="document-info">
                        <h4>Verifizierungsdokument</h4>
                        <p>Hochgeladen am <?php echo date('d.m.Y', strtotime($user['created_at'])); ?></p>
                        <small style="color: #6c757d;">Status: <?php echo $verification_status === 'approved' ? '✓ Verifiziert' : ($verification_status === 'rejected' ? '✗ Abgelehnt' : '🔄 Wird bearbeitet'); ?></small>
                    </div>
                    <div class="document-actions">
                        <a href="<?php echo htmlspecialchars($verification_document); ?>" target="_blank" class="btn-action btn-action-edit">
                            <i class="fas fa-eye"></i>
                            <span>Ansehen</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php else: ?>
        <div class="section-card">
            <div class="empty-state">
                <i class="fas fa-file-alt"></i>
                <p>Keine Dokumente vorhanden</p>
            </div>
        </div>
        <?php endif; ?>
        </div>

        <!-- Notifications Tab Content -->
        <div id="notifications-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-bell"></i>
                Benachrichtigungen
            </h2>
            
            <div class="notifications-list">
                <?php if (empty($unread_notifications)): ?>
                    <div class="empty-state">
                        <i class="fas fa-bell-slash"></i>
                        <p>Keine neuen Benachrichtigungen</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($unread_notifications as $notif): ?>
                        <div class="notification-item">
                            <div class="notification-icon">
                                <?php
                                $icons = [
                                    'system' => 'fas fa-info-circle',
                                    'appointment' => 'fas fa-calendar-check',
                                    'verification' => 'fas fa-check-circle',
                                    'warning' => 'fas fa-exclamation-triangle',
                                    'success' => 'fas fa-check-circle'
                                ];
                                $icon = $icons[$notif['type']] ?? 'fas fa-bell';
                                ?>
                                <i class="<?php echo $icon; ?>"></i>
                            </div>
                            <div class="notification-content">
                                <h4><?php echo htmlspecialchars($notif['title'] ?? 'Benachrichtigung'); ?></h4>
                                <p><?php echo htmlspecialchars($notif['message']); ?></p>
                                <small><?php echo date('d.m.Y H:i', strtotime($notif['created_at'])); ?></small>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Activity Tab Content -->
        <div id="activity-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-history"></i>
                Aktivitätsprotokoll
            </h2>
            
            <div class="activity-list">
                <?php if (empty($activity_log)): ?>
                    <div class="empty-state">
                        <i class="fas fa-history"></i>
                        <p>Noch keine Aktivitäten</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($activity_log as $activity): ?>
                        <div class="activity-item">
                            <div class="activity-icon">
                                <?php
                                $action_icons = [
                                    'profile_updated' => 'fas fa-user-edit',
                                    'vehicle_added' => 'fas fa-plus-circle',
                                    'vehicle_updated' => 'fas fa-edit',
                                    'vehicle_deleted' => 'fas fa-trash',
                                    'appointment_created' => 'fas fa-calendar-plus',
                                    'appointment_cancelled' => 'fas fa-calendar-times',
                                    'appointment_rescheduled' => 'fas fa-calendar-alt',
                                    'document_uploaded' => 'fas fa-file-upload',
                                    'password_changed' => 'fas fa-key',
                                    'login' => 'fas fa-sign-in-alt',
                                    'logout' => 'fas fa-sign-out-alt'
                                ];
                                $icon = $action_icons[$activity['action_type']] ?? 'fas fa-circle';
                                ?>
                                <i class="<?php echo $icon; ?>"></i>
                            </div>
                            <div class="activity-content">
                                <p><strong><?php echo htmlspecialchars($activity['description']); ?></strong></p>
                                <small><?php echo date('d.m.Y H:i', strtotime($activity['created_at'])); ?></small>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        </div>

        <!-- Security Tab Content -->
        <div id="security-tab" class="tab-content">
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-shield-alt"></i>
                Sicherheit & Privatsphäre
            </h2>
            
            <div class="security-section">
                <button class="btn btn-secondary" onclick="openPasswordModal()">
                    <i class="fas fa-key"></i>
                    Passwort ändern
                </button>
            </div>
            
            <div style="margin-top: 2rem; padding: 1.5rem; background: #f8f9fa; border-radius: 12px;">
                <h3 style="margin: 0 0 1rem 0; color: #003057;">
                    <i class="fas fa-info-circle"></i>
                    Konto-Informationen
                </h3>
                <div style="display: grid; gap: 1rem;">
                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; background: #fff; border-radius: 8px;">
                        <span style="color: #6c757d;">E-Mail-Adresse:</span>
                        <strong><?php echo htmlspecialchars($user['email']); ?></strong>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; background: #fff; border-radius: 8px;">
                        <span style="color: #6c757d;">Registriert am:</span>
                        <strong><?php echo date('d.m.Y', strtotime($user['created_at'])); ?></strong>
                    </div>
                    <div style="display: flex; justify-content: space-between; padding: 0.75rem; background: #fff; border-radius: 8px;">
                        <span style="color: #6c757d;">Registrierung via:</span>
                        <strong><?php 
                            $reg_source = $user['registration_source'] ?? 'unknown';
                            $source_labels = [
                                'web' => 'Web',
                                'ios' => 'iOS App',
                                'android' => 'Android App',
                                'apple_signin' => 'Apple Sign-In',
                                'google_signin' => 'Google Sign-In',
                                'unknown' => 'Unbekannt'
                            ];
                            echo $source_labels[$reg_source] ?? 'Unbekannt';
                        ?></strong>
                    </div>
                </div>
            </div>
        </div>
        </div>
    </div>

    <!-- Notification Banner -->
    <div id="notification-banner" class="notification-banner">
        <i class="fas"></i>
        <span id="notification-text"></span>
    </div>

    <!-- Modal für Fahrzeug hinzufügen/bearbeiten -->
    <div id="vehicle-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 class="modal-title" id="modal-title">Fahrzeug hinzufügen</h3>
                <button class="close-modal" onclick="closeVehicleModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="vehicle-form">
                <input type="hidden" id="vehicle-id" name="vehicle_id">
                
                <div class="form-group">
                    <label for="license-plate">Kennzeichen *</label>
                    <input type="text" id="license-plate" name="license_plate" required 
                           placeholder="z. B. M-AB 123">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-brand">Marke</label>
                    <input type="text" id="vehicle-brand" name="vehicle_brand" 
                           placeholder="z. B. Volvo">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-model">Modell</label>
                    <input type="text" id="vehicle-model" name="vehicle_model" 
                           placeholder="z. B. XC60">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-year">Baujahr</label>
                    <input type="number" id="vehicle-year" name="vehicle_year" 
                           placeholder="z. B. 2020" min="1900" max="<?php echo date('Y'); ?>">
                </div>
                
                <div class="form-group">
                    <label for="vehicle-notes">Notizen</label>
                    <textarea id="vehicle-notes" name="vehicle_notes" rows="3" 
                              placeholder="Zusätzliche Informationen..."></textarea>
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 0.5rem; cursor: pointer;">
                        <input type="checkbox" id="is-default" name="is_default" style="width: auto;">
                        Als Standardfahrzeug setzen
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeVehicleModal()">
                        Abbrechen
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i> Speichern
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal für Profil bearbeiten -->
    <div id="profile-modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Profil bearbeiten</h3>
                <button class="close-modal" onclick="closeProfileModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="profile-edit-form" style="padding: 2rem;">
                <div class="profile-image-upload">
                    <img id="profile-image-preview" src="<?php echo !empty($user['profile_image']) ? htmlspecialchars($user['profile_image']) : 'assets/images/default-avatar.png'; ?>" alt="Profilbild" class="profile-image-preview">
                    <input type="file" id="profile-image-input" name="profile_image" accept="image/*" style="display: none;" onchange="previewImage(event)">
                    <button type="button" class="btn-action btn-action-edit" style="margin-top: 1rem;" onclick="document.getElementById('profile-image-input').click()">
                        <i class="fas fa-camera"></i>
                        <span>Bild ändern</span>
                    </button>
                </div>
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1.5rem;">
                    <div class="form-group">
                        <label for="edit-first-name">Vorname *</label>
                        <input type="text" id="edit-first-name" name="first_name" required 
                               value="<?php echo htmlspecialchars($user['first_name']); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-last-name">Nachname *</label>
                        <input type="text" id="edit-last-name" name="last_name" required 
                               value="<?php echo htmlspecialchars($user['last_name']); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-phone">Telefonnummer</label>
                        <input type="tel" id="edit-phone" name="phone_number" 
                               value="<?php echo htmlspecialchars($user['phone_number'] ?? ''); ?>"
                               placeholder="z. B. +49 123 456 789">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-address">Adresse</label>
                        <input type="text" id="edit-address" name="address" 
                               value="<?php echo htmlspecialchars($user['address'] ?? ''); ?>"
                               placeholder="z. B. Musterstraße 123">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-postal-code">Postleitzahl</label>
                        <input type="text" id="edit-postal-code" name="postal_code" 
                               value="<?php echo htmlspecialchars($user['postal_code'] ?? ''); ?>"
                               placeholder="z. B. 12345">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit-city">Stadt</label>
                        <input type="text" id="edit-city" name="city" 
                               value="<?php echo htmlspecialchars($user['city'] ?? ''); ?>"
                               placeholder="z. B. München">
                    </div>
                </div>
                
                <div style="margin-top: 2rem; display: flex; gap: 1rem; justify-content: flex-end;">
                    <button type="button" class="btn-action" onclick="closeProfileModal()" style="background: #6c757d;">
                        <i class="fas fa-times"></i>
                        <span>Abbrechen</span>
                    </button>
                    <button type="submit" class="btn-action btn-action-edit">
                        <i class="fas fa-save"></i>
                        <span>Speichern</span>
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal für Passwort ändern -->
    <div id="password-modal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Passwort ändern</h3>
                <button class="close-modal" onclick="closePasswordModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <form id="password-change-form" style="padding: 2rem;">
                <div class="form-group">
                    <label for="current-password">Aktuelles Passwort *</label>
                    <input type="password" id="current-password" name="current_password" required>
                </div>
                
                <div class="form-group">
                    <label for="new-password">Neues Passwort *</label>
                    <input type="password" id="new-password" name="new_password" required minlength="6">
                    <small style="color: #6c757d;">Mindestens 6 Zeichen</small>
                </div>
                
                <div class="form-group">
                    <label for="confirm-password">Neues Passwort bestätigen *</label>
                    <input type="password" id="confirm-password" name="confirm_password" required>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closePasswordModal()">
                        Abbrechen
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-key"></i>
                        Passwort ändern
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script src="assets/js/main.js"></script>
    <script>
        let currentVehicleId = null;
        
        // Modal öffnen
        function openVehicleModal(vehicleId = null) {
            currentVehicleId = vehicleId;
            const modal = document.getElementById('vehicle-modal');
            const form = document.getElementById('vehicle-form');
            const title = document.getElementById('modal-title');
            
            if (vehicleId) {
                title.textContent = 'Fahrzeug bearbeiten';
                loadVehicleData(vehicleId);
            } else {
                title.textContent = 'Fahrzeug hinzufügen';
                form.reset();
            }
            
            modal.classList.add('active');
        }
        
        // Modal schließen
        function closeVehicleModal() {
            document.getElementById('vehicle-modal').classList.remove('active');
            document.getElementById('vehicle-form').reset();
            currentVehicleId = null;
        }
        
        // Fahrzeugdaten laden
        async function loadVehicleData(vehicleId) {
            try {
                const response = await fetch(`api/get_vehicle.php?id=${vehicleId}`);
                const data = await response.json();
                
                if (data.success) {
                    const vehicle = data.vehicle;
                    document.getElementById('vehicle-id').value = vehicle.id;
                    document.getElementById('license-plate').value = vehicle.license_plate;
                    document.getElementById('vehicle-brand').value = vehicle.vehicle_brand || '';
                    document.getElementById('vehicle-model').value = vehicle.vehicle_model || '';
                    document.getElementById('vehicle-year').value = vehicle.vehicle_year || '';
                    document.getElementById('vehicle-notes').value = vehicle.notes || '';
                    document.getElementById('is-default').checked = vehicle.is_default;
                }
            } catch (error) {
                showAlert('Fehler beim Laden der Fahrzeugdaten', 'error');
            }
        }
        
        // Fahrzeug speichern
        document.getElementById('vehicle-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = {};
            formData.forEach((value, key) => {
                data[key] = value;
            });
            
            try {
                const response = await fetch(currentVehicleId ? 'api/update_vehicle.php' : 'api/add_vehicle.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showAlert('Fahrzeug erfolgreich gespeichert!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showAlert(result.message || 'Fehler beim Speichern', 'error');
                }
            } catch (error) {
                showAlert('Verbindungsfehler', 'error');
            }
        });
        
        // Fahrzeug löschen
        async function deleteVehicle(vehicleId) {
            if (!confirm('Möchten Sie dieses Fahrzeug wirklich löschen?')) {
                return;
            }
            
            try {
                const response = await fetch('api/delete_vehicle.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({ vehicle_id: vehicleId })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showAlert('Fahrzeug gelöscht', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showAlert(result.message || 'Fehler beim Löschen', 'error');
                }
            } catch (error) {
                showAlert('Verbindungsfehler', 'error');
            }
        }
        
        // Alert anzeigen
        function showAlert(message, type = 'error') {
            const alert = document.getElementById('alert-message');
            alert.textContent = message;
            alert.className = `alert alert-${type} show`;
            setTimeout(() => {
                alert.classList.remove('show');
            }, 5000);
        }
        
        // Notification Banner anzeigen (für E-Mail-Bestätigungen etc.)
        function showNotification(message, type = 'success') {
            const banner = document.getElementById('notification-banner');
            const icon = banner.querySelector('i');
            const text = document.getElementById('notification-text');
            
            // Icon setzen
            if (type === 'success') {
                icon.className = 'fas fa-check-circle';
                banner.className = 'notification-banner success show';
            } else {
                icon.className = 'fas fa-exclamation-circle';
                banner.className = 'notification-banner error show';
            }
            
            text.textContent = message;
            
            // Auto-hide nach 5 Sekunden
            setTimeout(() => {
                banner.classList.remove('show');
            }, 5000);
        }
        
        // Modal schließen bei Klick außerhalb
        document.getElementById('vehicle-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeVehicleModal();
            }
        });
        
        // Profil-Modal
        function openProfileModal() {
            document.getElementById('profile-modal').classList.add('active');
        }
        
        function closeProfileModal() {
            document.getElementById('profile-modal').classList.remove('active');
        }
        
        function previewImage(event) {
            const reader = new FileReader();
            reader.onload = function() {
                document.getElementById('profile-image-preview').src = reader.result;
            };
            reader.readAsDataURL(event.target.files[0]);
        }
        
        // Profil speichern
        document.getElementById('profile-edit-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            
            try {
                const response = await fetch('api/update_profile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Profil erfolgreich aktualisiert!', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Speichern', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        });
        
        // Modal schließen bei Klick außerhalb
        document.getElementById('profile-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeProfileModal();
            }
        });
        
        // Passwort-Modal
        function openPasswordModal() {
            document.getElementById('password-modal').style.display = 'block';
        }
        
        function closePasswordModal() {
            document.getElementById('password-modal').style.display = 'none';
            document.getElementById('password-change-form').reset();
        }
        
        // Passwort-Änderungsformular
        document.getElementById('password-change-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const currentPassword = document.getElementById('current-password').value;
            const newPassword = document.getElementById('new-password').value;
            const confirmPassword = document.getElementById('confirm-password').value;
            
            // Validierung
            if (newPassword !== confirmPassword) {
                showNotification('Die Passwörter stimmen nicht überein', 'error');
                return;
            }
            
            if (newPassword.length < 6) {
                showNotification('Das Passwort muss mindestens 6 Zeichen lang sein', 'error');
                return;
            }
            
            try {
                const response = await fetch('api/change_password.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        current_password: currentPassword,
                        new_password: newPassword
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Passwort erfolgreich geändert!', 'success');
                    setTimeout(() => {
                        closePasswordModal();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Ändern des Passworts', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        });
        
        // Modal schließen bei Klick außerhalb
        document.getElementById('password-modal').addEventListener('click', function(e) {
            if (e.target === this) {
                closePasswordModal();
            }
        });
        
        // Profile Tabs
        function showProfileTab(tab) {
            const tabs = document.querySelectorAll('.tab-btn[data-tab]');
            const contents = document.querySelectorAll('.tab-content');
            
            // Alle Tabs deaktivieren
            tabs.forEach(btn => {
                if (btn.dataset.tab === tab) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            });
            
            // Alle Inhalte ausblenden
            contents.forEach(content => {
                if (content.id === tab + '-tab') {
                    content.classList.add('active');
                } else {
                    content.classList.remove('active');
                }
            });
        }
        
        // Appointment Cancellation
        async function cancelAppointment(appointmentId) {
            if (!confirm('Möchten Sie diesen Termin wirklich stornieren?')) {
                return;
            }
            
            try {
                const response = await fetch('api/cancel_appointment.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        appointment_id: appointmentId
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Termin erfolgreich storniert', 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(result.message || 'Fehler beim Stornieren', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        }
        
        // Appointment Tabs
        function showAppointmentTab(tab) {
            const cards = document.querySelectorAll('.appointment-card');
            const buttons = document.querySelectorAll('.tab-btn');
            
            // Buttons aktivieren/deaktivieren
            buttons.forEach(btn => {
                if (btn.dataset.tab === tab) {
                    btn.classList.add('active');
                } else {
                    btn.classList.remove('active');
                }
            });
            
            // Cards filtern
            cards.forEach(card => {
                const isPast = card.classList.contains('past');
                const cardDate = card.dataset.date;
                const currentTime = new Date().getTime();
                const appointmentTime = new Date(cardDate + 'T00:00:00').getTime();
                
                switch(tab) {
                    case 'all':
                        card.style.display = 'flex';
                        break;
                    case 'upcoming':
                        card.style.display = appointmentTime >= currentTime ? 'flex' : 'none';
                        break;
                    case 'past':
                        card.style.display = appointmentTime < currentTime ? 'flex' : 'none';
                        break;
                }
            });
        }
    </script>
</body>
</html>
