<?php
require_once 'config/database.php';

// Einlagerungsnummer aus URL-Parameter holen
$storageNumber = isset($_GET['lgs_id']) ? $_GET['lgs_id'] : '';
$error = '';
$data = null;

if (empty($storageNumber)) {
    $error = 'Keine Einlagerungsnummer angegeben';
} else {
    try {
        // API-Klasse einbinden und Daten abrufen
        require_once 'api/check_tire_status.php';
        
        $api = new ResyTireStatusAPI();
        $result = $api->checkTireStatus($storageNumber, '', '');
        
        if ($result && $result['success']) {
            $data = $result['data'];
        } else {
            $error = isset($result['error']) ? $result['error'] : 'Unbekannter Fehler beim Laden der Daten';
        }
    } catch (Exception $e) {
        $error = 'Fehler beim Laden der Daten: ' . $e->getMessage();
    }
}

// Resy-Detailseite URL für direkten Link
$resyDetailUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=edit&LGS_ID=' . str_replace('.', '', $storageNumber) . '&FN=WheelsetResult';
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Detailansicht - Einlagerungsnummer <?php echo htmlspecialchars($storageNumber); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            font-weight: 300;
        }
        
        .header .subtitle {
            font-size: 1.2em;
            opacity: 0.9;
        }
        
        .content {
            padding: 40px;
        }
        
        .error {
            background: #e74c3c;
            color: white;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            font-size: 1.1em;
        }
        
        .resy-link {
            background: #3498db;
            color: white;
            padding: 15px 30px;
            text-decoration: none;
            border-radius: 8px;
            display: inline-block;
            margin: 20px 0;
            font-weight: bold;
            transition: background 0.3s;
        }
        
        .resy-link:hover {
            background: #2980b9;
        }
        
        .data-section {
            margin: 30px 0;
            padding: 25px;
            background: #f8f9fa;
            border-radius: 10px;
            border-left: 5px solid #3498db;
        }
        
        .data-section h3 {
            color: #2c3e50;
            margin-bottom: 20px;
            font-size: 1.4em;
        }
        
        .data-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .data-item {
            background: white;
            padding: 15px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .data-item strong {
            color: #2c3e50;
            display: block;
            margin-bottom: 5px;
        }
        
        .tire-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .tire-card {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            border-top: 4px solid #e74c3c;
            position: relative;
        }
        
        /* Reifenstatus-Farbkodierung für Detail-Seite */
        .tire-card.tire-good {
            border-top: 4px solid #28a745;
            background: linear-gradient(135deg, #ffffff 0%, #f8fff8 100%);
        }
        
        .tire-card.tire-warning {
            border-top: 4px solid #ffc107;
            background: linear-gradient(135deg, #ffffff 0%, #fffbf0 100%);
        }
        
        .tire-card.tire-bad {
            border-top: 4px solid #dc3545;
            background: linear-gradient(135deg, #ffffff 0%, #fff5f5 100%);
        }
        
        .tire-status-badge {
            position: absolute;
            top: 15px;
            right: 15px;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            z-index: 5;
        }
        
        .tire-status-badge.status-success {
            background: #28a745;
            color: white;
        }
        
        .tire-status-badge.status-warning {
            background: #ffc107;
            color: #212529;
        }
        
        .tire-status-badge.status-danger {
            background: #dc3545;
            color: white;
        }
        
        /* Reifenbilder-Styles für Detail-Seite */
        .tire-images {
            margin: 15px 0;
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            align-items: center;
        }
        
        .tire-image {
            width: 60px;
            height: 60px;
            object-fit: cover;
            border-radius: 8px;
            border: 2px solid #dee2e6;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .tire-image:hover {
            transform: scale(1.1);
            border-color: #007bff;
            box-shadow: 0 4px 12px rgba(0, 123, 255, 0.3);
        }
        
        .tire-more-images-btn {
            background: #6c757d;
            color: white;
            border: none;
            padding: 8px 12px;
            border-radius: 6px;
            font-size: 11px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .tire-more-images-btn:hover {
            background: #5a6268;
            transform: translateY(-1px);
        }
        
        /* Bild-Modal-Styles für Detail-Seite */
        .image-modal {
            display: none;
            position: fixed;
            z-index: 10000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(5px);
        }
        
        .image-modal-content {
            position: relative;
            margin: 5% auto;
            padding: 20px;
            width: 90%;
            max-width: 800px;
            background: #ffffff;
            border-radius: 15px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
        }
        
        .image-modal-content img {
            width: 100%;
            height: auto;
            border-radius: 10px;
            max-height: 70vh;
            object-fit: contain;
        }
        
        .image-modal-close {
            position: absolute;
            top: 15px;
            right: 20px;
            color: #000;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            z-index: 10001;
            background: rgba(255, 255, 255, 0.8);
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        
        .image-modal-close:hover {
            background: rgba(255, 0, 0, 0.8);
            color: white;
            transform: scale(1.1);
        }
        
        .image-modal-title {
            color: #212529;
            text-align: center;
            margin-top: 15px;
            font-size: 16px;
            font-weight: 600;
        }
        
        .all-images-modal {
            max-width: 1200px;
        }
        
        .all-images-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }
        
        .all-image-item {
            text-align: center;
        }
        
        .all-image-item img {
            width: 100%;
            height: 120px;
            object-fit: cover;
            border-radius: 8px;
            border: 2px solid #dee2e6;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .all-image-item img:hover {
            transform: scale(1.05);
            border-color: #007bff;
            box-shadow: 0 4px 12px rgba(0, 123, 255, 0.3);
        }
        
        .image-info {
            color: #6c757d;
            font-size: 12px;
            margin-top: 5px;
        }
        
        .tire-card h4 {
            color: #2c3e50;
            margin-bottom: 15px;
            text-align: center;
        }
        
        .tire-specs {
            font-size: 0.9em;
        }
        
        .tire-specs div {
            margin: 8px 0;
            padding: 5px 0;
            border-bottom: 1px solid #ecf0f1;
        }
        
        .tire-specs div:last-child {
            border-bottom: none;
        }
        
        .back-link {
            display: inline-block;
            background: #95a5a6;
            color: white;
            padding: 12px 25px;
            text-decoration: none;
            border-radius: 6px;
            margin-top: 20px;
            transition: background 0.3s;
        }
        
        .back-link:hover {
            background: #7f8c8d;
        }
        
        .images-section {
            margin: 30px 0;
        }
        
        .image-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin: 20px 0;
        }
        
        .image-item {
            text-align: center;
        }
        
        .image-item img {
            max-width: 100%;
            height: auto;
            border-radius: 8px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }
        
        .image-caption {
            margin-top: 10px;
            font-size: 0.9em;
            color: #7f8c8d;
        }
        
        @media (max-width: 768px) {
            .container {
                margin: 10px;
                border-radius: 10px;
            }
            
            .header {
                padding: 20px;
            }
            
            .header h1 {
                font-size: 2em;
            }
            
            .content {
                padding: 20px;
            }
            
            .data-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Detailansicht</h1>
            <div class="subtitle">Einlagerungsnummer: <?php echo htmlspecialchars($storageNumber); ?></div>
        </div>
        
        <div class="content">
            <?php if ($error): ?>
                <div class="error">
                    <strong>Fehler:</strong> <?php echo htmlspecialchars($error); ?>
                </div>
                <a href="javascript:history.back()" class="back-link">← Zurück</a>
            <?php else: ?>
                <!-- Resy-Detailseite Link -->
                <div style="text-align: center; margin-bottom: 30px;">
                    <a href="<?php echo htmlspecialchars($resyDetailUrl); ?>" target="_blank" class="resy-link">
                        🔗 Zur Resy-Detailseite (<?php echo htmlspecialchars($storageNumber); ?>)
                    </a>
                </div>
                
                <!-- Einlagerungsinformationen -->
                <div class="data-section">
                    <h3>📦 Einlagerungsinformationen</h3>
                    <div class="data-grid">
                        <div class="data-item">
                            <strong>Einlagerungsnummer:</strong>
                            <?php echo htmlspecialchars($data['einlagerungsnummer'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>Satznummer:</strong>
                            <?php echo htmlspecialchars($data['satznummer'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>Eingangsdatum:</strong>
                            <?php echo htmlspecialchars($data['eingangsdatum'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>Serviceart:</strong>
                            <?php echo htmlspecialchars($data['serviceart'] ?? 'N/A'); ?>
                        </div>
                    </div>
                </div>
                
                <!-- Fahrzeugdaten -->
                <div class="data-section">
                    <h3>🚗 Fahrzeugdaten</h3>
                    <div class="data-grid">
                        <div class="data-item">
                            <strong>Kennzeichen:</strong>
                            <?php echo htmlspecialchars($data['kennzeichen'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>FIN:</strong>
                            <?php echo htmlspecialchars($data['fin'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>Haltername:</strong>
                            <?php echo htmlspecialchars($data['haltername'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>Fahrzeugtyp:</strong>
                            <?php echo htmlspecialchars($data['fahrzeugtyp'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>Baujahr:</strong>
                            <?php echo htmlspecialchars($data['baujahr'] ?? 'N/A'); ?>
                        </div>
                        <div class="data-item">
                            <strong>KM-Stand:</strong>
                            <?php echo htmlspecialchars($data['km_stand'] ?? 'N/A'); ?>
                        </div>
                    </div>
                </div>
                
                <!-- Reifendaten -->
                <div class="data-section">
                    <h3>🛞 Reifendaten</h3>
                    <div class="tire-grid">
                        <?php 
                        $tirePositions = ['vl' => 'Vorderlinks', 'vr' => 'Vorderrechts', 'hl' => 'Hinterlinks', 'hr' => 'Hinterrechts'];
                        foreach ($tirePositions as $pos => $label): 
                            $tire = $data['reifen_' . $pos] ?? null;
                            if ($tire && !empty($tire)):
                        ?>
                        <?php
                        // Bestimme Reifenstatus
                        $tireStatusClass = 'tire-good';
                        $tireStatusText = 'Keine Fehler';
                        
                        // Prüfe Empfehlungstext für "Austausch notwendig"
                        if (!empty($tire['empfehlung']) && stripos($tire['empfehlung'], 'austausch notwendig') !== false) {
                            $tireStatusClass = 'tire-bad';
                            $tireStatusText = 'Austausch notwendig';
                        } elseif (isset($tire['profiltiefe']) && is_array($tire['profiltiefe'])) {
                            // Berechne Durchschnittsprofiltiefe
                            $inner = floatval($tire['profiltiefe']['i'] ?? 0);
                            $middle = floatval($tire['profiltiefe']['m'] ?? 0);
                            $outer = floatval($tire['profiltiefe']['a'] ?? 0);
                            
                            if ($inner > 0 && $middle > 0 && $outer > 0) {
                                $avgDepth = ($inner + $middle + $outer) / 3;
                                if ($avgDepth < 2) {
                                    $tireStatusClass = 'tire-bad';
                                    $tireStatusText = 'Kritisch (<2mm)';
                                } elseif ($avgDepth < 4) {
                                    $tireStatusClass = 'tire-warning';
                                    $tireStatusText = 'Befriedigend (2-4mm)';
                                } else {
                                    $tireStatusClass = 'tire-good';
                                    $tireStatusText = 'Sehr gut (4+mm)';
                                }
                            }
                        }
                        ?>
                        <div class="tire-card <?php echo $tireStatusClass; ?>">
                            <div class="tire-status-badge <?php echo str_replace('tire-', 'status-', $tireStatusClass); ?>"><?php echo $tireStatusText; ?></div>
                            <h4><?php echo $label; ?></h4>
                            
                            <?php if (isset($tire['images']) && is_array($tire['images']) && !empty($tire['images'])): ?>
                            <div class="tire-images">
                                <?php 
                                $maxImages = min(3, count($tire['images']));
                                for ($i = 0; $i < $maxImages; $i++): 
                                    $image = $tire['images'][$i];
                                ?>
                                <img src="<?php echo htmlspecialchars($image['proxy_url']); ?>" 
                                     alt="Reifenbild <?php echo $label; ?> <?php echo $i + 1; ?>" 
                                     class="tire-image"
                                     title="Reifenbild <?php echo $label; ?> - LGFOTO_ID: <?php echo htmlspecialchars($image['lgfoto_id']); ?>"
                                     onclick="openImageModal('<?php echo htmlspecialchars($image['proxy_url']); ?>', 'Reifenbild <?php echo $label; ?> - LGFOTO_ID: <?php echo htmlspecialchars($image['lgfoto_id']); ?>')">
                                <?php endfor; ?>
                                
                                <?php if (count($tire['images']) > 3): ?>
                                <button class="tire-more-images-btn" onclick="showAllImages('<?php echo $label; ?>', <?php echo htmlspecialchars(json_encode($tire['images'])); ?>)">
                                    +<?php echo count($tire['images']) - 3; ?> weitere
                                </button>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>
                            <div class="tire-specs">
                                <div><strong>Marke:</strong> <?php echo htmlspecialchars($tire['marke'] ?? 'N/A'); ?></div>
                                <div><strong>Modell:</strong> <?php echo htmlspecialchars($tire['modell'] ?? 'N/A'); ?></div>
                                <div><strong>Größe:</strong> <?php echo htmlspecialchars($tire['groesse'] ?? 'N/A'); ?></div>
                                <div><strong>Art:</strong> <?php echo htmlspecialchars($tire['art'] ?? 'N/A'); ?></div>
                                <div><strong>DOT:</strong> <?php echo htmlspecialchars($tire['dot'] ?? 'N/A'); ?></div>
                                <div><strong>Luftdruck:</strong> <?php echo htmlspecialchars($tire['luftdruck'] ?? 'N/A'); ?></div>
                                <div><strong>Zustand:</strong> <?php echo htmlspecialchars($tire['zustand'] ?? 'N/A'); ?></div>
                                <?php if (isset($tire['profiltiefe']) && is_array($tire['profiltiefe'])): ?>
                                <div><strong>Profiltiefe:</strong> 
                                    I: <?php echo htmlspecialchars($tire['profiltiefe']['i'] ?? 'N/A'); ?>mm | 
                                    M: <?php echo htmlspecialchars($tire['profiltiefe']['m'] ?? 'N/A'); ?>mm | 
                                    A: <?php echo htmlspecialchars($tire['profiltiefe']['a'] ?? 'N/A'); ?>mm
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($tire['empfehlung'])): ?>
                                <div><strong>Empfehlung:</strong> <?php echo htmlspecialchars($tire['empfehlung']); ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php 
                            endif;
                        endforeach; 
                        ?>
                    </div>
                </div>
                
                <!-- Bilder -->
                <?php if (!empty($data['images']) && is_array($data['images'])): ?>
                <div class="data-section">
                    <h3>📸 Bilder</h3>
                    <div class="image-grid">
                        <?php foreach ($data['images'] as $image): ?>
                        <div class="image-item">
                            <img src="<?php echo htmlspecialchars($image['url'] ?? ''); ?>" 
                                 alt="<?php echo htmlspecialchars($image['description'] ?? 'Reifenbild'); ?>">
                            <div class="image-caption"><?php echo htmlspecialchars($image['description'] ?? ''); ?></div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Bemerkungen -->
                <?php if (!empty($data['bemerkung'])): ?>
                <div class="data-section">
                    <h3>📝 Bemerkungen</h3>
                    <p><?php echo nl2br(htmlspecialchars($data['bemerkung'])); ?></p>
                </div>
                <?php endif; ?>
                
                <!-- Debug-Informationen (nur wenn vorhanden) -->
                <?php if (!empty($data['debug_info'])): ?>
                <div class="data-section" style="background: #f1f2f6; border-left-color: #95a5a6;">
                    <h3>🔧 Debug-Informationen</h3>
                    <div class="data-item">
                        <strong>HTML-Länge:</strong> <?php echo htmlspecialchars($data['debug_info']['html_length'] ?? 'N/A'); ?>
                    </div>
                    <div class="data-item">
                        <strong>Geprüft am:</strong> <?php echo htmlspecialchars($data['geprueft_am'] ?? 'N/A'); ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <div style="text-align: center; margin-top: 40px;">
                    <a href="javascript:history.back()" class="back-link">← Zurück</a>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        function openImageModal(imageUrl, title) {
            // Erstelle Modal für Bildanzeige
            let modal = document.getElementById('imageModal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'imageModal';
                modal.className = 'image-modal';
                modal.innerHTML = `
                    <div class="image-modal-content">
                        <span class="image-modal-close">&times;</span>
                        <img id="modalImage" src="" alt="">
                        <div class="image-modal-title"></div>
                    </div>
                `;
                document.body.appendChild(modal);
                
                // Event Listeners
                modal.querySelector('.image-modal-close').onclick = closeImageModal;
                modal.onclick = function(e) {
                    if (e.target === modal) closeImageModal();
                };
            }
            
            // Setze Bild und Titel
            modal.querySelector('#modalImage').src = imageUrl;
            modal.querySelector('.image-modal-title').textContent = title;
            
            // Zeige Modal
            modal.style.display = 'block';
        }
        
        function closeImageModal() {
            const modal = document.getElementById('imageModal');
            if (modal) {
                modal.style.display = 'none';
            }
        }
        
        function showAllImages(position, images) {
            // Erstelle Modal für alle Bilder einer Position
            let modal = document.getElementById('allImagesModal');
            if (!modal) {
                modal = document.createElement('div');
                modal.id = 'allImagesModal';
                modal.className = 'image-modal';
                modal.innerHTML = `
                    <div class="image-modal-content all-images-modal">
                        <span class="image-modal-close">&times;</span>
                        <div class="image-modal-title">Alle Reifenbilder - ${position}</div>
                        <div class="all-images-grid"></div>
                    </div>
                `;
                document.body.appendChild(modal);
                
                // Event Listeners
                modal.querySelector('.image-modal-close').onclick = closeAllImagesModal;
                modal.onclick = function(e) {
                    if (e.target === modal) closeAllImagesModal();
                };
            }
            
            // Leere und fülle Grid
            const grid = modal.querySelector('.all-images-grid');
            grid.innerHTML = '';
            
            images.forEach((image, index) => {
                const imgContainer = document.createElement('div');
                imgContainer.className = 'all-image-item';
                imgContainer.innerHTML = `
                    <img src="${image.proxy_url}" alt="Reifenbild ${position} ${index + 1}" 
                         onclick="openImageModal('${image.proxy_url}', 'Reifenbild ${position} - LGFOTO_ID: ${image.lgfoto_id}')">
                    <div class="image-info">LGFOTO_ID: ${image.lgfoto_id}</div>
                `;
                grid.appendChild(imgContainer);
            });
            
            // Zeige Modal
            modal.style.display = 'block';
        }
        
        function closeAllImagesModal() {
            const modal = document.getElementById('allImagesModal');
            if (modal) {
                modal.style.display = 'none';
            }
        }
    </script>
</body>
</html>




