-- Multi-Autohaus System für Boxxenstopp
-- Erweitert das bestehende System um Multi-Autohaus-Funktionalität

USE boxxenstopp;

-- 1. Autohaus-Tabelle erstellen
CREATE TABLE IF NOT EXISTS autohaus (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    short_name VARCHAR(50) NOT NULL,
    address TEXT NOT NULL,
    phone VARCHAR(20) NOT NULL,
    email VARCHAR(100) NOT NULL,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- RESY-System Credentials
    resy_username VARCHAR(100) NOT NULL,
    resy_password VARCHAR(255) NOT NULL,
    resy_dealer_id VARCHAR(100) NOT NULL,
    resy_base_url VARCHAR(255) DEFAULT 'https://resy.dtm-reifen.de',
    
    -- Autohaus-spezifische Einstellungen
    timezone VARCHAR(50) DEFAULT 'Europe/Berlin',
    working_days JSON DEFAULT '["monday", "tuesday", "wednesday", "thursday", "friday"]',
    working_hours JSON DEFAULT '{"start": "08:00", "end": "18:00"}',
    
    UNIQUE KEY unique_short_name (short_name),
    UNIQUE KEY unique_resy_dealer_id (resy_dealer_id)
);

-- 2. Events-Tabelle um Autohaus-ID erweitern
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'events' 
     AND COLUMN_NAME = 'autohaus_id') = 0,
    'ALTER TABLE events ADD COLUMN autohaus_id INT NULL AFTER id',
    'SELECT "autohaus_id column already exists in events"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 3. Appointments-Tabelle um Autohaus-ID erweitern
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'appointments' 
     AND COLUMN_NAME = 'autohaus_id') = 0,
    'ALTER TABLE appointments ADD COLUMN autohaus_id INT NULL AFTER id',
    'SELECT "autohaus_id column already exists in appointments"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 4. Admin-Users-Tabelle um Autohaus-Berechtigungen erweitern
-- Prüfe ob Spalten bereits existieren
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'admin_users' 
     AND COLUMN_NAME = 'role') = 0,
    'ALTER TABLE admin_users ADD COLUMN role ENUM(''admin'', ''manager'', ''employee'') DEFAULT ''employee'' AFTER is_active',
    'SELECT "role column already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'admin_users' 
     AND COLUMN_NAME = 'autohaus_id') = 0,
    'ALTER TABLE admin_users ADD COLUMN autohaus_id INT NULL AFTER role',
    'SELECT "autohaus_id column already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'admin_users' 
     AND COLUMN_NAME = 'can_switch_autohaus') = 0,
    'ALTER TABLE admin_users ADD COLUMN can_switch_autohaus BOOLEAN DEFAULT FALSE AFTER autohaus_id',
    'SELECT "can_switch_autohaus column already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Foreign Key hinzufügen (nur wenn autohaus_id existiert)
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'admin_users' 
     AND COLUMN_NAME = 'autohaus_id' 
     AND REFERENCED_TABLE_NAME = 'autohaus') = 0,
    'ALTER TABLE admin_users ADD FOREIGN KEY (autohaus_id) REFERENCES autohaus(id) ON DELETE SET NULL',
    'SELECT "Foreign key already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 5. Storage-Tabelle um Autohaus-ID erweitern
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'storage' 
     AND COLUMN_NAME = 'autohaus_id') = 0,
    'ALTER TABLE storage ADD COLUMN autohaus_id INT NULL AFTER id',
    'SELECT "autohaus_id column already exists in storage"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 6. Foreign Keys hinzufügen (nach Erstellung der Autohaus-Tabelle)
-- Events Foreign Key
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'events' 
     AND COLUMN_NAME = 'autohaus_id' 
     AND REFERENCED_TABLE_NAME = 'autohaus') = 0,
    'ALTER TABLE events ADD FOREIGN KEY (autohaus_id) REFERENCES autohaus(id) ON DELETE CASCADE',
    'SELECT "Events foreign key already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Appointments Foreign Key
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'appointments' 
     AND COLUMN_NAME = 'autohaus_id' 
     AND REFERENCED_TABLE_NAME = 'autohaus') = 0,
    'ALTER TABLE appointments ADD FOREIGN KEY (autohaus_id) REFERENCES autohaus(id) ON DELETE CASCADE',
    'SELECT "Appointments foreign key already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- Storage Foreign Key
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'storage' 
     AND COLUMN_NAME = 'autohaus_id' 
     AND REFERENCED_TABLE_NAME = 'autohaus') = 0,
    'ALTER TABLE storage ADD FOREIGN KEY (autohaus_id) REFERENCES autohaus(id) ON DELETE CASCADE',
    'SELECT "Storage foreign key already exists"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 7. Standard-Autohäuser einfügen
INSERT INTO autohaus (name, short_name, address, phone, email, resy_username, resy_password, resy_dealer_id) VALUES
('B & E Volvo Kiel', 'Kiel', 'Musterstraße 1, 24103 Kiel', '+49 431 123456', 'kiel@volvo-autohaus.de', 'D.ULUDAG', 'Ikizler123!', '113 B & E / Kiel'),
('B & E Volvo Norderstedt', 'Norderstedt', 'Hauptstraße 15, 22846 Norderstedt', '+49 40 987654', 'norderstedt@volvo-autohaus.de', 'D.ULUDAG', 'Ikizler123!', '112 B & E / Norderstedt'),
('B & E Volvo Glinde', 'Glinde', 'Gewerbestraße 8, 21509 Glinde', '+49 40 555666', 'glinde@volvo-autohaus.de', 'D.ULUDAG', 'Ikizler123!', '114 B & E / Glinde');

-- 8. RESY Kunden-IDs für jedes Autohaus konfigurieren
INSERT INTO settings (setting_key, setting_value, description) VALUES
('resy_kd_id_1', '113', 'RESY Kunden-ID für B & E Volvo Kiel'),
('resy_kd_id_2', '112', 'RESY Kunden-ID für B & E Volvo Norderstedt'),
('resy_kd_id_3', '114', 'RESY Kunden-ID für B & E Volvo Glinde'),
('resy_username_1', 'D.ULUDAG', 'RESY Benutzername für B & E Volvo Kiel'),
('resy_username_2', 'D.ULUDAG', 'RESY Benutzername für B & E Volvo Norderstedt'),
('resy_username_3', 'D.ULUDAG', 'RESY Benutzername für B & E Volvo Glinde'),
('resy_password_1', 'Ikizler123!', 'RESY Passwort für B & E Volvo Kiel'),
('resy_password_2', 'Ikizler123!', 'RESY Passwort für B & E Volvo Norderstedt'),
('resy_password_3', 'Ikizler123!', 'RESY Passwort für B & E Volvo Glinde')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);

-- 9. Admin-User als Super-Admin setzen
UPDATE admin_users 
SET role = 'admin', can_switch_autohaus = TRUE 
WHERE username = 'admin';

-- 10. Indizes für bessere Performance
CREATE INDEX IF NOT EXISTS idx_events_autohaus ON events(autohaus_id);
CREATE INDEX IF NOT EXISTS idx_appointments_autohaus ON appointments(autohaus_id);
CREATE INDEX IF NOT EXISTS idx_storage_autohaus ON storage(autohaus_id);
CREATE INDEX IF NOT EXISTS idx_admin_users_autohaus ON admin_users(autohaus_id);

-- 11. Session-Tabelle für Autohaus-Auswahl erweitern
CREATE TABLE IF NOT EXISTS user_sessions (
    id VARCHAR(128) PRIMARY KEY,
    data TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- 12. Einstellungen für Multi-Autohaus-System
INSERT INTO settings (setting_key, setting_value, description) VALUES
('multi_autohaus_enabled', '1', 'Multi-Autohaus-System aktiviert'),
('default_autohaus_id', '1', 'Standard-Autohaus ID'),
('autohaus_selection_required', '1', 'Autohaus-Auswahl für Kunden erforderlich'),
('admin_can_switch_autohaus', '1', 'Admins können zwischen Autohäusern wechseln')
ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value);

-- 13. Tabelle für Reifenprüfungen
CREATE TABLE IF NOT EXISTS tire_checks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    autohaus_id INT NOT NULL,
    einlagerungsnummer VARCHAR(100) NOT NULL,
    status ENUM('checking', 'completed', 'error') DEFAULT 'checking',
    data JSON NULL,
    error TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (autohaus_id) REFERENCES autohaus(id) ON DELETE CASCADE,
    UNIQUE KEY unique_autohaus_einlagerung (autohaus_id, einlagerungsnummer),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at)
);

-- 14. Appointments-Tabelle um Einlagerungsnummer erweitern
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'appointments' 
     AND COLUMN_NAME = 'einlagerungsnummer') = 0,
    'ALTER TABLE appointments ADD COLUMN einlagerungsnummer VARCHAR(100) NULL AFTER autohaus_id',
    'SELECT "einlagerungsnummer column already exists in appointments"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 15. Appointments-Tabelle um Reifenstatus erweitern
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'appointments' 
     AND COLUMN_NAME = 'tire_status') = 0,
    'ALTER TABLE appointments ADD COLUMN tire_status JSON NULL AFTER einlagerungsnummer',
    'SELECT "tire_status column already exists in appointments"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;

-- 16. Appointments-Tabelle um Reifenempfehlung erweitern
SET @sql = (SELECT IF(
    (SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
     WHERE TABLE_SCHEMA = 'boxxenstopp' 
     AND TABLE_NAME = 'appointments' 
     AND COLUMN_NAME = 'tire_recommendation') = 0,
    'ALTER TABLE appointments ADD COLUMN tire_recommendation VARCHAR(255) NULL AFTER tire_status',
    'SELECT "tire_recommendation column already exists in appointments"'
));
PREPARE stmt FROM @sql;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;
