<?php
session_start();
require_once 'config/database.php';

// Get token from URL
$token = $_GET['token'] ?? '';

if (empty($token)) {
    die('Ungültiger Token');
}

// Validate token and get appointment ID
$appointmentId = null;
$tokenValid = false;
$tokenExpired = false;
$tokenUsed = false;

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    // Validate token
    $stmt = $conn->prepare("
        SELECT appointment_id, expires_at, used_at 
        FROM offer_tokens 
        WHERE token = ? AND expires_at > NOW()
    ");
    $stmt->execute([$token]);
    $tokenData = $stmt->fetch();
    
    if (!$tokenData) {
        // Check if token exists but is expired
        $stmt = $conn->prepare("
            SELECT appointment_id, expires_at, used_at 
            FROM offer_tokens 
            WHERE token = ?
        ");
        $stmt->execute([$token]);
        $expiredToken = $stmt->fetch();
        
        if ($expiredToken) {
            if ($expiredToken['used_at']) {
                $tokenUsed = true;
            } else {
                $tokenExpired = true;
            }
        }
        
        // Show appropriate message
        if ($tokenUsed) {
            die('
            <!DOCTYPE html>
            <html lang="de">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Angebot bereits angenommen</title>
                <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
                <style>
                    body { font-family: Inter, sans-serif; background: #f8f9fa; margin: 0; padding: 20px; }
                    .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; padding: 40px; text-align: center; box-shadow: 0 4px 20px rgba(0,0,0,0.08); }
                    .icon { font-size: 4rem; color: #28a745; margin-bottom: 20px; }
                    h1 { color: #2c3e50; margin-bottom: 20px; }
                    p { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
                    .contact { background: #e3f2fd; padding: 20px; border-radius: 8px; margin-top: 30px; }
                </style>
            </head>
            <body>
                <div class="container">
                    <div class="icon"><i class="fas fa-check-circle"></i></div>
                    <h1>Angebot bereits angenommen</h1>
                    <p>Sie haben das Angebot bereits angenommen. Falls Sie sich umentschieden haben, schreiben Sie uns gerne!</p>
                    <div class="contact">
                        <p><strong>Kontakt:</strong><br>
                        E-Mail: info@volvo-autohaus.de<br>
                        Telefon: +49 (0) 123 456 789</p>
                    </div>
                </div>
            </body>
            </html>');
        } else {
            die('
            <!DOCTYPE html>
            <html lang="de">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Token abgelaufen</title>
                <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
                <style>
                    body { font-family: Inter, sans-serif; background: #f8f9fa; margin: 0; padding: 20px; }
                    .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 16px; padding: 40px; text-align: center; box-shadow: 0 4px 20px rgba(0,0,0,0.08); }
                    .icon { font-size: 4rem; color: #dc3545; margin-bottom: 20px; }
                    h1 { color: #2c3e50; margin-bottom: 20px; }
                    p { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
                    .contact { background: #e3f2fd; padding: 20px; border-radius: 8px; margin-top: 30px; }
                </style>
            </head>
            <body>
                <div class="container">
                    <div class="icon"><i class="fas fa-clock"></i></div>
                    <h1>Token abgelaufen</h1>
                    <p>Der Link ist abgelaufen (24 Stunden gültig). Bitte kontaktieren Sie uns für ein neues Angebot.</p>
                    <div class="contact">
                        <p><strong>Kontakt:</strong><br>
                        E-Mail: info@volvo-autohaus.de<br>
                        Telefon: +49 (0) 123 456 789</p>
                    </div>
                </div>
            </body>
            </html>');
        }
    }
    
    $appointmentId = $tokenData['appointment_id'];
    $tokenValid = true;
    
    // Get appointment and offer data
    $stmt = $conn->prepare("
        SELECT 
            a.*,
            ah.name as autohaus_name,
            ah.address as autohaus_address,
            ah.phone as autohaus_phone,
            ah.email as autohaus_email
        FROM appointments a
        JOIN autohaus ah ON a.autohaus_id = ah.id
        WHERE a.id = ?
    ");
    
    $stmt->execute([$appointmentId]);
    $appointment = $stmt->fetch();
    
    if (!$appointment) {
        die('Termin nicht gefunden');
    }
    
    // Get offer data from tracking table
    $stmt = $conn->prepare("
        SELECT offer_data, created_at 
        FROM offer_tracking 
        WHERE appointment_id = ? AND offer_type = 'sent'
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    
    $stmt->execute([$appointmentId]);
    $offerData = $stmt->fetch();
    
    if (!$offerData) {
        die('Angebot nicht gefunden');
    }
    
    $offer = json_decode($offerData['offer_data'], true);
    $selectedTires = $offer['selectedTires'] ?? [];
    $newTires = $offer['newTires'] ?? [];
    $selectedServices = $offer['selectedServices'] ?? [];
    
    // Get services details
    $services = [];
    if (!empty($selectedServices)) {
        $placeholders = str_repeat('?,', count($selectedServices) - 1) . '?';
        $stmt = $conn->prepare("
            SELECT id, service_name, service_description, price 
            FROM autohaus_services 
            WHERE id IN ($placeholders) AND is_active = 1
        ");
        $stmt->execute($selectedServices);
        $services = $stmt->fetchAll();
    }
    
} catch (Exception $e) {
    die('Fehler: ' . $e->getMessage());
}

// Handle form submission
if ($_POST) {
    $customerTireSelection = $_POST['tire_selection'] ?? [];
    $customerServiceSelection = $_POST['service_selection'] ?? [];
    $customerNotes = $_POST['customer_notes'] ?? '';
    
    try {
        // Save customer selection
        $stmt = $conn->prepare("
            INSERT INTO customer_selections 
            (appointment_id, tire_selection, service_selection, customer_notes, created_at) 
            VALUES (?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $appointmentId,
            json_encode($customerTireSelection),
            json_encode($customerServiceSelection),
            $customerNotes
        ]);
        
        // Mark token as used
        $stmt = $conn->prepare("
            UPDATE offer_tokens 
            SET used_at = NOW() 
            WHERE token = ?
        ");
        $stmt->execute([$token]);
        
        $success = true;
        
    } catch (Exception $e) {
        $error = 'Fehler beim Speichern der Auswahl: ' . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kundenauswahl - Reifenangebot</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&family=Space+Grotesk:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f8f9fa;
            min-height: 100vh;
            line-height: 1.6;
        }
        
        /* Navigation */
        .navbar {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            position: sticky;
            top: 0;
            z-index: 1000;
        }
        
        .nav-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .nav-brand {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .logo-wrapper {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .nav-logo {
            width: 40px;
            height: 40px;
            object-fit: contain;
        }
        
        .brand-text h1 {
            font-size: 1.5rem;
            font-weight: 700;
            color: #2c3e50;
            margin: 0;
        }
        
        .brand-subtitle {
            font-size: 0.9rem;
            color: #6c757d;
            font-weight: 500;
        }
        
        .nav-menu {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 1rem;
            background: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 6px;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        
        .nav-link:hover {
            background: #0056b3;
            transform: translateY(-1px);
        }
        
        /* Hero Section */
        .hero {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 4rem 0;
            position: relative;
            overflow: hidden;
        }
        
        .hero-background {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.1"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.1"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.1"/><circle cx="10" cy="60" r="0.5" fill="white" opacity="0.1"/><circle cx="90" cy="40" r="0.5" fill="white" opacity="0.1"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            opacity: 0.3;
        }
        
        .hero-container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 2rem;
            position: relative;
            z-index: 1;
        }
        
        .hero-content {
            text-align: center;
        }
        
        .hero-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            background: rgba(255,255,255,0.2);
            padding: 0.5rem 1rem;
            border-radius: 25px;
            font-size: 0.9rem;
            font-weight: 500;
            margin-bottom: 1.5rem;
            backdrop-filter: blur(10px);
        }
        
        .hero-title {
            font-size: 3rem;
            font-weight: 800;
            margin-bottom: 1rem;
            line-height: 1.2;
        }
        
        .hero-description {
            font-size: 1.2rem;
            opacity: 0.95;
            margin-bottom: 2rem;
            max-width: 600px;
            margin-left: auto;
            margin-right: auto;
        }
        
        /* Main Content */
        .main-content {
            max-width: 900px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        .container {
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            overflow: hidden;
        }
        
        .content {
            padding: 2rem;
        }
        
        .section {
            margin-bottom: 2rem;
            padding: 1.5rem;
            border: 1px solid #e9ecef;
            border-radius: 12px;
            background: #f8f9fa;
        }
        
        .section h2 {
            color: #2c3e50;
            margin-bottom: 1rem;
            font-size: 1.25rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .section p {
            color: #6c757d;
            margin-bottom: 1rem;
            font-size: 0.95rem;
        }
        
        /* Selection Options */
        .tire-option, .service-option {
            display: flex;
            align-items: center;
            padding: 1.5rem;
            margin: 1rem 0;
            border: 2px solid #e9ecef;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            background: white;
            position: relative;
            overflow: hidden;
        }
        
        .tire-option::before, .service-option::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, #007bff, #0056b3);
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .tire-option:hover, .service-option:hover {
            border-color: #007bff;
            background-color: #f8f9ff;
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 123, 255, 0.15);
        }
        
        .tire-option:hover::before, .service-option:hover::before {
            opacity: 0.05;
        }
        
        .tire-option.selected, .service-option.selected {
            border-color: #28a745;
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(40, 167, 69, 0.2);
        }
        
        .tire-option input, .service-option input {
            margin-right: 1rem;
            transform: scale(1.2);
            position: relative;
            z-index: 1;
        }
        
        .tire-info, .service-info {
            flex: 1;
            position: relative;
            z-index: 1;
        }
        
        .tire-name, .service-name {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 0.5rem;
            font-size: 1rem;
        }
        
        .tire-details, .service-details {
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .price {
            font-weight: 700;
            color: #28a745;
            font-size: 1.1rem;
            position: relative;
            z-index: 1;
        }
        
        /* Form Elements */
        .notes {
            width: 100%;
            padding: 1rem;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 0.95rem;
            resize: vertical;
            min-height: 120px;
            font-family: inherit;
            transition: border-color 0.3s ease;
        }
        
        .notes:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
        }
        
        .submit-btn {
            background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
            color: white;
            padding: 1rem 2rem;
            border: none;
            border-radius: 8px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            width: 100%;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
        }
        
        .submit-btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }
        
        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0, 123, 255, 0.3);
        }
        
        .submit-btn:hover::before {
            left: 100%;
        }
        
        /* Status Messages */
        .success {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            color: #155724;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            border: 1px solid #c3e6cb;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .error {
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
            color: #721c24;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            border: 1px solid #f5c6cb;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .customer-info {
            background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
            padding: 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            border: 1px solid #bbdefb;
        }
        
        .customer-info strong {
            color: #1976d2;
            font-weight: 600;
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .nav-container {
                padding: 1rem;
            }
            
            .hero-title {
                font-size: 2rem;
            }
            
            .hero-description {
                font-size: 1rem;
            }
            
            .main-content {
                padding: 1rem;
            }
            
            .content {
                padding: 1rem;
            }
            
            .section {
                padding: 1rem;
            }
            
            .tire-option, .service-option {
                padding: 1rem;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="assets/images/volvo-logo.png" alt="Volvo" class="nav-logo">
                    <div class="brand-text">
                        <h1>Boxenstop</h1>
                        <span class="brand-subtitle">Volvo Service</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="index.php" class="nav-link">
                    <i class="fas fa-home"></i>
                    <span>Zurück zur Startseite</span>
                </a>
            </div>
        </div>
    </nav>

    <!-- Hero Section -->
    <section class="hero">
        <div class="hero-background"></div>
        <div class="hero-container">
            <div class="hero-content">
                <h1 class="hero-title">Kundenauswahl</h1>
                <p class="hero-description">
                    Wählen Sie Ihre gewünschten Reifen und Zusatzleistungen aus
                </p>
            </div>
        </div>
    </section>

    <!-- Main Content -->
    <div class="main-content">
        <div class="container">
            <div class="content">
            <?php if (isset($success)): ?>
                <div class="success" style="text-align: center; padding: 2rem;">
                    <i class="fas fa-check-circle" style="font-size: 3rem; color: #28a745; margin-bottom: 1rem;"></i>
                    <h2 style="color: #28a745; margin-bottom: 1rem;">✅ Auswahl erfolgreich übermittelt!</h2>
                    <p style="font-size: 1.1rem; margin-bottom: 1rem;">Vielen Dank für Ihre Auswahl. Sie erhalten eine Bestätigung per E-Mail.</p>
                    <p style="color: #6c757d; font-size: 0.9rem;">Das Angebot wurde angenommen und der Link ist nun ungültig.</p>
                </div>
                
                <script>
                    // Disable form after successful submission
                    document.addEventListener('DOMContentLoaded', function() {
                        const form = document.querySelector('form');
                        if (form) {
                            form.style.display = 'none';
                        }
                    });
                </script>
            <?php endif; ?>
            
            <?php if (isset($error)): ?>
                <div class="error">
                    <i class="fas fa-exclamation-circle"></i>
                    <div>
                        <strong>Fehler:</strong> <?= htmlspecialchars($error) ?>
                    </div>
                </div>
            <?php endif; ?>
            
            <div class="customer-info">
                <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                    <i class="fas fa-user" style="color: #1976d2; font-size: 1.2rem;"></i>
                    <div>
                        <strong>Kunde:</strong> <?= htmlspecialchars($appointment['customer_name']) ?>
                    </div>
                </div>
                <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                    <i class="fas fa-car" style="color: #1976d2; font-size: 1.2rem;"></i>
                    <div>
                        <strong>Kennzeichen:</strong> <?= htmlspecialchars($appointment['kennzeichen'] ?? $appointment['license_plate'] ?? 'Nicht angegeben') ?>
                    </div>
                </div>
                <div style="display: flex; align-items: center; gap: 1rem;">
                    <i class="fas fa-building" style="color: #1976d2; font-size: 1.2rem;"></i>
                    <div>
                        <strong>Autohaus:</strong> <?= htmlspecialchars($appointment['autohaus_name']) ?>
                    </div>
                </div>
            </div>
            
            <form method="POST">
                <!-- Selected Tires Section -->
                <?php if (!empty($selectedTires)): ?>
                <div class="section">
                    <h2><i class="fas fa-tools"></i> Ausgewählte Reifen (Austausch notwendig)</h2>
                    <p>Bitte wählen Sie aus, welche Reifen Sie tauschen möchten:</p>
                    
                    <?php foreach ($selectedTires as $index => $tire): ?>
                    <div class="tire-option" onclick="toggleSelection(this)">
                        <input type="checkbox" name="tire_selection[]" value="<?= $index ?>" id="tire_<?= $index ?>">
                        <div class="tire-info">
                            <div class="tire-name">
                                <?= htmlspecialchars($tire['brand'] ?? '') ?> <?= htmlspecialchars($tire['model'] ?? '') ?>
                            </div>
                            <div class="tire-details">
                                Position: <?= htmlspecialchars($tire['position'] ?? '') ?> | 
                                Größe: <?= htmlspecialchars($tire['size'] ?? '') ?> | 
                                Zustand: <?= htmlspecialchars($tire['recommendation'] ?? '') ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <!-- New Tires Section -->
                <?php if (!empty($newTires)): ?>
                <div class="section">
                    <h2><i class="fas fa-plus-circle"></i> Neue Reifen</h2>
                    <p>Diese Reifen werden als Ersatz angeboten:</p>
                    
                    <?php foreach ($newTires as $index => $tire): ?>
                    <div class="tire-option" onclick="toggleSelection(this)">
                        <input type="checkbox" name="tire_selection[]" value="new_<?= $index ?>" id="new_tire_<?= $index ?>">
                        <div class="tire-info">
                            <div class="tire-name">
                                <?= htmlspecialchars($tire['brand'] ?? '') ?> <?= htmlspecialchars($tire['model'] ?? '') ?>
                            </div>
                            <div class="tire-details">
                                Größe: <?= htmlspecialchars($tire['size'] ?? '') ?> | 
                                Anzahl: <?= htmlspecialchars($tire['quantity'] ?? '') ?> | 
                                Preis: <?= number_format($tire['price'] ?? 0, 2, ',', '.') ?> €
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <!-- Services Section -->
                <?php if (!empty($services)): ?>
                <div class="section">
                    <h2><i class="fas fa-cogs"></i> Zusatzleistungen</h2>
                    <p>Wählen Sie die gewünschten Zusatzleistungen aus:</p>
                    
                    <?php foreach ($services as $service): ?>
                    <div class="service-option" onclick="toggleSelection(this)">
                        <input type="checkbox" name="service_selection[]" value="<?= $service['id'] ?>" id="service_<?= $service['id'] ?>">
                        <div class="service-info">
                            <div class="service-name">
                                <?= htmlspecialchars($service['service_name']) ?>
                            </div>
                            <div class="service-details">
                                <?= htmlspecialchars($service['service_description']) ?>
                            </div>
                        </div>
                        <div class="price">
                            <?= number_format($service['price'], 2, ',', '.') ?> €
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <!-- Notes Section -->
                <div class="section">
                    <h2><i class="fas fa-comment-alt"></i> Anmerkungen</h2>
                    <p>Haben Sie besondere Wünsche oder Anmerkungen?</p>
                    <textarea name="customer_notes" class="notes" placeholder="Ihre Anmerkungen hier..."><?= htmlspecialchars($_POST['customer_notes'] ?? '') ?></textarea>
                </div>
                
                <button type="submit" class="submit-btn">
                    <i class="fas fa-check-circle"></i>
                    <span>Auswahl übermitteln</span>
                </button>
            </form>
            </div>
        </div>
    </div>
    
    <script>
        function toggleSelection(element) {
            const checkbox = element.querySelector('input[type="checkbox"]');
            checkbox.checked = !checkbox.checked;
            
            if (checkbox.checked) {
                element.classList.add('selected');
            } else {
                element.classList.remove('selected');
            }
        }
        
        // Initialize selected state
        document.addEventListener('DOMContentLoaded', function() {
            const checkboxes = document.querySelectorAll('input[type="checkbox"]');
            checkboxes.forEach(checkbox => {
                const option = checkbox.closest('.tire-option, .service-option');
                if (checkbox.checked) {
                    option.classList.add('selected');
                }
            });
        });
    </script>
</body>
</html>
