<?php
/**
 * Terminbuchungsseite
 */

session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    requireLoginRedirect();
    exit;
}

// Prüfe ob User verifiziert ist
$user = getCurrentUser();
if ($user) {
    if (!isset($user['verification_status']) || $user['verification_status'] !== 'approved') {
        header('Location: verification_pending.php');
        exit;
    }
}

$event_id = $_GET['event'] ?? null;

if (!$event_id) {
    header('Location: index.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Event-Details mit Autohaus-Information laden
$stmt = $conn->prepare("
    SELECT e.*, a.name as autohaus_name, a.short_name as autohaus_short_name, a.id as autohaus_id
    FROM events e 
    LEFT JOIN autohaus a ON e.autohaus_id = a.id 
    WHERE e.id = ? AND e.is_active = 1
");
$stmt->execute([$event_id]);
$event = $stmt->fetch();

// Services für das Autohaus laden
$services = [];
if ($event && $event['autohaus_id']) {
    $stmt = $conn->prepare("
        SELECT * FROM autohaus_services 
        WHERE autohaus_id = ? AND is_active = 1 
        ORDER BY sort_order, service_name
    ");
    $stmt->execute([$event['autohaus_id']]);
    $services = $stmt->fetchAll();
}

if (!$event) {
    header('Location: index.php');
    exit;
}

// User-Daten laden für automatisches Ausfüllen
$user = getCurrentUser();
$customer_name = '';
$customer_email = '';
$customer_phone = '';
$customer_license_plate = '';

// Fahrzeuge des Users laden
$user_vehicles = [];
$default_vehicle = null;

if ($user) {
    $customer_name = $user['first_name'] . ' ' . $user['last_name'];
    $customer_email = $user['email'] ?? '';
    $customer_phone = $user['phone_number'] ?? '';
    
    // Lade Fahrzeuge aus user_vehicles Tabelle
    $stmt = $conn->prepare("
        SELECT * FROM user_vehicles 
        WHERE user_id = ? AND is_active = 1 
        ORDER BY is_default DESC, created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $user_vehicles = $stmt->fetchAll();
    
    // Finde Standard-Fahrzeug oder nimm das erste
    foreach ($user_vehicles as $vehicle) {
        if ($vehicle['is_default']) {
            $default_vehicle = $vehicle;
            break;
        }
    }
    
    if (!$default_vehicle && !empty($user_vehicles)) {
        $default_vehicle = $user_vehicles[0];
    }
    
    // Wenn Fahrzeuge existieren, Standard-Fahrzeug verwenden, sonst das alte Kennzeichen
    if ($default_vehicle) {
        $customer_license_plate = $default_vehicle['license_plate'];
    } else {
        $customer_license_plate = $user['license_plate'] ?? '';
    }
}

// Erfolgs-/Fehlermeldungen werden jetzt per JavaScript angezeigt
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="googlebot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="bingbot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <title>Termin buchen - B&E Gruppe</title>
    <link rel="stylesheet" href="assets/css/style.css?v=<?php echo filemtime(__DIR__ . '/assets/css/style.css'); ?>">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* Einfache Input-Group Styles */
        .input-group {
            display: flex;
            gap: 0.5rem;
            align-items: stretch;
        }
        
        .input-group input {
            flex: 1;
        }
        
        .input-group button {
            white-space: nowrap;
            padding: 0.75rem 1rem;
        }
        
        @media (max-width: 768px) {
            .input-group {
                flex-direction: column;
            }
            
            /* Booking Form Mobile Optimierung */
            .booking-section {
                padding: 1rem;
            }
            
            .form-container {
                padding: 1.5rem;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .form-group label {
                font-size: 0.875rem;
            }
            
            .form-group input,
            .form-group select,
            .form-group textarea {
                padding: 0.875rem;
                font-size: 0.95rem;
            }
            
            /* Button Mobile Optimierung */
            .form-actions {
                flex-direction: column;
                gap: 1rem;
            }
            
            .form-actions button {
                width: 100%;
            }
            
            /* Time Slot Cards Mobile */
            .time-slots-grid {
                grid-template-columns: 1fr;
                gap: 0.75rem;
            }
            
            .slot-card {
                padding: 1rem;
            }
            
            .slot-card-header {
                flex-direction: column;
                gap: 0.5rem;
                align-items: flex-start;
            }
            
            .occupancy-display {
                padding: 0.5rem;
                font-size: 0.875rem;
            }
            
            /* Event Info Mobile */
            .event-banner {
                padding: 1rem;
            }
            
            .event-banner h2 {
                font-size: 1.25rem;
            }
            
            .event-banner p {
                font-size: 0.875rem;
            }
            
            .vehicle-selection {
                padding: 1rem;
            }
            
            .vehicle-card {
                padding: 1rem;
            }
            
            /* Autohaus Info Mobile */
            .autohaus-info-card {
                padding: 1rem;
            }
            
            .autohaus-info-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
        }
        
        @media (max-width: 480px) {
            .form-container {
                padding: 1rem;
            }
            
            .slot-card {
                padding: 0.875rem;
            }
            
            .occupancy-display {
                font-size: 0.75rem;
                padding: 0.375rem;
            }
            
            .form-actions button {
                font-size: 0.875rem;
                padding: 0.875rem 1rem;
            }
        }
        
        /* Verbessertes Navigation Design - Volvo Style */
        .nav-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 8px;
            text-decoration: none;
            font-weight: 500;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            border: 1px solid #e9ecef;
        }
        
        .nav-link i {
            font-size: 1rem;
        }
        
        .back-link {
            background: #6c757d;
            color: white;
            border-color: #6c757d;
        }
        
        .back-link:hover {
            background: #5a6268;
            border-color: #5a6268;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(108, 117, 125, 0.2);
        }
        
        .user-link {
            background: #003057;
            color: white;
            border-color: #003057;
        }
        
        .user-link:hover {
            background: #0056b3;
            border-color: #0056b3;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0, 48, 87, 0.2);
        }
        
        .logout-link {
            background: #dc3545;
            color: white;
            border-color: #dc3545;
        }
        
        .logout-link:hover {
            background: #c82333;
            border-color: #c82333;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(220, 53, 69, 0.2);
        }
        
        .admin-link {
            background: #fff;
            color: #495057;
            border-color: #dee2e6;
        }
        
        .admin-link:hover {
            background: #f8f9fa;
            border-color: #adb5bd;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        @media (max-width: 768px) {
            .nav-menu {
                flex-wrap: wrap;
                gap: 0.5rem;
            }
            
            .nav-link {
                padding: 0.625rem 1rem;
                font-size: 0.875rem;
            }
            
            .nav-link span {
                display: none;
            }
            
            .nav-link {
                width: 44px;
                height: 44px;
                justify-content: center;
                padding: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="assets/images/volvo-logo.png" alt="Volvo" class="nav-logo">
                    <div class="brand-text">
                        <h1>B&E Gruppe</h1>
                        <span class="brand-subtitle">B&E Gruppe - Volvo Service</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="index.php" class="nav-link back-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Zurück</span>
                </a>
                <a href="profile.php" class="nav-link user-link">
                    <i class="fas fa-user"></i>
                    <span>Profil</span>
                </a>
                <a href="api/logout.php" class="nav-link logout-link">
                    <i class="fas fa-sign-out-alt"></i>
                    <span>Abmelden</span>
                </a>
                <?php
                // Prüfe ob der eingeloggte User ein Admin ist
                $is_admin = false;
                if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] && isset($_SESSION['user_id'])) {
                    // Lade User-Daten
                    $stmt = $conn->prepare("SELECT id, email, admin_user_id FROM users WHERE id = ? AND is_active = 1");
                    $stmt->execute([$_SESSION['user_id']]);
                    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($user_data) {
                        // Prüfe ob User direkt mit Admin verknüpft ist
                        if (!empty($user_data['admin_user_id'])) {
                            // Prüfe ob verknüpfter Admin aktiv ist
                            $stmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ? AND is_active = 1");
                            $stmt->execute([$user_data['admin_user_id']]);
                            $admin_user = $stmt->fetch();
                            $is_admin = (bool)$admin_user;
                        } else {
                            // Fallback: Suche nach E-Mail in admin_users
                            $user_email = $user_data['email'] ?? '';
                            if (!empty($user_email)) {
                                $stmt = $conn->prepare("SELECT id FROM admin_users WHERE email = ? AND is_active = 1");
                                $stmt->execute([$user_email]);
                                $admin_user = $stmt->fetch();
                                $is_admin = (bool)$admin_user;
                            }
                        }
                    }
                }
                // Oder prüfe ob der User bereits als Admin eingeloggt ist
                if (!$is_admin && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
                    $is_admin = true;
                }
                
                if ($is_admin):
                ?>
                <a href="admin/" class="nav-link admin-link">
                    <i class="fas fa-cog"></i>
                    <span>Admin</span>
                </a>
                <?php endif; ?>
            </div>
        </div>
    </nav>

    <!-- Booking Section -->
    <section class="booking-section">
        <div class="section-container">
            <div class="booking-header">
                <h1 class="section-title">Termin buchen</h1>
                <p class="section-subtitle">Buchen Sie Ihren Reifenwechsel-Service</p>
                
                <div class="event-info-card">
                    <div class="event-info-header">
                        <h3><?php echo htmlspecialchars($event['name']); ?></h3>
                        <div class="event-info-badges">
                            <span class="info-badge">
                                <i class="fas fa-calendar"></i>
                                <?php echo date('d.m.Y', strtotime($event['event_date'])); ?>
                            </span>
                            <span class="info-badge">
                                <i class="fas fa-clock"></i>
                                <?php echo date('H:i', strtotime($event['start_time'])); ?> - <?php echo date('H:i', strtotime($event['end_time'])); ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Alerts werden per JavaScript angezeigt -->

            <div class="booking-layout">
                <div class="booking-main">
                    <form id="booking-form" method="POST" class="booking-form">
                    <input type="hidden" name="event_id" id="event_id" value="<?php echo (int)$event_id; ?>">
                    <input type="hidden" name="autohaus_id" id="autohaus_id" value="<?php echo $event['autohaus_id'] ?? ''; ?>">
                    <input type="hidden" name="event_name" id="event_name" value="<?php echo htmlspecialchars($event['name']); ?>">
                    <div class="form-grid">
                        <!-- Zeitslot Auswahl -->
                        <div class="form-group">
                            <label for="booking_date">Wählen Sie Ihr Datum</label>
                            <input type="date" id="booking_date" name="slot_date" required 
                                   min="<?php echo htmlspecialchars($event['event_date']); ?>" 
                                   max="<?php echo htmlspecialchars($event['event_date']); ?>" 
                                   value="<?php echo htmlspecialchars($event['event_date']); ?>">
                        </div>

                        <div class="form-group full-width">
                            <label>Auslastung</label>
                            <div id="occupancy-calendar" class="occupancy-calendar"></div>
                        </div>

                        <div class="form-group">
                            <label>Verfügbare Zeitslots</label>
                            <div id="time-slots" class="time-slots slot-grid">
                                <p class="loading">Lade verfügbare Zeitslots...</p>
                            </div>
                        </div>

                        <!-- Kundeninformationen -->
                        <?php if ($user): ?>
                        <div style="background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 8px; padding: 1rem; margin-bottom: 1.5rem;">
                            <i class="fas fa-info-circle" style="color: #0c5460; margin-right: 0.5rem;"></i>
                            <span style="color: #0c5460; font-weight: 500;">Ihre Daten wurden aus Ihrem Profil geladen</span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label for="customer_name">Ihr Name *</label>
                            <input type="text" id="customer_name" name="customer_name" required 
                                   placeholder="Vor- und Nachname"
                                   value="<?php echo htmlspecialchars($customer_name); ?>">
                        </div>

                        <?php if (!empty($user_vehicles)): ?>
                        <div class="form-group">
                            <label for="selected_vehicle">Fahrzeug auswählen *</label>
                            <select id="selected_vehicle" name="selected_vehicle" required onchange="updateLicensePlate()">
                                <?php foreach ($user_vehicles as $vehicle): ?>
                                    <option value="<?php echo $vehicle['id']; ?>" 
                                        data-license="<?php echo htmlspecialchars($vehicle['license_plate']); ?>"
                                        <?php echo $vehicle['is_default'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($vehicle['license_plate']); ?>
                                        <?php if ($vehicle['vehicle_brand']): ?>
                                            - <?php echo htmlspecialchars($vehicle['vehicle_brand']); ?>
                                        <?php endif; ?>
                                        <?php if ($vehicle['is_default']): ?>
                                            (Standard)
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">
                                Oder <a href="profile.php" style="color: #0056b3; text-decoration: underline;">Fahrzeug hinzufügen</a>
                            </small>
                        </div>
                        
                        <input type="hidden" id="license_plate" name="license_plate" value="<?php echo htmlspecialchars($customer_license_plate); ?>">
                        <?php else: ?>
                        <div class="form-group">
                            <label for="license_plate">Ihr Kennzeichen *</label>
                            <input type="text" id="license_plate" name="license_plate" required 
                                   placeholder="z. B. M-AB 123"
                                   value="<?php echo htmlspecialchars($customer_license_plate); ?>">
                            <small class="text-muted">
                                <a href="profile.php" style="color: #0056b3; text-decoration: underline;">Mehrere Fahrzeuge verwalten</a>
                            </small>
                        </div>
                        <?php endif; ?>

                        <div class="form-group">
                            <label for="phone">Ihre Telefonnummer *</label>
                            <input type="tel" id="phone" name="phone" required 
                                   placeholder="z. B. +49 123 456 789"
                                   value="<?php echo htmlspecialchars($customer_phone); ?>">
                            <small class="text-muted">Für WhatsApp-Benachrichtigungen und Bestätigungen</small>
                        </div>

                        <div class="form-group">
                            <label for="email">Ihre E-Mail-Adresse *</label>
                            <input type="email" id="email" name="email" required 
                                   placeholder="z. B. kunde@beispiel.de"
                                   value="<?php echo htmlspecialchars($customer_email); ?>">
                            <small class="text-muted">Für Bestätigungen und wichtige Informationen</small>
                        </div>

                        <!-- Reifen-Informationen -->
                        <div class="form-group">
                            <h4 class="question-title">Bringen Sie die Räder mit?</h4>
                            <div class="radio-group">
                                <label class="radio-label">
                                    <input type="radio" name="brings_wheels" value="1" checked>
                                    <span class="radio-text">Ja, ich bringe die Räder mit</span>
                                </label>
                                <label class="radio-label">
                                    <input type="radio" name="brings_wheels" value="0">
                                    <span class="radio-text">Nein, meine Räder sind eingelagert</span>
                                </label>
                            </div>
                            
                            <!-- Einlagerungsnummer - wird dynamisch angezeigt/versteckt -->
                            <div class="form-group" id="storage_number_group" style="display: none; margin-top: 1rem;">
                                <label for="storage_number">Einlagerungsnummer *</label>
                                <div class="input-group">
                                    <input type="text" id="storage_number" name="storage_number" 
                                           placeholder="z. B. 9.123.456" required>
                                    <button type="button" id="check_tire_status_btn" class="btn btn-outline-primary" onclick="checkTireStatus()">
                                        <i class="fas fa-search"></i>
                                        <span>Reifenzustand prüfen</span>
                                    </button>
                                </div>
                                <small class="text-muted">Ihre Einlagerungsnummer finden Sie auf der letzten Rechnung vom letzten Räderwechsel.</small>
                                
                                <!-- Reifenzustand-Anzeige -->
                                <div id="tire_status_display" class="tire-status-display" style="display: none;">
                                    <div class="tire-status-header">
                                        <h4><i class="fas fa-tire"></i> Reifenzustand</h4>
                                    </div>
                                    <div id="tire_status_content" class="tire-status-content">
                                        <!-- Wird dynamisch gefüllt -->
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Reifenstatus-Felder wurden entfernt -->
                        </div>

                        <!-- Tire Type Question - Will be toggled by JavaScript -->
                        <div class="form-group" id="tire_type_group">
                            <h4 class="question-title">Welche Art von Reifen/Räder bringen Sie?</h4>
                            <div class="radio-group">
                                <label class="radio-label">
                                    <input type="radio" name="tire_type" value="set" checked>
                                    <span class="radio-text">Kompletträder / Radsatz</span>
                                </label>
                                <label class="radio-label">
                                    <input type="radio" name="tire_type" value="loose">
                                    <span class="radio-text">Lose Reifen (nicht montiert)</span>
                                </label>
                            </div>
                            <small class="text-muted">Lose Reifen können am Tag des Services nicht gewechselt werden.</small>
                        </div>

                        <div class="form-group">
                            <label for="wheels_storage_action">Was sollen wir mit den abmontierten Rädern machen?</label>
                            <select id="wheels_storage_action" name="wheels_storage_action">
                                <option value="store">Einlagern</option>
                                <option value="vehicle">In Fahrzeug legen</option>
                            </select>
                        </div>

                        <?php if (!empty($services)): ?>
                        <div class="form-group">
                            <h4 class="question-title">Zusätzliche Services</h4>
                            <div class="checkbox-group">
                                <?php foreach ($services as $service): ?>
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="service_<?php echo $service['id']; ?>" value="1" data-service-id="<?php echo $service['id']; ?>" data-service-name="<?php echo htmlspecialchars($service['service_name']); ?>" data-service-price="<?php echo $service['price'] ?? 0; ?>" data-service-show-price="<?php echo $service['show_price'] ? 'true' : 'false'; ?>">
                                        <span class="checkbox-text">
                                            <?php echo htmlspecialchars($service['service_name']); ?>
                                            <?php if ($service['show_price'] && $service['price']): ?>
                                                <span class="service-price">(<?php echo number_format($service['price'], 2); ?> €)</span>
                                            <?php else: ?>
                                                <span class="service-price">(auf Anfrage)</span>
                                            <?php endif; ?>
                                        </span>
                                    </label>
                                    <?php if ($service['service_description']): ?>
                                        <small class="text-muted"><?php echo htmlspecialchars($service['service_description']); ?></small>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <div class="form-group">
                            <label for="seasonal_check">Jahreszeiten-Check</label>
                            <select id="seasonal_check" name="seasonal_check">
                                <option value="none">Kein Check</option>
                                <option value="spring">Frühjahres-Check</option>
                                <option value="winter">Winter-Check</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <h4 class="question-title">Haben Sie ein Radmutterschloss?</h4>
                            <div class="radio-group">
                                <label class="radio-label">
                                    <input type="radio" name="has_wheel_lock" value="1">
                                    <span class="radio-text">Ja</span>
                                </label>
                                <label class="radio-label">
                                    <input type="radio" name="has_wheel_lock" value="0" checked>
                                    <span class="radio-text">Nein</span>
                                </label>
                            </div>
                        </div>

                        <div class="form-group full-width">
                            <label for="info_text">Zusätzliche Informationen</label>
                            <textarea id="info_text" name="info_text" rows="3" 
                                      placeholder="Besondere Wünsche oder Hinweise (optional)"></textarea>
                        </div>
                    </div>

                        <div class="form-actions">
                            <button type="button" class="btn btn-secondary" onclick="history.back()">
                                <i class="fas fa-arrow-left"></i>
                                <span>Abbrechen</span>
                            </button>
                            <button type="submit" class="btn btn-primary" id="submit-btn" disabled>
                                <i class="fas fa-paper-plane"></i>
                                <span>Termin buchen</span>
                            </button>
                        </div>
                    </form>
                </div>

                <aside class="booking-sidebar">
                    <div class="summary-card">
                        <h3>Zusammenfassung</h3>
                        <div class="summary-content">
                            <div class="summary-row">
                                <span class="summary-label">Event</span>
                                <span class="summary-value"><?php echo htmlspecialchars($event['name']); ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">Datum</span>
                                <span class="summary-value"><?php echo date('d.m.Y', strtotime($event['event_date'])); ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">Zeit</span>
                                <span class="summary-value"><?php echo date('H:i', strtotime($event['start_time'])); ?> - <?php echo date('H:i', strtotime($event['end_time'])); ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">Gewählter Slot</span>
                                <span class="summary-value" id="summary-slot">Bitte Slot wählen</span>
                            </div>
                            <div class="summary-row" id="summary-services" style="display: none;">
                                <span class="summary-label">Zusätzliche Services</span>
                                <span class="summary-value" id="summary-wiper-blades">-</span>
                            </div>
                            <div class="summary-row" id="summary-pricing" style="display: none;">
                                <span class="summary-label">Preisübersicht</span>
                                <div class="summary-value" id="price-breakdown"></div>
                            </div>
                            <div class="summary-row summary-total" id="summary-total" style="display: none;">
                                <span class="summary-label"><strong>Gesamtpreis</strong></span>
                                <span class="summary-value" id="total-price">-</span>
                            </div>
                        </div>
                        <div class="summary-footer">
                            <p class="summary-note">Die Terminanfrage wird erst mit Klick auf „Terminanfrage senden" übermittelt.</p>
                            <div class="approval-info">
                                <i class="fab fa-whatsapp"></i>
                                <span>Ihre Anfrage wird von unserem Team geprüft und Sie erhalten eine Bestätigung per WhatsApp.</span>
                            </div>
                        </div>
                    </div>
                </aside>
            </div>
        </div>
    </section>

    <script>
        // Autohaus-spezifische Services an JavaScript übergeben
        window.autohausServices = <?php echo json_encode($services); ?>;
        
        // Fahrzeug-Auswahl-Update
        function updateLicensePlate() {
            const select = document.getElementById('selected_vehicle');
            const hiddenInput = document.getElementById('license_plate');
            
            if (select && hiddenInput) {
                const selectedOption = select.options[select.selectedIndex];
                const licensePlate = selectedOption.getAttribute('data-license');
                hiddenInput.value = licensePlate;
            }
        }
        
        // Handle brings_wheels radio button change
        document.addEventListener('DOMContentLoaded', function() {
            const bringsWheelsRadios = document.querySelectorAll('input[name="brings_wheels"]');
            const storageNumberGroup = document.getElementById('storage_number_group');
            const tireTypeGroup = document.getElementById('tire_type_group');
            
            function updateFormFields() {
                const bringsWheels = document.querySelector('input[name="brings_wheels"]:checked').value;
                
                if (bringsWheels === '0') {
                    // If "Nein, meine Räder sind eingelagert" is selected
                    storageNumberGroup.style.display = 'block';
                    document.getElementById('storage_number').required = true;
                    tireTypeGroup.style.display = 'none';
                    document.querySelector('input[name="tire_type"][value="set"]').checked = true;
                } else {
                    // If "Ja, ich bringe die Räder mit" is selected
                    storageNumberGroup.style.display = 'none';
                    document.getElementById('storage_number').required = false;
                    tireTypeGroup.style.display = 'block';
                }
            }
            
            // Add event listeners
            bringsWheelsRadios.forEach(radio => {
                radio.addEventListener('change', updateFormFields);
            });
            
            // Initialize form fields
            updateFormFields();
        });
        
        // Reifenzustand prüfen
        async function checkTireStatus() {
            const storageNumber = document.getElementById('storage_number').value.trim();
            const button = document.getElementById('check_tire_status_btn');
            const display = document.getElementById('tire_status_display');
            const content = document.getElementById('tire_status_content');
            
            if (!storageNumber) {
                alert('Bitte geben Sie zuerst eine Einlagerungsnummer ein.');
                return;
            }
            
            // Button deaktivieren und Loading anzeigen
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> <span>Prüfe...</span>';
            
            // Display anzeigen
            display.style.display = 'block';
            content.innerHTML = `
                <div class="loading-spinner">
                    <i class="fas fa-spinner fa-spin"></i>
                    <p>Reifenzustand wird abgerufen...</p>
                </div>
            `;
            
            try {
                const response = await fetch('api/check_tire_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        storage_number: storageNumber
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Zeige Erfolgsmeldung
                    content.innerHTML = `
                        <div class="success-message" style="margin-bottom: 1rem;">
                            <i class="fas fa-check-circle"></i>
                            <p><strong>Erfolg:</strong> ${data.message}</p>
                        </div>
                    `;
                    displayTireStatus(data.tire_data, data.source);
                } else {
                    // Formatiere mehrzeilige Fehlermeldungen
                    const errorText = data.error.replace(/\n/g, '<br>');
                    content.innerHTML = `
                        <div class="error-message">
                            <i class="fas fa-exclamation-triangle"></i>
                            <p><strong>Fehler:</strong></p>
                            <div style="margin-top: 0.5rem; text-align: left;">${errorText}</div>
                        </div>
                    `;
                }
            } catch (error) {
                content.innerHTML = `
                    <div class="error-message">
                        <i class="fas fa-exclamation-triangle"></i>
                        <p><strong>Verbindungsfehler:</strong> ${error.message}</p>
                        <small>Bitte überprüfen Sie Ihre Internetverbindung und versuchen Sie es erneut.</small>
                    </div>
                `;
            } finally {
                // Button wieder aktivieren
                button.disabled = false;
                button.innerHTML = '<i class="fas fa-search"></i> <span>Reifenzustand prüfen</span>';
            }
        }
        
        function displayTireStatus(tireData, source = 'unknown') {
            const content = document.getElementById('tire_status_content');
            
            if (!tireData || Object.keys(tireData).length === 0) {
                content.innerHTML = `
                    <div class="error-message">
                        <i class="fas fa-info-circle"></i>
                        <p>Keine Reifendaten für diese Einlagerungsnummer gefunden.</p>
                    </div>
                `;
                return;
            }
            
            // Zeige Hinweis für Datenquelle
            let sourceInfo = '';
            if (source === 'cache') {
                sourceInfo = `
                    <div class="info-message" style="margin-bottom: 1rem;">
                        <i class="fas fa-database"></i>
                        <p><strong>Gecachte Daten:</strong> Diese Daten wurden aus dem Cache geladen.</p>
                    </div>
                `;
            } else if (source === 'resy') {
                sourceInfo = `
                    <div class="success-message" style="margin-bottom: 1rem;">
                        <i class="fas fa-check-circle"></i>
                        <p><strong>Echte RESY-Daten:</strong> Diese Daten wurden direkt aus dem RESY-System abgerufen.</p>
                    </div>
                `;
            }
            
            let html = sourceInfo;
            const positions = ['VL', 'VR', 'HL', 'HR'];
            
            positions.forEach(position => {
                if (tireData[position]) {
                    const tire = tireData[position];
                    const condition = tire.condition || 'Unbekannt';
                    const conditionClass = getConditionClass(condition);
                    const conditionText = getConditionText(condition);
                    
                    html += `
                        <div class="tire-position-card ${conditionClass}">
                            <div class="tire-position-header">
                                <span class="tire-position-name">Position ${position}</span>
                                <span class="tire-status-badge ${conditionClass}">${conditionText}</span>
                            </div>
                            <div class="tire-details">
                                <div class="tire-detail">
                                    <span><strong>Marke:</strong></span>
                                    <span>${tire.brand || '-'}</span>
                                </div>
                                <div class="tire-detail">
                                    <span><strong>Modell:</strong></span>
                                    <span>${tire.model || '-'}</span>
                                </div>
                                <div class="tire-detail">
                                    <span><strong>Größe:</strong></span>
                                    <span>${tire.size || '-'}</span>
                                </div>
                                <div class="tire-detail">
                                    <span><strong>Typ:</strong></span>
                                    <span>${tire.tire_type || '-'}</span>
                                </div>
                                <div class="tire-detail">
                                    <span><strong>Profiltiefe:</strong></span>
                                    <span>${tire.profile_depth || '-'} mm</span>
                                </div>
                                <div class="tire-detail">
                                    <span><strong>Empfehlung:</strong></span>
                                    <span>${tire.recommendation || '-'}</span>
                                </div>
                            </div>
                        </div>
                    `;
                }
            });
            
            content.innerHTML = html;
        }
        
        function getConditionClass(condition) {
            if (condition === 'good') {
                return 'good';
            } else if (condition === 'warning') {
                return 'warning';
            } else if (condition === 'bad') {
                return 'bad';
            }
            return 'good'; // Default
        }
        
        function getConditionText(condition) {
            if (condition === 'good') {
                return 'Sehr gut';
            } else if (condition === 'warning') {
                return 'Leichte Fehler';
            } else if (condition === 'bad') {
                return 'Austausch notwendig';
            }
            return condition; // Fallback
        }
    </script>
    <script src="assets/js/booking.js?v=<?php echo filemtime(__DIR__ . '/assets/js/booking.js'); ?>"></script>
    
    <style>
        .approval-info {
            background: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
        }
        
        .approval-info i {
            color: #1976d2;
            margin-top: 0.125rem;
            flex-shrink: 0;
        }
        
        .approval-info span {
            color: #1565c0;
            font-size: 0.9rem;
            line-height: 1.4;
        }
        
        .service-price {
            color: #28a745;
            font-weight: 600;
            margin-left: 0.5rem;
        }
        
        /* Reifenzustand-Anzeige Styles */
        .tire-status-display {
            margin-top: 1.5rem;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-radius: 12px;
            border: 1px solid #dee2e6;
            overflow: hidden;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        }
        
        .tire-status-header {
            background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
            color: white;
            padding: 1rem 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .tire-status-header h4 {
            margin: 0;
            font-size: 1.1rem;
            font-weight: 600;
        }
        
        .tire-status-header i {
            font-size: 1.2rem;
        }
        
        .tire-status-content {
            padding: 1.5rem;
        }
        
        .tire-position-card {
            background: white;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            border-left: 4px solid #dee2e6;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }
        
        .tire-position-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .tire-position-card.good {
            border-left-color: #28a745;
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
        }
        
        .tire-position-card.warning {
            border-left-color: #ffc107;
            background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
        }
        
        .tire-position-card.bad {
            border-left-color: #dc3545;
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
        }
        
        .tire-position-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }
        
        .tire-position-name {
            font-weight: 600;
            font-size: 1rem;
            color: #495057;
        }
        
        .tire-status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.85rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .tire-status-badge.good {
            background: #28a745;
            color: white;
        }
        
        .tire-status-badge.warning {
            background: #ffc107;
            color: #212529;
        }
        
        .tire-status-badge.bad {
            background: #dc3545;
            color: white;
        }
        
        .tire-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 0.75rem;
            font-size: 0.9rem;
            color: #6c757d;
        }
        
        .tire-detail {
            display: flex;
            justify-content: space-between;
        }
        
        .tire-detail strong {
            color: #495057;
        }
        
        .loading-spinner {
            text-align: center;
            padding: 2rem;
            color: #6c757d;
        }
        
        .loading-spinner i {
            font-size: 2rem;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .error-message {
            background: #f8d7da;
            color: #721c24;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid #f5c6cb;
            text-align: center;
        }
        
        .error-message div {
            text-align: left;
            line-height: 1.5;
        }
        
        .success-message {
            background: #d4edda;
            color: #155724;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid #c3e6cb;
            text-align: center;
        }
        
        .info-message {
            background: #d1ecf1;
            color: #0c5460;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid #bee5eb;
            text-align: center;
        }
        
        @media (max-width: 768px) {
            .tire-details {
                grid-template-columns: 1fr;
            }
            
            .tire-position-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 0.5rem;
            }
        }
    </style>
</body>
</html>
