<?php
/**
 * WhatsApp-Benachrichtigungen mit HelloMateo API
 * Moderne Kommunikation per WhatsApp
 */

require_once '../config/database.php';

class WhatsAppNotifications {
    private $db;
    private $conn;
    private $apiKey;
    private $apiUrl;
    
    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
        $this->apiKey = $this->getWhatsAppApiKey();
        $this->apiUrl = 'https://api.hellomateo.com/v1/messages';
    }
    
    /**
     * Sendet eine WhatsApp-Benachrichtigung für Terminstatusänderungen
     */
    public function sendAppointmentNotification($appointmentId, $action) {
        try {
            // Hole Termindetails
            $appointment = $this->getAppointmentDetails($appointmentId);
            if (!$appointment) {
                return ['success' => false, 'error' => 'Termin nicht gefunden'];
            }
            
            // Prüfe ob Telefonnummer vorhanden ist
            if (empty($appointment['phone'])) {
                return ['success' => false, 'error' => 'Keine Telefonnummer für Termin vorhanden'];
            }
            
            // Prüfe ob WhatsApp-Benachrichtigungen aktiviert sind
            if (!$this->isWhatsAppNotificationsEnabled()) {
                return ['success' => true, 'message' => 'WhatsApp-Benachrichtigungen deaktiviert'];
            }
            
            // Erstelle WhatsApp-Nachricht basierend auf Aktion
            $message = $this->createWhatsAppMessage($appointment, $action);
            
            // Debug: Log WhatsApp-Details
            error_log("WhatsApp - Sende Nachricht an: " . $appointment['phone']);
            error_log("WhatsApp - Nachricht: " . $message);
            error_log("WhatsApp - Aktion: " . $action);
            
            // Sende WhatsApp-Nachricht
            $result = $this->sendWhatsAppMessage($appointment['phone'], $message);
            
            error_log("WhatsApp - Nachricht-Ergebnis: " . json_encode($result));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("WhatsApp - Nachricht-Fehler: " . $e->getMessage());
            return ['success' => false, 'error' => 'WhatsApp-Nachricht konnte nicht gesendet werden: ' . $e->getMessage()];
        }
    }
    
    /**
     * Holt Termindetails aus der Datenbank
     */
    private function getAppointmentDetails($appointmentId) {
        $stmt = $this->conn->prepare("
            SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time
            FROM appointments a 
            LEFT JOIN events e ON a.event_id = e.id 
            WHERE a.id = ?
        ");
        $stmt->execute([$appointmentId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    /**
     * Prüft ob WhatsApp-Benachrichtigungen aktiviert sind
     */
    private function isWhatsAppNotificationsEnabled() {
        $stmt = $this->conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'whatsapp_notifications_enabled'");
        $stmt->execute();
        $result = $stmt->fetch();
        return $result && $result['setting_value'] === '1';
    }
    
    /**
     * Holt WhatsApp-Einstellungen aus der Datenbank
     */
    private function getWhatsAppSettings() {
        $stmt = $this->conn->prepare("
            SELECT setting_key, setting_value FROM settings 
            WHERE setting_key IN ('whatsapp_api_key', 'whatsapp_phone_number', 'whatsapp_business_name', 'whatsapp_template_id', 'whatsapp_test_mode')
        ");
        $stmt->execute();
        $settings = [];
        while ($row = $stmt->fetch()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        return [
            'api_key' => $settings['whatsapp_api_key'] ?? '',
            'phone_number' => $settings['whatsapp_phone_number'] ?? '+49123456789',
            'business_name' => $settings['whatsapp_business_name'] ?? 'Boxenstop Volvo Service',
            'template_id' => $settings['whatsapp_template_id'] ?? '',
            'test_mode' => $settings['whatsapp_test_mode'] === '1'
        ];
    }
    
    /**
     * Erstellt WhatsApp-Nachricht basierend auf Aktion
     */
    private function createWhatsAppMessage($appointment, $action) {
        $eventDate = date('d.m.Y', strtotime($appointment['event_date']));
        $eventTime = date('H:i', strtotime($appointment['slot_time']));
        $companyName = $this->getCompanyName();
        
        switch ($action) {
            case 'approved':
                return "✅ *Termin bestätigt!*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Ihr Termin wurde erfolgreich bestätigt:\n\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Datum:* " . $eventDate . "\n" .
                       "🕐 *Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n" .
                       "🔢 *Einlagerungsnummer:* " . $appointment['storage_number'] . "\n\n" .
                       "Wir freuen uns auf Ihren Besuch!\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $companyName . " Team";
                       
            case 'rejected':
                return "❌ *Termin abgelehnt*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Leider können wir Ihren Termin nicht bestätigen:\n\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Datum:* " . $eventDate . "\n" .
                       "🕐 *Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n\n" .
                       "Bitte kontaktieren Sie uns für alternative Termine.\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $companyName . " Team";
                       
            case 'rescheduled':
                return "📅 *Termin verschoben*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Ihr Termin wurde verschoben:\n\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Neues Datum:* " . $eventDate . "\n" .
                       "🕐 *Neue Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n\n" .
                       "Bitte notieren Sie sich den neuen Termin.\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $companyName . " Team";
                       
            default:
                return "📧 *Termin-Update*\n\n" .
                       "Hallo " . $appointment['customer_name'] . ",\n\n" .
                       "Ihr Termin wurde aktualisiert:\n\n" .
                       "📅 *Event:* " . $appointment['event_name'] . "\n" .
                       "📅 *Datum:* " . $eventDate . "\n" .
                       "🕐 *Zeit:* " . $eventTime . "\n" .
                       "🚗 *Fahrzeug:* " . $appointment['license_plate'] . "\n\n" .
                       "Mit freundlichen Grüßen\n" .
                       "Ihr " . $companyName . " Team";
        }
    }
    
    /**
     * Holt Firmenname aus Einstellungen
     */
    private function getCompanyName() {
        $stmt = $this->conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'company_name'");
        $stmt->execute();
        $result = $stmt->fetch();
        return $result ? $result['setting_value'] : 'Boxenstop Volvo Service';
    }
    
    /**
     * Sendet WhatsApp-Nachricht über HelloMateo API
     */
    private function sendWhatsAppMessage($phoneNumber, $message) {
        try {
            $settings = $this->getWhatsAppSettings();
            
            // Test-Modus oder kein API Key
            if ($settings['test_mode'] || empty($settings['api_key'])) {
                return $this->sendWhatsAppTest($phoneNumber, $message, $settings);
            }
            
            // HelloMateo API Anfrage
            $data = [
                'to' => $phoneNumber,
                'message' => $message,
                'type' => 'text'
            ];
            
            // Template verwenden falls konfiguriert
            if (!empty($settings['template_id'])) {
                $data['template_id'] = $settings['template_id'];
                $data['type'] = 'template';
            }
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $this->apiUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $settings['api_key'],
                'Content-Type: application/json'
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200) {
                error_log("WhatsApp - Nachricht erfolgreich gesendet an: " . $phoneNumber);
                return ['success' => true, 'message' => 'WhatsApp-Nachricht erfolgreich gesendet an ' . $phoneNumber];
            } else {
                error_log("WhatsApp - Nachricht-Fehler für: " . $phoneNumber . " (HTTP: " . $httpCode . ")");
                return ['success' => false, 'error' => 'WhatsApp API Fehler (HTTP: ' . $httpCode . ')'];
            }
            
        } catch (Exception $e) {
            $errorMsg = 'WhatsApp-Fehler: ' . $e->getMessage();
            error_log("WhatsApp - Nachricht-Fehler für " . $phoneNumber . ": " . $errorMsg);
            return ['success' => false, 'error' => $errorMsg];
        }
    }
    
    /**
     * Test-Modus für WhatsApp (ohne API)
     */
    private function sendWhatsAppTest($phoneNumber, $message, $settings = []) {
        try {
            // Simuliere WhatsApp-Versendung
            error_log("WhatsApp Test - Nachricht an " . $phoneNumber . ": " . $message);
            error_log("WhatsApp Test - Einstellungen: " . json_encode($settings));
            
            // Hier würde normalerweise die HelloMateo API aufgerufen werden
            // Für den Test simulieren wir eine erfolgreiche Versendung
            
            $mode = $settings['test_mode'] ? 'Test-Modus' : 'Kein API-Key';
            return ['success' => true, 'message' => 'WhatsApp-Nachricht erfolgreich gesendet an ' . $phoneNumber . ' (' . $mode . ')'];
            
        } catch (Exception $e) {
            error_log("WhatsApp Test - Fehler für " . $phoneNumber . ": " . $e->getMessage());
            return ['success' => false, 'error' => 'WhatsApp Test-Fehler: ' . $e->getMessage()];
        }
    }
}
?>
