<?php
// update_user.php - Benutzerprofil aktualisieren
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

require_once 'config.php';

try {
    // Token aus Header extrahieren
    $headers = getallheaders();
    $auth_header = $headers['Authorization'] ?? '';
    
    if (!preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
        throw new Exception('Authorization token required');
    }
    
    $token = $matches[1];
    
    // Benutzer anhand Token finden
    $stmt = $pdo->prepare("SELECT * FROM users WHERE auth_token = ? AND is_active = 1 AND token_expires_at > NOW()");
    $stmt->execute([$token]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        throw new Exception('Invalid or expired token');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Invalid JSON input');
    }
    
    // Validierung der Eingabedaten
    $required_fields = ['first_name', 'last_name', 'email', 'phone_number', 'license_plate'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            throw new Exception("Field '$field' is required");
        }
    }
    
    // E-Mail-Validierung
    if (!filter_var($input['email'], FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Invalid email format');
    }
    
    // Prüfen ob E-Mail bereits von anderem Benutzer verwendet wird
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
    $stmt->execute([$input['email'], $user['id']]);
    if ($stmt->fetch()) {
        throw new Exception('Email already exists');
    }
    
    // Passwort-Hash aktualisieren (falls neues Passwort angegeben)
    $password_hash = $user['password_hash'];
    if (!empty($input['password'])) {
        $password_hash = password_hash($input['password'], PASSWORD_DEFAULT);
    }
    
    // Benutzer aktualisieren
    $sql = "UPDATE users SET 
        first_name = ?, 
        last_name = ?, 
        email = ?, 
        phone_number = ?, 
        license_plate = ?,
        preferred_dealership_id = ?, 
        profile_image_url = ?, 
        is_leasing_vehicle = ?,
        is_contract_vehicle = ?,
        password_hash = ?,
        updated_at = NOW()
    WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([
        $input['first_name'],
        $input['last_name'],
        $input['email'],
        $input['phone_number'],
        $input['license_plate'],
        $input['preferred_dealership_id'] ?? null,
        $input['profile_image_url'] ?? $user['profile_image_url'],
        $input['is_leasing_vehicle'] ?? false,
        $input['is_contract_vehicle'] ?? false,
        $password_hash,
        $user['id']
    ]);
    
    if ($result) {
        // Aktualisierte Benutzerdaten abrufen
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user['id']]);
        $updated_user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Passwort-Hash aus Antwort entfernen
        unset($updated_user['password_hash']);
        
        echo json_encode([
            'success' => true,
            'user' => $updated_user,
            'message' => 'Profile updated successfully'
        ]);
    } else {
        throw new Exception('Failed to update profile');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>
