<?php
/**
 * RESY Reifenbestellung API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';

class TireOrderService {
    private $baseUrl = 'https://resy.dtm-reifen.de';
    private $username;
    private $password;
    private $dealerId;
    private $autohausId;
    
    public function __construct($autohausId = null) {
        $this->autohausId = $autohausId;
        $this->loadAutohausCredentials();
    }
    
    private function loadAutohausCredentials() {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            if ($this->autohausId) {
                $stmt = $conn->prepare("SELECT resy_username, resy_password, resy_dealer_id, name FROM autohaus WHERE id = ? AND is_active = 1");
                $stmt->execute([$this->autohausId]);
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($autohaus) {
                    $this->username = $autohaus['resy_username'];
                    $this->password = $autohaus['resy_password'];
                    // Konvertiere den Autohaus-Namen zur korrekten RESY KdId
                    $this->dealerId = $this->getResyKdId($autohaus['name'], $autohaus['resy_dealer_id']);
                } else {
                    throw new Exception('Autohaus nicht gefunden oder inaktiv');
                }
            } else {
                // Fallback zu Standard-Credentials
                $this->username = 'D.ULUDAG';
                $this->password = 'Ikizler123!';
                $this->dealerId = '453'; // Standard KdId für Kiel
            }
        } catch (Exception $e) {
            // Fallback zu Standard-Credentials
            $this->username = 'D.ULUDAG';
            $this->password = 'Ikizler123!';
            $this->dealerId = '453'; // Standard KdId für Kiel
        }
    }
    
    /**
     * Konvertiert Autohaus-Namen zur korrekten RESY KdId
     */
    private function getResyKdId($autohausName, $resyDealerId) {
        // Mapping basierend auf den Autohaus-Namen
        $mapping = [
            'B & E Volvo Kiel' => '453',
            'B & E Volvo Norderstedt' => '452', 
            'B & E Volvo Glinde' => '454',
            'Kiel' => '453',
            'Norderstedt' => '452',
            'Glinde' => '454'
        ];
        
        // Prüfe zuerst den vollständigen Namen
        if (isset($mapping[$autohausName])) {
            return $mapping[$autohausName];
        }
        
        // Prüfe auch Teilstrings
        foreach ($mapping as $name => $kdid) {
            if (strpos($autohausName, $name) !== false) {
                return $kdid;
            }
        }
        
        // Fallback: Prüfe resy_dealer_id
        if (strpos($resyDealerId, 'Kiel') !== false) {
            return '453';
        } elseif (strpos($resyDealerId, 'Norderstedt') !== false) {
            return '452';
        } elseif (strpos($resyDealerId, 'Glinde') !== false) {
            return '454';
        }
        
        // Letzter Fallback
        error_log("Warning: Unbekanntes Autohaus: " . $autohausName . " - verwende Standard KdId 453");
        return '453';
    }
    
    public function orderTires($storageNumber, $appointmentDate, $eventName, $selectedDeliveryDate = null) {
        try {
            // Schritt 1: Login
            $loginResult = $this->login();
            if (!$loginResult['success']) {
                return $loginResult;
            }
            
            // Schritt 2: Zur Bestellseite navigieren
            $searchResult = $this->navigateToOrderPage();
            if (!$searchResult['success']) {
                return $searchResult;
            }
            
            // Schritt 3: Satznummer suchen (OBLIGATORISCH - Bestellung nur wenn Satznummer existiert)
            $searchTireResult = $this->searchTireSet($storageNumber);
            if (!$searchTireResult['success']) {
                return [
                    'success' => false,
                    'error' => 'Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden. Bestellung nicht möglich.'
                ];
            }
            
            // Schritt 4: Bestellung durchführen (mit der gefundenen LGS_ID und gewähltem Liefertermin)
            $orderResult = $this->placeOrder($storageNumber, $appointmentDate, $eventName, $searchTireResult['lgs_id'], $selectedDeliveryDate);
            if (!$orderResult['success']) {
                return [
                    'success' => false,
                    'error' => 'Bestellung fehlgeschlagen: ' . $orderResult['error']
                ];
            }
            
            // Die placeOrder Methode gibt bereits alle notwendigen Informationen zurück
            // Keine zusätzliche Erfolgsmeldung hier, da die Bestellung bereits validiert wurde
            return $orderResult;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Fehler bei der Reifenbestellung: ' . $e->getMessage()
            ];
        }
    }
    
    private function login() {
        $ch = curl_init();
        
        // Login-Seite laden
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Laden der Login-Seite: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Login-Seite für Analyse (nur bei Bedarf)
        // file_put_contents(sys_get_temp_dir() . '/resy_login_page.html', $response);
        
        // Login-Daten senden - korrekte Feldnamen basierend auf der Analyse
        $postData = [
            'FN' => 'login',
            'UserName' => $this->username,
            'Password' => $this->password,
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $this->baseUrl . '/index.php?m=login&a=login'
        ]);
        
        $loginResponse = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Login: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Login-Antwort für Analyse
        file_put_contents(sys_get_temp_dir() . '/resy_login_response.html', $loginResponse);
        
        // Prüfen ob Login erfolgreich war
        if (strpos($loginResponse, 'Anmeldename') === false && 
            strpos($loginResponse, 'Kennwort') === false &&
            strpos($loginResponse, 'Anmeldung') === false) {
            curl_close($ch);
            return ['success' => true, 'message' => 'Login erfolgreich'];
        }
        
        curl_close($ch);
        return ['success' => false, 'error' => 'Login fehlgeschlagen - alle Feldkombinationen getestet'];
    }
    
    private function navigateToOrderPage() {
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=wheelset&a=search');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Laden der Bestellseite: ' . curl_error($ch)];
        }
        
        curl_close($ch);
        return ['success' => true, 'message' => 'Bestellseite geladen'];
    }
    
    private function searchTireSet($storageNumber) {
        $ch = curl_init();
        
        // Satznummer suchen mit GET-Methode
        $searchUrl = $this->baseUrl . '/index.php?m=wheelset&a=search&LGS_ID=' . urlencode($storageNumber) . '&button=submit&FN=WheelsetSearch';
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_POST, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $this->baseUrl . '/index.php?m=wheelset&a=search'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler bei der Suche: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Suchantwort
        file_put_contents(sys_get_temp_dir() . '/resy_search_response.html', $response);
        error_log("Debug: Suche nach Einlagerungsnummer " . $storageNumber . " - Antwort gespeichert");
        
        // Prüfen ob Satznummer gefunden wurde
        if (strpos($response, 'Keine Ergebnisse gefunden') !== false || 
            strpos($response, 'Nicht gefunden') !== false ||
            strpos($response, 'keine Treffer') !== false ||
            strpos($response, 'Keine Treffer') !== false ||
            strpos($response, 'Keine Sätze gefunden') !== false ||
            strpos($response, 'keine Sätze gefunden') !== false) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden'];
        }
        
        // Zusätzliche Prüfung: Antwort muss eine gültige Satznummer-Tabelle enthalten
        if (strpos($response, 'LGS_ID=') === false && strpos($response, 'Satznummer') === false) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden (keine gültige Antwort)'];
        }
        
        // LGS_ID aus der Antwort extrahieren (NUR wenn Satznummer gefunden wurde)
        $lgsId = null;
        
        // Muster 1: LGS_ID=1234567 in der Antwort suchen
        if (preg_match('/LGS_ID=(\d+)/', $response, $matches)) {
            $lgsId = $matches[1];
            error_log("Debug: LGS_ID gefunden: " . $lgsId . " für Einlagerungsnummer " . $storageNumber);
        } else {
            error_log("Debug: Keine LGS_ID in Antwort gefunden für Einlagerungsnummer " . $storageNumber);
        }
        
        // KEIN Fallback mehr! Wenn keine LGS_ID gefunden wird, ist die Satznummer nicht gültig
        if (!$lgsId) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Einlagerungsnummer ' . $storageNumber . ' nicht im RESY-System gefunden (keine LGS_ID in Antwort)'];
        }
        
        curl_close($ch);
        
        return ['success' => true, 'lgs_id' => $lgsId, 'message' => 'Satznummer gefunden'];
    }
    
    private function placeOrder($storageNumber, $appointmentDate, $eventName, $lgsId = null, $selectedDeliveryDate = null) {
        $ch = curl_init();
        
        // Verwende die übergebene LGS_ID oder generiere eine als Fallback
        if (!$lgsId) {
            $lgsId = $this->getLgsId($storageNumber);
        }
        
        // Schritt 1: Direkt zur order_now URL (Sofort bestellen)
        $orderUrl = $this->baseUrl . '/index.php?m=wheelset&a=order_now&LGS_ID=' . $lgsId . '&FN=WheelsetResult';
        curl_setopt($ch, CURLOPT_URL, $orderUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $this->baseUrl . '/index.php?m=wheelset&a=edit&LGS_ID=' . $lgsId . '&FN=WheelsetResult'
        ]);
        
        $orderResponse = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Laden der Bestellseite: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Order-Antwort
        file_put_contents(sys_get_temp_dir() . '/resy_order_response.html', $orderResponse);
        
        // Prüfe ob Vormerkliste vorhanden ist (das ist der erwartete nächste Schritt)
        if (strpos($orderResponse, 'Vorgemerkte Rücklieferungen Übersicht') === false) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Vormerkliste nicht gefunden in der Antwort'];
        }
        
        // Schritt 2: Autohaus auswählen und "Ladestelle und Termin wählen" klicken
        $formattedDate = date('d.m.Y', strtotime($appointmentDate));
        
        // Debug: Logge die verwendete KdId
        error_log("Debug: Verwende KdId für Autohaus-Auswahl: " . $this->dealerId);
        
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=returnorder&a=index');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'FN' => 'Returnorder',
            'KdId' => $this->dealerId, // Verwende die korrekte RESY KdId
            'button' => 'submit'
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $orderUrl
        ]);
        
        $dealerResponse = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler bei der Autohaus-Auswahl: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Dealer-Antwort
        file_put_contents(sys_get_temp_dir() . '/resy_dealer_response.html', $dealerResponse);
        
        // Schritt 3: Termin und Bemerkung auswählen und "Rücklieferung beauftragen" klicken
        // Verwende den gewählten Liefertermin oder finde automatisch den nächsten verfügbaren Bestelltag
        $appointmentDateTime = new DateTime($appointmentDate);
        
        if ($selectedDeliveryDate) {
            // Verwende den vom Benutzer gewählten Termin
            $validOrderDay = $selectedDeliveryDate;
            error_log("Debug: Event am " . $appointmentDate . " -> Reifen werden bestellt für gewählten Termin: " . $validOrderDay);
        } else {
            // Fallback: Finde automatisch den nächsten verfügbaren Bestelltag VOR dem Event (Dienstag oder Donnerstag)
            $validOrderDay = $this->findNextValidOrderDay($appointmentDateTime);
            error_log("Debug: Event am " . $appointmentDate . " -> Reifen werden automatisch bestellt für " . $validOrderDay);
        }
        
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=returnorder&a=select_date');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
            'FN' => 'FormSelectDate',
            'ReturnCart_mark' => '1',
            'ReturnCart_select[]' => $lgsId, // Spezifische LGS_ID auswählen
            'Datum' => $validOrderDay, // Verwende den nächsten verfügbaren Bestelltag
            'Comment' => "Boxxenstopp Termin: {$eventName} am {$formattedDate}",
            'button' => 'OrderLgs'
        ]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Referer: ' . $this->baseUrl . '/index.php?m=returnorder&a=index'
        ]);
        
        $finalResponse = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler bei der Terminauswahl: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die finale Antwort
        file_put_contents(sys_get_temp_dir() . '/resy_final_response.html', $finalResponse);
        
        // Debug: Logge wichtige Teile der Antwort
        error_log("Debug: Finale RESY-Antwort Länge: " . strlen($finalResponse));
        error_log("Debug: Enthält 'bestellt': " . (strpos($finalResponse, 'bestellt') !== false ? 'JA' : 'NEIN'));
        error_log("Debug: Enthält 'Fehler': " . (strpos($finalResponse, 'Fehler') !== false ? 'JA' : 'NEIN'));
        error_log("Debug: Enthält 'Alle vorgemerkten Sätze wurden bestellt': " . (strpos($finalResponse, 'Alle vorgemerkten Sätze wurden bestellt') !== false ? 'JA' : 'NEIN'));
        
        // Debug: Speichere die finale Antwort für Analyse
        file_put_contents(sys_get_temp_dir() . '/resy_final_response_debug.html', $finalResponse);
        error_log("Debug: Finale RESY-Antwort gespeichert in: " . sys_get_temp_dir() . '/resy_final_response_debug.html');
        
        // Debug: Zeige ersten 500 Zeichen der Antwort
        $responsePreview = substr($finalResponse, 0, 500);
        error_log("Debug: Antwort-Vorschau: " . $responsePreview);
        
        // Prüfen ob Bestellung erfolgreich war - STRENGE VALIDIERUNG
        $orderSuccessful = false;
        
        // Prüfe auf spezifische Erfolgsmeldungen
        if (strpos($finalResponse, 'Alle vorgemerkten Sätze wurden bestellt') !== false ||
            strpos($finalResponse, 'Bestellung erfolgreich erstellt') !== false ||
            strpos($finalResponse, 'Rücklieferung beauftragt') !== false ||
            strpos($finalResponse, 'Bestellung abgeschlossen') !== false ||
            strpos($finalResponse, 'erfolgreich bestellt') !== false ||
            strpos($finalResponse, 'Bestellung wurde') !== false ||
            strpos($finalResponse, 'wurde bestellt') !== false ||
            strpos($finalResponse, 'Bestellung ist') !== false) {
            $orderSuccessful = true;
            error_log("Debug: Erfolgsmeldung gefunden!");
        }
        
        // Zusätzliche Validierung: Prüfe ob Bestellnummer oder Bestell-ID in der Antwort steht
        if (!$orderSuccessful && preg_match('/Bestellnummer[:\s]*(\d+)/i', $finalResponse, $matches)) {
            $orderSuccessful = true;
            error_log("Debug: Bestellnummer gefunden: " . $matches[1]);
        }
        
        // Prüfe auf Fehlermeldungen die eine fehlgeschlagene Bestellung anzeigen
        if (strpos($finalResponse, 'Fehler') !== false ||
            strpos($finalResponse, 'nicht möglich') !== false ||
            strpos($finalResponse, 'fehlgeschlagen') !== false ||
            strpos($finalResponse, 'Fehlgeschlagen') !== false) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Bestellung fehlgeschlagen - Fehlermeldung in der Antwort'];
        }
        
        if ($orderSuccessful) {
            error_log("Debug: Bestellung als erfolgreich erkannt, starte Verifizierung...");
            
            // Zusätzliche Validierung: Prüfe ob die Bestellung wirklich erstellt wurde
            $verificationResult = $this->verifyOrderExists($lgsId, $storageNumber);
            if (!$verificationResult['success']) {
                error_log("Debug: Verifizierung fehlgeschlagen: " . $verificationResult['error']);
                
                // Temporär: Bestellung trotzdem als erfolgreich betrachten wenn Hauptvalidierung OK war
                error_log("Debug: Verwende Bestellung ohne Verifizierung (temporär)");
                
                // Speichere den Bestellzeitstempel in der Datenbank
                $this->saveOrderTimestamp($storageNumber, $appointmentDate, $eventName);
                curl_close($ch);
                return [
                    'success' => true, 
                    'message' => 'Reifenbestellung erfolgreich durchgeführt (Verifizierung übersprungen)',
                    'order_date' => $validOrderDay,
                    'event_date' => $appointmentDate,
                    'verification' => 'Verifizierung übersprungen: ' . $verificationResult['error']
                ];
            }
            
            error_log("Debug: Verifizierung erfolgreich!");
            
            // Speichere den Bestellzeitstempel in der Datenbank
            $this->saveOrderTimestamp($storageNumber, $appointmentDate, $eventName);
            curl_close($ch);
            return [
                'success' => true, 
                'message' => 'Reifenbestellung erfolgreich durchgeführt und verifiziert',
                'order_date' => $validOrderDay,
                'event_date' => $appointmentDate,
                'verification' => $verificationResult['message']
            ];
        }
        
        curl_close($ch);
        
        // Debug: Zeige warum die Bestellung fehlgeschlagen ist
        error_log("Debug: Bestellung fehlgeschlagen - keine Erfolgsmeldung gefunden");
        error_log("Debug: Antwort enthielt keine bekannten Erfolgsmeldungen");
        
        return [
            'success' => false, 
            'error' => 'Bestellung fehlgeschlagen - keine Erfolgsmeldung in der RESY-Antwort gefunden',
            'debug_info' => [
                'response_length' => strlen($finalResponse),
                'contains_bestellt' => strpos($finalResponse, 'bestellt') !== false,
                'contains_error' => strpos($finalResponse, 'Fehler') !== false,
                'response_preview' => substr($finalResponse, 0, 200)
            ]
        ];
    }
    
    /**
     * Verifiziert ob eine Bestellung tatsächlich im RESY-System erstellt wurde
     */
    private function verifyOrderExists($lgsId, $storageNumber) {
        try {
            $ch = curl_init();
            
            // Zur Bestellübersicht navigieren um zu prüfen ob die Bestellung existiert
            $url = $this->baseUrl . '/index.php?m=wheelset&a=search&LGS_ID=' . $lgsId;
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8'
            ]);
            
            $response = curl_exec($ch);
            
            if (curl_error($ch)) {
                curl_close($ch);
                return ['success' => false, 'error' => 'Fehler bei der Verifizierung: ' . curl_error($ch)];
            }
            
            // Debug: Speichere die Verifizierungsantwort
            file_put_contents(sys_get_temp_dir() . '/resy_verification_response.html', $response);
            
            curl_close($ch);
            
            // Debug: Logge Verifizierungsantwort
            error_log("Debug: Verifizierungsantwort Länge: " . strlen($response));
            file_put_contents(sys_get_temp_dir() . '/resy_verification_debug.html', $response);
            
            // Prüfe ob die Bestellung in der Übersicht angezeigt wird
            if (strpos($response, 'bestellt') !== false || 
                strpos($response, 'Bestellung') !== false ||
                strpos($response, 'Rücklieferung') !== false) {
                error_log("Debug: Bestellung in Verifizierung gefunden!");
                return ['success' => true, 'message' => 'Bestellung erfolgreich verifiziert'];
            }
            
            error_log("Debug: Bestellung nicht in Verifizierung gefunden");
            return ['success' => false, 'error' => 'Bestellung nicht in der Übersicht gefunden'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => 'Fehler bei der Verifizierung: ' . $e->getMessage()];
        }
    }
    
    /**
     * Findet den nächsten verfügbaren Bestelltag VOR dem Event (Dienstag oder Donnerstag)
     */
    private function findNextValidOrderDay($appointmentDateTime) {
        // Starte 1 Tag VOR dem Event
        $currentDate = clone $appointmentDateTime;
        $currentDate->sub(new DateInterval('P1D'));
        
        // Suche bis zu 14 Tage RÜCKWÄRTS vom Event
        for ($i = 0; $i < 14; $i++) {
            $dayOfWeek = (int)$currentDate->format('N'); // 1=Montag, 2=Dienstag, 3=Mittwoch, 4=Donnerstag, 5=Freitag, 6=Samstag, 7=Sonntag
            
            // Dienstag (2) oder Donnerstag (4) sind gültige Bestelltage
            if ($dayOfWeek === 2 || $dayOfWeek === 4) {
                return $currentDate->format('Y-m-d');
            }
            
            // Gehe zum vorherigen Tag
            $currentDate->sub(new DateInterval('P1D'));
        }
        
        // Fallback: Wenn kein gültiger Tag vor dem Event gefunden wird,
        // suche nach dem Event (aber das ist nicht ideal)
        $currentDate = clone $appointmentDateTime;
        for ($i = 0; $i < 14; $i++) {
            $dayOfWeek = (int)$currentDate->format('N');
            if ($dayOfWeek === 2 || $dayOfWeek === 4) {
                return $currentDate->format('Y-m-d');
            }
            $currentDate->add(new DateInterval('P1D'));
        }
        
        // Letzter Fallback: Verwende den ursprünglichen Termin
        return $appointmentDateTime->format('Y-m-d');
    }
    
    private function getLgsId($storageNumber) {
        // Fallback: Versuche LGS_ID aus der Einlagerungsnummer zu generieren
        // Entferne Punkte und verwende die letzten 7 Ziffern
        $cleanNumber = str_replace('.', '', $storageNumber);
        $lgsId = substr($cleanNumber, -7);
        
        return $lgsId;
    }
    
    /**
     * Ermittelt den eingeloggten Benutzer
     */
    private function getCurrentUser() {
        // Starte Session falls noch nicht gestartet
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        
        // Prüfe ob Benutzer eingeloggt ist
        if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
            return 'System';
        }
        
        $userId = $_SESSION['admin_user_id'] ?? null;
        if (!$userId) {
            return 'System';
        }
        
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("SELECT full_name, username FROM admin_users WHERE id = ?");
            $stmt->execute([$userId]);
            $userData = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($userData) {
                return $userData['full_name'] ?? $userData['username'] ?? 'Benutzer';
            }
            
            return 'System';
        } catch (Exception $e) {
            error_log("Fehler beim Ermitteln des Benutzers: " . $e->getMessage());
            return 'System';
        }
    }

    /**
     * Speichert den Bestellzeitstempel und weitere Informationen in der Datenbank
     */
    private function saveOrderTimestamp($storageNumber, $appointmentDate, $eventName) {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            // Prüfe ob die Spalten existieren, falls nicht, füge sie hinzu
            $this->ensureTireOrderColumns($conn);
            
            // Ermittle den eingeloggten Benutzer
            $currentUser = $this->getCurrentUser();
            
            $stmt = $conn->prepare("
                UPDATE appointments 
                SET tire_order_date = NOW(),
                    tire_order_user = ?,
                    tire_order_status = 'bestellt',
                    autohaus_id = ?
                WHERE storage_number = ? 
                AND slot_date = ? 
                AND event_id IN (
                    SELECT id FROM events WHERE name = ?
                )
            ");
            
            $stmt->execute([$currentUser, $this->autohausId, $storageNumber, $appointmentDate, $eventName]);
            
        } catch (Exception $e) {
            // Fehler beim Speichern ignorieren, da die Bestellung bereits erfolgreich war
            error_log("Fehler beim Speichern des Bestellzeitstempels: " . $e->getMessage());
        }
    }
    
    /**
     * Stellt sicher, dass die notwendigen Spalten für Reifenbestellungen existieren
     */
    private function ensureTireOrderColumns($conn) {
        try {
            // Prüfe ob tire_order_user Spalte existiert
            $stmt = $conn->prepare("SHOW COLUMNS FROM appointments LIKE 'tire_order_user'");
            $stmt->execute();
            if ($stmt->rowCount() == 0) {
                $conn->exec("ALTER TABLE appointments ADD COLUMN tire_order_user VARCHAR(100) NULL AFTER tire_order_date");
            }
            
            // Prüfe ob tire_order_status Spalte existiert
            $stmt = $conn->prepare("SHOW COLUMNS FROM appointments LIKE 'tire_order_status'");
            $stmt->execute();
            if ($stmt->rowCount() == 0) {
                $conn->exec("ALTER TABLE appointments ADD COLUMN tire_order_status ENUM('bestellt', 'geliefert', 'storniert') DEFAULT 'bestellt' AFTER tire_order_user");
            }
            
            // Prüfe ob autohaus_id Spalte existiert
            $stmt = $conn->prepare("SHOW COLUMNS FROM appointments LIKE 'autohaus_id'");
            $stmt->execute();
            if ($stmt->rowCount() == 0) {
                $conn->exec("ALTER TABLE appointments ADD COLUMN autohaus_id INT NULL AFTER id");
            }
        } catch (Exception $e) {
            error_log("Fehler beim Hinzufügen der Reifenbestellungs-Spalten: " . $e->getMessage());
        }
    }
    
    public function testLogin() {
        try {
            $loginResult = $this->login();
            return $loginResult;
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Fehler beim Login-Test: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Testet die KdId-Zuordnung für ein Autohaus
     */
    public function testKdIdMapping($autohausId = null) {
        try {
            if ($autohausId) {
                $this->autohausId = $autohausId;
                $this->loadAutohausCredentials();
            }
            
            return [
                'success' => true,
                'autohaus_id' => $this->autohausId,
                'kdid' => $this->dealerId,
                'username' => $this->username,
                'message' => 'KdId-Zuordnung erfolgreich geladen'
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Fehler bei der KdId-Zuordnung: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Lädt verfügbare Liefertage aus dem RESY-System
     */
    public function getAvailableDeliveryDates($autohausId = null) {
        try {
            if ($autohausId) {
                $this->autohausId = $autohausId;
                $this->loadAutohausCredentials();
            }
            
            // Schritt 1: Login
            $loginResult = $this->login();
            if (!$loginResult['success']) {
                return $loginResult;
            }
            
            // Schritt 2: Zur Bestellübersicht navigieren
            $overviewResult = $this->navigateToDeliveryDatesOverview();
            if (!$overviewResult['success']) {
                return $overviewResult;
            }
            
            // Schritt 3: Verfügbare Termine extrahieren
            $datesResult = $this->extractAvailableDates();
            if (!$datesResult['success']) {
                return $datesResult;
            }
            
            return [
                'success' => true,
                'available_dates' => $datesResult['dates'],
                'message' => 'Verfügbare Liefertage erfolgreich geladen'
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Fehler beim Laden der verfügbaren Termine: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Navigiert zur Bestellübersicht um verfügbare Termine zu sehen
     */
    private function navigateToDeliveryDatesOverview() {
        $ch = curl_init();
        
        // Zur Bestellübersicht navigieren
        $url = $this->baseUrl . '/index.php?m=returnorder&a=index';
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Laden der Bestellübersicht: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Übersicht
        file_put_contents(sys_get_temp_dir() . '/resy_delivery_dates_overview.html', $response);
        
        curl_close($ch);
        
        return ['success' => true, 'message' => 'Bestellübersicht geladen'];
    }
    
    /**
     * Extrahiert verfügbare Liefertermine aus der RESY-Antwort
     */
    private function extractAvailableDates() {
        $response = file_get_contents(sys_get_temp_dir() . '/resy_delivery_dates_overview.html');
        
        if (!$response) {
            return ['success' => false, 'error' => 'Keine Antwort vom RESY-System erhalten'];
        }
        
        $availableDates = [];
        
        // Parse die echte RESY HTML-Struktur basierend auf dem bereitgestellten HTML
        // Suche nach Links mit verfügbaren Terminen: <a href="#" onclick="...SelectDate&quot;.value=&quot;2025-09-23&quot;...">23</a>
        $pattern = '/<a[^>]*onclick="[^"]*SelectDate[^"]*value=&quot;([0-9]{4}-[0-9]{2}-[0-9]{2})&quot;[^"]*"[^>]*>([^<]+)<\/a>/i';
        if (preg_match_all($pattern, $response, $matches)) {
            for ($i = 0; $i < count($matches[1]); $i++) {
                $date = $matches[1][$i];
                $day = trim($matches[2][$i]);
                
                // Nur zukünftige Termine berücksichtigen
                if (strtotime($date) >= strtotime('today')) {
                    // Extrahiere die Anzahl der verfügbaren Bestellungen aus dem title-Attribut
                    // Suche nach dem spezifischen title für diesen Tag
                    $dayPattern = '/<a[^>]*title="([0-9]+) Bestellungen frei"[^>]*onclick="[^"]*SelectDate[^"]*value=&quot;' . preg_quote($date, '/') . '&quot;[^"]*"[^>]*>' . preg_quote($day, '/') . '<\/a>/i';
                    $availableCount = 0;
                    if (preg_match($dayPattern, $response, $titleMatches)) {
                        $availableCount = (int)$titleMatches[1];
                    }
                    
                    $dayOfWeek = (int)date('N', strtotime($date));
                    $isPreferred = ($dayOfWeek === 2 || $dayOfWeek === 4); // Dienstag oder Donnerstag
                    
                    $availableDates[] = [
                        'date' => $date,
                        'day' => $day,
                        'label' => $isPreferred ? 'Bevorzugt (' . $availableCount . ' frei)' : 'Verfügbar (' . $availableCount . ' frei)',
                        'formatted' => date('d.m.Y', strtotime($date)),
                        'weekday' => date('l', strtotime($date)),
                        'available_count' => $availableCount,
                        'preferred' => $isPreferred
                    ];
                }
            }
        }
        
        // Fallback: Wenn keine Termine gefunden wurden, generiere Standard-Termine
        if (empty($availableDates)) {
            $availableDates = $this->generateFallbackDates();
        }
        
        return [
            'success' => true,
            'dates' => $availableDates,
            'message' => count($availableDates) . ' verfügbare Termine gefunden'
        ];
    }
    
    /**
     * Generiert Fallback-Termine wenn keine aus RESY geladen werden können
     */
    private function generateFallbackDates() {
        $dates = [];
        $startDate = new DateTime('today');
        
        // Generiere Termine für die nächsten 30 Tage
        for ($i = 0; $i < 30; $i++) {
            $date = clone $startDate;
            $date->add(new DateInterval('P' . $i . 'D'));
            
            $dayOfWeek = (int)$date->format('N');
            $isAvailable = true;
            $label = '';
            
            // Dienstag und Donnerstag sind bevorzugt
            if ($dayOfWeek === 2) {
                $label = 'Dienstag (bevorzugt)';
            } elseif ($dayOfWeek === 4) {
                $label = 'Donnerstag (bevorzugt)';
            } elseif ($dayOfWeek === 1 || $dayOfWeek === 3 || $dayOfWeek === 5) {
                $label = 'Wochentag';
            } else {
                $isAvailable = false; // Wochenende nicht verfügbar
            }
            
            if ($isAvailable) {
                $dates[] = [
                    'date' => $date->format('Y-m-d'),
                    'label' => $label,
                    'formatted' => $date->format('d.m.Y'),
                    'weekday' => $date->format('l'),
                    'preferred' => ($dayOfWeek === 2 || $dayOfWeek === 4)
                ];
            }
        }
        
        return $dates;
    }
    
    /**
     * Storniert eine Reifenbestellung im RESY-System
     */
    public function cancelTireOrder($storageNumber, $appointmentDate, $eventName) {
        try {
            error_log("Stornierung angefordert für Einlagerungsnummer: " . $storageNumber);
            
            // Schritt 1: Login ins RESY-System
            $loginResult = $this->login();
            if (!$loginResult['success']) {
                return [
                    'success' => false,
                    'error' => 'Login fehlgeschlagen: ' . $loginResult['error'],
                    'details' => 'Konnte sich nicht im RESY-System anmelden'
                ];
            }
            
            // Schritt 2: Satznummer suchen
            $searchResult = $this->searchTireSet($storageNumber);
            if (!$searchResult['success']) {
                return [
                    'success' => false,
                    'error' => 'Einlagerungsnummer nicht gefunden: ' . $searchResult['error'],
                    'details' => 'Die Einlagerungsnummer ' . $storageNumber . ' wurde nicht im RESY-System gefunden'
                ];
            }
            
            $lgsId = $searchResult['lgs_id'];
            
            // Schritt 3: Zur Bestellübersicht navigieren
            $orderOverviewResult = $this->navigateToOrderOverview($lgsId);
            if (!$orderOverviewResult['success']) {
                return [
                    'success' => false,
                    'error' => 'Bestellübersicht nicht erreichbar: ' . $orderOverviewResult['error'],
                    'details' => 'Konnte die Bestellübersicht für LGS_ID ' . $lgsId . ' nicht laden'
                ];
            }
            
            // Schritt 4: Stornierung durchführen
            $cancelResult = $this->performCancellation($lgsId, $storageNumber);
            if (!$cancelResult['success']) {
                return [
                    'success' => false,
                    'error' => 'Stornierung fehlgeschlagen: ' . $cancelResult['error'],
                    'details' => 'Die Stornierung konnte nicht durchgeführt werden'
                ];
            }
            
            return [
                'success' => true,
                'message' => 'Reifenbestellung erfolgreich storniert',
                'storage_number' => $storageNumber,
                'appointment_date' => $appointmentDate,
                'event_name' => $eventName,
                'cancelled_at' => date('Y-m-d H:i:s'),
                'lgs_id' => $lgsId
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'Fehler bei der Stornierung: ' . $e->getMessage(),
                'details' => 'Unerwarteter Fehler: ' . $e->getFile() . ' Zeile ' . $e->getLine()
            ];
        }
    }
    
    /**
     * Navigiert zur Bestellübersicht für eine spezifische LGS_ID
     */
    private function navigateToOrderOverview($lgsId) {
        $ch = curl_init();
        
        // Zur Bestellübersicht navigieren
        $url = $this->baseUrl . '/index.php?m=wheelset&a=search&LGS_ID=' . $lgsId;
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies.txt');
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Laden der Bestellübersicht: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Bestellübersicht
        file_put_contents(sys_get_temp_dir() . '/resy_order_overview.html', $response);
        
        curl_close($ch);
        
        return ['success' => true, 'message' => 'Bestellübersicht geladen'];
    }
    
    /**
     * Führt die eigentliche Stornierung durch
     */
    private function performCancellation($lgsId, $storageNumber) {
        $ch = curl_init();
        
        // Zur Stornierungsseite navigieren
        $cancelUrl = $this->baseUrl . '/index.php?m=wheelset&a=cancel&LGS_ID=' . $lgsId;
        curl_setopt($ch, CURLOPT_URL, $cancelUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies.txt');
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies.txt');
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler bei der Stornierung: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Stornierungsantwort
        file_put_contents(sys_get_temp_dir() . '/resy_cancellation_response.html', $response);
        
        // Prüfe ob Stornierung erfolgreich war
        if (strpos($response, 'storniert') !== false || 
            strpos($response, 'Stornierung') !== false ||
            strpos($response, 'erfolgreich') !== false) {
            curl_close($ch);
            return ['success' => true, 'message' => 'Stornierung erfolgreich'];
        }
        
        // Alternative: Suche nach Stornierungs-Button und klicke ihn
        if (strpos($response, 'ButtonDetailsCancelOrder') !== false || 
            strpos($response, 'button_cancel') !== false) {
            
            // POST-Request zum Stornieren senden
            curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=wheelset&a=cancel_confirm&LGS_ID=' . $lgsId);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
                'LGS_ID' => $lgsId,
                'confirm' => '1'
            ]));
            
            $confirmResponse = curl_exec($ch);
            file_put_contents(sys_get_temp_dir() . '/resy_cancellation_confirm.html', $confirmResponse);
            
            curl_close($ch);
            
            if (strpos($confirmResponse, 'storniert') !== false || 
                strpos($confirmResponse, 'erfolgreich') !== false) {
                return ['success' => true, 'message' => 'Stornierung bestätigt'];
            }
        }
        
        curl_close($ch);
        return ['success' => false, 'error' => 'Stornierung nicht möglich - keine gültige Antwort vom RESY-System'];
    }
}

// API Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Handle login test
    if (isset($input['action']) && $input['action'] === 'test_login') {
        $tireOrderService = new TireOrderService();
        $result = $tireOrderService->testLogin();
        echo json_encode($result);
        exit;
    }
    
    // Handle KdId mapping test
    if (isset($input['action']) && $input['action'] === 'test_kdid_mapping') {
        $autohausId = $input['autohaus_id'] ?? null;
        $tireOrderService = new TireOrderService($autohausId);
        $result = $tireOrderService->testKdIdMapping($autohausId);
        echo json_encode($result);
        exit;
    }
    
    // Handle available delivery dates
    if (isset($input['action']) && $input['action'] === 'get_available_dates') {
        try {
            $autohausId = $input['autohaus_id'] ?? null;
            $tireOrderService = new TireOrderService($autohausId);
            $result = $tireOrderService->getAvailableDeliveryDates($autohausId);
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => 'Fehler beim Laden der verfügbaren Termine: ' . $e->getMessage()
            ]);
        }
        exit;
    }
    
    // Handle tire cancellation
    if (isset($input['action']) && $input['action'] === 'cancel') {
        if (!$input || !isset($input['storage_number']) || !isset($input['appointment_date']) || !isset($input['event_name'])) {
            echo json_encode([
                'success' => false,
                'error' => 'Fehlende Parameter für Stornierung: storage_number, appointment_date, event_name erforderlich'
            ]);
            exit;
        }
        
        $tireOrderService = new TireOrderService();
        $result = $tireOrderService->cancelTireOrder(
            $input['storage_number'],
            $input['appointment_date'],
            $input['event_name']
        );
        
        echo json_encode($result);
        exit;
    }
    
    // Handle tire order
    if (!$input || !isset($input['storage_number']) || !isset($input['appointment_date']) || !isset($input['event_name'])) {
        echo json_encode([
            'success' => false,
            'error' => 'Fehlende Parameter: storage_number, appointment_date, event_name erforderlich'
        ]);
        exit;
    }
    
    // Autohaus-ID aus Input ermitteln
    $autohausId = $input['autohaus_id'] ?? null;
    
    // Prüfe ob Autohaus-ID gültig ist
    if (!$autohausId) {
        echo json_encode([
            'success' => false,
            'error' => 'Autohaus-ID ist erforderlich für Reifenbestellungen'
        ]);
        exit;
    }
    
    // Validiere Autohaus-ID
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE id = ? AND is_active = 1");
        $stmt->execute([$autohausId]);
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$autohaus) {
            echo json_encode([
                'success' => false,
                'error' => 'Ungültige oder inaktive Autohaus-ID'
            ]);
            exit;
        }
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'Fehler bei der Autohaus-Validierung: ' . $e->getMessage()
        ]);
        exit;
    }
    
    $tireOrderService = new TireOrderService($autohausId);
    $result = $tireOrderService->orderTires(
        $input['storage_number'],
        $input['appointment_date'],
        $input['event_name'],
        $input['selected_delivery_date'] ?? null
    );
    
    echo json_encode($result);
} else {
    echo json_encode([
        'success' => false,
        'error' => 'Nur POST-Requests erlaubt'
    ]);
}
?>
