<?php
// Disable error display to prevent HTML output
error_reporting(E_ALL);
ini_set('display_errors', 0);

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

try {
    $database = new Database();
    $conn = $database->getConnection();
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['offer_data']) || !isset($input['recipient_email'])) {
        throw new Exception('Angebotsdaten und E-Mail-Adresse sind erforderlich');
    }
    
    $offerData = $input['offer_data'];
    $recipientEmail = $input['recipient_email'];
    $letterContent = $input['letter_content'] ?? '';
    $emailSubject = $input['email_subject'] ?? 'Reifenangebot - Boxenstop';
    $emailMessage = $input['email_message'] ?? '';
    
    // Get appointment and autohaus data
    $appointmentId = $offerData['appointmentId'];
    
        $stmt = $conn->prepare("
            SELECT
                a.*,
                ah.name as autohaus_name,
                ah.address as autohaus_address,
                ah.phone as autohaus_phone,
                ah.email as autohaus_email,
                ah.email_from_name,
                ah.email_smtp_host,
                ah.email_smtp_port,
                ah.email_smtp_username,
                ah.email_smtp_password,
                ah.email_smtp_encryption
            FROM appointments a
            JOIN autohaus ah ON a.autohaus_id = ah.id
            WHERE a.id = ?
        ");

        $stmt->execute([$appointmentId]);
        $appointment = $stmt->fetch();

        if (!$appointment) {
            throw new Exception('Termin nicht gefunden');
        }
    
    // Generate PDF content using the real PDF generator
    $pdfContent = generatePDFContentForEmail($appointment, $offerData, $letterContent, $conn);
    $filename = 'angebot_' . $appointmentId . '_' . date('Y-m-d_H-i-s') . '.pdf';
    $filepath = '../temp/' . $filename;
    
    if (!is_dir('../temp')) {
        mkdir('../temp', 0755, true);
    }
    
    file_put_contents($filepath, $pdfContent);
    
    // Send email
    $emailResult = sendOfferEmail($appointment, $recipientEmail, $filepath, $filename, $emailSubject, $emailMessage);
    
    // Clean up temp file
    if (file_exists($filepath)) {
        unlink($filepath);
    }
    
    if ($emailResult['success']) {
        echo json_encode([
            'success' => true,
            'message' => 'E-Mail erfolgreich versendet'
        ]);
    } else {
        throw new Exception($emailResult['error']);
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
} catch (Error $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Interner Serverfehler: ' . $e->getMessage()
    ]);
}

function sendOfferEmail($appointment, $recipientEmail, $pdfPath, $filename, $emailSubject = null, $emailMessage = '') {
    try {
        // Check if SMTP settings are configured
        if (empty($appointment['email_smtp_host']) || empty($appointment['email_smtp_username'])) {
            throw new Exception('SMTP-Konfiguration für dieses Autohaus nicht vollständig. Bitte SMTP-Daten in der Datenbank konfigurieren.');
        }
        
        // Use PHPMailer for SMTP email sending
        return sendEmailWithPHPMailer($appointment, $recipientEmail, $pdfPath, $filename, $emailSubject, $emailMessage);
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function sendEmailWithPHPMailer($appointment, $recipientEmail, $pdfPath, $filename, $emailSubject = null, $emailMessage = '') {
    require_once '../vendor/autoload.php';
    
    $mail = new PHPMailer\PHPMailer\PHPMailer(true);
    
    try {
        // Server settings - optimized for Outlook SMTP
        $mail->isSMTP();
        $mail->Host = $appointment['email_smtp_host'];
        $mail->SMTPAuth = true;
        $mail->Username = $appointment['email_smtp_username'];
        $mail->Password = $appointment['email_smtp_password'];
        $mail->SMTPSecure = $appointment['email_smtp_encryption'] ?? 'tls';
        $mail->Port = $appointment['email_smtp_port'] ?? 587;
        
        // Outlook-specific settings
        $mail->SMTPOptions = array(
            'ssl' => array(
                'verify_peer' => false,
                'verify_peer_name' => false,
                'allow_self_signed' => true
            )
        );
        
        // Enable debug output for troubleshooting
        $mail->SMTPDebug = 0; // Set to 2 for detailed debug output
        $mail->Debugoutput = 'error_log';
        
        // Recipients - use SMTP username as sender
        $mail->setFrom($appointment['email_smtp_username'], $appointment['email_from_name'] ?? $appointment['autohaus_name']);
        $mail->addAddress($recipientEmail);
        
        // Add reply-to header
        $mail->addReplyTo($appointment['email_smtp_username'], $appointment['email_from_name'] ?? $appointment['autohaus_name']);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = $emailSubject ?: 'Reifenangebot - ' . ($appointment['fahrzeughalter'] ?? 'Kunde');
        $mail->Body = generateEmailBody($appointment, $emailMessage);
        
        // Set charset
        $mail->CharSet = 'UTF-8';
        
        // Attach PDF
        $mail->addAttachment($pdfPath, $filename);
        
        $mail->send();
        
        return [
            'success' => true,
            'message' => 'E-Mail erfolgreich versendet'
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => 'E-Mail konnte nicht versendet werden: ' . $e->getMessage()
        ];
    }
}

function sendEmailBasic($appointment, $recipientEmail, $pdfPath, $filename, $emailSubject = null, $emailMessage = '') {
    // Basic email sending without PHPMailer
    $subject = $emailSubject ?: 'Reifenangebot - ' . ($appointment['fahrzeughalter'] ?? 'Kunde');
    $body = generateEmailBody($appointment, $emailMessage);
    
    $headers = [
        'From: ' . ($appointment['email_from_name'] ?? $appointment['autohaus_name']) . ' <' . $appointment['autohaus_email'] . '>',
        'Reply-To: ' . $appointment['autohaus_email'],
        'Content-Type: text/html; charset=UTF-8',
        'MIME-Version: 1.0'
    ];
    
    if (mail($recipientEmail, $subject, $body, implode("\r\n", $headers))) {
        return [
            'success' => true,
            'message' => 'E-Mail erfolgreich versendet'
        ];
    } else {
        return [
            'success' => false,
            'error' => 'E-Mail konnte nicht versendet werden'
        ];
    }
}

function generateEmailBody($appointment, $customMessage = '') {
    $selectionToken = generateSelectionToken($appointment['id']);
    $baseUrl = getBaseUrl();
    
    return '
    <!DOCTYPE html>
    <html lang="de">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Reifenangebot - ' . htmlspecialchars($appointment['autohaus_name']) . '</title>
        <style>
            @import url("https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap");
            
            body {
                font-family: "Inter", -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
                line-height: 1.6;
                color: #2c3e50;
                background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
                margin: 0;
                padding: 20px;
            }
            
            .email-container {
                max-width: 600px;
                margin: 0 auto;
                background: white;
                border-radius: 16px;
                box-shadow: 0 10px 40px rgba(0,0,0,0.1);
                overflow: hidden;
            }
            
            .header {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 40px 30px;
                text-align: center;
                position: relative;
                overflow: hidden;
            }
            
            .header::before {
                content: "";
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: url("data:image/svg+xml,<svg xmlns=\'http://www.w3.org/2000/svg\' viewBox=\'0 0 100 100\'><defs><pattern id=\'grain\' width=\'100\' height=\'100\' patternUnits=\'userSpaceOnUse\'><circle cx=\'25\' cy=\'25\' r=\'1\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'75\' cy=\'75\' r=\'1\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'50\' cy=\'10\' r=\'0.5\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'10\' cy=\'60\' r=\'0.5\' fill=\'white\' opacity=\'0.1\'/><circle cx=\'90\' cy=\'40\' r=\'0.5\' fill=\'white\' opacity=\'0.1\'/></pattern></defs><rect width=\'100\' height=\'100\' fill=\'url(%23grain)\'/></svg>");
                opacity: 0.3;
            }
            
            .header-content {
                position: relative;
                z-index: 1;
            }
            
            .header h1 {
                font-size: 28px;
                font-weight: 700;
                margin: 0 0 10px 0;
                text-shadow: 0 2px 4px rgba(0,0,0,0.1);
            }
            
            .header p {
                font-size: 16px;
                opacity: 0.95;
                margin: 0;
                font-weight: 400;
            }
            
            .content {
                padding: 40px 30px;
            }
            
            .greeting {
                font-size: 18px;
                color: #2c3e50;
                margin-bottom: 20px;
                font-weight: 500;
            }
            
            .custom-message {
                background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
                padding: 20px;
                border-radius: 12px;
                margin: 20px 0;
                border-left: 4px solid #1976d2;
                box-shadow: 0 4px 12px rgba(25, 118, 210, 0.1);
            }
            
            .custom-message p {
                margin: 0;
                color: #1976d2;
                font-weight: 500;
            }
            
            .offer-info {
                background: #f8f9fa;
                padding: 20px;
                border-radius: 12px;
                margin: 20px 0;
                border: 1px solid #e9ecef;
            }
            
            .offer-info h3 {
                color: #2c3e50;
                margin: 0 0 15px 0;
                font-size: 18px;
                font-weight: 600;
            }
            
            .license-plate {
                background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
                color: white;
                padding: 15px 20px;
                border-radius: 8px;
                text-align: center;
                font-size: 20px;
                font-weight: 700;
                margin: 15px 0;
                box-shadow: 0 4px 12px rgba(40, 167, 69, 0.2);
            }
            
            .selection-card {
                background: linear-gradient(135deg, #fff 0%, #f8f9fa 100%);
                padding: 30px;
                border-radius: 16px;
                margin: 30px 0;
                border: 2px solid #007bff;
                box-shadow: 0 8px 25px rgba(0, 123, 255, 0.1);
                position: relative;
                overflow: hidden;
            }
            
            .selection-card::before {
                content: "";
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                height: 4px;
                background: linear-gradient(90deg, #007bff, #0056b3);
            }
            
            .selection-card h3 {
                color: #2c3e50;
                margin: 0 0 20px 0;
                font-size: 22px;
                font-weight: 700;
                display: flex;
                align-items: center;
                gap: 10px;
            }
            
            .selection-card h3::before {
                content: "🎯";
                font-size: 24px;
            }
            
            .selection-steps {
                display: grid;
                gap: 20px;
                margin: 20px 0;
            }
            
            .step {
                display: flex;
                align-items: flex-start;
                gap: 15px;
                padding: 15px;
                background: white;
                border-radius: 8px;
                border: 1px solid #e9ecef;
                transition: all 0.3s ease;
            }
            
            .step:hover {
                box-shadow: 0 4px 12px rgba(0,0,0,0.1);
                transform: translateY(-2px);
            }
            
            .step-icon {
                width: 40px;
                height: 40px;
                background: linear-gradient(135deg, #007bff, #0056b3);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                color: white;
                font-size: 18px;
                font-weight: 700;
                flex-shrink: 0;
            }
            
            .step-content h4 {
                margin: 0 0 8px 0;
                color: #2c3e50;
                font-size: 16px;
                font-weight: 600;
            }
            
            .step-content p {
                margin: 0;
                color: #6c757d;
                font-size: 14px;
            }
            
            .cta-button {
                text-align: center;
                margin: 30px 0;
            }
            
            .cta-button a {
                background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
                color: white;
                padding: 18px 40px;
                text-decoration: none;
                border-radius: 12px;
                display: inline-block;
                font-weight: 700;
                font-size: 18px;
                box-shadow: 0 8px 25px rgba(0, 123, 255, 0.3);
                transition: all 0.3s ease;
                position: relative;
                overflow: hidden;
            }
            
            .cta-button a::before {
                content: "";
                position: absolute;
                top: 0;
                left: -100%;
                width: 100%;
                height: 100%;
                background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
                transition: left 0.5s;
            }
            
            .cta-button a:hover {
                transform: translateY(-3px);
                box-shadow: 0 12px 35px rgba(0, 123, 255, 0.4);
            }
            
            .cta-button a:hover::before {
                left: 100%;
            }
            
            .disclaimer {
                background: #fff3cd;
                border: 1px solid #ffeaa7;
                border-radius: 8px;
                padding: 15px;
                margin: 20px 0;
                color: #856404;
                font-size: 14px;
            }
            
            .footer {
                background: #f8f9fa;
                padding: 30px;
                text-align: center;
                border-top: 1px solid #e9ecef;
            }
            
            .footer p {
                margin: 0 0 10px 0;
                color: #6c757d;
            }
            
            .footer .signature {
                font-weight: 600;
                color: #2c3e50;
                margin: 15px 0;
            }
            
            .footer .autohaus-name {
                color: #007bff;
                font-weight: 700;
                font-size: 16px;
            }
            
            .footer .disclaimer-text {
                font-size: 12px;
                color: #6c757d;
                margin-top: 20px;
                padding-top: 20px;
                border-top: 1px solid #e9ecef;
            }
            
            @media (max-width: 600px) {
                body {
                    padding: 10px;
                }
                
                .content {
                    padding: 20px;
                }
                
                .header {
                    padding: 30px 20px;
                }
                
                .header h1 {
                    font-size: 24px;
                }
                
                .cta-button a {
                    padding: 15px 30px;
                    font-size: 16px;
                }
            }
        </style>
    </head>
    <body>
        <div class="email-container">
            <div class="header">
                <div class="header-content">
                    <h1>🚗 Reifenangebot</h1>
                    <p>Professioneller Service für Ihr Fahrzeug</p>
                </div>
            </div>
            
            <div class="content">
                <div class="greeting">
                    Sehr geehrte Damen und Herren,
                </div>
                
                ' . ($customMessage ? '<div class="custom-message"><p>' . nl2br(htmlspecialchars($customMessage)) . '</p></div>' : '') . '
                
                <div class="offer-info">
                    <h3>📋 Ihr Reifenangebot</h3>
                    <p>anbei erhalten Sie Ihr Reifenangebot für Ihr Fahrzeug mit dem Kennzeichen:</p>
                    <div class="license-plate">' . htmlspecialchars($appointment['license_plate'] ?? 'Nicht angegeben') . '</div>
                    <p>Das Angebot ist als PDF-Datei angehängt.</p>
                </div>
                
                <div class="selection-card">
                    <h3>Kundenauswahl</h3>
                    <p style="margin-bottom: 20px; color: #6c757d;">Bitte wählen Sie aus den folgenden Optionen aus:</p>
                    
                    <div class="selection-steps">
                        <div class="step">
                            <div class="step-icon">🔧</div>
                            <div class="step-content">
                                <h4>Reifen auswählen</h4>
                                <p>Klicken Sie auf die gewünschten Reifen, um Ihre Auswahl zu treffen.</p>
                            </div>
                        </div>
                        
                        <div class="step">
                            <div class="step-icon">⚙️</div>
                            <div class="step-content">
                                <h4>Zusatzleistungen auswählen</h4>
                                <p>Wählen Sie die gewünschten Zusatzleistungen aus.</p>
                            </div>
                        </div>
                        
                        <div class="step">
                            <div class="step-icon">📝</div>
                            <div class="step-content">
                                <h4>Anmerkungen hinzufügen</h4>
                                <p>Fügen Sie optional besondere Wünsche oder Anmerkungen hinzu.</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="cta-button">
                        <a href="' . $baseUrl . '/customer_selection.php?token=' . $selectionToken . '">
                            🚗 Auswahl treffen
                        </a>
                    </div>
                    
                    <div class="disclaimer">
                        <strong>⏰ Wichtiger Hinweis:</strong> Der Link ist 24 Stunden gültig. Ihre Auswahl wird automatisch an uns übermittelt und Sie erhalten eine Bestätigung.
                    </div>
                </div>
                
                <p style="margin-top: 30px; color: #6c757d;">Bei Fragen stehen wir Ihnen gerne zur Verfügung.</p>
            </div>
            
            <div class="footer">
                <p class="signature">Mit freundlichen Grüßen</p>
                <p class="autohaus-name">' . htmlspecialchars($appointment['email_from_name'] ?? $appointment['autohaus_name']) . '</p>
                <p>' . htmlspecialchars($appointment['autohaus_name']) . '</p>
                <p class="disclaimer-text">
                    Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.
                </p>
            </div>
        </div>
    </body>
    </html>';
}


function generateSelectionToken($appointmentId) {
    // Generate a secure token for customer selection with 24h expiry
    $token = bin2hex(random_bytes(32));
    $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));
    
    // Save token to database
    global $conn;
    $stmt = $conn->prepare("
        INSERT INTO offer_tokens (appointment_id, token, expires_at) 
        VALUES (?, ?, ?)
        ON DUPLICATE KEY UPDATE 
        token = VALUES(token), 
        expires_at = VALUES(expires_at),
        used_at = NULL
    ");
    $stmt->execute([$appointmentId, $token, $expiresAt]);
    
    return $token;
}

function getBaseUrl() {
    // For production, use the correct base URL
    $baseUrl = 'http://100.82.207.72:8080';
    
    // For development, try to detect automatically
    if (strpos($_SERVER['HTTP_HOST'] ?? '', 'localhost') !== false || 
        strpos($_SERVER['HTTP_HOST'] ?? '', '127.0.0.1') !== false) {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        
        // Check if port is already included in HTTP_HOST
        if (strpos($host, ':') !== false) {
            return $protocol . '://' . $host;
        }
        
        $port = $_SERVER['SERVER_PORT'] ?? '';
        $port = ($port && $port != '80' && $port != '443') ? ':' . $port : '';
        return $protocol . '://' . $host . $port;
    }
    
    return $baseUrl;
}

function generatePDFContentForEmail($appointment, $offerData, $letterContent, $conn) {
    // Use the same simple PDF generation as the main API
    return generateSimplePDF($appointment, $offerData['selectedTires'] ?? [], $offerData['newTires'] ?? [], $offerData['selectedServices'] ?? [], $conn);
}

function generateSimplePDF($appointment, $selectedTires, $newTires, $selectedServices, $conn) {
    // Create PDF structure
    $pdf = "%PDF-1.4\n";
    $pdf .= "1 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Catalog\n";
    $pdf .= "/Pages 2 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "2 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Pages\n";
    $pdf .= "/Kids [3 0 R]\n";
    $pdf .= "/Count 1\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    // Create content with proper UTF-8 handling
    $pdfContent = "";
    $yPosition = 750;
    
    // Company header (larger font)
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 16 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(B&E Autocentrum GmbH) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    // Date
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Kiel, " . date('d.m.Y') . ") Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 30;
    
    // Greeting
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Sehr geehrte Damen und Herren,) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 30;
    
    // Main text paragraphs
    $paragraphs = [
        "die Vorbereitungen für unseren nächsten Boxenstopp laufen bereits und dabei ist uns leider aufgefallen, dass die bei uns eingelagerten Reifen ersetzt werden müssten.",
        "Daher erhalten Sie mit diesem Schreiben das beigefügte Angebot."
    ];
    
    foreach ($paragraphs as $paragraph) {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(" . escapePDFText($paragraph) . ") Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 30;
    }
    
    $yPosition -= 20;
    
    // Selected tires section
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 14 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Auszutauschender Reifen:) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 25;
    
    if (!empty($selectedTires)) {
        foreach ($selectedTires as $tire) {
            $tireText = "- " . $tire['brand'] . " " . $tire['model'] . " (" . $tire['size'] . ") - Position: " . $tire['position'] . " - " . $tire['recommendation'];
            $pdfContent .= "BT\n";
            $pdfContent .= "/F1 12 Tf\n";
            $pdfContent .= "50 " . $yPosition . " Td\n";
            $pdfContent .= "(" . escapePDFText($tireText) . ") Tj\n";
            $pdfContent .= "ET\n";
            $yPosition -= 20;
        }
    } else {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(Keine Reifen ausgewählt) Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 20;
    }
    
    $yPosition -= 20;
    
    // New tires section
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 14 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Neue Reifen:) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 25;
    
    if (!empty($newTires)) {
        foreach ($newTires as $tire) {
            $tireText = "- " . $tire['brand'] . " (" . $tire['size'] . ") - " . $tire['quantity'] . "x " . number_format($tire['price'], 2, ',', '.') . "€";
            $pdfContent .= "BT\n";
            $pdfContent .= "/F1 12 Tf\n";
            $pdfContent .= "50 " . $yPosition . " Td\n";
            $pdfContent .= "(" . escapePDFText($tireText) . ") Tj\n";
            $pdfContent .= "ET\n";
            $yPosition -= 20;
        }
    } else {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(Keine neuen Reifen hinzugefügt) Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 20;
    }
    
    $yPosition -= 20;
    
    // Costs section
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 14 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Kosten inkl. Reifenmontage:) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 25;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Zzgl.) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Rädermontage                    49,99€) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Einlagerung der Sommerreifen     85,00€) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    // Services
    if (!empty($selectedServices)) {
        foreach ($selectedServices as $serviceId) {
            $service = getServiceById($serviceId, $conn);
            if ($service) {
                $serviceText = str_pad($service['service_name'], 30) . number_format($service['price'], 2, ',', '.') . "€";
                $pdfContent .= "BT\n";
                $pdfContent .= "/F1 12 Tf\n";
                $pdfContent .= "50 " . $yPosition . " Td\n";
                $pdfContent .= "(" . escapePDFText($serviceText) . ") Tj\n";
                $pdfContent .= "ET\n";
                $yPosition -= 20;
            }
        }
    } else {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(Keine Services ausgewählt) Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 20;
    }
    
    $yPosition -= 30;
    
    // Contact info
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Für Rückfragen stehen wir Ihnen gerne unter Tel: 0431/54 63 00 zur Verfügung.) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 40;
    
    // Signature
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Mit freundlichem Gruß) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Deniz Uludag) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(B&E Autocentrum GmbH) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 40;
    
    // Disclaimer (smaller font)
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 10 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Dieses Dokument wurde maschinell erstellt und ist ohne Unterschrift gültig) Tj\n";
    $pdfContent .= "ET\n";
    
    $pdf .= "3 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Page\n";
    $pdf .= "/Parent 2 0 R\n";
    $pdf .= "/MediaBox [0 0 612 792]\n";
    $pdf .= "/Contents 4 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "4 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Length " . strlen($pdfContent) . "\n";
    $pdf .= ">>\n";
    $pdf .= "stream\n";
    $pdf .= $pdfContent;
    $pdf .= "endstream\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "xref\n";
    $pdf .= "0 5\n";
    $pdf .= "0000000000 65535 f \n";
    $pdf .= "0000000009 00000 n \n";
    $pdf .= "0000000058 00000 n \n";
    $pdf .= "0000000115 00000 n \n";
    $pdf .= "0000000204 00000 n \n";
    $pdf .= "trailer\n";
    $pdf .= "<<\n";
    $pdf .= "/Size 5\n";
    $pdf .= "/Root 1 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "startxref\n";
    $pdf .= "0\n";
    $pdf .= "%%EOF\n";
    
    return $pdf;
}

function escapePDFText($text) {
    // Ensure proper UTF-8 encoding
    $text = mb_convert_encoding($text, 'UTF-8', 'UTF-8');
    
    // Convert German characters to their Latin-1 equivalents
    $replacements = [
        'ä' => chr(228),
        'ö' => chr(246),
        'ü' => chr(252),
        'Ä' => chr(196),
        'Ö' => chr(214),
        'Ü' => chr(220),
        'ß' => chr(223),
        '€' => chr(128)
    ];
    
    foreach ($replacements as $char => $replacement) {
        $text = str_replace($char, $replacement, $text);
    }
    
    // Escape special PDF characters
    $text = str_replace(['(', ')', '\\'], ['\\(', '\\)', '\\\\'], $text);
    
    return $text;
}

function getServiceById($serviceId, $conn) {
    $stmt = $conn->prepare("SELECT service_name, price FROM autohaus_services WHERE id = ? AND is_active = 1");
    $stmt->execute([$serviceId]);
    return $stmt->fetch();
}

function createProfessionalPDF($appointment, $selectedTires, $newTires, $selectedServices) {
    // Create a professional PDF structure directly from data
    $pdf = "%PDF-1.4\n";
    $pdf .= "1 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Catalog\n";
    $pdf .= "/Pages 2 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "2 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Pages\n";
    $pdf .= "/Kids [3 0 R]\n";
    $pdf .= "/Count 1\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    // Create PDF content directly from data
    $pdfContent = "";
    $yPosition = 750; // Start from top
    
    // Company header (larger font)
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 16 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(B&E Autocentrum GmbH) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    // Date
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Kiel, " . date('d.m.Y') . ") Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 30;
    
    // Greeting
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Sehr geehrte Damen und Herren,) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 30;
    
    // Main text paragraphs
    $paragraphs = [
        "die Vorbereitungen für unseren nächsten Boxenstopp laufen bereits und dabei ist uns leider aufgefallen, dass die bei uns eingelagerten Reifen ersetzt werden müssten.",
        "Daher erhalten Sie mit diesem Schreiben das beigefügte Angebot."
    ];
    
    foreach ($paragraphs as $paragraph) {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(" . addslashes($paragraph) . ") Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 30;
    }
    
    $yPosition -= 20; // Extra space
    
    // Selected tires section
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 14 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Auszutauschender Reifen:) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 25;
    
    if (!empty($selectedTires)) {
        foreach ($selectedTires as $tire) {
            $tireText = "- " . $tire['brand'] . " " . $tire['model'] . " (" . $tire['size'] . ") - Position: " . $tire['position'] . " - " . $tire['recommendation'];
            $pdfContent .= "BT\n";
            $pdfContent .= "/F1 12 Tf\n";
            $pdfContent .= "50 " . $yPosition . " Td\n";
            $pdfContent .= "(" . addslashes($tireText) . ") Tj\n";
            $pdfContent .= "ET\n";
            $yPosition -= 20;
        }
    } else {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(Keine Reifen ausgewählt) Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 20;
    }
    
    $yPosition -= 20; // Extra space
    
    // New tires section
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 14 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Neue Reifen:) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 25;
    
    if (!empty($newTires)) {
        foreach ($newTires as $tire) {
            $tireText = "- " . $tire['brand'] . " (" . $tire['size'] . ") - " . $tire['quantity'] . "x " . number_format($tire['price'], 2, ',', '.') . "€";
            $pdfContent .= "BT\n";
            $pdfContent .= "/F1 12 Tf\n";
            $pdfContent .= "50 " . $yPosition . " Td\n";
            $pdfContent .= "(" . addslashes($tireText) . ") Tj\n";
            $pdfContent .= "ET\n";
            $yPosition -= 20;
        }
    } else {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(Keine neuen Reifen hinzugefügt) Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 20;
    }
    
    $yPosition -= 20; // Extra space
    
    // Costs section
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 14 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Kosten inkl. Reifenmontage:) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 25;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Zzgl.) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Rädermontage                    49,99€) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Einlagerung der Sommerreifen     85,00€) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    // Services
    if (!empty($selectedServices)) {
        foreach ($selectedServices as $serviceId) {
            $service = getServiceById($serviceId);
            if ($service) {
                $serviceText = str_pad($service['service_name'], 30) . number_format($service['price'], 2, ',', '.') . "€";
                $pdfContent .= "BT\n";
                $pdfContent .= "/F1 12 Tf\n";
                $pdfContent .= "50 " . $yPosition . " Td\n";
                $pdfContent .= "(" . addslashes($serviceText) . ") Tj\n";
                $pdfContent .= "ET\n";
                $yPosition -= 20;
            }
        }
    } else {
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 12 Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        $pdfContent .= "(Keine Services ausgewählt) Tj\n";
        $pdfContent .= "ET\n";
        $yPosition -= 20;
    }
    
    $yPosition -= 30; // Extra space
    
    // Contact info
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Für Rückfragen stehen wir Ihnen gerne unter Tel: 0431/54 63 00 zur Verfügung.) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 40;
    
    // Signature
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Mit freundlichem Gruß) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Deniz Uludag) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 20;
    
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 12 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(B&E Autocentrum GmbH) Tj\n";
    $pdfContent .= "ET\n";
    $yPosition -= 40;
    
    // Disclaimer (smaller font)
    $pdfContent .= "BT\n";
    $pdfContent .= "/F1 10 Tf\n";
    $pdfContent .= "50 " . $yPosition . " Td\n";
    $pdfContent .= "(Dieses Dokument wurde maschinell erstellt und ist ohne Unterschrift gültig) Tj\n";
    $pdfContent .= "ET\n";
    
    $pdf .= "3 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Page\n";
    $pdf .= "/Parent 2 0 R\n";
    $pdf .= "/MediaBox [0 0 612 792]\n";
    $pdf .= "/Contents 4 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "4 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Length " . strlen($pdfContent) . "\n";
    $pdf .= ">>\n";
    $pdf .= "stream\n";
    $pdf .= $pdfContent;
    $pdf .= "endstream\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "xref\n";
    $pdf .= "0 5\n";
    $pdf .= "0000000000 65535 f \n";
    $pdf .= "0000000009 00000 n \n";
    $pdf .= "0000000058 00000 n \n";
    $pdf .= "0000000115 00000 n \n";
    $pdf .= "0000000204 00000 n \n";
    $pdf .= "trailer\n";
    $pdf .= "<<\n";
    $pdf .= "/Size 5\n";
    $pdf .= "/Root 1 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "startxref\n";
    $pdf .= "0\n";
    $pdf .= "%%EOF\n";
    
    return $pdf;
}

// Function already declared above, removing duplicate

function createSimplePDF($content, $appointment) {
    // Create a properly formatted PDF with correct line spacing
    $pdf = "%PDF-1.4\n";
    $pdf .= "1 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Catalog\n";
    $pdf .= "/Pages 2 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "2 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Pages\n";
    $pdf .= "/Kids [3 0 R]\n";
    $pdf .= "/Count 1\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    // Convert content to PDF format with proper line spacing
    $lines = explode("\n", $content);
    $pdfContent = "";
    $yPosition = 750; // Start from top
    
    foreach ($lines as $index => $line) {
        // Handle empty lines
        if (trim($line) === '') {
            $yPosition -= 12; // Standard gap for empty lines
            continue;
        }
        
        // Determine font size and formatting based on content
        $fontSize = 12;
        $isBold = false;
        
        if (strpos($line, 'B&E Autocentrum GmbH') !== false) {
            $fontSize = 16;
            $isBold = true;
        } elseif (strpos($line, 'Auszutauschender Reifen:') !== false || 
                  strpos($line, 'Neue Reifen:') !== false || 
                  strpos($line, 'Kosten inkl. Reifenmontage:') !== false) {
            $fontSize = 14;
            $isBold = true;
        } elseif (strpos($line, 'Dieses Dokument wurde maschinell erstellt') !== false) {
            $fontSize = 10;
        }
        
        $pdfContent .= "BT\n";
        $pdfContent .= "/F1 {$fontSize} Tf\n";
        $pdfContent .= "50 " . $yPosition . " Td\n";
        
        // Escape special characters for PDF
        $escapedLine = str_replace(['(', ')', '\\'], ['\\(', '\\)', '\\\\'], $line);
        $pdfContent .= "({$escapedLine}) Tj\n";
        $pdfContent .= "ET\n";
        
        $yPosition -= ($fontSize + 6); // Move down based on font size
    }
    
    $pdf .= "3 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Type /Page\n";
    $pdf .= "/Parent 2 0 R\n";
    $pdf .= "/MediaBox [0 0 612 792]\n";
    $pdf .= "/Contents 4 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "4 0 obj\n";
    $pdf .= "<<\n";
    $pdf .= "/Length " . strlen($pdfContent) . "\n";
    $pdf .= ">>\n";
    $pdf .= "stream\n";
    $pdf .= $pdfContent;
    $pdf .= "endstream\n";
    $pdf .= "endobj\n\n";
    
    $pdf .= "xref\n";
    $pdf .= "0 5\n";
    $pdf .= "0000000000 65535 f \n";
    $pdf .= "0000000009 00000 n \n";
    $pdf .= "0000000058 00000 n \n";
    $pdf .= "0000000115 00000 n \n";
    $pdf .= "0000000204 00000 n \n";
    $pdf .= "trailer\n";
    $pdf .= "<<\n";
    $pdf .= "/Size 5\n";
    $pdf .= "/Root 1 0 R\n";
    $pdf .= ">>\n";
    $pdf .= "startxref\n";
    $pdf .= "0\n";
    $pdf .= "%%EOF\n";
    
    return $pdf;
}

function generateOfferHTML($appointment, $offerData, $letterContent, $conn) {
    // Convert HTML to plain text if needed
    $processedContent = $letterContent;
    
    // If content contains HTML, convert to plain text for PDF
    if (strpos($processedContent, '<') !== false) {
        $processedContent = strip_tags($processedContent);
        $processedContent = html_entity_decode($processedContent, ENT_QUOTES, 'UTF-8');
    }
    
    // Replace common placeholders
    $processedContent = str_replace('[haltername]', $appointment['fahrzeughalter'] ?? '[Haltername]', $processedContent);
    $processedContent = str_replace('[kundenname]', $appointment['fahrzeughalter'] ?? '[Kundenname]', $processedContent);
    $processedContent = str_replace('[reifenart]', 'Winterreifen', $processedContent);
    $processedContent = str_replace('[kennzeichen]', $appointment['kennzeichen'] ?? '[Kennzeichen]', $processedContent);
    $processedContent = str_replace('[briefanrede]', 'Sehr geehrte Damen und Herren', $processedContent);
    
    // Add new tires section
    $newTiresSection = '';
    if (!empty($offerData['newTires'])) {
        $newTiresSection = "\n\nNeue Reifen:\n";
        foreach ($offerData['newTires'] as $tire) {
            $newTiresSection .= "- {$tire['brandModel']} ({$tire['size']}) - {$tire['quantity']}x " . number_format($tire['total'], 2, ',', '.') . "€\n";
        }
    }
    
    // Add services section
    $servicesSection = '';
    $zzglServices = '';
    if (!empty($offerData['selectedServices'])) {
        $servicesSection = "\n\nZusatzleistungen:\n";
        $zzglServices = "\nZzgl.\n";
        
        // Get services from database
        $serviceIds = implode(',', array_map('intval', $offerData['selectedServices']));
        $stmt = $conn->prepare("
            SELECT service_name, price 
            FROM autohaus_services 
            WHERE id IN ($serviceIds) AND is_active = 1
        ");
        $stmt->execute();
        $services = $stmt->fetchAll();
        
        foreach ($services as $service) {
            $price = number_format($service['price'], 2, ',', '.');
            $servicesSection .= "- {$service['service_name']}: {$price}€\n";
            $zzglServices .= str_pad($service['service_name'], 30) . "{$price}€\n";
        }
    } else {
        // Default services if none selected
        $zzglServices = "\nZzgl.\nRädermontage                    49,99€\nEinlagerung der Sommerreifen     85,00€\n";
    }
    
    $processedContent .= $newTiresSection . $servicesSection . $zzglServices;
    
    // Convert to HTML
    $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Reifenangebot</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 40px;
            color: #333;
        }
        .header {
            margin-bottom: 30px;
        }
        .content {
            white-space: pre-line;
        }
        .footer {
            margin-top: 30px;
            font-size: 0.9em;
            color: #666;
        }
    </style>
</head>
<body>
    <div class="header">
        <strong>' . ($appointment['autohaus_name'] ?? 'B&E Autocentrum GmbH') . '</strong><br>
        ' . ($appointment['autohaus_address'] ?? '') . '<br>
        Tel: ' . ($appointment['autohaus_phone'] ?? '0431/54 63 00') . '
    </div>
    
    <div class="content">' . htmlspecialchars($processedContent) . '</div>
    
    <div class="footer">
        Dieses Dokument wurde maschinell erstellt und ist ohne Unterschrift gültig
    </div>
</body>
</html>';
    
    return $html;
}
?>
