<?php
/**
 * RESY Data Extraction API - Real Sample Version
 * Verwendet die echte LGS_ID aus dem ursprünglichen Sample (9718476)
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true,
        'cookie_file' => '/tmp/resy_cookies.txt'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    curl_setopt($ch, CURLOPT_COOKIEJAR, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $options['cookie_file']);
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// RESY Login durchführen
function performResyLogin($baseUrl, $username, $password) {
    debugLog("=== RESY LOGIN START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    // Cookie-Datei löschen für sauberen Start
    $cookieFile = '/tmp/resy_cookies.txt';
    if (file_exists($cookieFile)) {
        unlink($cookieFile);
    }
    
    // Schritt 1: Login-Seite laden
    $loginUrl = "$baseUrl/index.php?m=login&a=login";
    debugLog("Lade Login-Seite: $loginUrl");
    
    $response = httpRequest($loginUrl, ['cookie_file' => $cookieFile]);
    if ($response['http_code'] !== 200) {
        throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
    }
    
    debugLog("Login-Seite erfolgreich geladen");
    
    // Schritt 2: Login-Formular analysieren
    $dom = new DOMDocument();
    @$dom->loadHTML($response['body']);
    $xpath = new DOMXPath($dom);
    
    // Suche nach dem Login-Formular
    $forms = $xpath->query('//form');
    debugLog("Gefundene Formulare: " . $forms->length);
    
    $loginForm = null;
    for ($i = 0; $i < $forms->length; $i++) {
        $form = $forms->item($i);
        $usernameField = $xpath->query('.//input[@name="UserName"]', $form);
        if ($usernameField->length > 0) {
            debugLog("Login-Formular gefunden (Form $i)");
            $loginForm = $form;
            break;
        }
    }
    
    if (!$loginForm) {
        throw new Exception("Login-Formular nicht gefunden");
    }
    
    $action = $loginForm->getAttribute('action');
    if (empty($action)) {
        $action = $loginUrl;
    }
    
    // Stelle sicher, dass die Action-URL vollständig ist
    if (strpos($action, 'http') !== 0) {
        if (strpos($action, '/') === 0) {
            $action = $baseUrl . $action;
        } else {
            $action = $baseUrl . '/' . $action;
        }
    }
    
    debugLog("Login-Formular gefunden, Action: $action");
    
    // Schritt 3: Versteckte Felder sammeln
    $hiddenFields = [];
    $hiddenInputs = $xpath->query('.//input[@type="hidden"]', $loginForm);
    foreach ($hiddenInputs as $input) {
        $name = $input->getAttribute('name');
        $value = $input->getAttribute('value');
        if ($name) {
            $hiddenFields[$name] = $value;
            debugLog("Verstecktes Feld gefunden: $name = $value");
        }
    }
    
    // Schritt 4: Login-Daten senden
    $loginData = array_merge($hiddenFields, [
        'UserName' => $username,
        'Password' => $password,
        'button' => 'submit'
    ]);
    
    $loginDataString = http_build_query($loginData);
    debugLog("Login-Daten: " . str_replace($password, '***', $loginDataString));
    
    debugLog("Sende Login-Daten...");
    
    $loginResponse = httpRequest($action, [
        'method' => 'POST',
        'data' => $loginDataString,
        'headers' => [
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: ' . $loginUrl,
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1'
        ],
        'cookie_file' => $cookieFile
    ]);
    
    debugLog("Login-Response Code: " . $loginResponse['http_code']);
    
    // Schritt 5: Login-Erfolg prüfen
    $responseBody = $loginResponse['body'];
    
    // Prüfe auf Login-Formular
    if (strpos($responseBody, 'Anmeldename') !== false || 
        strpos($responseBody, 'UserName') !== false ||
        strpos($responseBody, 'Password') !== false) {
        debugLog("Login fehlgeschlagen - Anmeldeformular noch sichtbar");
        throw new Exception("Login fehlgeschlagen - Anmeldeformular noch sichtbar");
    }
    
    // Prüfe auf Fehlermeldungen
    if (strpos($responseBody, 'Fehler') !== false || 
        strpos($responseBody, 'Error') !== false ||
        strpos($responseBody, 'falsch') !== false ||
        strpos($responseBody, 'ungültig') !== false) {
        debugLog("Login fehlgeschlagen - Fehlermeldung erkannt");
        throw new Exception("Login fehlgeschlagen - Fehlermeldung erkannt");
    }
    
    debugLog("✅ Login erfolgreich!");
    debugLog("=== RESY LOGIN END ===");
    
    return $loginResponse;
}

// Zur Detailseite mit echter LGS_ID navigieren
function navigateToRealDetail($baseUrl, $realLgsId = '9718476') {
    debugLog("=== REAL DETAIL NAVIGATION START ===");
    debugLog("Echte LGS_ID: $realLgsId");
    
    // Verwende die echte LGS_ID aus dem ursprünglichen Sample
    $detailUrl = "$baseUrl/index.php?m=wheelset&a=edit&LGS_ID=$realLgsId&FN=WheelsetResult";
    debugLog("Detail-URL: $detailUrl");
    
    try {
        $response = httpRequest($detailUrl, ['cookie_file' => '/tmp/resy_cookies.txt']);
        debugLog("Response Code: " . $response['http_code']);
        debugLog("Response Length: " . strlen($response['body']));
        
        if ($response['http_code'] === 200) {
            // Prüfe ob es eine Detailseite ist
            if (strpos($response['body'], 'WheelHALTERNAME') !== false) {
                debugLog("✅ Echte Detailseite gefunden!");
                debugLog("=== REAL DETAIL NAVIGATION END ===");
                return $response;
            }
            
            // Prüfe ob es eine Login-Seite ist
            if (strpos($response['body'], 'Anmeldename') !== false) {
                debugLog("⚠️ Login-Seite erkannt");
            }
            
            // Prüfe ob es eine Fehlerseite ist
            if (strpos($response['body'], 'Fehler') !== false || strpos($response['body'], 'Error') !== false) {
                debugLog("⚠️ Fehlerseite erkannt");
            }
            
            debugLog("Response Preview: " . substr($response['body'], 0, 500));
        }
    } catch (Exception $e) {
        debugLog("URL fehlgeschlagen: " . $e->getMessage());
    }
    
    throw new Exception("Keine Detailseite für echte LGS_ID $realLgsId gefunden");
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    debugLog("=== HTML-ANALYSE FÜR FELD: $field ===");
    
    // Mapping der Feldnamen zu HTML-IDs
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID', // Autohaus ist die Ladestelle
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    debugLog("Suche nach Element mit ID: $htmlId");
    
    // Suche nach dem Element mit der entsprechenden ID
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        debugLog("Element mit ID '$htmlId' nicht gefunden für Feld '$field'");
        return '';
    }
    
    // Extrahiere den Textinhalt
    $value = trim($element->textContent);
    
    // Für Bemerkung-Felder: Suche nach textarea
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
            debugLog("Bemerkung aus textarea extrahiert: '$value'");
        }
    }
    
    debugLog("Feld '$field' (ID: $htmlId) extrahiert: '$value'");
    
    // Wenn Wert leer ist, suche nach <b> Tags innerhalb des Elements
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
            debugLog("Wert aus <b> Tag extrahiert: '$value'");
        }
    }
    
    debugLog("=== ENDE HTML-ANALYSE FÜR FELD: $field ===");
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== REAL SAMPLE RESY API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'real_sample':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Real Sample für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId) {
                throw new Exception('Appointment ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            // Schritt 1: Login durchführen
            performResyLogin($autohaus['resy_base_url'], $autohaus['resy_username'], $autohaus['resy_password']);
            
            // Schritt 2: Zur echten Detailseite navigieren (ignoriere die eingegebene LGS_ID)
            $detailResponse = navigateToRealDetail($autohaus['resy_base_url'], '9718476');
            
            $result = [
                'success' => true,
                'detailHtml' => $detailResponse['body'],
                'url' => $detailResponse['url'],
                'approach' => 'real_sample',
                'message' => 'Erfolgreich eingeloggt und zur echten Detailseite navigiert (LGS_ID: 9718476)',
                'realLgsId' => '9718476'
            ];
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== REAL SAMPLE RESY API Request Ended ===");
?>
