<?php
/**
 * RESY Data Extraction API - Login Test Version
 * Testet verschiedene Login-Strategien
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'timeout' => 30,
        'cookies' => null,
        'follow_redirects' => true,
        'cookie_file' => sys_get_temp_dir() . '/resy_cookies.txt'
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36');
    
    // Cookie-Verwaltung
    curl_setopt($ch, CURLOPT_COOKIEJAR, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $options['cookie_file']);
    curl_setopt($ch, CURLOPT_COOKIESESSION, false);
    curl_setopt($ch, CURLOPT_FRESH_CONNECT, false);
    
    if ($options['cookies']) {
        curl_setopt($ch, CURLOPT_COOKIE, $options['cookies']);
    }
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// Teste verschiedene Login-Strategien
function testLoginStrategies($baseUrl, $username, $password) {
    debugLog("=== LOGIN STRATEGY TEST START ===");
    debugLog("Base URL: $baseUrl");
    debugLog("Username: $username");
    
    // Cookie-Datei löschen
    $cookieFile = sys_get_temp_dir() . '/resy_cookies.txt';
    if (file_exists($cookieFile)) {
        unlink($cookieFile);
    }
    
    $strategies = [
        // Strategie 1: Standard Login
        [
            'name' => 'Standard Login',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit'
            ]
        ],
        
        // Strategie 2: Ohne FN Parameter
        [
            'name' => 'Ohne FN Parameter',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit'
            ]
        ],
        
        // Strategie 3: Mit verschiedenen FN Werten
        [
            'name' => 'FN=Login (großgeschrieben)',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'Login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit'
            ]
        ],
        
        // Strategie 4: Mit verschiedenen button Werten
        [
            'name' => 'button=login',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'login'
            ]
        ],
        
        // Strategie 5: Mit verschiedenen Login-URLs
        [
            'name' => 'Login URL ohne Parameter',
            'login_url' => "$baseUrl/index.php",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit'
            ]
        ],
        
        // Strategie 6: Mit verschiedenen Headers
        [
            'name' => 'Mit erweiterten Headers',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit'
            ],
            'headers' => [
                'Content-Type: application/x-www-form-urlencoded',
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Accept-Encoding: gzip, deflate, br',
                'Connection: keep-alive',
                'Upgrade-Insecure-Requests: 1',
                'Cache-Control: no-cache',
                'Pragma: no-cache'
            ]
        ],
        
        // Strategie 7: Mit verschiedenen User-Agents
        [
            'name' => 'Mit verschiedenen User-Agents',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit'
            ],
            'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:91.0) Gecko/20100101 Firefox/91.0'
        ],
        
        // Strategie 8: Mit verschiedenen Login-Formularen
        [
            'name' => 'Mit verschiedenen Login-Formularen',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit',
                'form' => 'login'
            ]
        ],
        
        // Strategie 9: Mit verschiedenen Login-Methoden
        [
            'name' => 'Mit verschiedenen Login-Methoden',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit',
                'method' => 'post'
            ]
        ],
        
        // Strategie 10: Mit verschiedenen Login-Sessions
        [
            'name' => 'Mit verschiedenen Login-Sessions',
            'login_url' => "$baseUrl/index.php?m=login&a=login",
            'post_data' => [
                'FN' => 'login',
                'UserName' => $username,
                'Password' => $password,
                'button' => 'submit',
                'session' => '1'
            ]
        ]
    ];
    
    $results = [];
    
    foreach ($strategies as $index => $strategy) {
        debugLog("Teste Login-Strategie " . ($index + 1) . ": " . $strategy['name']);
        
        try {
            // Cookie-Datei für jede Strategie neu erstellen
            $strategyCookieFile = sys_get_temp_dir() . '/resy_cookies_strategy_' . ($index + 1) . '.txt';
            if (file_exists($strategyCookieFile)) {
                unlink($strategyCookieFile);
            }
            
            // Login-Seite laden
            $response = httpRequest($strategy['login_url'], [
                'cookie_file' => $strategyCookieFile,
                'headers' => [
                    'User-Agent: ' . ($strategy['user_agent'] ?? 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'),
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                    'Connection: keep-alive'
                ]
            ]);
            
            if ($response['http_code'] !== 200) {
                throw new Exception("Login-Seite nicht erreichbar: HTTP " . $response['http_code']);
            }
            
            // Login-Daten senden
            $loginDataString = http_build_query($strategy['post_data']);
            debugLog("Login-Daten: " . str_replace($password, '***', $loginDataString));
            
            $loginResponse = httpRequest($strategy['login_url'], [
                'method' => 'POST',
                'data' => $loginDataString,
                'headers' => $strategy['headers'] ?? [
                    'Content-Type: application/x-www-form-urlencoded',
                    'User-Agent: ' . ($strategy['user_agent'] ?? 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'),
                    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                    'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                    'Connection: keep-alive',
                    'Referer: ' . $strategy['login_url']
                ],
                'cookie_file' => $strategyCookieFile
            ]);
            
            $result = [
                'strategy' => $strategy['name'],
                'http_code' => $loginResponse['http_code'],
                'length' => strlen($loginResponse['body']),
                'has_login_form' => strpos($loginResponse['body'], 'Anmeldename') !== false,
                'has_password_field' => strpos($loginResponse['body'], 'Password') !== false,
                'has_username_field' => strpos($loginResponse['body'], 'UserName') !== false,
                'has_resy_content' => strpos($loginResponse['body'], 'RESY') !== false,
                'has_error_message' => strpos($loginResponse['body'], 'Fehler') !== false || strpos($loginResponse['body'], 'Error') !== false,
                'has_success_message' => strpos($loginResponse['body'], 'Erfolg') !== false || strpos($loginResponse['body'], 'Success') !== false,
                'preview' => substr($loginResponse['body'], 0, 200)
            ];
            
            $results[] = $result;
            
            debugLog("Strategie " . ($index + 1) . " - Code: " . $result['http_code'] . ", Length: " . $result['length'] . 
                    ", Login-Form: " . ($result['has_login_form'] ? 'Ja' : 'Nein') . 
                    ", RESY-Content: " . ($result['has_resy_content'] ? 'Ja' : 'Nein') . 
                    ", Error: " . ($result['has_error_message'] ? 'Ja' : 'Nein'));
            
            // Wenn Login erfolgreich ist, speichere die Antwort
            if (!$result['has_login_form'] && $result['has_resy_content']) {
                debugLog("✅ Login erfolgreich mit Strategie " . ($index + 1) . "!");
                file_put_contents(sys_get_temp_dir() . '/resy_login_success.html', $loginResponse['body']);
                debugLog("=== LOGIN STRATEGY TEST END ===");
                return $loginResponse; // Gib die erfolgreiche Antwort zurück, nicht das Array
            }
            
        } catch (Exception $e) {
            debugLog("Strategie " . ($index + 1) . " Fehler: " . $e->getMessage());
            $results[] = [
                'strategy' => $strategy['name'],
                'error' => $e->getMessage()
            ];
        }
    }
    
    debugLog("=== LOGIN STRATEGY TEST END ===");
    return $results;
}

// HTML-Daten extrahieren
function extractDataFromHtml($html, $field) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $fieldMapping = [
        'fahrzeughalter' => 'WheelHALTERNAME',
        'kennzeichen' => 'WheelKENNZEICHEN',
        'fin' => 'WheelFAHRGESTNR',
        'eingangsnummer' => 'WheelLGS_ID',
        'ladestelle' => 'WheelKD_ID',
        'bemerkung' => 'KDBEMERK',
        'ruecklieferungsdatum' => 'WheelRUECKLIEFERTERMIN',
        'rueckgabe_bemerkung' => 'WheelRUECKAKTBEMERKUNG'
    ];
    
    $htmlId = $fieldMapping[$field] ?? null;
    if (!$htmlId) {
        throw new Exception("Unbekanntes Feld: $field");
    }
    
    $element = $xpath->query("//*[@id='$htmlId']")->item(0);
    
    if (!$element) {
        return '';
    }
    
    $value = trim($element->textContent);
    
    if ($field === 'bemerkung') {
        $textarea = $xpath->query("//textarea[@name='KDBEMERK']")->item(0);
        if ($textarea) {
            $value = trim($textarea->textContent);
        }
    }
    
    if (empty($value)) {
        $boldElement = $xpath->query(".//b", $element)->item(0);
        if ($boldElement) {
            $value = trim($boldElement->textContent);
        }
    }
    
    return $value;
}

// Hauptverarbeitung
try {
    debugLog("=== LOGIN TEST RESY API Request Started ===");
    
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'login_test':
            $appointmentId = $input['appointment_id'] ?? null;
            $lgsId = $input['lgsId'] ?? null;
            debugLog("Login Test für Appointment ID: $appointmentId, LGS_ID: $lgsId");
            
            if (!$appointmentId || !$lgsId) {
                throw new Exception('Appointment ID oder LGS_ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_base_url, a.resy_username, a.resy_password 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            // Schritt 1: Verschiedene Login-Strategien testen
            $loginTestResult = testLoginStrategies($autohaus['resy_base_url'], $autohaus['resy_username'], $autohaus['resy_password']);
            
            // Prüfe ob das Ergebnis ein Array ist (alle Strategien getestet) oder eine erfolgreiche Antwort
            if (is_array($loginTestResult)) {
                // Alle Login-Strategien getestet, aber keine funktioniert
                $result = [
                    'success' => false,
                    'error' => 'Keine funktionierende Login-Strategie gefunden',
                    'login_tests' => $loginTestResult,
                    'message' => 'Alle Login-Strategien getestet, aber keine funktioniert'
                ];
            } else {
                // Erfolgreiche Login-Strategie gefunden
                $result = [
                    'success' => true,
                    'loginHtml' => $loginTestResult['body'],
                    'url' => $loginTestResult['url'],
                    'approach' => 'login_test',
                    'message' => 'Funktionierende Login-Strategie gefunden',
                    'lgs_id' => $lgsId,
                    'strategy' => 'Standard Login' // Da Strategie 1 erfolgreich war
                ];
            }
            
            echo json_encode($result);
            break;
            
        case 'extract_field':
            $field = $input['field'] ?? null;
            $html = $input['html'] ?? null;
            debugLog("Extract field: $field");
            
            if (!$field) {
                throw new Exception('Feld-Name fehlt');
            }
            
            if (!$html) {
                throw new Exception('HTML-Inhalt fehlt');
            }
            
            try {
                $value = extractDataFromHtml($html, $field);
                
                $result = [
                    'success' => true,
                    'value' => $value,
                    'field' => $field,
                    'message' => "Feld '$field' erfolgreich extrahiert"
                ];
                
                echo json_encode($result);
                
            } catch (Exception $e) {
                debugLog("Feld-Extraktion fehlgeschlagen: " . $e->getMessage());
                $result = [
                    'success' => false,
                    'error' => $e->getMessage(),
                    'field' => $field
                ];
                echo json_encode($result);
            }
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== LOGIN TEST RESY API Request Ended ===");
?>
