<?php
require_once __DIR__ . '/../config/database.php';

api_cors();

$script = basename($_SERVER['SCRIPT_NAME']);
$allowList = [
  'login_user.php',
  'register_user.php',
  'apple_callback.php',
];

// Interner Web-Flow via Session weiterhin zulassen
// WICHTIG: Session start nur wenn noch nicht gestartet
if (session_status() === PHP_SESSION_NONE) {
    @session_start();
}

// Wenn User oder Admin eingeloggt ist, Guard komplett überspringen
if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in']) {
    // User ist eingeloggt - keine API-Key-Prüfung nötig - Guard überspringen
    error_log("API Guard: User session detected, skipping guard");
    return;
}

if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
    // Admin ist eingeloggt - keine API-Key-Prüfung nötig - Guard überspringen
    error_log("API Guard: Admin session detected, skipping guard");
    return;
}

// Ausnahmen (öffentlich)
if (in_array($script, $allowList, true)) return;

// Scope-Mapping pro Datei
$scopeMap = [
  'get_available_slots.php' => ['slots:read'],
  // 'appointments.php' - KEINE Scope-Prüfung wenn über Session eingeloggt
  'cancel_appointment.php'  => ['appointments:write'],
  'get_user_profile.php'    => ['profile:read'],
  'update_profile.php'      => ['profile:write'],
  'get_vehicle.php'         => ['vehicles:read'],
  'add_vehicle.php'         => ['vehicles:write'],
  'update_vehicle.php'      => ['vehicles:write'],
  'delete_vehicle.php'      => ['vehicles:write'],
  'get_user_notifications.php' => ['notifications:read'],
  'mark_notification_read.php' => ['notifications:write'],
  'get_activity_log.php'    => ['activity:read'],
  'admin_approve_user.php'  => ['admin:verify'],
  'admin_reject_user.php'   => ['admin:verify'],
  'link_user_to_admin.php'  => ['admin:link'],
  'get_customer_profile.php'=> ['admin:verify'],
  'upload_autohaus_footer_image.php' => ['admin:settings'],
  // Sonstige API-Dateien können hier ergänzt werden
];
$requiredScopes = $scopeMap[$script] ?? [];

// Wenn keine Session und keine Ausnahme, API-Key prüfen
$apiKey = api_get_header('X-Api-Key');
if (!$apiKey) api_deny(401, 'Missing X-Api-Key');

$db = new Database();
$conn = $db->getConnection();

function load_api_key($conn, $plain) {
  $hash = hash('sha256', $plain);
  $stmt = $conn->prepare("SELECT * FROM api_keys WHERE key_hash = ? AND is_active = 1");
  $stmt->execute([$hash]);
  return $stmt->fetch(PDO::FETCH_ASSOC) ?: null;
}

function has_scopes($required, $keyScopes) {
  if (empty($required)) return true;
  foreach ($required as $s) if (!in_array($s, $keyScopes, true)) return false;
  return true;
}

function rate_limit_enforce($conn, $row) {
  $limit = (int)($row['rate_limit_per_min'] ?? 60);
  if ($limit <= 0) return;
  $stmt = $conn->prepare("SELECT COUNT(*) FROM api_key_usage WHERE api_key_id = ? AND request_ts > (NOW() - INTERVAL 1 MINUTE)");
  $stmt->execute([$row['id']]);
  $count = (int)$stmt->fetchColumn();
  if ($count >= $limit) api_deny(429, 'Rate limit exceeded');
}

function verify_signature_if_present($conn, $row) {
  $ts = api_get_header('X-Timestamp');
  $nonce = api_get_header('X-Nonce');
  $sig = api_get_header('X-Signature');
  $secretHash = $row['secret_hash'] ?? null;
  if (!$ts || !$nonce || !$sig || !$secretHash) return; // nicht erzwungen
  if (abs(time() - (int)$ts) > 300) api_deny(401, 'Timestamp skew');

  // Replay prüfen
  $stmt = $conn->prepare("SELECT 1 FROM api_replay_nonce WHERE api_key_id = ? AND nonce = ?");
  $stmt->execute([$row['id'], $nonce]);
  if ($stmt->fetch()) api_deny(401, 'Replay detected');

  // TODO: Secret-Auflösung (ENV/KMS). Hier nur Platzhalter – Implementierung kundenspezifisch.
  // $secretPlain = getenv('API_SECRET_' . $row['id']);
  // $method = $_SERVER['REQUEST_METHOD'];
  // $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
  // $body = file_get_contents('php://input');
  // $base = $method . '|' . $path . '|' . $ts . '|' . $nonce . '|' . $body;
  // $expected = base64_encode(hash_hmac('sha256', $base, $secretPlain, true));
  // if (!hash_equals($expected, $sig)) api_deny(401, 'Invalid signature');

  $stmt = $conn->prepare("INSERT INTO api_replay_nonce (api_key_id, nonce, ts) VALUES (?, ?, NOW())");
  $stmt->execute([$row['id'], $nonce]);
}

function log_usage_shutdown($conn, $row) {
  $status = http_response_code() ?: 200;
  $endpoint = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
  $stmt = $conn->prepare("INSERT INTO api_key_usage (api_key_id, endpoint, http_method, status_code, ip, bytes_in, bytes_out) VALUES (?, ?, ?, ?, ?, ?, ?)");
  $bytesIn = (int)($_SERVER['CONTENT_LENGTH'] ?? 0);
  $bytesOut = function_exists('ob_get_length') ? (int)ob_get_length() : 0;
  $stmt->execute([$row['id'], $endpoint, $_SERVER['REQUEST_METHOD'], $status, $_SERVER['REMOTE_ADDR'] ?? null, $bytesIn, $bytesOut]);
}

// Wenn ein API-Key vorhanden ist, authentifizieren und loggen
if ($apiKey) {
    $row = load_api_key($conn, $apiKey);
    if (!$row) api_deny(401, 'Invalid API key');

    $scopes = json_decode($row['scopes'] ?? '[]', true);
    if (!has_scopes($requiredScopes, $scopes)) api_deny(403, 'Insufficient scope');

    rate_limit_enforce($conn, $row);
    verify_signature_if_present($conn, $row);

    // Zusätzlich sofort loggen (falls Shutdown nicht greift)
    try {
      $endpoint = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
      $stmt = $conn->prepare("INSERT INTO api_key_usage (api_key_id, endpoint, http_method, status_code, ip, bytes_in, bytes_out) VALUES (?, ?, ?, ?, ?, ?, ?)");
      $bytesIn = (int)($_SERVER['CONTENT_LENGTH'] ?? 0);
      $stmt->execute([$row['id'], $endpoint, $_SERVER['REQUEST_METHOD'], 200, $_SERVER['REMOTE_ADDR'] ?? null, $bytesIn, 0]);
    } catch (Throwable $e) { /* ignore */ }

    register_shutdown_function('log_usage_shutdown', $conn, $row);
}
