<?php
session_start();
require_once '../config/database.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT full_name, username, role FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
    $current_user_role = $user_data['role'] ?? 'admin';
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action']) && $_POST['action'] === 'add_autohaus') {
            // Prüfe ob alle erforderlichen Autohaus-Felder vorhanden und nicht leer sind
            if (isset($_POST['autohaus_name']) && !empty(trim($_POST['autohaus_name'])) &&
                isset($_POST['autohaus_short_name']) && !empty(trim($_POST['autohaus_short_name'])) &&
                isset($_POST['autohaus_address']) && !empty(trim($_POST['autohaus_address'])) &&
                isset($_POST['autohaus_phone']) && !empty(trim($_POST['autohaus_phone'])) &&
                isset($_POST['autohaus_email']) && !empty(trim($_POST['autohaus_email']))) {
                
                // Neues Autohaus hinzufügen
                $stmt = $conn->prepare("INSERT INTO autohaus (name, short_name, address, phone, email, resy_username, resy_password, resy_dealer_id, timezone, working_days, working_hours, price_wheel_change, price_storage, price_spring_check, price_winter_check, email_smtp_host, email_smtp_port, email_smtp_username, email_smtp_password, email_smtp_encryption, email_from_name, teams_webhook_url, teams_channel_name, teams_notifications_enabled, teams_mention_users) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    trim($_POST['autohaus_name']),
                    trim($_POST['autohaus_short_name']),
                    trim($_POST['autohaus_address']),
                    trim($_POST['autohaus_phone']),
                    trim($_POST['autohaus_email']),
                    trim($_POST['resy_username'] ?? ''),
                    trim($_POST['resy_password'] ?? ''),
                    trim($_POST['resy_dealer_id'] ?? ''),
                    $_POST['timezone'] ?? 'Europe/Berlin',
                    json_encode(explode(',', $_POST['working_days'] ?? 'monday,tuesday,wednesday,thursday,friday')),
                    json_encode(['start' => $_POST['working_start'] ?? '08:00', 'end' => $_POST['working_end'] ?? '18:00']),
                    $_POST['price_wheel_change'] ?? 0.00,
                    $_POST['price_storage'] ?? 0.00,
                    $_POST['price_spring_check'] ?? 0.00,
                    $_POST['price_winter_check'] ?? 0.00,
                    trim($_POST['email_smtp_host'] ?? ''),
                    $_POST['email_smtp_port'] ?? 587,
                    trim($_POST['email_smtp_username'] ?? ''),
                    trim($_POST['email_smtp_password'] ?? ''),
                    $_POST['email_smtp_encryption'] ?? 'tls',
                    trim($_POST['email_from_name'] ?? ''),
                    trim($_POST['teams_webhook_url'] ?? ''),
                    trim($_POST['teams_channel_name'] ?? ''),
                    isset($_POST['teams_notifications_enabled']) ? 1 : 0,
                    trim($_POST['teams_mention_users'] ?? '')
                ]);
                $_SESSION['success_message'] = 'Autohaus erfolgreich hinzugefügt';
            } else {
                $_SESSION['error_message'] = 'Autohaus konnte nicht hinzugefügt werden: Fehlende oder leere Pflichtfelder';
            }
            header('Location: settings.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'update_autohaus') {
            // Prüfe ob alle erforderlichen Autohaus-Felder vorhanden und nicht leer sind
            if (isset($_POST['autohaus_id']) && !empty($_POST['autohaus_id']) &&
                isset($_POST['autohaus_name']) && !empty(trim($_POST['autohaus_name'])) &&
                isset($_POST['autohaus_short_name']) && !empty(trim($_POST['autohaus_short_name'])) &&
                isset($_POST['autohaus_address']) && !empty(trim($_POST['autohaus_address'])) &&
                isset($_POST['autohaus_phone']) && !empty(trim($_POST['autohaus_phone'])) &&
                isset($_POST['autohaus_email']) && !empty(trim($_POST['autohaus_email']))) {
                
                // Autohaus aktualisieren
                $stmt = $conn->prepare("UPDATE autohaus SET name=?, short_name=?, address=?, phone=?, email=?, resy_username=?, resy_password=?, resy_dealer_id=?, timezone=?, working_days=?, working_hours=?, price_wheel_change=?, price_storage=?, price_spring_check=?, price_winter_check=?, email_smtp_host=?, email_smtp_port=?, email_smtp_username=?, email_smtp_password=?, email_smtp_encryption=?, email_from_name=?, teams_webhook_url=?, teams_channel_name=?, teams_notifications_enabled=?, teams_mention_users=?, is_active=? WHERE id=?");
                $stmt->execute([
                    trim($_POST['autohaus_name']),
                    trim($_POST['autohaus_short_name']),
                    trim($_POST['autohaus_address']),
                    trim($_POST['autohaus_phone']),
                    trim($_POST['autohaus_email']),
                    trim($_POST['resy_username'] ?? ''),
                    trim($_POST['resy_password'] ?? ''),
                    trim($_POST['resy_dealer_id'] ?? ''),
                    $_POST['timezone'] ?? 'Europe/Berlin',
                    json_encode(explode(',', $_POST['working_days'] ?? 'monday,tuesday,wednesday,thursday,friday')),
                    json_encode(['start' => $_POST['working_start'] ?? '08:00', 'end' => $_POST['working_end'] ?? '18:00']),
                    $_POST['price_wheel_change'] ?? 0.00,
                    $_POST['price_storage'] ?? 0.00,
                    $_POST['price_spring_check'] ?? 0.00,
                    $_POST['price_winter_check'] ?? 0.00,
                    trim($_POST['email_smtp_host'] ?? ''),
                    $_POST['email_smtp_port'] ?? 587,
                    trim($_POST['email_smtp_username'] ?? ''),
                    trim($_POST['email_smtp_password'] ?? ''),
                    $_POST['email_smtp_encryption'] ?? 'tls',
                    trim($_POST['email_from_name'] ?? ''),
                    trim($_POST['teams_webhook_url'] ?? ''),
                    trim($_POST['teams_channel_name'] ?? ''),
                    isset($_POST['teams_notifications_enabled']) ? 1 : 0,
                    trim($_POST['teams_mention_users'] ?? ''),
                    isset($_POST['is_active']) ? 1 : 0,
                    $_POST['autohaus_id']
                ]);
                $_SESSION['success_message'] = 'Autohaus erfolgreich aktualisiert';
            } else {
                $_SESSION['error_message'] = 'Autohaus konnte nicht aktualisiert werden: Fehlende oder leere Pflichtfelder';
            }
            header('Location: settings.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'delete_autohaus') {
            // Autohaus löschen
            $stmt = $conn->prepare("DELETE FROM autohaus WHERE id = ?");
            $stmt->execute([$_POST['autohaus_id']]);
            $_SESSION['success_message'] = 'Autohaus erfolgreich gelöscht';
            header('Location: settings.php');
            exit;
        }
        
        // Service-Verwaltung
        if (isset($_POST['action']) && $_POST['action'] === 'add_service') {
            // Prüfe ob alle erforderlichen Service-Felder vorhanden sind
            if (isset($_POST['service_autohaus_id']) && !empty($_POST['service_autohaus_id']) && 
                isset($_POST['service_name']) && !empty($_POST['service_name'])) {
                
                // Neuen Service hinzufügen
                $stmt = $conn->prepare("INSERT INTO autohaus_services (autohaus_id, service_name, service_description, price, show_price, is_active, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $_POST['service_autohaus_id'],
                    $_POST['service_name'],
                    $_POST['service_description'] ?? '',
                    $_POST['service_price'] ?: null,
                    isset($_POST['service_show_price']) ? 1 : 0,
                    isset($_POST['service_is_active']) ? 1 : 0,
                    $_POST['service_sort_order'] ?: 0
                ]);
                $_SESSION['success_message'] = 'Service erfolgreich hinzugefügt';
            } else {
                $_SESSION['error_message'] = 'Service konnte nicht hinzugefügt werden: Fehlende Pflichtfelder';
            }
            header('Location: settings.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'update_service') {
            // Prüfe ob alle erforderlichen Service-Felder vorhanden sind
            if (isset($_POST['service_id']) && !empty($_POST['service_id']) &&
                isset($_POST['service_name']) && !empty($_POST['service_name'])) {
                
                // Service aktualisieren
                $stmt = $conn->prepare("UPDATE autohaus_services SET service_name=?, service_description=?, price=?, show_price=?, is_active=?, sort_order=? WHERE id=?");
                $stmt->execute([
                    $_POST['service_name'],
                    $_POST['service_description'] ?? '',
                    $_POST['service_price'] ?: null,
                    isset($_POST['service_show_price']) ? 1 : 0,
                    isset($_POST['service_is_active']) ? 1 : 0,
                    $_POST['service_sort_order'] ?: 0,
                    $_POST['service_id']
                ]);
                $_SESSION['success_message'] = 'Service erfolgreich aktualisiert';
            } else {
                $_SESSION['error_message'] = 'Service konnte nicht aktualisiert werden: Fehlende Pflichtfelder';
            }
            header('Location: settings.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'delete_service') {
            // Service löschen
            $stmt = $conn->prepare("DELETE FROM autohaus_services WHERE id = ?");
            $stmt->execute([$_POST['service_id']]);
            $_SESSION['success_message'] = 'Service erfolgreich gelöscht';
            header('Location: settings.php');
            exit;
        }
        
        // Teams-Einstellungen
        if (isset($_POST['action']) && $_POST['action'] === 'update_teams_settings') {
            // Teams-Einstellungen für Autohaus aktualisieren
            if (isset($_POST['autohaus_id']) && !empty($_POST['autohaus_id'])) {
                $stmt = $conn->prepare("UPDATE autohaus SET teams_webhook_url=?, teams_channel_name=?, teams_notifications_enabled=?, teams_mention_users=? WHERE id=?");
                $stmt->execute([
                    trim($_POST['teams_webhook_url'] ?? ''),
                    trim($_POST['teams_channel_name'] ?? ''),
                    isset($_POST['teams_notifications_enabled']) ? 1 : 0,
                    trim($_POST['teams_mention_users'] ?? ''),
                    $_POST['autohaus_id']
                ]);
                $_SESSION['success_message'] = 'Teams-Einstellungen erfolgreich gespeichert';
            } else {
                $_SESSION['error_message'] = 'Autohaus-ID ist erforderlich';
            }
            header('Location: settings.php');
            exit;
        }
        
        if (isset($_POST['action']) && $_POST['action'] === 'update_settings') {
            // Prüfe welche Felder gesendet wurden und verarbeite nur diese
            $settings = [];
            // Launch-Datum und Freitext
            $settings['launch_date'] = $_POST['launch_date'] ?? '';
            if (empty($settings['launch_date']) && isset($_POST['launch_date_text'])) {
                $settings['launch_date_text'] = $_POST['launch_date_text'];
            } else {
                $settings['launch_date_text'] = '';
            }
            // Unternehmensdaten (nur wenn diese Felder vorhanden und nicht leer sind)
            if (isset($_POST['company_name']) && !empty($_POST['company_name'])) {
                $settings['company_name'] = $_POST['company_name'];
                $settings['company_phone'] = $_POST['company_phone'] ?? '';
                $settings['company_email'] = $_POST['company_email'] ?? '';
                $settings['company_address'] = $_POST['company_address'] ?? '';
            }
            // WhatsApp-Einstellungen (nur wenn diese Felder vorhanden sind)
            if (isset($_POST['whatsapp_phone_number']) && !empty($_POST['whatsapp_phone_number'])) {
                $settings['whatsapp_notifications_enabled'] = isset($_POST['whatsapp_notifications_enabled']) ? '1' : '0';
                $settings['whatsapp_api_key'] = $_POST['whatsapp_api_key'] ?? '';
                $settings['whatsapp_phone_number'] = $_POST['whatsapp_phone_number'] ?? '';
                $settings['whatsapp_business_name'] = $_POST['whatsapp_business_name'] ?? '';
                $settings['whatsapp_template_id'] = $_POST['whatsapp_template_id'] ?? '';
                $settings['whatsapp_webhook_url'] = $_POST['whatsapp_webhook_url'] ?? '';
                $settings['whatsapp_test_mode'] = isset($_POST['whatsapp_test_mode']) ? '1' : '0';
            }
            // System-Einstellungen (nur wenn diese Felder vorhanden sind)
            if (isset($_POST['app_name']) && !empty($_POST['app_name'])) {
                $settings['app_name'] = $_POST['app_name'];
                $settings['timezone'] = $_POST['timezone'] ?? 'Europe/Berlin';
                $settings['date_format'] = $_POST['date_format'] ?? 'd.m.Y';
                $settings['time_format'] = $_POST['time_format'] ?? 'H:i';
                $settings['language'] = $_POST['language'] ?? 'de';
                $settings['max_appointments_per_slot'] = $_POST['max_appointments_per_slot'] ?? '5';
                $settings['booking_advance_days'] = $_POST['booking_advance_days'] ?? '30';
                $settings['cancellation_hours'] = $_POST['cancellation_hours'] ?? '24';
                // Wartungsmodus-Checkbox
                $settings['maintenance_mode'] = isset($_POST['maintenance_mode']) ? '1' : '0';
            }
            
            // Speichere nur die gesendeten Einstellungen
            foreach ($settings as $key => $value) {
                $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
                $stmt->execute([$key, $value, $value]);
            }
            
            // Handle logo upload (nur wenn Unternehmensdaten gespeichert werden)
            if (isset($_POST['company_name']) && !empty($_POST['company_name']) && isset($_FILES['company_logo']) && $_FILES['company_logo']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = '../assets/images/';
                $fileName = 'company-logo-' . time() . '.' . pathinfo($_FILES['company_logo']['name'], PATHINFO_EXTENSION);
                $uploadPath = $uploadDir . $fileName;
                
                if (move_uploaded_file($_FILES['company_logo']['tmp_name'], $uploadPath)) {
                    $stmt = $conn->prepare("INSERT INTO settings (setting_key, setting_value) VALUES ('company_logo', ?) ON DUPLICATE KEY UPDATE setting_value = ?");
                    $stmt->execute([$uploadPath, $uploadPath]);
                }
            }
            
            $_SESSION['success_message'] = 'Einstellungen erfolgreich gespeichert';
            header('Location: settings.php');
            exit;
        }
    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Fehler beim Speichern: ' . $e->getMessage();
    }
}

// Get current settings
$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings");
$stmt->execute();
$results = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

$defaults = [
    'price_wheel_change' => '25.00',
    'price_storage' => '15.00',
    'price_spring_check' => '35.00',
    'price_winter_check' => '35.00',
    'company_name' => 'Volvo Autohaus',
    'company_logo' => 'assets/images/volvo-logo.png',
    'company_phone' => '+49 123 456789',
    'company_email' => 'info@volvo-autohaus.de',
    'company_address' => 'Musterstraße 123, 12345 Musterstadt',
    'app_name' => 'Boxenstop',
    'timezone' => 'Europe/Berlin',
    'date_format' => 'd.m.Y',
    'time_format' => 'H:i',
    'language' => 'de',
    'max_appointments_per_slot' => '5',
    'booking_advance_days' => '30',
    'cancellation_hours' => '24',
    // WhatsApp-Einstellungen
    'whatsapp_notifications_enabled' => '0',
    'whatsapp_api_key' => '',
    'whatsapp_phone_number' => '+49123456789',
    'whatsapp_business_name' => 'Boxenstop Volvo Service',
    'whatsapp_template_id' => '',
    'whatsapp_webhook_url' => '',
    'whatsapp_test_mode' => '1'
];

$settings = array_merge($defaults, $results);

// Autohäuser laden
$stmt = $conn->prepare("SELECT * FROM autohaus ORDER BY name ASC");
$stmt->execute();
$autohauser = $stmt->fetchAll();

// Services pro Autohaus laden
$autohaus_services = [];
foreach ($autohauser as $autohaus) {
    $stmt = $conn->prepare("
        SELECT * FROM autohaus_services 
        WHERE autohaus_id = ? AND is_active = 1 
        ORDER BY sort_order ASC, service_name ASC
    ");
    $stmt->execute([$autohaus['id']]);
    $autohaus_services[$autohaus['id']] = $stmt->fetchAll();
}

// Services laden
$stmt = $conn->prepare("
    SELECT s.*, a.name as autohaus_name, a.short_name as autohaus_short_name 
    FROM autohaus_services s 
    LEFT JOIN autohaus a ON s.autohaus_id = a.id 
    ORDER BY a.name, s.sort_order, s.service_name
");
$stmt->execute();
$services = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Debug: Prüfe ob Services geladen wurden
error_log("Services loaded: " . count($services));

?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Einstellungen - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/global-dark-theme.css">
    <link rel="stylesheet" href="assets/css/unified-sidebar.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="admin-body" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">
            <!-- Settings Header with Hero Section -->
            <header class="settings-header">
                <div class="settings-hero">
                    <div class="settings-hero-content">
                        <div class="settings-title-section">
                            <h1 class="settings-title">
                                <i class="fas fa-cog"></i> Einstellungen
                            </h1>
                            <p class="settings-subtitle">Systemkonfiguration und Verwaltung</p>
                            <div class="settings-meta">
                                <div class="meta-item">
                                    <i class="fas fa-building"></i>
                                    <span>Autohaus-Verwaltung</span>
                                </div>
                                <div class="meta-item">
                                    <i class="fas fa-cogs"></i>
                                    <span>Service-Konfiguration</span>
                                </div>
                                <div class="meta-item">
                                    <i class="fas fa-globe"></i>
                                    <span>System-Einstellungen</span>
                                </div>
                            </div>
                        </div>
                        <div class="settings-actions">
                            <button class="btn-primary" onclick="showAddAutohausForm()">
                                <i class="fas fa-plus"></i>
                                Neues Autohaus
                            </button>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Settings Content -->
            <div class="settings-content">
                <?php if (isset($_SESSION['success_message'])): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($_SESSION['error_message'])): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
                    </div>
                <?php endif; ?>

                <!-- Modern Tab Navigation -->
                <div class="modern-tabs">
                    <div class="tab-nav">
                        <button type="button" class="tab-nav-button active" data-tab="autohaus-section">
                            <div class="tab-icon">
                                <i class="fas fa-building"></i>
                            </div>
                            <div class="tab-content">
                                <span class="tab-title">Autohaus-Verwaltung</span>
                                <span class="tab-subtitle">Autohäuser und Standorte</span>
                            </div>
                        </button>
                        <button type="button" class="tab-nav-button" data-tab="services-section">
                            <div class="tab-icon">
                                <i class="fas fa-cogs"></i>
                            </div>
                            <div class="tab-content">
                                <span class="tab-title">Service-Verwaltung</span>
                                <span class="tab-subtitle">Services und Preise</span>
                            </div>
                        </button>
                        <button type="button" class="tab-nav-button" data-tab="company-section">
                            <div class="tab-icon">
                                <i class="fas fa-building"></i>
                            </div>
                            <div class="tab-content">
                                <span class="tab-title">Unternehmensdaten</span>
                                <span class="tab-subtitle">Firmeninformationen</span>
                            </div>
                        </button>
                        <button type="button" class="tab-nav-button" data-tab="whatsapp-section">
                            <div class="tab-icon">
                                <i class="fab fa-whatsapp"></i>
                            </div>
                            <div class="tab-content">
                                <span class="tab-title">WhatsApp</span>
                                <span class="tab-subtitle">HelloMateo Integration</span>
                            </div>
                        </button>
                        <button type="button" class="tab-nav-button" data-tab="teams-section">
                            <div class="tab-icon">
                                <i class="fab fa-microsoft"></i>
                            </div>
                            <div class="tab-content">
                                <span class="tab-title">Microsoft Teams</span>
                                <span class="tab-subtitle">Benachrichtigungen</span>
                            </div>
                        </button>
                        <button type="button" class="tab-nav-button" data-tab="system-section">
                            <div class="tab-icon">
                                <i class="fas fa-cogs"></i>
                            </div>
                            <div class="tab-content">
                                <span class="tab-title">System</span>
                                <span class="tab-subtitle">Allgemeine Einstellungen</span>
                            </div>
                        </button>
                    </div>
                </div>

                <!-- Unternehmensdaten -->
                <form method="POST" enctype="multipart/form-data" id="company-form" class="settings-form" novalidate>
                    <input type="hidden" name="action" value="update_settings">
                    <div class="settings-section" id="company-section">
                        <div class="section-header">
                            <h2><i class="fas fa-building"></i> Unternehmensdaten</h2>
                            <button type="button" onclick="submitCompanySettings()" class="btn btn-primary">
                                <i class="fas fa-save"></i> Speichern
                            </button>
                        </div>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="launch_date">Launch-Datum (für Onlinewelt/ICS)</label>
                                <input type="date" id="launch_date" name="launch_date" value="<?php echo htmlspecialchars($settings['launch_date'] ?? ''); ?>">
                                <small class="text-muted">Wird für den Kalendereintrag auf der Vorschaltseite verwendet.</small>
                            </div>
                            <?php if (empty($settings['launch_date'])): ?>
                            <div class="form-group" style="background:#e6f7e6;border:2px solid #25d366;padding:1em;border-radius:8px;">
                                <label for="launch_date_text" style="color:#25d366;font-weight:700;">Alternativ-Text für Launch (wenn kein Datum):</label>
                                <input type="text" id="launch_date_text" name="launch_date_text" value="<?php echo htmlspecialchars($settings['launch_date_text'] ?? ''); ?>" placeholder="z.B. Frühjahr 2026, Sommer, Bald verfügbar ..." style="border:1.5px solid #25d366;">
                                <small class="text-muted">Dieser Text erscheint auf der Vorschaltseite und im Kalender, wenn kein Datum gesetzt ist. <br>Beispiel: <b>Frühjahr 2026</b> oder <b>Bald verfügbar</b></small>
                            </div>
                            <?php endif; ?>
                            <div class="form-group">
                                <label for="company_name">Firmenname</label>
                                <input type="text" id="company_name" name="company_name" 
                                       value="<?php echo htmlspecialchars($settings['company_name']); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="company_phone">Telefon</label>
                                <input type="tel" id="company_phone" name="company_phone" 
                                       value="<?php echo htmlspecialchars($settings['company_phone']); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="company_email">E-Mail</label>
                                <input type="email" id="company_email" name="company_email" 
                                       value="<?php echo htmlspecialchars($settings['company_email']); ?>" required>
                            </div>
                            
                            <div class="form-group full-width">
                                <label for="company_address">Adresse</label>
                                <textarea id="company_address" name="company_address" rows="3" required><?php echo htmlspecialchars($settings['company_address']); ?></textarea>
                            </div>
                            
                            <div class="form-group">
                                <label for="company_logo">Firmenlogo</label>
                                <input type="file" id="company_logo" name="company_logo" accept="image/*">
                                <?php if ($settings['company_logo']): ?>
                                    <div class="current-logo">
                                        <img src="../<?php echo htmlspecialchars($settings['company_logo']); ?>" alt="Aktuelles Logo" style="max-width: 200px; margin-top: 10px;">
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </form>

                    <!-- WhatsApp-Einstellungen -->
                <form method="POST" enctype="multipart/form-data" id="whatsapp-form" class="settings-form" novalidate>
                    <input type="hidden" name="action" value="update_settings">
                    <div class="settings-section" id="whatsapp-section">
                        <div class="section-header">
                            <h2><i class="fab fa-whatsapp"></i> WhatsApp-Einstellungen (HelloMateo)</h2>
                            <button type="button" onclick="submitWhatsAppSettings()" class="btn btn-primary">
                                <i class="fas fa-save"></i> Speichern
                            </button>
                        </div>
                        
                        <!-- Globale WhatsApp-Einstellungen -->
                        <div class="whatsapp-global-settings">
                            <h3><i class="fas fa-globe"></i> Globale WhatsApp-Einstellungen</h3>
                            <p class="text-muted">Diese Einstellungen gelten für alle Autohäuser, sofern keine spezifischen Einstellungen konfiguriert sind.</p>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="whatsapp_notifications_enabled">WhatsApp-Benachrichtigungen aktivieren</label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="whatsapp_notifications_enabled" value="1" 
                                           <?php echo ($settings['whatsapp_notifications_enabled'] ?? '0') === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-text">WhatsApp-Benachrichtigungen für Termine senden</span>
                                </label>
                            </div>
                            
                            <div class="form-group">
                                <label for="whatsapp_api_key">HelloMateo API-Key</label>
                                <input type="password" id="whatsapp_api_key" name="whatsapp_api_key" 
                                       value="<?php echo htmlspecialchars($settings['whatsapp_api_key'] ?? ''); ?>" 
                                       placeholder="Ihr HelloMateo API-Key (optional)">
                                <small class="text-muted">Holen Sie sich Ihren API-Key von <a href="https://hellomateo.com" target="_blank">hellomateo.com</a> (kann leer bleiben)</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="whatsapp_phone_number">WhatsApp-Nummer *</label>
                                <input type="tel" id="whatsapp_phone_number" name="whatsapp_phone_number" 
                                       value="<?php echo htmlspecialchars($settings['whatsapp_phone_number'] ?? '+49123456789'); ?>" 
                                       placeholder="+49123456789" required>
                                <small class="text-muted">Ihre WhatsApp-Geschäftsnummer (mit Ländervorwahl)</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="whatsapp_business_name">Geschäftsname *</label>
                                <input type="text" id="whatsapp_business_name" name="whatsapp_business_name" 
                                       value="<?php echo htmlspecialchars($settings['whatsapp_business_name'] ?? 'Boxenstop Volvo Service'); ?>" 
                                       placeholder="Ihr Firmenname" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="whatsapp_template_id">Nachrichten-Template ID</label>
                                <input type="text" id="whatsapp_template_id" name="whatsapp_template_id" 
                                       value="<?php echo htmlspecialchars($settings['whatsapp_template_id'] ?? ''); ?>" 
                                       placeholder="Template ID (optional)">
                                <small class="text-muted">Optional: Verwenden Sie ein vordefiniertes Template</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="whatsapp_webhook_url">Webhook-URL</label>
                                <input type="url" id="whatsapp_webhook_url" name="whatsapp_webhook_url" 
                                       value="<?php echo htmlspecialchars($settings['whatsapp_webhook_url'] ?? ''); ?>" 
                                       placeholder="https://ihre-domain.de/webhook/whatsapp">
                                <small class="text-muted">Optional: Für eingehende Nachrichten</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="whatsapp_test_mode">Test-Modus</label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="whatsapp_test_mode" value="1" 
                                           <?php echo ($settings['whatsapp_test_mode'] ?? '1') === '1' ? 'checked' : ''; ?>>
                                    <span class="checkbox-text">Test-Modus aktivieren (keine echten Nachrichten)</span>
                                </label>
                            </div>
                            
                            <div class="form-group full-width">
                                <button type="button" class="btn btn-success" onclick="testWhatsApp()">
                                    <i class="fab fa-whatsapp"></i>
                                    WhatsApp-Nachricht testen
                                </button>
                                <button type="button" class="btn btn-info" onclick="testWhatsAppTemplate()" style="margin-left: 10px;">
                                    <i class="fas fa-file-alt"></i>
                                    Template testen
                                </button>
                            </div>
                        </div>
                        </div>
                        
                        <!-- Autohaus-spezifische WhatsApp-Einstellungen -->
                        <div class="whatsapp-autohaus-settings">
                            <h3><i class="fas fa-building"></i> Autohaus-spezifische WhatsApp-Einstellungen</h3>
                            <p class="text-muted">Konfigurieren Sie individuelle WhatsApp-Einstellungen für jedes Autohaus.</p>
                            
                            <?php foreach ($autohauser as $autohaus): ?>
                            <div class="autohaus-whatsapp-card">
                                <div class="autohaus-header">
                                    <h4><i class="fas fa-building"></i> <?php echo htmlspecialchars($autohaus['name']); ?></h4>
                                    <div class="whatsapp-status">
                                        <?php if ($autohaus['whatsapp_notifications_enabled'] ?? false): ?>
                                            <span class="status-badge status-active">
                                                <i class="fas fa-check-circle"></i> Aktiv
                                            </span>
                                        <?php else: ?>
                                            <span class="status-badge status-inactive">
                                                <i class="fas fa-times-circle"></i> Inaktiv
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <form method="POST" class="autohaus-whatsapp-form" data-autohaus-id="<?php echo $autohaus['id']; ?>">
                                    <input type="hidden" name="action" value="update_autohaus_whatsapp">
                                    <input type="hidden" name="autohaus_id" value="<?php echo $autohaus['id']; ?>">
                                    
                                    <div class="form-grid">
                                        <div class="form-group checkbox-group">
                                            <label class="checkbox-label">
                                                <input type="checkbox" name="whatsapp_notifications_enabled" value="1" 
                                                       <?php echo ($autohaus['whatsapp_notifications_enabled'] ?? false) ? 'checked' : ''; ?>>
                                                <span class="checkbox-text">WhatsApp-Benachrichtigungen für dieses Autohaus aktivieren</span>
                                            </label>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="whatsapp_api_key_<?php echo $autohaus['id']; ?>">HelloMateo API-Key</label>
                                            <input type="password" id="whatsapp_api_key_<?php echo $autohaus['id']; ?>" 
                                                   name="whatsapp_api_key" 
                                                   value="<?php echo htmlspecialchars($autohaus['whatsapp_api_key'] ?? ''); ?>" 
                                                   placeholder="Ihr HelloMateo API-Key">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="whatsapp_phone_number_<?php echo $autohaus['id']; ?>">WhatsApp-Nummer</label>
                                            <input type="tel" id="whatsapp_phone_number_<?php echo $autohaus['id']; ?>" 
                                                   name="whatsapp_phone_number" 
                                                   value="<?php echo htmlspecialchars($autohaus['whatsapp_phone_number'] ?? ''); ?>" 
                                                   placeholder="+49123456789">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="whatsapp_business_name_<?php echo $autohaus['id']; ?>">Geschäftsname</label>
                                            <input type="text" id="whatsapp_business_name_<?php echo $autohaus['id']; ?>" 
                                                   name="whatsapp_business_name" 
                                                   value="<?php echo htmlspecialchars($autohaus['whatsapp_business_name'] ?? $autohaus['name']); ?>" 
                                                   placeholder="Autohaus Name">
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="whatsapp_template_id_<?php echo $autohaus['id']; ?>">Template ID</label>
                                            <input type="text" id="whatsapp_template_id_<?php echo $autohaus['id']; ?>" 
                                                   name="whatsapp_template_id" 
                                                   value="<?php echo htmlspecialchars($autohaus['whatsapp_template_id'] ?? ''); ?>" 
                                                   placeholder="Template ID (optional)">
                                        </div>
                                        
                                        <div class="form-group checkbox-group">
                                            <label class="checkbox-label">
                                                <input type="checkbox" name="whatsapp_test_mode" value="1" 
                                                       <?php echo ($autohaus['whatsapp_test_mode'] ?? true) ? 'checked' : ''; ?>>
                                                <span class="checkbox-text">Test-Modus aktivieren</span>
                                            </label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-actions">
                                        <button type="button" class="btn btn-primary" onclick="saveAutohausWhatsAppSettings(<?php echo $autohaus['id']; ?>)">
                                            <i class="fas fa-save"></i> WhatsApp-Einstellungen speichern
                                        </button>
                                        <button type="button" class="btn btn-success" onclick="testAutohausWhatsApp(<?php echo $autohaus['id']; ?>)">
                                            <i class="fab fa-whatsapp"></i> WhatsApp-Nachricht testen
                                        </button>
                                    </div>
                                </form>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </form>

                <!-- Autohaus-Verwaltung -->
                <div class="settings-section" id="autohaus-section">
                        <h2><i class="fas fa-building"></i> Autohaus-Verwaltung</h2>
                        
                        <!-- Autohaus-Liste -->
                        <div class="autohaus-list">
                            <div class="list-header">
                                <h3>Verfügbare Autohäuser</h3>
                                <button type="button" class="btn btn-primary" onclick="showAddAutohausForm()">
                                    <i class="fas fa-plus"></i> Neues Autohaus
                                </button>
                            </div>
                            
                            <div class="autohaus-grid">
                                <?php foreach ($autohauser as $autohaus): ?>
                                    <div class="autohaus-card">
                                        <div class="autohaus-header">
                                            <h4><?php echo htmlspecialchars($autohaus['name']); ?></h4>
                                            <div class="autohaus-actions">
                                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="editAutohaus(<?php echo htmlspecialchars(json_encode($autohaus)); ?>)">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-outline-danger" onclick="deleteAutohaus(<?php echo $autohaus['id']; ?>, '<?php echo htmlspecialchars($autohaus['name']); ?>')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                        
                                        <div class="autohaus-info">
                                            <div class="info-item">
                                                <i class="fas fa-map-marker-alt"></i>
                                                <span><?php echo htmlspecialchars($autohaus['address']); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <i class="fas fa-phone"></i>
                                                <span><?php echo htmlspecialchars($autohaus['phone']); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <i class="fas fa-envelope"></i>
                                                <span><?php echo htmlspecialchars($autohaus['email']); ?></span>
                                            </div>
                                            <div class="info-item">
                                                <i class="fas fa-key"></i>
                                                <span>RESY: <?php echo htmlspecialchars($autohaus['resy_dealer_id']); ?></span>
                                            </div>
                                        </div>
                                        
                                        <!-- Preise anzeigen -->
                                        <div class="autohaus-prices">
                                            <h5><i class="fas fa-euro-sign"></i> Standard-Preise</h5>
                                            <div class="price-grid">
                                                <div class="price-item">
                                                    <span class="price-label">Reifenwechsel:</span>
                                                    <span class="price-value"><?php echo number_format($autohaus['price_wheel_change'] ?? 0, 2); ?> €</span>
                                                </div>
                                                <div class="price-item">
                                                    <span class="price-label">Einlagerung:</span>
                                                    <span class="price-value"><?php echo number_format($autohaus['price_storage'] ?? 0, 2); ?> €</span>
                                                </div>
                                                <div class="price-item">
                                                    <span class="price-label">Frühjahrscheck:</span>
                                                    <span class="price-value"><?php echo number_format($autohaus['price_spring_check'] ?? 0, 2); ?> €</span>
                                                </div>
                                                <div class="price-item">
                                                    <span class="price-label">Wintercheck:</span>
                                                    <span class="price-value"><?php echo number_format($autohaus['price_winter_check'] ?? 0, 2); ?> €</span>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <!-- Services anzeigen -->
                                        <?php if (!empty($autohaus_services[$autohaus['id']])): ?>
                                        <div class="autohaus-services">
                                            <h5><i class="fas fa-cogs"></i> Zusätzliche Services</h5>
                                            <div class="services-grid">
                                                <?php foreach ($autohaus_services[$autohaus['id']] as $service): ?>
                                                <div class="service-item">
                                                    <div class="service-info">
                                                        <span class="service-name"><?php echo htmlspecialchars($service['service_name']); ?></span>
                                                        <?php if ($service['service_description']): ?>
                                                            <span class="service-description"><?php echo htmlspecialchars($service['service_description']); ?></span>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="service-price">
                                                        <?php if ($service['show_price'] && $service['price']): ?>
                                                            <span class="price-value"><?php echo number_format($service['price'], 2); ?> €</span>
                                                        <?php else: ?>
                                                            <span class="price-value price-on-request">auf Anfrage</span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                                <?php endforeach; ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <div class="autohaus-status">
                                            <span class="status-badge <?php echo $autohaus['is_active'] ? 'status-active' : 'status-inactive'; ?>">
                                                <?php echo $autohaus['is_active'] ? 'Aktiv' : 'Inaktiv'; ?>
                                            </span>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                    </div>


                <!-- Service-Verwaltung -->
                <div class="settings-section" id="services-section">
                    <div class="section-header">
                        <h2><i class="fas fa-cogs"></i> Service-Verwaltung</h2>
                        <p>Verwalten Sie zusätzliche Services für jedes Autohaus</p>
                        <button type="button" class="btn btn-primary" onclick="showAddServiceForm()">
                            <i class="fas fa-plus"></i> Neuer Service
                        </button>
                    </div>

                    <!-- Services-Liste -->
                    <div class="services-list">
                        <?php if (empty($services)): ?>
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle"></i>
                                Keine Services gefunden. <a href="#" onclick="showAddServiceForm()">Ersten Service hinzufügen</a>
                            </div>
                        <?php else: ?>
                        <?php foreach ($services as $service): ?>
                            <div class="service-card" data-service-id="<?php echo $service['id']; ?>">
                                <div class="service-info">
                                    <h3><?php echo htmlspecialchars($service['service_name']); ?></h3>
                                    <p class="service-description"><?php echo htmlspecialchars($service['service_description']); ?></p>
                                    <div class="service-details">
                                        <span class="autohaus-badge"><?php echo htmlspecialchars($service['autohaus_name']); ?></span>
                                        <span class="service-price">
                                            <?php if ($service['show_price'] && $service['price']): ?>
                                                <?php echo number_format($service['price'], 2); ?> €
                                            <?php else: ?>
                                                auf Anfrage
                                            <?php endif; ?>
                                        </span>
                                        <span class="service-status <?php echo $service['is_active'] ? 'active' : 'inactive'; ?>">
                                            <?php echo $service['is_active'] ? 'Aktiv' : 'Inaktiv'; ?>
                                        </span>
                                    </div>
                                </div>
                                <div class="service-actions">
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="editService(<?php echo htmlspecialchars(json_encode($service)); ?>)">
                                        <i class="fas fa-edit"></i> Bearbeiten
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="deleteService(<?php echo $service['id']; ?>)">
                                        <i class="fas fa-trash"></i> Löschen
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </div>

                    <!-- Service-Formular (versteckt) -->
                    <div id="service-form" class="service-form" style="display: none;">
                        <h3 id="service-form-title">Neuen Service hinzufügen</h3>
                        
                        <form method="POST" id="service-form-element">
                            <div id="service-form-content">
                                <input type="hidden" name="action" id="service-form-action" value="add_service">
                                <input type="hidden" name="service_id" id="service-form-id" value="">
                            
                            <div class="form-grid">
                                <div class="form-group">
                                    <label for="service_autohaus_id">Autohaus *</label>
                                    <select id="service_autohaus_id" name="service_autohaus_id" data-required="true">
                                        <option value="">Autohaus auswählen</option>
                                        <?php foreach ($autohauser as $autohaus): ?>
                                            <option value="<?php echo $autohaus['id']; ?>"><?php echo htmlspecialchars($autohaus['name']); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label for="service_name">Service-Name *</label>
                                    <input type="text" id="service_name" name="service_name" data-required="true">
                                </div>
                                
                                <div class="form-group full-width">
                                    <label for="service_description">Beschreibung</label>
                                    <textarea id="service_description" name="service_description" rows="3"></textarea>
                                </div>
                                
                                <div class="form-group">
                                    <label for="service_price">Preis (€)</label>
                                    <input type="number" id="service_price" name="service_price" step="0.01" min="0">
                                    <small class="text-muted">Leer lassen für "auf Anfrage"</small>
                                </div>
                                
                                <div class="form-group">
                                    <label for="service_sort_order">Reihenfolge</label>
                                    <input type="number" id="service_sort_order" name="service_sort_order" min="0" value="0">
                                </div>
                                
                                <div class="form-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="service_show_price" name="service_show_price" checked>
                                        <span class="checkbox-text">Preis anzeigen</span>
                                    </label>
                                </div>
                                
                                <div class="form-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="service_is_active" name="service_is_active" checked>
                                        <span class="checkbox-text">Service aktiv</span>
                                    </label>
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <button type="button" class="btn btn-primary" onclick="submitServiceForm()">
                                    <i class="fas fa-save"></i> Speichern
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="hideServiceForm()">
                                    <i class="fas fa-times"></i> Abbrechen
                                </button>
                            </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Microsoft Teams-Einstellungen -->
                <div class="settings-section" id="teams-section">
                    <div class="section-header">
                        <h2><i class="fab fa-microsoft"></i> Microsoft Teams Integration</h2>
                        <p>Konfigurieren Sie Teams-Benachrichtigungen für angenommene Angebote</p>
                    </div>
                    
                    <!-- Teams-Konfiguration pro Autohaus -->
                    <div class="teams-configuration">
                        <?php foreach ($autohauser as $autohaus): ?>
                        <div class="teams-autohaus-card">
                            <div class="autohaus-header">
                                <h3><i class="fas fa-building"></i> <?php echo htmlspecialchars($autohaus['name']); ?></h3>
                                <div class="teams-status">
                                    <?php if ($autohaus['teams_notifications_enabled']): ?>
                                        <span class="status-badge status-active">
                                            <i class="fas fa-check-circle"></i> Aktiv
                                        </span>
                                    <?php else: ?>
                                        <span class="status-badge status-inactive">
                                            <i class="fas fa-times-circle"></i> Inaktiv
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <form method="POST" class="teams-form" data-autohaus-id="<?php echo $autohaus['id']; ?>">
                                <input type="hidden" name="action" value="update_teams_settings">
                                <input type="hidden" name="autohaus_id" value="<?php echo $autohaus['id']; ?>">
                                
                                <div class="form-grid">
                                    <div class="form-group checkbox-group">
                                        <label class="checkbox-label">
                                            <input type="checkbox" name="teams_notifications_enabled" value="1" 
                                                   <?php echo $autohaus['teams_notifications_enabled'] ? 'checked' : ''; ?>>
                                            <span class="checkbox-text">Teams-Benachrichtigungen aktivieren</span>
                                        </label>
                                    </div>
                                    
                                    <div class="form-group full-width">
                                        <label for="teams_webhook_url_<?php echo $autohaus['id']; ?>">Teams Webhook URL *</label>
                                        <input type="url" id="teams_webhook_url_<?php echo $autohaus['id']; ?>" 
                                               name="teams_webhook_url" 
                                               value="<?php echo htmlspecialchars($autohaus['teams_webhook_url'] ?? ''); ?>" 
                                               placeholder="https://outlook.office.com/webhook/...">
                                        <small class="text-muted">
                                            <i class="fas fa-info-circle"></i> 
                                            Webhook-URL aus Microsoft Teams Channel Connector kopieren
                                        </small>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label for="teams_channel_name_<?php echo $autohaus['id']; ?>">Channel-Name</label>
                                        <input type="text" id="teams_channel_name_<?php echo $autohaus['id']; ?>" 
                                               name="teams_channel_name" 
                                               value="<?php echo htmlspecialchars($autohaus['teams_channel_name'] ?? ''); ?>" 
                                               placeholder="z.B. Angebote, Service, etc.">
                                    </div>
                                    
                                    <div class="form-group">
                                        <label for="teams_mention_users_<?php echo $autohaus['id']; ?>">Benutzer erwähnen</label>
                                        <input type="text" id="teams_mention_users_<?php echo $autohaus['id']; ?>" 
                                               name="teams_mention_users" 
                                               value="<?php echo htmlspecialchars($autohaus['teams_mention_users'] ?? ''); ?>" 
                                               placeholder="z.B. @Max Mustermann, @Anna Schmidt">
                                        <small class="text-muted">
                                            <i class="fas fa-info-circle"></i> 
                                            Komma-getrennte Liste von Benutzern, die erwähnt werden sollen
                                        </small>
                                    </div>
                                </div>
                                
                                <div class="form-actions">
                                    <button type="button" class="btn btn-primary" onclick="saveTeamsSettings(<?php echo $autohaus['id']; ?>)">
                                        <i class="fas fa-save"></i> Teams-Einstellungen speichern
                                    </button>
                                    <button type="button" class="btn btn-success" onclick="testTeamsNotification(<?php echo $autohaus['id']; ?>)">
                                        <i class="fab fa-microsoft"></i> Teams-Nachricht testen
                                    </button>
                                </div>
                            </form>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <!-- Teams-Hilfe -->
                    <div class="teams-help">
                        <h4><i class="fas fa-question-circle"></i> Teams-Integration einrichten</h4>
                        <div class="help-steps">
                            <div class="help-step">
                                <span class="step-number">1</span>
                                <div class="step-content">
                                    <h5>Webhook in Teams erstellen</h5>
                                    <p>Gehen Sie zu Ihrem Teams-Channel → Einstellungen → Connectors → Incoming Webhook</p>
                                </div>
                            </div>
                            <div class="help-step">
                                <span class="step-number">2</span>
                                <div class="step-content">
                                    <h5>Webhook-URL kopieren</h5>
                                    <p>Kopieren Sie die generierte Webhook-URL und fügen Sie sie oben ein</p>
                                </div>
                            </div>
                            <div class="help-step">
                                <span class="step-number">3</span>
                                <div class="step-content">
                                    <h5>Testen</h5>
                                    <p>Klicken Sie auf "Teams-Nachricht testen" um die Integration zu prüfen</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- System-Einstellungen -->
                <form method="POST" enctype="multipart/form-data" id="system-form" class="settings-form" novalidate>
                    <input type="hidden" name="action" value="update_settings">
                    <div class="settings-section" id="system-section">
                        <div class="section-header">
                            <h2><i class="fas fa-cogs"></i> System-Einstellungen</h2>
                            <button type="button" onclick="submitSystemSettings()" class="btn btn-primary">
                                <i class="fas fa-save"></i> Speichern
                            </button>
                        </div>
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="app_name">Anwendungsname</label>
                                <input type="text" id="app_name" name="app_name" 
                                       value="<?php echo htmlspecialchars($settings['app_name'] ?? 'Boxenstop'); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="system_timezone">Zeitzone</label>
                                <select id="system_timezone" name="timezone">
                                    <option value="Europe/Berlin" <?php echo ($settings['timezone'] ?? 'Europe/Berlin') === 'Europe/Berlin' ? 'selected' : ''; ?>>Europa/Berlin (MEZ/MESZ)</option>
                                    <option value="Europe/Vienna" <?php echo ($settings['timezone'] ?? 'Europe/Berlin') === 'Europe/Vienna' ? 'selected' : ''; ?>>Europa/Wien</option>
                                    <option value="Europe/Zurich" <?php echo ($settings['timezone'] ?? 'Europe/Berlin') === 'Europe/Zurich' ? 'selected' : ''; ?>>Europa/Zürich</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="date_format">Datumsformat</label>
                                <select id="date_format" name="date_format">
                                    <option value="d.m.Y" <?php echo ($settings['date_format'] ?? 'd.m.Y') === 'd.m.Y' ? 'selected' : ''; ?>>DD.MM.YYYY</option>
                                    <option value="Y-m-d" <?php echo ($settings['date_format'] ?? 'd.m.Y') === 'Y-m-d' ? 'selected' : ''; ?>>YYYY-MM-DD</option>
                                    <option value="m/d/Y" <?php echo ($settings['date_format'] ?? 'd.m.Y') === 'm/d/Y' ? 'selected' : ''; ?>>MM/DD/YYYY</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="time_format">Zeitformat</label>
                                <select id="time_format" name="time_format">
                                    <option value="H:i" <?php echo ($settings['time_format'] ?? 'H:i') === 'H:i' ? 'selected' : ''; ?>>24-Stunden (HH:MM)</option>
                                    <option value="h:i A" <?php echo ($settings['time_format'] ?? 'H:i') === 'h:i A' ? 'selected' : ''; ?>>12-Stunden (HH:MM AM/PM)</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="language">Sprache</label>
                                <select id="language" name="language">
                                    <option value="de" <?php echo ($settings['language'] ?? 'de') === 'de' ? 'selected' : ''; ?>>Deutsch</option>
                                    <option value="en" <?php echo ($settings['language'] ?? 'de') === 'en' ? 'selected' : ''; ?>>English</option>
                                    <option value="fr" <?php echo ($settings['language'] ?? 'de') === 'fr' ? 'selected' : ''; ?>>Français</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="max_appointments_per_slot">Max. Termine pro Zeitslot</label>
                                <input type="number" id="max_appointments_per_slot" name="max_appointments_per_slot" 
                                       value="<?php echo htmlspecialchars($settings['max_appointments_per_slot'] ?? '5'); ?>" 
                                       min="1" max="20" required>
                            </div>
                            <div class="form-group checkbox-group">
                                <label class="checkbox-label">
                                    <input type="checkbox" id="maintenance_mode" name="maintenance_mode" value="1" <?php echo (!empty($settings['maintenance_mode']) && $settings['maintenance_mode'] == '1') ? 'checked' : ''; ?>>
                                <span class="checkbox-text"><i class="fas fa-tools"></i> Wartungsmodus aktivieren (Kundenseite offline)</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label for="launch_date">Launch-Datum (für Onlinewelt/ICS)</label>
                            <input type="date" id="launch_date" name="launch_date" value="<?php echo htmlspecialchars($settings['launch_date'] ?? ''); ?>">
                            <small class="text-muted">Wird für den Kalendereintrag auf der Vorschaltseite verwendet.</small>
                        </div>
                        <div class="form-group" style="background:#e6f7e6;border:2px solid #25d366;padding:1em;border-radius:8px;">
                            <label for="launch_date_text" style="color:#25d366;font-weight:700;">Alternativ-Text für Launch (wenn kein Datum):</label>
                            <input type="text" id="launch_date_text" name="launch_date_text" value="<?php echo htmlspecialchars($settings['launch_date_text'] ?? ''); ?>" placeholder="z.B. Frühjahr 2026, Sommer, Bald verfügbar ..." style="border:1.5px solid #25d366;">
                            <small class="text-muted">Dieser Text erscheint auf der Vorschaltseite und im Kalender, wenn kein Datum gesetzt ist. <br>Beispiel: <b>Frühjahr 2026</b> oder <b>Bald verfügbar</b></small>
                        </div>
                            
                            <div class="form-group">
                                <label for="booking_advance_days">Buchungsvorlauf (Tage)</label>
                                <input type="number" id="booking_advance_days" name="booking_advance_days" 
                                       value="<?php echo htmlspecialchars($settings['booking_advance_days'] ?? '30'); ?>" 
                                       min="1" max="365" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="cancellation_hours">Stornierungsfrist (Stunden)</label>
                                <input type="number" id="cancellation_hours" name="cancellation_hours" 
                                       value="<?php echo htmlspecialchars($settings['cancellation_hours'] ?? '24'); ?>" 
                                       min="1" max="168" required>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </main>
    </div>

    <!-- Autohaus Modal -->
    <div id="autohaus-modal" class="modal-overlay" style="display: none;">
        <div class="modal-container">
            <div class="modal-header">
                <h3 id="modal-title">Neues Autohaus hinzufügen</h3>
                <button type="button" class="modal-close" onclick="closeAutohausModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            
            <div class="modal-body">
                <form method="POST" id="autohaus-form-element">
                    <input type="hidden" name="action" id="form-action" value="add_autohaus">
                    <input type="hidden" name="autohaus_id" id="form-autohaus-id" value="">
                
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="autohaus_name">Autohaus-Name *</label>
                            <input type="text" id="autohaus_name" name="autohaus_name" data-required="true">
                        </div>
                        
                        <div class="form-group">
                            <label for="autohaus_short_name">Kurzname *</label>
                            <input type="text" id="autohaus_short_name" name="autohaus_short_name" data-required="true">
                        </div>
                        
                        <div class="form-group">
                            <label for="autohaus_phone">Telefon *</label>
                            <input type="tel" id="autohaus_phone" name="autohaus_phone" data-required="true">
                        </div>
                        
                        <div class="form-group">
                            <label for="autohaus_email">E-Mail *</label>
                            <input type="email" id="autohaus_email" name="autohaus_email" data-required="true">
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="autohaus_address">Adresse *</label>
                            <textarea id="autohaus_address" name="autohaus_address" rows="3" data-required="true"></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="resy_username">RESY Benutzername *</label>
                            <input type="text" id="resy_username" name="resy_username" data-required="true">
                        </div>
                        
                        <div class="form-group">
                            <label for="resy_password">RESY Passwort *</label>
                            <input type="password" id="resy_password" name="resy_password" data-required="true">
                        </div>
                        
                        <div class="form-group">
                            <label for="resy_dealer_id">RESY Dealer-ID *</label>
                            <input type="text" id="resy_dealer_id" name="resy_dealer_id" data-required="true">
                        </div>
                        
                        <div class="form-group">
                            <label for="timezone">Zeitzone</label>
                            <select id="timezone" name="timezone">
                                <option value="Europe/Berlin">Europa/Berlin (MEZ/MESZ)</option>
                                <option value="Europe/Vienna">Europa/Wien</option>
                                <option value="Europe/Zurich">Europa/Zürich</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="working_days">Arbeitstage</label>
                            <input type="text" id="working_days" name="working_days" placeholder="monday,tuesday,wednesday,thursday,friday">
                            <small class="text-muted">Komma-getrennt: monday,tuesday,wednesday,thursday,friday</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="working_start">Arbeitsbeginn</label>
                            <input type="time" id="working_start" name="working_start" value="08:00">
                        </div>
                        
                        <div class="form-group">
                            <label for="working_end">Arbeitsende</label>
                            <input type="time" id="working_end" name="working_end" value="18:00">
                        </div>
                        
                        <div class="form-group full-width">
                            <label class="checkbox-label">
                                <input type="checkbox" id="is_active" name="is_active" value="1" checked>
                                <span class="checkbox-text">Autohaus ist aktiv</span>
                            </label>
                        </div>
                    </div>
                    
                    <!-- Footer-Bild-Einstellungen -->
                    <h4 style="margin-top: 2rem; margin-bottom: 1rem; color: #495057; border-bottom: 2px solid #e9ecef; padding-bottom: 0.5rem;">
                        <i class="fas fa-image"></i> Footer-Bild
                    </h4>
                    
                    <div class="form-grid">
                        <div class="form-group full-width">
                            <label for="footer_image_alt">Alt-Text für Footer-Bild</label>
                            <input type="text" id="footer_image_alt" name="footer_image_alt" placeholder="Autohaus Exterieur">
                            <small class="text-muted">Beschreibung des Bildes für Barrierefreiheit</small>
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="footer_image_input">Footer-Bild hochladen</label>
                            <input type="file" id="footer_image_input" name="footer_image" accept="image/jpeg,image/jpg,image/png,image/webp" style="display: none;" onchange="previewFooterImage(this)">
                            <div style="display: flex; gap: 0.5rem; align-items: center;">
                                <button type="button" class="btn btn-secondary" onclick="document.getElementById('footer_image_input').click()">
                                    <i class="fas fa-upload"></i> Bild auswählen
                                </button>
                                <button type="button" class="btn btn-primary" id="footer_upload_btn" onclick="uploadFooterImage()" style="display: none;">
                                    <i class="fas fa-cloud-upload-alt"></i> Hochladen
                                </button>
                            </div>
                            <div id="footer_image_preview" style="margin-top: 1rem; display: none;">
                                <img id="footer_image_preview_img" src="" alt="Vorschau" style="max-width: 100%; max-height: 300px; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                                <button type="button" class="btn btn-danger btn-sm" onclick="removeFooterImage()" style="margin-top: 0.5rem;">
                                    <i class="fas fa-trash"></i> Bild entfernen
                                </button>
                            </div>
                            <small class="text-muted">Max. 5MB - JPG, PNG oder WEBP</small>
                        </div>
                    </div>
                    
                    <!-- Preiseinstellungen -->
                    <h4 style="margin-top: 2rem; margin-bottom: 1rem; color: #495057; border-bottom: 2px solid #e9ecef; padding-bottom: 0.5rem;">
                        <i class="fas fa-euro-sign"></i> Preiseinstellungen
                    </h4>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="price_wheel_change">Reifenwechsel (€)</label>
                            <input type="number" id="price_wheel_change" name="price_wheel_change" step="0.01" min="0" value="25.00">
                        </div>
                        
                        <div class="form-group">
                            <label for="price_storage">Einlagerung (€)</label>
                            <input type="number" id="price_storage" name="price_storage" step="0.01" min="0" value="15.00">
                        </div>
                        
                        <div class="form-group">
                            <label for="price_spring_check">Frühjahrscheck (€)</label>
                            <input type="number" id="price_spring_check" name="price_spring_check" step="0.01" min="0" value="35.00">
                        </div>
                        
                        <div class="form-group">
                            <label for="price_winter_check">Wintercheck (€)</label>
                            <input type="number" id="price_winter_check" name="price_winter_check" step="0.01" min="0" value="35.00">
                        </div>
                    </div>
                    
                    <!-- E-Mail-Einstellungen -->
                    <h4 style="margin-top: 2rem; margin-bottom: 1rem; color: #495057; border-bottom: 2px solid #e9ecef; padding-bottom: 0.5rem;">
                        <i class="fas fa-envelope"></i> E-Mail-Einstellungen
                    </h4>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="email_smtp_host">SMTP-Server</label>
                            <input type="text" id="email_smtp_host" name="email_smtp_host" placeholder="smtp.gmail.com">
                        </div>
                        
                        <div class="form-group">
                            <label for="email_smtp_port">SMTP-Port</label>
                            <input type="number" id="email_smtp_port" name="email_smtp_port" value="587" min="1" max="65535">
                        </div>
                        
                        <div class="form-group">
                            <label for="email_smtp_username">E-Mail-Benutzername</label>
                            <input type="text" id="email_smtp_username" name="email_smtp_username" placeholder="ihre-email@domain.com">
                        </div>
                        
                        <div class="form-group">
                            <label for="email_smtp_password">E-Mail-Passwort</label>
                            <input type="password" id="email_smtp_password" name="email_smtp_password" placeholder="App-Passwort oder Passwort">
                        </div>
                        
                        <div class="form-group">
                            <label for="email_smtp_encryption">Verschlüsselung</label>
                            <select id="email_smtp_encryption" name="email_smtp_encryption">
                                <option value="tls">TLS</option>
                                <option value="ssl">SSL</option>
                                <option value="">Keine</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="email_from_name">Absender-Name</label>
                            <input type="text" id="email_from_name" name="email_from_name" placeholder="Autohaus Name">
                        </div>
                    </div>
                    
                    <!-- Microsoft Teams-Einstellungen -->
                    <h4 style="margin-top: 2rem; margin-bottom: 1rem; color: #495057; border-bottom: 2px solid #e9ecef; padding-bottom: 0.5rem;">
                        <i class="fab fa-microsoft"></i> Microsoft Teams-Einstellungen
                    </h4>
                    
                    <div class="form-grid">
                        <div class="form-group checkbox-group">
                            <label class="checkbox-label">
                                <input type="checkbox" id="teams_notifications_enabled" name="teams_notifications_enabled" value="1">
                                <span class="checkbox-text">Teams-Benachrichtigungen aktivieren</span>
                            </label>
                        </div>
                        
                        <div class="form-group full-width">
                            <label for="teams_webhook_url">Teams Webhook URL</label>
                            <input type="url" id="teams_webhook_url" name="teams_webhook_url" placeholder="https://outlook.office.com/webhook/...">
                            <small class="text-muted">
                                <i class="fas fa-info-circle"></i> 
                                Webhook-URL aus Microsoft Teams Channel Connector kopieren
                            </small>
                        </div>
                        
                        <div class="form-group">
                            <label for="teams_channel_name">Channel-Name</label>
                            <input type="text" id="teams_channel_name" name="teams_channel_name" placeholder="z.B. Angebote, Service, etc.">
                        </div>
                        
                        <div class="form-group">
                            <label for="teams_mention_users">Benutzer erwähnen</label>
                            <input type="text" id="teams_mention_users" name="teams_mention_users" placeholder="z.B. @Max Mustermann, @Anna Schmidt">
                            <small class="text-muted">
                                <i class="fas fa-info-circle"></i> 
                                Komma-getrennte Liste von Benutzern
                            </small>
                        </div>
                    </div>
                </form>
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="closeAutohausModal()">
                    <i class="fas fa-times"></i> Abbrechen
                </button>
                <button type="button" class="btn btn-primary" onclick="submitAutohausForm()">
                    <i class="fas fa-save"></i> Speichern
                </button>
            </div>
        </div>
    </div>

    <style>
        /* Volvo-Apple Hybrid Design System */
        :root {
            /* Volvo Brand Colors */
            --volvo-blue: #0066CC;
            --volvo-blue-dark: #004C99;
            --volvo-blue-light: #E6F2FF;
            --volvo-gray: #2C3E50;
            --volvo-gray-light: #ECF0F1;
            --volvo-accent: #F39C12;
            
            /* Apple-inspired Neutrals */
            --apple-white: #FFFFFF;
            --apple-gray-1: #F2F2F7;
            --apple-gray-2: #E5E5EA;
            --apple-gray-3: #C7C7CC;
            --apple-gray-4: #8E8E93;
            --apple-gray-5: #636366;
            --apple-gray-6: #48484A;
            --apple-black: #1C1C1E;
            
            /* Design System */
            --color-primary: var(--volvo-blue);
            --color-primary-hover: var(--volvo-blue-dark);
            --color-primary-light: var(--volvo-blue-light);
            --color-secondary: var(--apple-gray-5);
            --color-accent: var(--volvo-accent);
            --color-success: #34C759;
            --color-warning: #FF9500;
            --color-error: #FF3B30;
            
            /* Surfaces */
            --surface-primary: var(--apple-white);
            --surface-secondary: var(--apple-gray-1);
            --surface-tertiary: var(--apple-gray-2);
            --surface-elevated: var(--apple-white);
            
            /* Text */
            --text-primary: var(--apple-black);
            --text-secondary: var(--apple-gray-5);
            --text-tertiary: var(--apple-gray-4);
            
            /* Borders & Radii */
            --border-color: var(--apple-gray-3);
            --border-radius: 12px;
            --border-radius-large: 20px;
            
            /* Spacing (Apple's 8pt grid system) */
            --spacing-xs: 4px;
            --spacing-sm: 8px;
            --spacing-md: 16px;
            --spacing-lg: 24px;
            --spacing-xl: 32px;
            --spacing-xxl: 48px;
            
            /* Typography (SF Pro Display inspired) */
            --font-family: 'SF Pro Display', 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            --font-size-xs: 12px;
            --font-size-sm: 14px;
            --font-size-md: 16px;
            --font-size-lg: 18px;
            --font-size-xl: 24px;
            --font-size-xxl: 32px;
            --font-weight-regular: 400;
            --font-weight-medium: 500;
            --font-weight-semibold: 600;
            --font-weight-bold: 700;
            --font-weight-heavy: 800;
            
            /* Shadows (Subtle Apple-style) */
            --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.04), 0 1px 2px rgba(0, 0, 0, 0.06);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.04), 0 2px 4px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.04), 0 4px 6px rgba(0, 0, 0, 0.05);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.04), 0 10px 10px rgba(0, 0, 0, 0.04);
            
            /* Transitions (Apple's ease curves) */
            --transition-fast: 0.15s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-normal: 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            --transition-slow: 0.5s cubic-bezier(0.4, 0, 0.2, 1);
        }

        /* Dark Theme Variables */
        [data-theme="dark"] {
            --surface-primary: #1C1C1E;
            --surface-secondary: #2C2C2E;
            --surface-tertiary: #3A3A3C;
            --surface-elevated: #2C2C2E;
            --text-primary: #FFFFFF;
            --text-secondary: #8E8E93;
            --text-tertiary: #636366;
            --border-color: #48484A;
            --color-primary: #0A84FF;
            --color-primary-hover: #0051D5;
            --color-primary-light: rgba(10, 132, 255, 0.1);
        }

        /* Global Reset & Base Styles */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: var(--font-family);
            background: var(--surface-secondary);
            color: var(--text-primary);
            line-height: 1.6;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        .admin-container {
            display: flex;
            min-height: 100vh;
        }

        /* Layout */
        .admin-main {
            flex: 1;
            padding: var(--spacing-xl);
            overflow-y: auto;
            background: var(--surface-secondary);
            transition: margin-left var(--transition-normal);
            min-height: 100vh;
        }

        .modern-sidebar.collapsed + .admin-main {
            margin-left: 80px;
        }

        /* Settings Header with Hero Section */
        .settings-header {
            position: relative;
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color);
        }

        .settings-hero {
            position: relative;
            padding: var(--spacing-xxl) var(--spacing-xl);
            background: linear-gradient(135deg, var(--color-primary) 0%, var(--volvo-blue-dark) 100%);
            overflow: hidden;
        }

        .settings-hero::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse"><path d="M 10 0 L 0 0 0 10" fill="none" stroke="rgba(255,255,255,0.05)" stroke-width="0.5"/></pattern></defs><rect width="100" height="100" fill="url(%23grid)"/></svg>');
            opacity: 0.3;
        }

        .settings-hero-content {
            position: relative;
            display: flex;
            align-items: center;
            justify-content: space-between;
            max-width: 1200px;
            margin: 0 auto;
            z-index: 1;
        }

        .settings-title-section {
            color: white;
        }

        .settings-title {
            font-size: var(--font-size-xxl);
            font-weight: var(--font-weight-bold);
            margin-bottom: var(--spacing-xs);
            color: white;
        }

        .settings-subtitle {
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-medium);
            color: rgba(255, 255, 255, 0.9);
            margin-bottom: var(--spacing-md);
        }

        .settings-meta {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-sm);
        }

        .meta-item {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            color: rgba(255, 255, 255, 0.8);
            font-size: var(--font-size-sm);
        }

        .meta-item i {
            width: 16px;
            text-align: center;
        }

        /* Modern Button System */
        .btn-primary,
        .btn-secondary {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-sm);
            padding: var(--spacing-md) var(--spacing-lg);
            border: none;
            border-radius: var(--border-radius);
            font-family: var(--font-family);
            font-size: var(--font-size-md);
            font-weight: var(--font-weight-semibold);
            text-decoration: none;
            cursor: pointer;
            transition: all var(--transition-normal);
            white-space: nowrap;
            position: relative;
            overflow: hidden;
        }

        .btn-primary {
            background: white;
            color: var(--color-primary);
            box-shadow: var(--shadow-md);
        }

        .btn-primary:hover {
            background: var(--apple-gray-1);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .btn-secondary {
            background: var(--color-primary);
            color: white;
            box-shadow: var(--shadow-md);
        }

        .btn-secondary:hover {
            background: var(--color-primary-hover);
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        /* Settings Content */
        .settings-content {
            padding: var(--spacing-xl);
            max-width: 1200px;
            margin: 0 auto;
        }

        /* Alert System */
        .alert {
            display: flex;
            align-items: flex-start;
            gap: var(--spacing-md);
            padding: var(--spacing-md) var(--spacing-lg);
            margin-bottom: var(--spacing-lg);
            border-radius: var(--border-radius);
            border: 1px solid;
            position: relative;
            backdrop-filter: blur(10px);
        }

        .alert-success {
            background: rgba(52, 199, 89, 0.1);
            border-color: rgba(52, 199, 89, 0.3);
            color: var(--color-success);
        }

        .alert-error {
            background: rgba(255, 59, 48, 0.1);
            border-color: rgba(255, 59, 48, 0.3);
            color: var(--color-error);
        }

        .alert i {
            font-size: var(--font-size-lg);
            margin-top: 2px;
        }
        
        .settings-section {
            display: none;
            animation: fadeIn 0.3s ease-in-out;
        }
        
        .settings-section.active {
            display: block;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Form Styles */
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: var(--spacing-lg);
            margin-top: var(--spacing-lg);
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
        }
        
        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--spacing-lg);
            padding-bottom: var(--spacing-md);
            border-bottom: 2px solid var(--border-color);
        }
        
        .section-header h2 {
            margin: 0;
            color: var(--text-primary);
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
        }
        
        .form-group label {
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin-bottom: var(--spacing-sm);
            font-size: var(--font-size-sm);
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            padding: var(--spacing-md);
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            font-size: var(--font-size-md);
            font-family: var(--font-family);
            color: var(--text-primary);
            background: var(--surface-primary);
            transition: all var(--transition-normal);
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px var(--color-primary-light);
        }
        
        .current-logo {
            margin-top: 1rem;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 6px;
            border: 1px solid #dee2e6;
        }
        
        .current-logo img {
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        /* User Menu Styles */
        .nav-user-section {
            margin-top: auto;
            padding-top: 1rem;
            border-top: 1px solid #e9ecef;
        }
        
        .user-menu {
            position: relative;
        }
        
        .user-menu-toggle {
            width: 100%;
            background: none;
            border: none;
            padding: 0.75rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            border-radius: 8px;
            transition: all 0.3s ease;
            cursor: pointer;
            color: #6c757d;
        }
        
        .user-menu-toggle:hover {
            background: #f8f9fa;
            color: #495057;
        }
        
        .user-avatar {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            background: #007bff;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.9rem;
        }
        
        .user-name {
            flex: 1;
            font-weight: 500;
            text-align: left;
        }
        
        .user-menu-toggle .fa-chevron-down {
            font-size: 0.8rem;
            transition: transform 0.3s ease;
        }
        
        .user-menu.active .fa-chevron-down {
            transform: rotate(180deg);
        }
        
        .user-dropdown {
            position: absolute;
            bottom: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transform: translateY(10px);
            transition: all 0.3s ease;
            margin-bottom: 0.5rem;
        }
        
        .user-menu.active .user-dropdown {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }
        
        .user-dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: #495057;
            text-decoration: none;
            transition: all 0.3s ease;
            border-bottom: 1px solid #f8f9fa;
        }
        
        .user-dropdown-item:last-child {
            border-bottom: none;
        }
        
        .user-dropdown-item:hover {
            background: #f8f9fa;
            color: #007bff;
            text-decoration: none;
        }
        
        .user-dropdown-item.active {
            background: #e3f2fd;
            color: #007bff;
        }
        
        .user-dropdown-item.logout:hover {
            color: #dc3545;
        }
        
        .user-dropdown-item i {
            width: 16px;
            font-size: 0.9rem;
        }
        
        .user-dropdown-divider {
            height: 1px;
            background: #e9ecef;
            margin: 0.25rem 0;
        }
        
        /* Checkbox Styles */
        .checkbox-label {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            cursor: pointer;
            padding: 0.5rem 0;
        }
        
        .checkbox-label input[type="checkbox"] {
            width: 18px;
            height: 18px;
            margin: 0;
            cursor: pointer;
        }
        
        .checkbox-text {
            font-weight: 400;
            color: #495057;
        }
        
        .form-group.full-width {
            grid-column: 1 / -1;
        }
        
        /* Modern Tab Navigation Styles */
        .modern-tabs {
            margin-bottom: var(--spacing-xl);
        }

        .tab-nav {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--spacing-md);
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            padding: var(--spacing-lg);
            box-shadow: var(--shadow-lg);
            border: 1px solid var(--border-color);
        }

        .tab-nav-button {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            padding: var(--spacing-lg);
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            background: var(--surface-secondary);
            cursor: pointer;
            transition: all var(--transition-normal);
            text-align: left;
            position: relative;
            overflow: hidden;
        }

        .tab-nav-button::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(135deg, var(--color-primary), var(--volvo-blue-dark));
            opacity: 0;
            transition: opacity var(--transition-normal);
        }

        .tab-nav-button:hover {
            border-color: var(--color-primary);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .tab-nav-button:hover::before {
            opacity: 0.05;
        }

        .tab-nav-button.active {
            border-color: var(--color-primary);
            background: var(--color-primary-light);
            box-shadow: var(--shadow-md);
        }

        .tab-nav-button.active::before {
            opacity: 0.1;
        }

        .tab-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--border-radius);
            background: var(--surface-tertiary);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: var(--font-size-lg);
            color: var(--color-primary);
            transition: all var(--transition-normal);
            position: relative;
            z-index: 1;
        }

        .tab-nav-button:hover .tab-icon {
            background: var(--color-primary);
            color: white;
            transform: scale(1.1);
        }

        .tab-nav-button.active .tab-icon {
            background: var(--color-primary);
            color: white;
        }

        .tab-content {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
            position: relative;
            z-index: 1;
        }

        .tab-title {
            font-size: var(--font-size-md);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            transition: color var(--transition-normal);
        }

        .tab-subtitle {
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
            transition: color var(--transition-normal);
        }

        .tab-nav-button:hover .tab-title,
        .tab-nav-button.active .tab-title {
            color: var(--color-primary);
        }

        .tab-nav-button:hover .tab-subtitle,
        .tab-nav-button.active .tab-subtitle {
            color: var(--text-primary);
        }

        @media (max-width: 768px) {
            .tab-nav {
                grid-template-columns: 1fr;
                gap: var(--spacing-sm);
                padding: var(--spacing-md);
            }
            
            .tab-nav-button {
                padding: var(--spacing-md);
            }

            .tab-icon {
                width: 40px;
                height: 40px;
                font-size: var(--font-size-md);
            }
        }
        
        /* Autohaus-Verwaltung Styles */
        .autohaus-list {
            margin-top: 2rem;
        }
        
        .list-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #e9ecef;
        }
        
        .autohaus-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
            gap: 1.5rem;
        }
        
        .autohaus-card {
            background: var(--surface-primary);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius-large);
            padding: var(--spacing-xl);
            box-shadow: var(--shadow-lg);
            transition: all var(--transition-slow);
            position: relative;
            overflow: hidden;
            backdrop-filter: blur(10px);
        }
        
        .autohaus-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: linear-gradient(90deg, #007bff, #28a745, #20c997);
            opacity: 0;
            transition: opacity 0.4s ease;
        }
        
        .autohaus-card::after {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 100%;
            height: 100%;
            background: radial-gradient(circle, rgba(0,123,255,0.05) 0%, transparent 70%);
            opacity: 0;
            transition: opacity 0.4s ease;
        }
        
        .autohaus-card:hover {
            box-shadow: 0 12px 40px rgba(0,0,0,0.15);
            transform: translateY(-5px) scale(1.02);
            border-color: #007bff;
        }
        
        .autohaus-card:hover::before {
            opacity: 1;
        }
        
        .autohaus-card:hover::after {
            opacity: 1;
        }
        
        .autohaus-prices {
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid #e9ecef;
        }
        
        .autohaus-services {
            margin-top: 1rem;
            padding-top: 1rem;
            border-top: 1px solid #e9ecef;
        }
        
        .autohaus-prices h5 {
            margin: 0 0 1.25rem 0;
            color: #2c3e50;
            font-size: 1.1rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            letter-spacing: -0.3px;
        }
        
        .autohaus-prices h5 i {
            color: #28a745;
            font-size: 1rem;
            background: rgba(40, 167, 69, 0.1);
            padding: 0.5rem;
            border-radius: 50%;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .autohaus-services h5 {
            margin: 0 0 1.25rem 0;
            color: #2c3e50;
            font-size: 1.1rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            letter-spacing: -0.3px;
        }
        
        .autohaus-services h5 i {
            color: #007bff;
            font-size: 1rem;
            background: rgba(0, 123, 255, 0.1);
            padding: 0.5rem;
            border-radius: 50%;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .services-grid {
            display: grid;
            gap: 0.75rem;
        }
        
        .service-item {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            padding: 1rem;
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border-radius: 12px;
            border: 1px solid #e9ecef;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        
        .service-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 3px;
            height: 100%;
            background: linear-gradient(180deg, #007bff, #20c997);
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .service-item:hover {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-color: #007bff;
            transform: translateY(-3px);
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.15);
        }
        
        .service-item:hover::before {
            opacity: 1;
        }
        
        .service-info {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 0.25rem;
        }
        
        .service-name {
            font-weight: 600;
            color: #495057;
            font-size: 0.9rem;
            margin: 0;
        }
        
        .service-description {
            font-size: 0.8rem;
            color: #6c757d;
            margin: 0;
            line-height: 1.3;
        }
        
        .service-price {
            margin-left: 1rem;
            text-align: right;
            min-width: 80px;
        }
        
        .service-price .price-value {
            font-weight: 700;
            font-size: 0.9rem;
            color: #28a745;
        }
        
        .service-price .price-on-request {
            color: #6c757d;
            font-style: italic;
            font-weight: 500;
        }
        
        .price-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            gap: 0.75rem;
        }
        
        .price-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.75rem 1rem;
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border-radius: 10px;
            border: 1px solid #e9ecef;
            font-size: 0.9rem;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        
        .price-item::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            width: 3px;
            height: 100%;
            background: linear-gradient(180deg, #28a745, #20c997);
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .price-item:hover {
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-color: #28a745;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.15);
        }
        
        .price-item:hover::before {
            opacity: 1;
        }
        
        .price-label {
            color: #495057;
            font-weight: 500;
            margin: 0;
            flex: 1;
        }
        
        .price-value {
            color: #28a745;
            font-weight: 700;
            font-size: 0.9rem;
            margin: 0;
            text-align: right;
            min-width: 60px;
        }
        
        .autohaus-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #f1f3f4;
            position: relative;
        }
        
        .autohaus-header::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 60px;
            height: 2px;
            background: linear-gradient(90deg, #007bff, #28a745);
            border-radius: 1px;
        }
        
        .autohaus-header h4 {
            margin: 0;
            color: #2c3e50;
            font-size: 1.3rem;
            font-weight: 700;
            letter-spacing: -0.5px;
            line-height: 1.2;
        }
        
        .autohaus-actions {
            display: flex;
            gap: 0.75rem;
            align-items: center;
        }
        
        .autohaus-info {
            margin-bottom: 1.5rem;
            background: rgba(248, 249, 250, 0.5);
            border-radius: 12px;
            padding: 1rem;
            border: 1px solid #f1f3f4;
        }
        
        .info-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            margin-bottom: 0.75rem;
            color: #495057;
            font-size: 0.95rem;
            font-weight: 500;
            padding: 0.5rem 0;
            transition: all 0.2s ease;
        }
        
        .info-item:last-child {
            margin-bottom: 0;
        }
        
        .info-item:hover {
            color: #007bff;
            transform: translateX(5px);
        }
        
        .info-item i {
            width: 20px;
            height: 20px;
            color: #007bff;
            background: rgba(0, 123, 255, 0.1);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.8rem;
            flex-shrink: 0;
        }
        
        /* Service-Verwaltung Styles */
        .services-list {
            display: grid;
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .service-card {
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 1.5rem;
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            transition: all 0.3s ease;
        }
        
        .service-card:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            border-color: #007bff;
        }
        
        .service-info h3 {
            margin: 0 0 0.5rem 0;
            color: #333;
            font-size: 1.1rem;
        }
        
        .service-description {
            color: #666;
            margin: 0 0 1rem 0;
            font-size: 0.9rem;
        }
        
        .service-details {
            display: flex;
            gap: 1rem;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .service-price {
            font-weight: 600;
            color: #28a745;
        }
        
        .service-status {
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .service-status.active {
            background: #d4edda;
            color: #155724;
        }
        
        .service-status.inactive {
            background: #f8d7da;
            color: #721c24;
        }
        
        .service-actions {
            display: flex;
            gap: 0.5rem;
        }
        
        .service-form {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            padding: 2rem;
            margin-top: 1rem;
        }
        
        .service-form h3 {
            margin: 0 0 1.5rem 0;
            color: #333;
        }
        
        .info-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            margin-bottom: 0.5rem;
            color: #6c757d;
            font-size: 0.9rem;
        }
        
        .info-item i {
            width: 16px;
            color: #007bff;
        }
        
        .autohaus-status {
            text-align: right;
        }
        
        .status-badge {
            padding: 0.4rem 1rem;
            border-radius: 25px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            transition: all 0.2s ease;
        }
        
        .status-active {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border: none;
            box-shadow: 0 2px 4px rgba(40, 167, 69, 0.3);
        }
        
        .status-active:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(40, 167, 69, 0.4);
        }
        
        .status-inactive {
            background: linear-gradient(135deg, #dc3545, #e74c3c);
            color: white;
            border: none;
            box-shadow: 0 2px 4px rgba(220, 53, 69, 0.3);
        }
        
        .status-inactive:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(220, 53, 69, 0.4);
        }
        
        .autohaus-form {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 2rem;
            margin-top: 2rem;
        }
        
        .form-actions {
            display: flex;
            gap: 1rem;
            margin-top: 2rem;
            padding-top: 1rem;
            border-top: 1px solid #dee2e6;
        }
        
        .btn-sm {
            padding: 0.375rem 0.75rem;
            font-size: 0.875rem;
        }
        
        .btn-outline-primary {
            color: var(--color-primary);
            border: 2px solid var(--color-primary);
            background: transparent;
            border-radius: var(--border-radius);
            padding: var(--spacing-sm) var(--spacing-md);
            font-weight: var(--font-weight-semibold);
            font-family: var(--font-family);
            transition: all var(--transition-normal);
            position: relative;
            overflow: hidden;
            z-index: 1;
        }
        
        .btn-outline-primary::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, #007bff, #20c997);
            transition: left 0.3s ease;
            z-index: 0;
        }
        
        .btn-outline-primary:hover {
            color: white;
            border-color: #007bff;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 123, 255, 0.3);
        }
        
        .btn-outline-primary:hover::before {
            left: 0;
        }
        
        .btn-outline-danger {
            color: #dc3545;
            border: 2px solid #dc3545;
            background: transparent;
            border-radius: 8px;
            padding: 0.5rem 0.75rem;
            font-weight: 600;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
            z-index: 1;
        }
        
        .btn-outline-danger::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, #dc3545, #e74c3c);
            transition: left 0.3s ease;
            z-index: 0;
        }
        
        .btn-outline-danger:hover {
            color: white;
            border-color: #dc3545;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
        }
        
        .btn-outline-danger:hover::before {
            left: 0;
        }
        
        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.6);
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
            backdrop-filter: blur(4px);
            animation: modalFadeIn 0.3s ease-out;
        }
        
        .modal-container {
            background: var(--surface-primary);
            border-radius: var(--border-radius-large);
            box-shadow: var(--shadow-xl);
            border: 1px solid var(--border-color);
            max-width: 800px;
            width: 90%;
            max-height: 90vh;
            overflow: hidden;
            animation: modalSlideIn 0.3s ease-out;
            position: relative;
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-lg) var(--spacing-xl);
            border-bottom: 1px solid var(--border-color);
            background: var(--surface-secondary);
            border-radius: var(--border-radius-large) var(--border-radius-large) 0 0;
        }
        
        .modal-header h3 {
            margin: 0;
            color: var(--text-primary);
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            font-family: var(--font-family);
        }
        
        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        
        .modal-close:hover {
            background: #e9ecef;
            color: #495057;
        }
        
        .modal-body {
            padding: var(--spacing-xl);
            max-height: 60vh;
            overflow-y: auto;
            background: var(--surface-primary);
        }
        
        .modal-footer {
            display: flex;
            justify-content: flex-end;
            gap: var(--spacing-md);
            padding: var(--spacing-lg) var(--spacing-xl);
            border-top: 1px solid var(--border-color);
            background: var(--surface-secondary);
            border-radius: 0 0 var(--border-radius-large) var(--border-radius-large);
        }
        
        @keyframes modalFadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        @keyframes modalSlideIn {
            from { 
                opacity: 0;
                transform: translateY(-50px) scale(0.95);
            }
            to { 
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }
        
        /* Teams-Sektion Styles */
        .teams-configuration {
            display: grid;
            gap: 2rem;
            margin-top: 2rem;
        }
        
        .teams-autohaus-card {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border: 1px solid #e9ecef;
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        
        .teams-autohaus-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: linear-gradient(90deg, #0078D4, #00BCF2);
            opacity: 0;
            transition: opacity 0.4s ease;
        }
        
        .teams-autohaus-card:hover {
            box-shadow: 0 12px 40px rgba(0,0,0,0.15);
            transform: translateY(-5px) scale(1.02);
            border-color: #0078D4;
        }
        
        .teams-autohaus-card:hover::before {
            opacity: 1;
        }
        
        .teams-autohaus-card .autohaus-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #f1f3f4;
        }
        
        .teams-autohaus-card .autohaus-header h3 {
            margin: 0;
            color: #2c3e50;
            font-size: 1.3rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .teams-autohaus-card .autohaus-header h3 i {
            color: #0078D4;
            font-size: 1.1rem;
        }
        
        .teams-status {
            display: flex;
            align-items: center;
        }
        
        .teams-form {
            margin-top: 1rem;
        }
        
        .teams-help {
            margin-top: 3rem;
            padding: 2rem;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-radius: 16px;
            border: 1px solid #dee2e6;
        }
        
        .teams-help h4 {
            margin: 0 0 1.5rem 0;
            color: #2c3e50;
            font-size: 1.2rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .teams-help h4 i {
            color: #0078D4;
            font-size: 1.1rem;
        }
        
        .help-steps {
            display: grid;
            gap: 1.5rem;
        }
        
        .help-step {
            display: flex;
            align-items: flex-start;
            gap: 1rem;
            padding: 1rem;
            background: white;
            border-radius: 12px;
            border: 1px solid #e9ecef;
            transition: all 0.3s ease;
        }
        
        .help-step:hover {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            transform: translateY(-2px);
        }
        
        .step-number {
            width: 32px;
            height: 32px;
            background: linear-gradient(135deg, #0078D4, #00BCF2);
            color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 0.9rem;
            flex-shrink: 0;
        }
        
        .step-content h5 {
            margin: 0 0 0.5rem 0;
            color: #2c3e50;
            font-size: 1rem;
            font-weight: 600;
        }
        
        .step-content p {
            margin: 0;
            color: #6c757d;
            font-size: 0.9rem;
            line-height: 1.4;
        }
        
        /* WhatsApp Autohaus Settings Styles */
        .whatsapp-global-settings {
            margin-bottom: 3rem;
            padding: 2rem;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-radius: 16px;
            border: 1px solid #dee2e6;
        }
        
        .whatsapp-global-settings h3 {
            margin: 0 0 1rem 0;
            color: #2c3e50;
            font-size: 1.2rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .whatsapp-global-settings h3 i {
            color: #25D366;
            font-size: 1.1rem;
        }
        
        .whatsapp-autohaus-settings {
            margin-top: 2rem;
        }
        
        .whatsapp-autohaus-settings h3 {
            margin: 0 0 1rem 0;
            color: #2c3e50;
            font-size: 1.2rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .whatsapp-autohaus-settings h3 i {
            color: #25D366;
            font-size: 1.1rem;
        }
        
        .autohaus-whatsapp-card {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            border: 1px solid #e9ecef;
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(0,0,0,0.08);
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }
        
        .autohaus-whatsapp-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: linear-gradient(90deg, #25D366, #128C7E);
            opacity: 0;
            transition: opacity 0.4s ease;
        }
        
        .autohaus-whatsapp-card:hover {
            box-shadow: 0 12px 40px rgba(0,0,0,0.15);
            transform: translateY(-5px) scale(1.02);
            border-color: #25D366;
        }
        
        .autohaus-whatsapp-card:hover::before {
            opacity: 1;
        }
        
        .autohaus-whatsapp-card .autohaus-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #f1f3f4;
        }
        
        .autohaus-whatsapp-card .autohaus-header h4 {
            margin: 0;
            color: #2c3e50;
            font-size: 1.3rem;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .autohaus-whatsapp-card .autohaus-header h4 i {
            color: #25D366;
            font-size: 1.1rem;
        }
        
        .whatsapp-status {
            display: flex;
            align-items: center;
        }
        
        .autohaus-whatsapp-form {
            margin-top: 1rem;
        }
        
        /* Responsive Modal */
        @media (max-width: 768px) {
            .modal-container {
                width: 95%;
                margin: 1rem;
            }
            
            .modal-header,
            .modal-body,
            .modal-footer {
                padding: 1rem;
            }
            
            .modal-body {
                max-height: 50vh;
            }
            
            .teams-configuration {
                gap: 1.5rem;
            }
            
            .teams-autohaus-card {
                padding: 1.5rem;
            }
            
            .help-steps {
                gap: 1rem;
            }
            
            .help-step {
                padding: 0.75rem;
            }
            
            .autohaus-whatsapp-card {
                padding: 1.5rem;
            }
            
            .whatsapp-global-settings {
                padding: 1.5rem;
            }
        }
    </style>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sections = document.querySelectorAll('.settings-section');
            const tabButtons = document.querySelectorAll('.tab-nav-button');
            
            // Show autohaus-section by default (first tab)
            const autohausSection = document.getElementById('autohaus-section');
            if (autohausSection) {
                autohausSection.classList.add('active');
            }
            
            // Make sure first tab button is active
            const firstTabButton = document.querySelector('.tab-nav-button[data-tab="autohaus-section"]');
            if (firstTabButton) {
                firstTabButton.classList.add('active');
            }
            
            // Tab functionality
            tabButtons.forEach(button => {
                button.addEventListener('click', function() {
                    const targetTab = this.getAttribute('data-tab');
                    
                    // Remove active class from all tabs and sections
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    sections.forEach(section => section.classList.remove('active'));
                    
                    // Add active class to clicked tab and corresponding section
                    this.classList.add('active');
                    const targetSection = document.getElementById(targetTab);
                    if (targetSection) {
                        targetSection.classList.add('active');
                    }
                });
            });
            
            // User menu toggle
            const userMenuToggle = document.querySelector('.user-menu-toggle');
            const userMenu = document.querySelector('.user-menu');
            
            if (userMenuToggle && userMenu) {
                userMenuToggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    userMenu.classList.toggle('active');
                });
                
            // Close menu when clicking outside
            document.addEventListener('click', function(e) {
                if (!userMenu.contains(e.target)) {
                    userMenu.classList.remove('active');
                }
            });
        }
        
        // Modal Event Listeners
        const modal = document.getElementById('autohaus-modal');
        if (modal) {
            // Schließe Modal beim Klicken außerhalb
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    closeAutohausModal();
                }
            });
            
            // Schließe Modal mit ESC-Taste
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && modal.style.display === 'flex') {
                    closeAutohausModal();
                }
            });
        }
        });
        
        // WhatsApp-Nachricht testen
        window.testWhatsApp = function() {
            if (!confirm('WhatsApp-Nachricht testen?\n\nDies sendet eine Test-Nachricht an die konfigurierte WhatsApp-Nummer.')) {
                return;
            }
            
            // AJAX-Anfrage für WhatsApp-Test
            fetch('test_whatsapp.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'test_whatsapp'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ WhatsApp-Nachricht erfolgreich gesendet!\n\nEmpfänger: ' + data.recipient + '\n\nPrüfen Sie Ihr WhatsApp!');
                } else {
                    alert('❌ WhatsApp-Fehler: ' + data.error);
                }
            })
            .catch(error => {
                alert('❌ Fehler: ' + error.message);
            });
        };
        
        // WhatsApp-Template testen
        window.testWhatsAppTemplate = function() {
            if (!confirm('WhatsApp-Template testen?\n\nDies testet die Template-Funktionalität mit HelloMateo.')) {
                return;
            }
            
            // AJAX-Anfrage für Template-Test
            fetch('test_whatsapp_template.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'test_whatsapp_template'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ WhatsApp-Template erfolgreich getestet!\n\nTemplate: ' + data.template_id + '\n\nStatus: ' + data.status);
                } else {
                    alert('❌ Template-Fehler: ' + data.error);
                }
            })
            .catch(error => {
                alert('❌ Fehler: ' + error.message);
            });
        };
        
        // Autohaus-Verwaltung Funktionen
        window.showAddAutohausForm = function() {
            const modalTitle = document.getElementById('modal-title');
            const formAction = document.getElementById('form-action');
            const formAutohausId = document.getElementById('form-autohaus-id');
            const autohausFormElement = document.getElementById('autohaus-form-element');
            const modal = document.getElementById('autohaus-modal');
            
            if (!modalTitle || !formAction || !formAutohausId || !autohausFormElement || !modal) {
                console.error('Autohaus-Modal Elemente nicht gefunden!');
                alert('Fehler: Modal-Elemente nicht gefunden. Bitte Seite neu laden.');
                return;
            }
            
            modalTitle.textContent = 'Neues Autohaus hinzufügen';
            formAction.value = 'add_autohaus';
            formAutohausId.value = '';
            autohausFormElement.reset();
            
            // Setze required-Attribute für sichtbare Felder
            const autohausFields = ['autohaus_name', 'autohaus_short_name', 'autohaus_phone', 'autohaus_email', 'autohaus_address', 'resy_username', 'resy_password', 'resy_dealer_id'];
            autohausFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field) {
                    field.setAttribute('required', 'required');
                }
            });
            
            // Zeige Modal
            modal.style.display = 'flex';
            document.body.style.overflow = 'hidden'; // Verhindere Scrollen im Hintergrund
            
            // Fokus auf erstes Eingabefeld
            const firstInput = document.getElementById('autohaus_name');
            if (firstInput) {
                setTimeout(() => firstInput.focus(), 100);
            }
        };
        
        window.closeAutohausModal = function() {
            const modal = document.getElementById('autohaus-modal');
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = ''; // Erlaube wieder Scrollen
                
                // Entferne required-Attribute wenn Modal geschlossen wird
                const autohausFields = ['autohaus_name', 'autohaus_short_name', 'autohaus_phone', 'autohaus_email', 'autohaus_address', 'resy_username', 'resy_password', 'resy_dealer_id'];
                autohausFields.forEach(fieldName => {
                    const field = document.getElementById(fieldName);
                    if (field) {
                        field.removeAttribute('required');
                        field.style.borderColor = '#dee2e6'; // Reset border color
                    }
                });
            }
        };
        
        window.submitAutohausForm = function() {
            // Validiere die Autohaus-Formularfelder
            const requiredFields = [
                'autohaus_name',
                'autohaus_short_name',
                'autohaus_address',
                'autohaus_phone',
                'autohaus_email'
            ];
            
            let isValid = true;
            let firstInvalidField = null;
            
            requiredFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field && !field.value.trim()) {
                    isValid = false;
                    if (!firstInvalidField) {
                        firstInvalidField = field;
                    }
                    field.style.borderColor = '#dc3545';
                } else if (field) {
                    field.style.borderColor = '#dee2e6';
                }
            });
            
            if (!isValid) {
                alert('Bitte füllen Sie alle Pflichtfelder aus.');
                if (firstInvalidField) {
                    firstInvalidField.focus();
                }
                return false;
            }
            
            // Verwende das separate Autohaus-Formular
            const form = document.getElementById('autohaus-form-element');
            if (form) {
                form.submit();
            } else {
                console.error('Autohaus-Formular nicht gefunden!');
                alert('Fehler: Formular nicht gefunden. Bitte Seite neu laden.');
            }
        };
        
        window.editAutohaus = function(autohaus) {
            const modalTitle = document.getElementById('modal-title');
            const formAction = document.getElementById('form-action');
            const formAutohausId = document.getElementById('form-autohaus-id');
            const modal = document.getElementById('autohaus-modal');
            
            if (!modalTitle || !formAction || !formAutohausId || !modal) {
                console.error('Autohaus-Modal Elemente nicht gefunden!');
                alert('Fehler: Modal-Elemente nicht gefunden. Bitte Seite neu laden.');
                return;
            }
            
            modalTitle.textContent = 'Autohaus bearbeiten';
            formAction.value = 'update_autohaus';
            formAutohausId.value = autohaus.id;
            
            // Formular mit Autohaus-Daten füllen
            document.getElementById('autohaus_name').value = autohaus.name;
            document.getElementById('autohaus_short_name').value = autohaus.short_name;
            document.getElementById('autohaus_phone').value = autohaus.phone;
            document.getElementById('autohaus_email').value = autohaus.email;
            document.getElementById('autohaus_address').value = autohaus.address;
            document.getElementById('resy_username').value = autohaus.resy_username;
            document.getElementById('resy_password').value = autohaus.resy_password;
            document.getElementById('resy_dealer_id').value = autohaus.resy_dealer_id;
            document.getElementById('timezone').value = autohaus.timezone || 'Europe/Berlin';
            
            // Arbeitstage parsen
            const workingDays = JSON.parse(autohaus.working_days || '["monday","tuesday","wednesday","thursday","friday"]');
            document.getElementById('working_days').value = workingDays.join(',');
            
            // Arbeitszeiten parsen
            const workingHours = JSON.parse(autohaus.working_hours || '{"start":"08:00","end":"18:00"}');
            document.getElementById('working_start').value = workingHours.start;
            document.getElementById('working_end').value = workingHours.end;
            
            // Preise füllen
            document.getElementById('price_wheel_change').value = autohaus.price_wheel_change || '0.00';
            document.getElementById('price_storage').value = autohaus.price_storage || '0.00';
            document.getElementById('price_spring_check').value = autohaus.price_spring_check || '0.00';
            document.getElementById('price_winter_check').value = autohaus.price_winter_check || '0.00';
            
            // E-Mail-Einstellungen füllen
            document.getElementById('email_smtp_host').value = autohaus.email_smtp_host || '';
            document.getElementById('email_smtp_port').value = autohaus.email_smtp_port || 587;
            document.getElementById('email_smtp_username').value = autohaus.email_smtp_username || '';
            document.getElementById('email_smtp_password').value = autohaus.email_smtp_password || '';
            document.getElementById('email_smtp_encryption').value = autohaus.email_smtp_encryption || 'tls';
            document.getElementById('email_from_name').value = autohaus.email_from_name || '';
            
            // Teams-Einstellungen füllen
            document.getElementById('teams_notifications_enabled').checked = autohaus.teams_notifications_enabled == 1;
            document.getElementById('teams_webhook_url').value = autohaus.teams_webhook_url || '';
            document.getElementById('teams_channel_name').value = autohaus.teams_channel_name || '';
            document.getElementById('teams_mention_users').value = autohaus.teams_mention_users || '';
            
            document.getElementById('is_active').checked = autohaus.is_active == 1;
            
            // Footer-Bild-Einstellungen füllen
            if (document.getElementById('footer_image_alt')) {
                document.getElementById('footer_image_alt').value = autohaus.footer_image_alt || '';
            }
            
            // Aktuelles Footer-Bild anzeigen wenn vorhanden
            if (autohaus.footer_image) {
                const previewDiv = document.getElementById('footer_image_preview');
                const previewImg = document.getElementById('footer_image_preview_img');
                const uploadBtn = document.getElementById('footer_upload_btn');
                
                if (previewImg) {
                    previewImg.src = '../' + autohaus.footer_image;
                    if (previewDiv) {
                        previewDiv.style.display = 'block';
                    }
                }
            }
            
            // Setze required-Attribute für sichtbare Felder
            const autohausFields = ['autohaus_name', 'autohaus_short_name', 'autohaus_phone', 'autohaus_email', 'autohaus_address', 'resy_username', 'resy_password', 'resy_dealer_id'];
            autohausFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field) {
                    field.setAttribute('required', 'required');
                }
            });
            
            // Zeige Modal
            modal.style.display = 'flex';
            document.body.style.overflow = 'hidden';
            
            // Fokus auf erstes Eingabefeld
            const firstInput = document.getElementById('autohaus_name');
            if (firstInput) {
                setTimeout(() => firstInput.focus(), 100);
            }
        };
        
        window.deleteAutohaus = function(id, name) {
            if (!confirm('Autohaus "' + name + '" wirklich löschen?\n\nDiese Aktion kann nicht rückgängig gemacht werden!')) {
                return;
            }
            
            // Formular erstellen und senden
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = '<input type="hidden" name="action" value="delete_autohaus">' +
                           '<input type="hidden" name="autohaus_id" value="' + id + '">';
            document.body.appendChild(form);
            form.submit();
        };
        
        // Service-Verwaltung Funktionen
        window.showAddServiceForm = function() {
            document.getElementById('service-form-title').textContent = 'Neuen Service hinzufügen';
            document.getElementById('service-form-action').value = 'add_service';
            document.getElementById('service-form-id').value = '';
            
            // Reset alle Formular-Felder manuell
            document.getElementById('service_autohaus_id').value = '';
            document.getElementById('service_name').value = '';
            document.getElementById('service_description').value = '';
            document.getElementById('service_price').value = '';
            document.getElementById('service_sort_order').value = '0';
            document.getElementById('service_show_price').checked = true;
            document.getElementById('service_is_active').checked = true;
            
            // Setze required-Attribute für sichtbare Felder
            const serviceFields = ['service_autohaus_id', 'service_name'];
            serviceFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field) {
                    field.setAttribute('required', 'required');
                }
            });
            
            document.getElementById('service-form').style.display = 'block';
            document.getElementById('service-form').scrollIntoView({ behavior: 'smooth' });
        };
        
        window.hideServiceForm = function() {
            document.getElementById('service-form').style.display = 'none';
            
            // Entferne required-Attribute wenn Formular versteckt wird
            const serviceFields = ['service_autohaus_id', 'service_name'];
            serviceFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field) {
                    field.removeAttribute('required');
                }
            });
        };
        
        window.editService = function(service) {
            document.getElementById('service-form-title').textContent = 'Service bearbeiten';
            document.getElementById('service-form-action').value = 'update_service';
            document.getElementById('service-form-id').value = service.id;
            document.getElementById('service_autohaus_id').value = service.autohaus_id;
            document.getElementById('service_name').value = service.service_name;
            document.getElementById('service_description').value = service.service_description || '';
            document.getElementById('service_price').value = service.price || '';
            document.getElementById('service_sort_order').value = service.sort_order || 0;
            document.getElementById('service_show_price').checked = service.show_price == 1;
            document.getElementById('service_is_active').checked = service.is_active == 1;
            
            // Setze required-Attribute für sichtbare Felder
            const serviceFields = ['service_autohaus_id', 'service_name'];
            serviceFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field) {
                    field.setAttribute('required', 'required');
                }
            });
            
            document.getElementById('service-form').style.display = 'block';
            document.getElementById('service-form').scrollIntoView({ behavior: 'smooth' });
        };
        
        window.deleteService = function(serviceId) {
            if (confirm('Sind Sie sicher, dass Sie diesen Service löschen möchten?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="action" value="delete_service"><input type="hidden" name="service_id" value="' + serviceId + '">';
                document.body.appendChild(form);
                form.submit();
            }
        };
        
        window.submitServiceForm = function() {
            // Validiere die Service-Formularfelder
            const requiredFields = [
                'service_autohaus_id',
                'service_name'
            ];
            
            let isValid = true;
            let firstInvalidField = null;
            
            requiredFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field && !field.value.trim()) {
                    isValid = false;
                    if (!firstInvalidField) {
                        firstInvalidField = field;
                    }
                    field.style.borderColor = '#dc3545';
                } else if (field) {
                    field.style.borderColor = '#dee2e6';
                }
            });
            
            if (!isValid) {
                alert('Bitte füllen Sie alle Pflichtfelder aus.');
                if (firstInvalidField) {
                    firstInvalidField.focus();
                }
                return false;
            }
            
            // Verwende das separate Service-Formular
            const form = document.getElementById('service-form-element');
            if (form) {
                form.submit();
            }
        };
        
        // Separate Speichern-Funktionen für jede Sektion
        window.submitCompanySettings = function() {
            const requiredFields = [
                'company_name',
                'company_phone', 
                'company_email',
                'company_address'
            ];
            
            let isValid = true;
            let firstInvalidField = null;
            
            requiredFields.forEach(fieldName => {
                const field = document.getElementById(fieldName);
                if (field && !field.value.trim()) {
                    isValid = false;
                    if (!firstInvalidField) {
                        firstInvalidField = field;
                    }
                    field.style.borderColor = '#dc3545';
                } else if (field) {
                    field.style.borderColor = '#dee2e6';
                }
            });
            
            if (!isValid) {
                alert('Bitte füllen Sie alle Pflichtfelder aus.');
                if (firstInvalidField) {
                    firstInvalidField.focus();
                }
                return false;
            }
            
            // Sende das Unternehmensdaten-Formular ab
            document.getElementById('company-form').submit();
        };
        
        window.submitWhatsAppSettings = function() {
            // Sende das WhatsApp-Formular ab
            document.getElementById('whatsapp-form').submit();
        };
        
        window.submitSystemSettings = function() {
            // Sende das System-Formular ab
            document.getElementById('system-form').submit();
        };
        
        // Teams-Einstellungen speichern
        window.saveTeamsSettings = function(autohausId) {
            const form = document.querySelector(`form[data-autohaus-id="${autohausId}"]`);
            if (form) {
                form.submit();
            }
        };
        
        // Teams-Nachricht testen
        window.testTeamsNotification = function(autohausId) {
            if (!confirm('Teams-Nachricht testen?\n\nDies sendet eine Test-Nachricht an den konfigurierten Teams-Channel.')) {
                return;
            }
            
            const form = document.querySelector(`form[data-autohaus-id="${autohausId}"]`);
            const webhookUrl = form.querySelector('input[name="teams_webhook_url"]').value;
            
            if (!webhookUrl) {
                alert('❌ Bitte geben Sie zuerst eine Teams Webhook-URL ein!');
                return;
            }
            
            // AJAX-Anfrage für Teams-Test
            fetch('test_teams.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'test_teams',
                    autohaus_id: autohausId
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ Teams-Nachricht erfolgreich gesendet!\n\nChannel: ' + data.channel + '\n\nPrüfen Sie Ihren Teams-Channel!');
                } else {
                    alert('❌ Teams-Fehler: ' + data.error);
                }
            })
            .catch(error => {
                alert('❌ Fehler: ' + error.message);
            });
        };
        
        // Autohaus-spezifische WhatsApp-Einstellungen speichern
        window.saveAutohausWhatsAppSettings = function(autohausId) {
            const form = document.querySelector(`form[data-autohaus-id="${autohausId}"]`);
            if (form) {
                form.submit();
            }
        };
        
        // Autohaus-spezifische WhatsApp-Nachricht testen
        window.testAutohausWhatsApp = function(autohausId) {
            if (!confirm('WhatsApp-Nachricht für dieses Autohaus testen?\n\nDies sendet eine Test-Nachricht an die konfigurierte WhatsApp-Nummer.')) {
                return;
            }
            
            // AJAX-Anfrage für Autohaus-WhatsApp-Test
            fetch('test_autohaus_whatsapp.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'test_autohaus_whatsapp',
                    autohaus_id: autohausId
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('✅ WhatsApp-Nachricht erfolgreich gesendet!\n\nAutohaus: ' + data.autohaus_name + '\nEmpfänger: ' + data.recipient + '\n\nPrüfen Sie Ihr WhatsApp!');
                } else {
                    alert('❌ WhatsApp-Fehler: ' + data.error);
                }
            })
            .catch(error => {
                alert('❌ Fehler: ' + error.message);
            });
        };
        
        // Footer-Bild Vorschau
        window.previewFooterImage = function(input) {
            if (input.files && input.files[0]) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    document.getElementById('footer_image_preview_img').src = e.target.result;
                    document.getElementById('footer_image_preview').style.display = 'block';
                    document.getElementById('footer_upload_btn').style.display = 'inline-block';
                };
                reader.readAsDataURL(input.files[0]);
            }
        };
        
        // Footer-Bild entfernen
        window.removeFooterImage = function() {
            document.getElementById('footer_image_input').value = '';
            document.getElementById('footer_image_preview').style.display = 'none';
            document.getElementById('footer_image_preview_img').src = '';
            document.getElementById('footer_upload_btn').style.display = 'none';
        };
        
        // Autohaus-Footer-Bild hochladen
        window.uploadFooterImage = function() {
            const fileInput = document.getElementById('footer_image_input');
            const altTextInput = document.getElementById('footer_image_alt');
            const autohausId = document.getElementById('form-autohaus-id').value;
            
            if (!fileInput.files || !fileInput.files[0]) {
                alert('Bitte wählen Sie ein Bild aus');
                return;
            }
            
            if (!autohausId || autohausId === '') {
                alert('Fehler: Autohaus-ID nicht gefunden. Bitte speichern Sie zunächst das Autohaus.');
                return;
            }
            
            const formData = new FormData();
            formData.append('autohaus_id', autohausId);
            formData.append('footer_image', fileInput.files[0]);
            formData.append('image_alt', altTextInput.value);
            
            console.log('Upload start...', {
                autohaus_id: autohausId,
                file: fileInput.files[0].name,
                alt: altTextInput.value
            });
            
            fetch('../api/upload_autohaus_footer_image.php', {
                method: 'POST',
                body: formData
            })
            .then(response => {
                console.log('Response status:', response.status);
                console.log('Response headers:', response.headers);
                return response.text().then(text => {
                    console.log('Response text:', text);
                    try {
                        return JSON.parse(text);
                    } catch (e) {
                        console.error('JSON parse error:', e);
                        throw new Error('Keine gültige JSON-Antwort vom Server');
                    }
                });
            })
            .then(data => {
                console.log('Upload response:', data);
                if (data.success) {
                    alert('Bild erfolgreich hochgeladen!');
                    location.reload();
                } else {
                    alert('Fehler: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Upload error:', error);
                alert('Fehler: ' + error.message);
            });
        };
        
    </script>
    
    <script src="../assets/js/main.js"></script>
    <script src="assets/js/admin.js"></script>
    <script src="assets/js/unified-sidebar.js"></script>
    
</body>
</html>
