# Sicherheitsanalyse Boxenstop - Empfehlungen

## ✅ Bereits implementiert

1. **API-Key-System** mit Scope-basierten Berechtigungen
2. **HMAC-Signatur** Unterstützung (optional für WRITE-APIs)
3. **Rate Limiting** pro API-Key konfigurierbar
4. **Prepared Statements** für SQL-Abfragen (PDO)
5. **CORS-Header** korrekt konfiguriert
6. **Session-Authentifizierung** für interne Web-Flows
7. **Upload-Validierung**: Dateityp + Größe (5-10MB)

---

## 🔴 Kritische Lücken (SOFORT beheben)

### 1. SQL-Injection Risiko

**Problem**: Direkt `$conn->exec()` ohne Prepared Statements in manchen Skripten  
**Dateien**: `apply_database_indexes.php`, `setup/*.php`, Tools  
**Risiko**: Hoch (Setup-Skripte aber selten aufgerufen)  
**Fix**: Alles auf Prepared Statements umstellen, keine User-Input in SQL

```php
// ❌ SCHLECHT
$conn->exec("UPDATE users SET email = '". $_POST['email'] ."'");

// ✅ GUT
$stmt = $conn->prepare("UPDATE users SET email = ?");
$stmt->execute([$_POST['email']]);
```

### 2. Upload-Schwachstelle (Path Traversal)

**Problem**: Dateien werden über HTTP zugänglich  
**Dateien**: `uploads/verification_documents/`, `uploads/profile/`  
**Risiko**: Hoch  
**Fix**:

- `.htaccess` in `uploads/` mit `Deny from all`
- Dateien nur via Proxy-Script ausliefern (`view_document.php`)
- Dateinamen sanitizen: `basename()`, keine `../`

```apache
# uploads/.htaccess
Order Deny,Allow
Deny from all
```

### 3. Session Fixation Risiko

**Problem**: Session-ID wird nicht bei Login rotiert  
**Dateien**: `api/login_user.php`, `api/register_user.php`  
**Risiko**: Mittel  
**Fix**: Nach erfolgreicher Auth: `session_regenerate_id(true)`

```php
session_start();
// ... Auth-Logik ...
session_regenerate_id(true); // ID rotieren
```

### 4. CSRF fehlt komplett

**Problem**: Keine CSRF-Token für POST-Requests  
**Risiko**: Hoch (Admin-Aktionen, Profil-Updates)  
**Fix**: Token bei jedem Formular erzeugen und validieren

```php
// Generator
$_SESSION['csrf_token'] = bin2hex(random_bytes(32));

// Validator
if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
    die('CSRF token mismatch');
}
```

---

## 🟡 Wichtige Verbesserungen

### 5. API Guard unvollständig

**Problem**: Nur wenige Endpunkte laden `_guard_bootstrap.php`  
**Dateien**: Die meisten API-Dateien fehlt der Guard-Call  
**Risiko**: Mittel  
**Fix**: Guard in alle sensiblen Endpunkte einbinden

```php
require_once __DIR__ . '/_guard_bootstrap.php';
```

### 6. Kein Rate Limiting für Web-Login

**Problem**: Login-Endpunkt kann bruteforced werden  
**Dateien**: `api/login_user.php`  
**Risiko**: Mittel  
**Fix**: Rate Limit implementieren (z. B. 5 Versuche/15min pro IP)

```php
$stmt = $conn->prepare("SELECT COUNT(*) FROM login_attempts
    WHERE ip = ? AND timestamp > (NOW() - INTERVAL 15 MINUTE)");
$stmt->execute([$_SERVER['REMOTE_ADDR']]);
if ($stmt->fetchColumn() >= 5) {
    die('Zu viele Versuche');
}
```

### 7. Password Hashing nicht überall

**Problem**: `admin_users` nutzt vermutlich `password_hash()`, aber prüfen  
**Risiko**: Mittel  
**Fix**: Überall `password_hash()` + `password_verify()` verwenden

### 8. API Secret in DB speichern (Klartext)

**Problem**: `api_keys.secret_key` wird als Klartext gespeichert  
**Risiko**: Hoch (wenn DB kompromittiert)  
**Fix**: Secret in ENV-Variable oder verschlüsselt speichern (Vault/KMS)

```php
// Statt DB
$secret = getenv('API_SECRET_' . $api_key_id);

// Oder: Client generiert Secret selbst, nur Hash in DB
```

### 9. Fehlende Content-Security-Policy (CSP)

**Problem**: Keine CSP-Header → XSS-Risiko  
**Dateien**: Alle Frontend-Seiten  
**Risiko**: Mittel  
**Fix**: CSP-Header setzen

```php
header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline' https://cdnjs.cloudflare.com;");
```

### 10. Keine 2FA für Admins

**Problem**: Admin-Login nur via Passwort  
**Dateien**: `admin/login.php`  
**Risiko**: Mittel  
**Fix**: TOTP 2FA implementieren (z. B. via `google2fa` Package)

---

## 🟢 Nice-to-have Verbesserungen

### 11. Error Logging & Monitoring

- Fehler-Logs zu DB hinzufügen
- Alerting bei Security-Incidents
- Sentry/Bugsnag Integration

### 12. Security Headers

```php
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Strict-Transport-Security: max-age=31536000');
```

### 13. Input Sanitization

- Alle User-Inputs filtern: `filter_var()`, `htmlspecialchars()`
- Whitelist statt Blacklist

### 14. File Upload Härtung

```php
// Nach Upload prüfen:
$finfo = finfo_open(FILEINFO_MIME_TYPE);
$mime = finfo_file($finfo, $file_path);
if ($mime !== 'image/jpeg') {
    unlink($file_path);
    die('Invalid file type');
}
```

### 15. Audit Logging

- Alle sensitiven Aktionen loggen (wer, wann, was)
- DB-Tabelle `audit_log` erstellen

---

## 📋 Implementierungs-Priorität

**PHASE 1 (sofort):**

1. SQL-Injection beheben (Prepared Statements überall)
2. Upload `.htaccess` Protection
3. Session-Regeneration bei Login
4. Rate Limiting für Login-Endpunkt

**PHASE 2 (diese Woche):** 5. CSRF-Token System 6. API Guard in alle Endpunkte 7. Password Hashing verifizieren 8. Security Headers setzen

**PHASE 3 (diesen Monat):** 9. API Secret aus DB entfernen (→ ENV) 10. 2FA für Admins 11. CSP Header 12. File Upload Härtung (MIME-Check)

**PHASE 4 (optional):** 13. Monitoring/Alerting 14. Input Sanitization Library 15. Audit Log System

---

## 🔧 Tools & Checks

```bash
# OWASP ZAP Scan
zap-cli quick-scan --self-contained http://localhost/boxenstop

# PHP Security Checker
composer require --dev sensiolabs/security-checker
vendor/bin/security-checker security:check

# SQL-Injection Scanner
sqlmap -u "http://localhost/boxenstop/api/login_user.php" --data "email=test&password=test"
```

---

**Stand**: 2025-10-27  
**Kritikalität**: Mittel-Hoch  
**Empfehlung**: Phase 1 & 2 sofort implementieren
