# API Dokumentation: Available Time Slots

## Endpoint

`GET /api/get_available_slots.php`

## Beschreibung

Diese API liefert verfügbare Zeitslots für ein Event an einem bestimmten Datum. Sie ist für mobile Apps, Web-Apps oder andere Services konzipiert, die Termine buchen möchten.

## Parameter (Query String)

| Parameter  | Typ     | Erforderlich | Beschreibung                   |
| ---------- | ------- | ------------ | ------------------------------ |
| `event_id` | integer | Ja           | Die ID des Events              |
| `date`     | string  | Ja           | Das Datum im Format YYYY-MM-DD |

## Erfolgreiche Antwort (200 OK)

```json
{
  "success": true,
  "event": {
    "id": 10,
    "name": "Reifenwechsel-Service",
    "event_date": "2025-11-01",
    "start_time": "08:00:00",
    "end_time": "18:00:00",
    "slot_duration": 30,
    "max_monteure": 5
  },
  "available_slots": [
    {
      "time": "08:00:00",
      "formatted_time": "08:00",
      "available": true,
      "available_count": 3,
      "booked": 2,
      "max_count": 5
    },
    {
      "time": "08:30:00",
      "formatted_time": "08:30",
      "available": false,
      "available_count": 0,
      "booked": 5,
      "max_count": 5
    }
  ],
  "date": "2025-11-01",
  "total_slots": 20,
  "total_available": 45
}
```

## Fehler-Antworten

### 400 Bad Request - Fehlende Parameter

```json
{
  "success": false,
  "error": "Event ID and date are required",
  "message": "Please provide both event_id and date parameters"
}
```

### 400 Bad Request - Ungültiges Datum

```json
{
  "success": false,
  "error": "Invalid date format",
  "message": "Date must be in YYYY-MM-DD format"
}
```

### 404 Not Found - Event nicht gefunden

```json
{
  "success": false,
  "error": "Event not found",
  "message": "The requested event does not exist or is inactive"
}
```

### 500 Internal Server Error

```json
{
  "success": false,
  "error": "Internal server error",
  "message": "Detailed error message"
}
```

## Verwendung

### cURL Beispiel

```bash
curl "https://be-automobileapps.de/api/get_available_slots.php?event_id=10&date=2025-11-01"
```

### JavaScript/Fetch Beispiel

```javascript
async function getAvailableSlots(eventId, date) {
  const response = await fetch(
    `api/get_available_slots.php?event_id=${eventId}&date=${date}`
  );
  const data = await response.json();

  if (data.success) {
    console.log("Available slots:", data.available_slots);
    return data.available_slots;
  } else {
    console.error("Error:", data.error);
  }
}
```

### iOS Swift Beispiel

```swift
func getAvailableSlots(eventId: Int, date: String) async throws -> [TimeSlot] {
    guard let url = URL(string: "https://be-automobileapps.de/api/get_available_slots.php?event_id=\(eventId)&date=\(date)") else {
        throw URLError(.badURL)
    }

    let (data, _) = try await URLSession.shared.data(from: url)
    let response = try JSONDecoder().decode(SlotsResponse.self, from: data)

    return response.availableSlots
}
```

### Android Kotlin Beispiel

```kotlin
suspend fun getAvailableSlots(eventId: Int, date: String): List<TimeSlot> {
    val url = "https://be-automobileapps.de/api/get_available_slots.php?event_id=$eventId&date=$date"
    val response = httpClient.get(url)
    val data = JSONObject(response.bodyAsText())

    return data.getJSONArray("available_slots").map { slot ->
        TimeSlot(
            time = slot.getString("time"),
            formattedTime = slot.getString("formatted_time"),
            available = slot.getBoolean("available"),
            availableCount = slot.getInt("available_count"),
            booked = slot.getInt("booked"),
            maxCount = slot.getInt("max_count")
        )
    }
}
```

## Datenstrukturen

### Available Slot

| Feld              | Typ     | Beschreibung                         |
| ----------------- | ------- | ------------------------------------ |
| `time`            | string  | Zeit im Format HH:MM:SS              |
| `formatted_time`  | string  | Formatierte Zeit für Anzeige (HH:MM) |
| `available`       | boolean | Ob der Slot verfügbar ist            |
| `available_count` | integer | Anzahl verfügbarer Plätze            |
| `booked`          | integer | Anzahl gebuchter Plätze              |
| `max_count`       | integer | Maximale Anzahl Plätze pro Slot      |

### Event

| Feld            | Typ     | Beschreibung                   |
| --------------- | ------- | ------------------------------ |
| `id`            | integer | Event-ID                       |
| `name`          | string  | Name des Events                |
| `event_date`    | string  | Datum des Events               |
| `start_time`    | string  | Startzeit (HH:MM:SS)           |
| `end_time`      | string  | Endzeit (HH:MM:SS)             |
| `slot_duration` | integer | Dauer eines Slots in Minuten   |
| `max_monteure`  | integer | Maximale Anzahl Monteure/Slots |

## Hinweise

- Die API berücksichtigt Pausen- und Blockierungszeiten automatisch
- Als "gebucht" zählen alle Status mit `pending`, `approved`, `booked`, `vorbereitet`, `completed`
- Inaktive Events werden nicht zurückgegeben (404)
- CORS ist für alle Ursprünge aktiviert (`*`)

## Integration mit bestehender Booking API

Diese API kann zusammen mit der bestehenden Booking API (`api/appointments.php`) verwendet werden:

1. Rufe zunächst `get_available_slots.php` auf, um verfügbare Zeitslots zu erhalten
2. Zeige dem Benutzer die verfügbaren Slots
3. Verwende `api/appointments.php` (POST) zum Buchen des ausgewählten Slots
