# Admin-User Linking Dokumentation

## Übersicht

Das System ermöglicht es, ein Kundenkonto (`users`) mit einem Admin-Konto (`admin_users`) zu verknüpfen. Wenn ein Kunde mit seinem Kundenkonto eingeloggt ist und verknüpft ist, sieht er automatisch den Admin-Button in der Navigation.

## Datenbank-Struktur

### Erweiterte `users` Tabelle

```sql
ALTER TABLE users ADD COLUMN admin_user_id INT NULL AFTER id;
ALTER TABLE users ADD CONSTRAINT fk_users_admin_user FOREIGN KEY (admin_user_id) REFERENCES admin_users(id) ON DELETE SET NULL;
ALTER TABLE users ADD INDEX idx_admin_user_id (admin_user_id);
```

### Verknüpfung

- `users.admin_user_id` → `admin_users.id`
- Optional: `NULL` = kein Admin-Account
- Foreign Key: Wenn Admin gelöscht wird, wird die Verknüpfung auf `NULL` gesetzt

## Verwendung

### 1. Manuell verknüpfen (SQL)

```sql
-- Verknüpfe User mit ID 57 mit Admin mit ID 1
UPDATE users SET admin_user_id = 1 WHERE id = 57;
```

### 2. Via API verknüpfen

```javascript
// POST /api/link_user_to_admin.php
fetch("api/link_user_to_admin.php", {
  method: "POST",
  headers: {
    "Content-Type": "application/json",
  },
  body: JSON.stringify({
    user_id: 57,
    admin_user_id: 1,
  }),
})
  .then((response) => response.json())
  .then((data) => {
    console.log(data);
  });
```

### 3. Via Admin Dashboard

**Neu**: Eine vollständige Benutzeroberfläche wurde erstellt!

1. Öffne `/admin/customer_users.php`
2. Finde das Kundenkonto in der Liste
3. Klicke auf das "Link"-Icon neben dem Konto
4. Wähle den Admin aus dem Dropdown-Menü
5. Klicke auf "Verknüpfen"

Die Verknüpfung wird sofort gespeichert und der Admin-Button erscheint beim nächsten Login des Kunden.

## Admin-Button Logik

Der Admin-Button wird angezeigt, wenn:

1. **Direkte Verknüpfung**: Der User hat `admin_user_id` gesetzt und der verknüpfte Admin ist aktiv
2. **E-Mail-Vergleich**: Der User hat keine direkte Verknüpfung, aber seine E-Mail existiert in `admin_users` und ist aktiv
3. **Admin-Session**: Der User ist bereits als Admin eingeloggt (`admin_logged_in`)

### Code-Beispiel (index.php und booking.php)

```php
// Prüfe ob der eingeloggte User ein Admin ist
$is_admin = false;
if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] && isset($_SESSION['user_id'])) {
    // Lade User-Daten
    $stmt = $conn->prepare("SELECT id, email, admin_user_id FROM users WHERE id = ? AND is_active = 1");
    $stmt->execute([$_SESSION['user_id']]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($user_data) {
        // Prüfe ob User direkt mit Admin verknüpft ist
        if (!empty($user_data['admin_user_id'])) {
            // Prüfe ob verknüpfter Admin aktiv ist
            $stmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ? AND is_active = 1");
            $stmt->execute([$user_data['admin_user_id']]);
            $admin_user = $stmt->fetch();
            $is_admin = (bool)$admin_user;
        } else {
            // Fallback: Suche nach E-Mail in admin_users
            $user_email = $user_data['email'] ?? '';
            if (!empty($user_email)) {
                $stmt = $conn->prepare("SELECT id FROM admin_users WHERE email = ? AND is_active = 1");
                $stmt->execute([$user_email]);
                $admin_user = $stmt->fetch();
                $is_admin = (bool)$admin_user;
            }
        }
    }
}
// Oder prüfe ob der User bereits als Admin eingeloggt ist
if (!$is_admin && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
    $is_admin = true;
}

if ($is_admin):
?>
    <a href="admin/" class="nav-link admin-link">
        <i class="fas fa-cog"></i>
        <span>Admin</span>
    </a>
<?php endif; ?>
```

## Anwendungsfälle

### Fall 1: Kunde und Admin mit derselben E-Mail

- Kunde registriert sich mit `john@example.com`
- Admin-Konto existiert bereits mit `john@example.com`
- **Ergebnis**: Admin-Button wird automatisch angezeigt

### Fall 2: Kunde und Admin manuell verknüpft

- Kunde hat E-Mail `john@example.com`
- Admin hat E-Mail `admin@example.com`
- Admin verknüpft Konto manuell via SQL oder API
- **Ergebnis**: Admin-Button wird angezeigt

### Fall 3: Separates Admin-Login

- Admin loggt sich direkt als Admin ein (`admin/login.php`)
- **Ergebnis**: Admin-Button wird angezeigt

## API-Endpunkt

### POST /api/link_user_to_admin.php

**Request:**

```json
{
  "user_id": 57,
  "admin_user_id": 1
}
```

**Response (Success):**

```json
{
  "success": true,
  "message": "User erfolgreich mit Admin verknüpft"
}
```

**Response (Error):**

```json
{
  "success": false,
  "message": "Error message"
}
```

## Sicherheit

- Nur eingeloggte Admins können User verknüpfen
- Verknüpfungen werden validiert (beide IDs müssen existieren)
- Admin muss aktiv sein (`is_active = 1`)
- User muss aktiv sein (`is_active = 1`)

## Entfernen der Verknüpfung

```sql
-- Entferne Verknüpfung
UPDATE users SET admin_user_id = NULL WHERE id = 57;
```

## Testing

1. Erstelle einen Test-User
2. Erstelle einen Admin in `admin_users`
3. Verknüpfe sie manuell via SQL oder API
4. Logge dich als Test-User ein
5. Überprüfe ob der Admin-Button angezeigt wird
