<?php
/**
 * Performance Monitoring System for Boxenstop
 * Measures and logs page load times, database queries, and memory usage
 */

class PerformanceMonitor {
    private static $start_time;
    private static $queries = [];
    private static $memory_start;
    private static $enabled = true;
    
    public static function start() {
        if (!self::$enabled) return;
        
        self::$start_time = microtime(true);
        self::$memory_start = memory_get_usage();
        
        // Log start
        error_log("PERFORMANCE: Page load started at " . date('Y-m-d H:i:s'));
    }
    
    public static function logQuery($sql, $params = [], $execution_time = null) {
        if (!self::$enabled) return;
        
        self::$queries[] = [
            'sql' => $sql,
            'params' => $params,
            'execution_time' => $execution_time ?: microtime(true),
            'memory' => memory_get_usage()
        ];
    }
    
    public static function end() {
        if (!self::$enabled) return;
        
        $end_time = microtime(true);
        $total_time = $end_time - self::$start_time;
        $memory_peak = memory_get_peak_usage();
        $memory_used = $memory_peak - self::$memory_start;
        
        $performance_data = [
            'total_time' => round($total_time, 4),
            'query_count' => count(self::$queries),
            'memory_peak' => $memory_peak,
            'memory_used' => $memory_used,
            'queries' => self::$queries,
            'timestamp' => date('Y-m-d H:i:s'),
            'url' => $_SERVER['REQUEST_URI'] ?? 'unknown'
        ];
        
        // Log to file
        $log_entry = json_encode($performance_data) . "\n";
        $log_file = __DIR__ . '/../logs/performance.log';
        
        // Ensure logs directory exists
        $log_dir = dirname($log_file);
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
        
        // Log slow pages
        if ($total_time > 1.0) {
            error_log("PERFORMANCE WARNING: Slow page load ({$total_time}s) - " . ($_SERVER['REQUEST_URI'] ?? 'unknown'));
        }
        
        return $performance_data;
    }
    
    public static function getStats() {
        if (!file_exists('logs/performance.log')) return [];
        
        $lines = file('logs/performance.log', FILE_IGNORE_NEW_LINES);
        $stats = [];
        
        foreach ($lines as $line) {
            $data = json_decode($line, true);
            if ($data) {
                $stats[] = $data;
            }
        }
        
        return array_slice($stats, -50); // Last 50 entries
    }
    
    public static function disable() {
        self::$enabled = false;
    }
    
    public static function enable() {
        self::$enabled = true;
    }
}

// Auto-start monitoring
PerformanceMonitor::start();
?>
