<?php
/**
 * Browser-Simulation für RESY-System ohne externe Abhängigkeiten
 * Diese Lösung simuliert einen echten Browser-Workflow
 */

echo "=== BROWSER-SIMULATION FÜR RESY-SYSTEM ===\n";

$testConfig = [
    'base_url' => 'https://resy.dtm-reifen.de',
    'username' => 'D.ULUDAG',
    'password' => 'Ikizler123!',
    'working_lgs_id' => '9716706'
];

/**
 * Simuliert einen echten Browser mit vollständiger Session-Verwaltung
 */
function simulateBrowserWorkflow($baseUrl, $username, $password, $targetLgsId) {
    echo "--- Schritt 1: Browser-Session initialisieren ---\n";
    
    $cookieFile = sys_get_temp_dir() . '/resy_browser_simulation.txt';
    
    // Lösche alte Session
    if (file_exists($cookieFile)) {
        unlink($cookieFile);
    }
    
    $ch = curl_init();
    
    // Browser-ähnliche Konfiguration
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_COOKIEJAR, $cookieFile);
    curl_setopt($ch, CURLOPT_COOKIEFILE, $cookieFile);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8,application/signed-exchange;v=b3;q=0.7',
        'Accept-Language: de-DE,de;q=0.9,en-US;q=0.8,en;q=0.7',
        'Accept-Encoding: gzip, deflate, br, zstd',
        'Connection: keep-alive',
        'Upgrade-Insecure-Requests: 1',
        'Sec-Fetch-Dest: document',
        'Sec-Fetch-Mode: navigate',
        'Sec-Fetch-Site: none',
        'Sec-Fetch-User: ?1'
    ]);
    
    echo "--- Schritt 2: Hauptseite besuchen (wie ein echter Browser) ---\n";
    $mainPageUrl = $baseUrl . '/resyweb/index.php';
    curl_setopt($ch, CURLOPT_URL, $mainPageUrl);
    
    $mainPageResponse = curl_exec($ch);
    $mainPageCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $mainPageFinalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    
    echo "Hauptseite: HTTP $mainPageCode, Final URL: $mainPageFinalUrl\n";
    echo "Content Length: " . strlen($mainPageResponse) . " Zeichen\n";
    
    // Prüfe ob wir zur Login-Seite umgeleitet wurden
    $needsLogin = strpos($mainPageResponse, 'Anmeldename') !== false || 
                  strpos($mainPageFinalUrl, 'login') !== false;
    
    if ($needsLogin) {
        echo "--- Schritt 3: Login-Seite besuchen ---\n";
        $loginPageUrl = $baseUrl . '/index.php?m=login&a=login';
        curl_setopt($ch, CURLOPT_URL, $loginPageUrl);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en-US;q=0.8,en;q=0.7',
            'Accept-Encoding: gzip, deflate, br, zstd',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $mainPageFinalUrl
        ]);
        
        $loginPageResponse = curl_exec($ch);
        $loginPageCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        echo "Login-Seite: HTTP $loginPageCode\n";
        
        echo "--- Schritt 4: Login durchführen ---\n";
        $postData = [
            'FN' => 'login',
            'UserName' => $username,
            'Password' => $password,
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en-US;q=0.8,en;q=0.7',
            'Accept-Encoding: gzip, deflate, br, zstd',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $loginPageUrl
        ]);
        
        $loginResponse = curl_exec($ch);
        $loginCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $loginFinalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
        
        echo "Login: HTTP $loginCode, Final URL: $loginFinalUrl\n";
        
        // Prüfe Login-Erfolg
        $loginSuccessful = strpos($loginResponse, 'Anmeldename') === false && 
                          strpos($loginFinalUrl, 'login') === false;
        
        if (!$loginSuccessful) {
            curl_close($ch);
            throw new Exception("Login fehlgeschlagen");
        }
        
        echo "✅ Login erfolgreich!\n";
    } else {
        echo "✅ Bereits eingeloggt!\n";
    }
    
    echo "--- Schritt 5: Navigiere zur Suchseite (wie ein echter Browser) ---\n";
    
    // Simuliere Browser-Navigation: Erst zur Hauptseite, dann zur Suchseite
    $searchPageUrl = $baseUrl . '/resyweb/index.php?m=wheelset&a=search';
    
    curl_setopt($ch, CURLOPT_POST, false);
    curl_setopt($ch, CURLOPT_POSTFIELDS, null);
    curl_setopt($ch, CURLOPT_URL, $searchPageUrl);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
        'Accept-Language: de-DE,de;q=0.9,en-US;q=0.8,en;q=0.7',
        'Accept-Encoding: gzip, deflate, br, zstd',
        'Connection: keep-alive',
        'Upgrade-Insecure-Requests: 1',
        'Referer: ' . $baseUrl . '/resyweb/index.php'
    ]);
    
    $searchPageResponse = curl_exec($ch);
    $searchPageCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $searchPageFinalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    
    echo "Suchseite: HTTP $searchPageCode, Final URL: $searchPageFinalUrl\n";
    echo "Content Length: " . strlen($searchPageResponse) . " Zeichen\n";
    
    // Speichere die Suchseite für Analyse
    file_put_contents(sys_get_temp_dir() . "/resy_browser_search_page.html", $searchPageResponse);
    
    // Prüfe ob wir zur Login-Seite umgeleitet wurden
    $redirectedToLogin = strpos($searchPageResponse, 'Anmeldename') !== false || 
                        strpos($searchPageFinalUrl, 'login') !== false;
    
    if ($redirectedToLogin) {
        curl_close($ch);
        throw new Exception("Session verloren - zur Login-Seite umgeleitet");
    }
    
    echo "--- Schritt 6: Analysiere Suchformular ---\n";
    
    // Extrahiere das Suchformular
    if (preg_match('/<form[^>]*action=["\']([^"\']*)["\'][^>]*>(.*?)<\/form>/is', $searchPageResponse, $formMatch)) {
        $formAction = $formMatch[1];
        $formContent = $formMatch[2];
        
        echo "Formular Action: $formAction\n";
        
        // Extrahiere alle Input-Felder
        preg_match_all('/<input[^>]*name=["\']([^"\']*)["\'][^>]*>/i', $formContent, $inputMatches);
        echo "Input-Felder gefunden: " . count($inputMatches[1]) . "\n";
        
        // Erstelle Suchdaten
        $searchData = [
            'LGS_ID' => $targetLgsId,
            'button' => 'submit',
            'FN' => 'WheelsetSearch'
        ];
        
        // Füge versteckte Felder hinzu
        preg_match_all('/<input[^>]*type=["\']hidden["\'][^>]*name=["\']([^"\']*)["\'][^>]*value=["\']([^"\']*)["\'][^>]*>/i', $formContent, $hiddenMatches);
        for ($i = 0; $i < count($hiddenMatches[1]); $i++) {
            $hiddenName = $hiddenMatches[1][$i];
            $hiddenValue = $hiddenMatches[2][$i];
            $searchData[$hiddenName] = $hiddenValue;
        }
        
        echo "--- Schritt 7: Sende Suchformular ---\n";
        
        $formUrl = $baseUrl . '/resyweb/' . ltrim($formAction, './');
        echo "Formular-URL: $formUrl\n";
        
        curl_setopt($ch, CURLOPT_URL, $formUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($searchData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en-US;q=0.8,en;q=0.7',
            'Accept-Encoding: gzip, deflate, br, zstd',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $searchPageFinalUrl
        ]);
        
        $searchResultResponse = curl_exec($ch);
        $searchResultCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $searchResultFinalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
        
        echo "Suchergebnis: HTTP $searchResultCode, Final URL: $searchResultFinalUrl\n";
        echo "Content Length: " . strlen($searchResultResponse) . " Zeichen\n";
        
        // Speichere das Suchergebnis
        file_put_contents(sys_get_temp_dir() . "/resy_browser_search_result.html", $searchResultResponse);
        
        curl_close($ch);
        
        // Analysiere das Suchergebnis
        $hasWheelsetData = strpos($searchResultResponse, 'WheelHALTERNAME') !== false || 
                          strpos($searchResultResponse, 'WheelKENNZEICHEN') !== false ||
                          strpos($searchResultResponse, 'WheelFAHRGESTNR') !== false;
        $hasTargetLgsId = strpos($searchResultResponse, $targetLgsId) !== false;
        $hasLoginForm = strpos($searchResultResponse, 'Anmeldename') !== false;
        
        echo "\n=== SUCHEERGEBNIS-ANALYSE ===\n";
        echo "Enthält Wheelset-Daten: " . ($hasWheelsetData ? 'Ja' : 'Nein') . "\n";
        echo "Enthält Ziel-LGS_ID: " . ($hasTargetLgsId ? 'Ja' : 'Nein') . "\n";
        echo "Ist Login-Seite: " . ($hasLoginForm ? 'Ja' : 'Nein') . "\n";
        
        if ($hasWheelsetData && $hasTargetLgsId) {
            echo "\n🎉 ERFOLG! Browser-Simulation funktioniert!\n";
            echo "Wheelset-Daten für LGS_ID $targetLgsId gefunden!\n";
            
            // Extrahiere einige Daten
            if (preg_match('/WheelHALTERNAME[^>]*>([^<]+)/', $searchResultResponse, $matches)) {
                echo "Fahrzeughalter: " . trim($matches[1]) . "\n";
            }
            if (preg_match('/WheelKENNZEICHEN[^>]*>([^<]+)/', $searchResultResponse, $matches)) {
                echo "Kennzeichen: " . trim($matches[1]) . "\n";
            }
            
            return [
                'success' => true,
                'lgs_id' => $targetLgsId,
                'search_method' => 'browser_simulation',
                'message' => "LGS_ID $targetLgsId mit Browser-Simulation gefunden",
                'response' => [
                    'body' => $searchResultResponse,
                    'http_code' => $searchResultCode,
                    'final_url' => $searchResultFinalUrl
                ]
            ];
            
        } elseif ($hasLoginForm) {
            throw new Exception("Session verloren während der Suche");
        } else {
            throw new Exception("Keine Wheelset-Daten gefunden");
        }
        
    } else {
        curl_close($ch);
        throw new Exception("Kein Suchformular gefunden");
    }
}

try {
    echo "Teste Browser-Simulation mit LGS_ID: " . $testConfig['working_lgs_id'] . "\n\n";
    
    $result = simulateBrowserWorkflow(
        $testConfig['base_url'],
        $testConfig['username'],
        $testConfig['password'],
        $testConfig['working_lgs_id']
    );
    
    echo "\n✅ Browser-Simulation erfolgreich!\n";
    echo "Suchmethode: " . $result['search_method'] . "\n";
    echo "Nachricht: " . $result['message'] . "\n";
    
} catch (Exception $e) {
    echo "\n❌ Browser-Simulation fehlgeschlagen: " . $e->getMessage() . "\n";
}

echo "\n=== BROWSER-SIMULATION ABGESCHLOSSEN ===\n";
echo "HTML-Dateien gespeichert in: " . sys_get_temp_dir() . "/\n";
?>
