<?php
/**
 * Database Performance Indexes Application Script
 * Run this script to apply performance indexes to the database
 */

require_once 'config/database.php';

echo "<h2>Datenbank Performance-Indizes anwenden</h2>\n";

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    // Enable buffered queries to avoid the unbuffered query error
    $conn->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);
    
    if (!$conn) {
        throw new Exception("Keine Datenbankverbindung möglich");
    }
    
    echo "<p>✓ Datenbankverbindung erfolgreich</p>\n";
    
    // Read the SQL file
    $sqlFile = 'database_performance_indexes.sql';
    if (!file_exists($sqlFile)) {
        throw new Exception("SQL-Datei nicht gefunden: $sqlFile");
    }
    
    $sql = file_get_contents($sqlFile);
    $queries = array_filter(array_map('trim', explode(';', $sql)));
    
    echo "<p>✓ SQL-Datei gelesen (" . count($queries) . " Abfragen)</p>\n";
    
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($queries as $query) {
        if (empty($query) || strpos($query, '--') === 0) {
            continue; // Skip empty lines and comments
        }
        
        try {
            // Use query() for ANALYZE statements to avoid buffering issues
            if (stripos($query, 'ANALYZE') === 0) {
                $stmt = $conn->query($query);
                $stmt->fetchAll(); // Consume the result
                $successCount++;
                
                if (preg_match('/ANALYZE TABLE\s+(\w+)/i', $query, $matches)) {
                    echo "<p>✓ Tabelle analysiert: " . $matches[1] . "</p>\n";
                }
            } else {
                $conn->exec($query);
                $successCount++;
                
                // Extract index name for display
                if (preg_match('/CREATE INDEX.*?(\w+)\s+ON/i', $query, $matches)) {
                    echo "<p>✓ Index erstellt: " . $matches[1] . "</p>\n";
                }
            }
            
        } catch (PDOException $e) {
            $errorCount++;
            echo "<p style='color: red;'>✗ Fehler bei Abfrage: " . htmlspecialchars($e->getMessage()) . "</p>\n";
            echo "<p style='color: gray;'>SQL: " . htmlspecialchars(substr($query, 0, 100)) . "...</p>\n";
        }
    }
    
    echo "<hr>\n";
    echo "<h3>Zusammenfassung:</h3>\n";
    echo "<p>✓ Erfolgreich: $successCount</p>\n";
    echo "<p>✗ Fehler: $errorCount</p>\n";
    
    if ($errorCount === 0) {
        echo "<p style='color: green; font-weight: bold;'>🎉 Alle Indizes erfolgreich angewendet!</p>\n";
        echo "<p>Die Datenbankabfragen sollten jetzt deutlich schneller sein.</p>\n";
    } else {
        echo "<p style='color: orange;'>⚠️ Einige Indizes konnten nicht erstellt werden. Das ist normal, wenn sie bereits existieren.</p>\n";
    }
    
    // Test query performance
    echo "<hr>\n";
    echo "<h3>Performance-Test:</h3>\n";
    
    try {
        $startTime = microtime(true);
        $stmt = $conn->query("SELECT COUNT(*) FROM events WHERE is_active = 1");
        $result = $stmt->fetchColumn();
        $stmt->closeCursor(); // Close the cursor to avoid buffering issues
        $endTime = microtime(true);
        
        $queryTime = ($endTime - $startTime) * 1000; // Convert to milliseconds
        echo "<p>Test-Abfrage: " . round($queryTime, 2) . "ms</p>\n";
        
        if ($queryTime < 10) {
            echo "<p style='color: green;'>✓ Sehr gute Performance!</p>\n";
        } elseif ($queryTime < 50) {
            echo "<p style='color: orange;'>⚠️ Mittlere Performance</p>\n";
        } else {
            echo "<p style='color: red;'>✗ Langsame Performance - weitere Optimierung erforderlich</p>\n";
        }
    } catch (PDOException $e) {
        echo "<p style='color: red;'>✗ Performance-Test fehlgeschlagen: " . htmlspecialchars($e->getMessage()) . "</p>\n";
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>Fehler: " . htmlspecialchars($e->getMessage()) . "</p>\n";
}

echo "<hr>\n";
echo "<p><a href='admin/performance.php'>→ Zum Performance Dashboard</a></p>\n";
echo "<p><a href='admin/index.php'>→ Zum Admin Dashboard</a></p>\n";
?>
