<?php
/**
 * Schlaue RESY API - Folgt exakt dem RESY-Workflow
 * 
 * Workflow:
 * 1. Login auf https://resy.dtm-reifen.de/index.php?m=login&a=login
 * 2. Submit mit Button "Anmelden"
 * 3. Navigiere zu "Sätze / Rücklieferungen bestellen"
 * 4. Wähle passendes Autohaus aus KD_ID Select
 * 5. Submit mit Button "Suchen"
 * 6. Parse alle Ergebnisse mit Pagination
 */

require_once 'cache_manager.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

session_start();
require_once dirname(__DIR__) . '/config/database.php';

/**
 * Schlaue RESY API - Folgt exakt dem Workflow
 */
class SmartResyAPI {
    private $autohausId;
    private $autohausConfig;
    private $cacheManager;
    private $startTime;
    private $debugLog = [];
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $this->startTime = microtime(true);
        
        // Initialisiere Cache
        try {
            if (isset($GLOBALS['conn']) && $GLOBALS['conn']) {
                $this->cacheManager = new CacheManager($GLOBALS['conn']);
            }
        } catch (Exception $e) {
            $this->debugLog("Cache nicht verfügbar: " . $e->getMessage());
        }
        
        $this->loadAutohausConfig();
    }
    
    /**
     * Lädt Autohaus-Konfiguration
     */
    private function loadAutohausConfig() {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id, name, short_name, 
                    resy_username, resy_password, resy_dealer_id, resy_base_url
                FROM autohaus 
                WHERE id = ? AND is_active = 1
            ");
            $stmt->execute([$this->autohausId]);
            $this->autohausConfig = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$this->autohausConfig) {
                throw new Exception('Autohaus nicht gefunden');
            }
            
            // Konvertiere zu korrekter RESY Dealer-ID
            $autohausName = $this->autohausConfig['name'];
            $mapping = [
                'B & E Volvo Kiel' => '453',
                'B & E Volvo Norderstedt' => '452', 
                'B & E Volvo Glinde' => '454',
                'Kiel' => '453',
                'Norderstedt' => '452',
                'Glinde' => '454'
            ];
            
            // Prüfe zuerst den vollständigen Namen
            if (isset($mapping[$autohausName])) {
                $this->autohausConfig['resy_dealer_id'] = $mapping[$autohausName];
            } else {
                // Prüfe auch Teilstrings
                foreach ($mapping as $name => $kdid) {
                    if (strpos($autohausName, $name) !== false) {
                        $this->autohausConfig['resy_dealer_id'] = $kdid;
                        break;
                    }
                }
            }
            
            $this->debugLog("Autohaus geladen: " . $this->autohausConfig['name'] . " (Dealer ID: " . $this->autohausConfig['resy_dealer_id'] . ")");
            
        } catch (Exception $e) {
            $this->debugLog("Fehler beim Laden der Autohaus-Konfiguration: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Hauptmethode - Folgt exakt dem RESY-Workflow
     */
    public function getWheelsets() {
        try {
            $this->debugLog("=== STARTE RESY-WORKFLOW ===");
            
            // 1. Prüfe Cache zuerst
            $cachedResult = $this->checkCache();
            if ($cachedResult) {
                return $cachedResult;
            }
            
            // 2. Führe exakten RESY-Workflow durch
            $wheelsets = $this->executeResyWorkflow();
            
            // 3. Validiere und bereinige Daten
            $validatedWheelsets = $this->validateWheelsets($wheelsets);
            
            // 4. Speichere im Cache
            $this->saveToCache($validatedWheelsets);
            
            $loadTime = round(microtime(true) - $this->startTime, 2);
            
            $this->debugLog("=== WORKFLOW ERFOLGREICH ABGESCHLOSSEN ===");
            $this->debugLog("Geladene Reifensätze: " . count($validatedWheelsets));
            $this->debugLog("Ladezeit: " . $loadTime . " Sekunden");
            
            return [
                'success' => true,
                'wheelsets' => $validatedWheelsets,
                'count' => count($validatedWheelsets),
                'autohaus_id' => $this->autohausId,
                'autohaus_name' => $this->autohausConfig['name'],
                'dealer_id' => $this->autohausConfig['resy_dealer_id'],
                'load_time' => $loadTime,
                'cached' => false,
                'cache_time' => 'Neu geladen',
                'debug_log' => $this->debugLog
            ];
            
        } catch (Exception $e) {
            $this->debugLog("=== FEHLER IM WORKFLOW ===");
            $this->debugLog("Fehler: " . $e->getMessage());
            return $this->handleError($e);
        }
    }
    
    /**
     * Führt den exakten RESY-Workflow durch
     */
    private function executeResyWorkflow() {
        $ch = curl_init();
        $allWheelsets = [];
        
        try {
            // Schritt 1: Login-Seite laden
            $this->debugLog("Schritt 1: Lade Login-Seite");
            $this->loadLoginPage($ch);
            
            // Schritt 2: Login durchführen
            $this->debugLog("Schritt 2: Führe Login durch");
            $this->performLogin($ch);
            
            // Schritt 3: Navigiere zu "Sätze / Rücklieferungen bestellen"
            $this->debugLog("Schritt 3: Navigiere zu Wheelset-Suche");
            $this->navigateToWheelsetSearch($ch);
            
            // Schritt 4: Wähle Autohaus aus und suche
            $this->debugLog("Schritt 4: Wähle Autohaus " . $this->autohausConfig['resy_dealer_id'] . " aus und suche");
            $this->selectAutohausAndSearch($ch);
            
            // Schritt 5: Lade alle Ergebnisse mit Pagination
            $this->debugLog("Schritt 5: Lade alle Ergebnisse mit Pagination");
            $allWheelsets = $this->loadAllResultsWithPagination($ch);
            
        } finally {
            curl_close($ch);
        }
        
        return $allWheelsets;
    }
    
    /**
     * Schritt 1: Login-Seite laden
     */
    private function loadLoginPage($ch) {
        $loginUrl = 'https://resy.dtm-reifen.de/index.php?m=login&a=login';
        
        curl_setopt($ch, CURLOPT_URL, $loginUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $cookieFile = dirname(__DIR__) . '/temp/resy_cookies_' . $this->autohausId . '.txt';
        curl_setopt($ch, CURLOPT_COOKIEJAR, $cookieFile);
        curl_setopt($ch, CURLOPT_COOKIEFILE, $cookieFile);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_ENCODING, 'gzip,deflate');
        
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Fehler beim Laden der Login-Seite: ' . curl_error($ch));
        }
        
        // Debug: Speichere Login-Seite
        $debugDir = dirname(__DIR__) . '/temp';
        if (!is_dir($debugDir)) {
            mkdir($debugDir, 0755, true);
        }
        file_put_contents($debugDir . '/resy_login_page_' . $this->autohausId . '.html', $response);
        $this->debugLog("Login-Seite geladen und gespeichert");
        
        // Prüfe ob Login-Formular vorhanden ist (verschiedene Varianten)
        $loginFormFound = false;
        $loginPatterns = [
            'Anmeldename',
            'UserName',
            'username',
            'login',
            'Login',
            'Anmeldung'
        ];
        
        foreach ($loginPatterns as $pattern) {
            if (strpos($response, $pattern) !== false) {
                $loginFormFound = true;
                $this->debugLog("Login-Formular gefunden mit Pattern: $pattern");
                break;
            }
        }
        
        if (!$loginFormFound) {
            $this->debugLog("Kein Login-Formular gefunden. Response-Länge: " . strlen($response));
            $this->debugLog("Response-Start: " . substr($response, 0, 500));
            throw new Exception('Login-Formular nicht gefunden - möglicherweise ist die Seite nicht erreichbar');
        }
        
        $this->debugLog("Login-Formular gefunden");
    }
    
    /**
     * Schritt 2: Login durchführen
     */
    private function performLogin($ch) {
        $loginUrl = 'https://resy.dtm-reifen.de/index.php?m=login&a=login';
        
        $postData = [
            'FN' => 'login',
            'UserName' => $this->autohausConfig['resy_username'],
            'Password' => $this->autohausConfig['resy_password'],
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $loginUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $loginUrl
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Fehler beim Login: ' . curl_error($ch));
        }
        
        // Debug: Speichere Login-Response
        $debugDir = dirname(__DIR__) . '/temp';
        file_put_contents($debugDir . '/resy_login_response_' . $this->autohausId . '.html', $response);
        $this->debugLog("Login-Response gespeichert");
        
        // Prüfe Login-Erfolg
        if (strpos($response, 'Anmeldename') !== false) {
            throw new Exception('Login fehlgeschlagen - Login-Formular noch vorhanden');
        }
        
        // Prüfe auf Erfolgs-Indikatoren
        if (strpos($response, 'RESY Websystem') === false && strpos($response, 'Lagerbestand') === false) {
            throw new Exception('Login fehlgeschlagen - Keine Erfolgs-Indikatoren gefunden');
        }
        
        $this->debugLog("Login erfolgreich");
    }
    
    /**
     * Schritt 3: Navigiere zu "Sätze / Rücklieferungen bestellen"
     */
    private function navigateToWheelsetSearch($ch) {
        $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search';
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_POST, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Fehler beim Navigieren zur Wheelset-Suche: ' . curl_error($ch));
        }
        
        // Debug: Speichere Wheelset-Suche-Seite
        $debugDir = dirname(__DIR__) . '/temp';
        file_put_contents($debugDir . '/resy_wheelset_search_' . $this->autohausId . '.html', $response);
        $this->debugLog("Wheelset-Suche-Seite geladen");
        
        // Prüfe ob KD_ID Select vorhanden ist
        if (strpos($response, 'KD_ID') === false) {
            throw new Exception('KD_ID Select nicht gefunden');
        }
        
        $this->debugLog("KD_ID Select gefunden");
    }
    
    /**
     * Schritt 4: Wähle Autohaus aus und suche
     */
    private function selectAutohausAndSearch($ch) {
        $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search';
        
        // Verwende POST für die Suche (wie ursprünglich)
        $postData = [
            'FN' => 'WheelsetSearch',
            'a' => 'search',
            'm' => 'wheelset',
            'KD_ID' => $this->autohausConfig['resy_dealer_id'],
            'STATUS_ID' => '20', // eingelagert
            'Page' => '1',
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $searchUrl
        ]);
        
        $this->debugLog("Suche mit POST: " . http_build_query($postData));
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            throw new Exception('Fehler beim Suchen: ' . curl_error($ch));
        }
        
        // Debug: Speichere erste Suchergebnisse
        $debugDir = dirname(__DIR__) . '/temp';
        file_put_contents($debugDir . '/resy_search_results_' . $this->autohausId . '.html', $response);
        $this->debugLog("Suchergebnisse geladen für Autohaus " . $this->autohausConfig['resy_dealer_id']);
        
        // Prüfe ob Ergebnisse vorhanden sind
        if (strpos($response, 'WheelsetSearchResult') === false) {
            $this->debugLog("Keine WheelsetSearchResult Tabelle gefunden - möglicherweise keine Ergebnisse");
        } else {
            $this->debugLog("WheelsetSearchResult Tabelle gefunden - Ergebnisse vorhanden");
        }
        
        // Prüfe auf Pagination
        if (preg_match('/Seite\s+(\d+)\s+von\s+(\d+)/', $response, $matches)) {
            $currentPage = $matches[1];
            $totalPages = $matches[2];
            $this->debugLog("Pagination gefunden: Seite $currentPage von $totalPages");
        } else {
            $this->debugLog("Keine Pagination gefunden");
        }
        
        // Prüfe auf Anzahl der Ergebnisse
        if (preg_match('/(\d+)\s+Reifensätze\s+gefunden/', $response, $matches)) {
            $totalResults = $matches[1];
            $this->debugLog("Gesamtanzahl Reifensätze: $totalResults");
        }
    }
    
    /**
     * Schritt 5: Lade alle Ergebnisse mit Pagination
     */
    private function loadAllResultsWithPagination($ch) {
        $allWheelsets = [];
        $page = 1;
        $maxPages = 100; // Erhöhtes Sicherheitslimit
        $consecutiveEmptyPages = 0;
        $totalPages = null;
        
        // Lade erste Seite um Gesamtanzahl zu ermitteln
        $this->debugLog("Lade erste Seite um Gesamtanzahl zu ermitteln");
        $firstPageWheelsets = $this->loadSinglePage($ch, 1);
        $allWheelsets = array_merge($allWheelsets, $firstPageWheelsets);
        $this->debugLog("Seite 1: " . count($firstPageWheelsets) . " Reifensätze, Total: " . count($allWheelsets));
        
        // Versuche Gesamtanzahl der Seiten zu ermitteln
        $totalPages = $this->getTotalPages($ch);
        if ($totalPages) {
            $this->debugLog("Gesamtanzahl Seiten ermittelt: $totalPages");
            $maxPages = min($totalPages, $maxPages);
        }
        
        // Lade weitere Seiten
        $page = 2;
        $lastPageCount = count($firstPageWheelsets);
        
        while ($page <= $maxPages) {
            $this->debugLog("Lade Seite $page von $maxPages");
            
            $pageWheelsets = $this->loadSinglePage($ch, $page);
            
            if (empty($pageWheelsets)) {
                $consecutiveEmptyPages++;
                if ($consecutiveEmptyPages >= 2) {
                    $this->debugLog("2 aufeinanderfolgende leere Seiten, stoppe bei Seite $page");
                    break;
                }
            } else {
                $consecutiveEmptyPages = 0;
                
                // Prüfe ob es die gleichen Daten wie die vorherige Seite sind
                if (count($pageWheelsets) == $lastPageCount) {
                    // Vergleiche erste paar Satznummern
                    $sameData = true;
                    for ($i = 0; $i < min(5, count($pageWheelsets)); $i++) {
                        if (!isset($pageWheelsets[$i]['satznummer']) || !isset($allWheelsets[$i]['satznummer'])) {
                            $sameData = false;
                            break;
                        }
                        if ($pageWheelsets[$i]['satznummer'] !== $allWheelsets[$i]['satznummer']) {
                            $sameData = false;
                            break;
                        }
                    }
                    
                    if ($sameData) {
                        $this->debugLog("Seite $page enthält die gleichen Daten wie Seite 1, stoppe Pagination");
                        break;
                    }
                }
                
                $allWheelsets = array_merge($allWheelsets, $pageWheelsets);
                $lastPageCount = count($pageWheelsets);
                $this->debugLog("Seite $page: " . count($pageWheelsets) . " Reifensätze, Total: " . count($allWheelsets));
            }
            
            $page++;
            
            // Kurze Pause zwischen Requests
            usleep(100000); // 0.1 Sekunden
        }
        
        $this->debugLog("Pagination abgeschlossen: " . count($allWheelsets) . " Reifensätze in " . ($page - 1) . " Seiten");
        return $allWheelsets;
    }
    
    /**
     * Ermittelt die Gesamtanzahl der Seiten
     */
    private function getTotalPages($ch) {
        // Lade erste Seite und suche nach Pagination-Info
        $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search';
        
        $postData = [
            'FN' => 'WheelsetSearch',
            'a' => 'search',
            'm' => 'wheelset',
            'KD_ID' => $this->autohausConfig['resy_dealer_id'],
            'STATUS_ID' => '20', // eingelagert
            'Page' => '1',
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            return null;
        }
        
        // Suche nach Pagination-Info
        if (preg_match('/Seite\s+(\d+)\s+von\s+(\d+)/', $response, $matches)) {
            return (int)$matches[2];
        }
        
        // Suche nach "Seite X von Y" Pattern
        if (preg_match('/Seite\s+\d+\s+von\s+(\d+)/', $response, $matches)) {
            return (int)$matches[1];
        }
        
        return null;
    }
    
    /**
     * Lädt eine einzelne Seite
     */
    private function loadSinglePage($ch, $page) {
        $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search';
        
        // Verwende POST für Pagination (wie ursprünglich)
        $postData = [
            'FN' => 'WheelsetSearch',
            'a' => 'search',
            'm' => 'wheelset',
            'KD_ID' => $this->autohausConfig['resy_dealer_id'],
            'STATUS_ID' => '20', // eingelagert
            'Page' => $page,
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $searchUrl
        ]);
        
        $this->debugLog("Lade Seite $page mit POST: " . http_build_query($postData));
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            $this->debugLog("Fehler bei Seite $page: " . curl_error($ch));
            return [];
        }
        
        // Debug: Speichere Seiten-Response (nur erste 5 Seiten)
        if ($page <= 5) {
            $debugDir = dirname(__DIR__) . '/temp';
            file_put_contents($debugDir . '/resy_page_' . $page . '_' . $this->autohausId . '.html', $response);
        }
        
        return $this->parsePageResponse($response);
    }
    
    /**
     * Parst die Antwort einer Seite
     */
    private function parsePageResponse($html) {
        $wheelsets = [];
        
        $this->debugLog("Parse Page Response - HTML Länge: " . strlen($html));
        
        // Debug: Speichere HTML für Analyse
        $debugDir = dirname(__DIR__) . '/temp';
        file_put_contents($debugDir . '/debug_parse_' . time() . '.html', $html);
        
        // Suche nach verschiedenen Tabellen-Patterns
        $tablePatterns = [
            '/<table[^>]*id="WheelsetSearchResult"[^>]*>(.*?)<\/table>/s',
            '/<table[^>]*class="[^"]*table[^"]*"[^>]*>(.*?)<\/table>/s',
            '/<table[^>]*>(.*?)<\/table>/s'
        ];
        
        $tableFound = false;
        foreach ($tablePatterns as $pattern) {
            if (preg_match($pattern, $html, $tableMatch)) {
                $tableHtml = $tableMatch[1];
                $this->debugLog("Tabelle gefunden mit Pattern");
                $tableFound = true;
                break;
            }
        }
        
        if ($tableFound) {
            // Parse Tabellenzeilen
            if (preg_match_all('/<tr[^>]*>(.*?)<\/tr>/s', $tableHtml, $rowMatches)) {
                $this->debugLog("Gefundene Zeilen: " . count($rowMatches[1]));
                
                foreach ($rowMatches[1] as $index => $rowHtml) {
                    if ($index === 0) continue; // Überspringe Header
                    
                    // Parse Zellen
                    if (preg_match_all('/<td[^>]*>(.*?)<\/td>/s', $rowHtml, $cellMatches)) {
                        $cells = $cellMatches[1];
                        
                        if (count($cells) >= 4) {
                            $wheelset = [
                                'satznummer' => $this->cleanCell($cells[0]),
                                'kennzeichen' => $this->cleanCell($cells[1]),
                                'fahrzeug' => $this->cleanCell($cells[2]),
                                'halter' => $this->cleanCell($cells[3] ?? 'Unbekannt'),
                                'eingangsdatum' => $this->cleanCell($cells[4] ?? ''),
                                'status' => 'eingelagert'
                            ];
                            
                            // Validiere Reifensatz - strenge Validierung
                            if (!empty($wheelset['satznummer']) && 
                                $wheelset['satznummer'] !== '&nbsp;' && 
                                $wheelset['satznummer'] !== '&nbsp' &&
                                $wheelset['satznummer'] !== '' &&
                                $wheelset['satznummer'] !== ' ' &&
                                is_numeric($wheelset['satznummer']) &&
                                strlen($wheelset['satznummer']) >= 6) {
                                
                                $wheelsets[] = $wheelset;
                                if (count($wheelsets) <= 5) { // Debug nur erste 5
                                    $this->debugLog("Reifensatz gefunden: " . $wheelset['satznummer'] . " - " . $wheelset['kennzeichen']);
                                }
                            } else {
                                if ($index <= 5) { // Debug nur erste 5 ungültige
                                    $this->debugLog("Ungültiger Reifensatz $index: Satznummer='" . $wheelset['satznummer'] . "', Kennzeichen='" . $wheelset['kennzeichen'] . "'");
                                }
                            }
                        }
                    }
                }
            }
        } else {
            $this->debugLog("Keine Tabelle gefunden, suche nach anderen Patterns");
            
            // Fallback: Suche nach LGS_ID Pattern
            if (preg_match_all('/LGS_ID[^>]*>([^<]+)</', $html, $matches)) {
                $this->debugLog("LGS_ID Pattern gefunden: " . count($matches[1]) . " Einträge");
                $lgsData = $matches[1];
                $chunks = array_chunk($lgsData, 5);
                
                foreach ($chunks as $index => $chunk) {
                    if ($index === 0) continue; // Überspringe Header
                    
                    if (count($chunk) >= 5) {
                        $wheelset = [
                            'satznummer' => trim($chunk[0]),
                            'kennzeichen' => trim($chunk[1]),
                            'fahrzeug' => trim($chunk[2]),
                            'halter' => $this->findCustomerName($html, trim($chunk[0])),
                            'eingangsdatum' => trim($chunk[4]),
                            'status' => 'eingelagert'
                        ];
                        
                        // Validiere Reifensatz
                        if (!empty($wheelset['satznummer']) && $wheelset['satznummer'] !== '&nbsp;') {
                            $wheelsets[] = $wheelset;
                        }
                    }
                }
            }
            
            // Weitere Fallback-Patterns
            if (empty($wheelsets)) {
                // Suche nach Satznummer-Pattern
                if (preg_match_all('/(\d{7,})/', $html, $matches)) {
                    $this->debugLog("Satznummer-Pattern gefunden: " . count($matches[1]) . " Einträge");
                    foreach ($matches[1] as $satznummer) {
                        $wheelset = [
                            'satznummer' => $satznummer,
                            'kennzeichen' => 'Unbekannt',
                            'fahrzeug' => 'Unbekannt',
                            'halter' => 'Unbekannt',
                            'eingangsdatum' => date('d.m.Y'),
                            'status' => 'eingelagert'
                        ];
                        $wheelsets[] = $wheelset;
                    }
                }
            }
        }
        
        $this->debugLog("Geparste Reifensätze: " . count($wheelsets));
        return $wheelsets;
    }
    
    /**
     * Bereinigt Tabellenzellen
     */
    private function cleanCell($cell) {
        $cleaned = strip_tags($cell);
        $cleaned = trim($cleaned);
        $cleaned = str_replace('&nbsp;', '', $cleaned);
        $cleaned = str_replace('&nbsp', '', $cleaned);
        $cleaned = str_replace('&amp;', '&', $cleaned);
        $cleaned = str_replace('&lt;', '<', $cleaned);
        $cleaned = str_replace('&gt;', '>', $cleaned);
        $cleaned = str_replace('&quot;', '"', $cleaned);
        $cleaned = preg_replace('/\s+/', ' ', $cleaned); // Mehrfache Leerzeichen zu einem
        $cleaned = trim($cleaned);
        return $cleaned;
    }
    
    /**
     * Findet Kundenname in HTML
     */
    private function findCustomerName($html, $satznummer) {
        // Suche nach Kundenname in der Nähe der Satznummer
        $patterns = [
            '/[^>]*' . preg_quote($satznummer, '/') . '[^<]*<[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            '/<div[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            '/<td[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                $candidate = trim(strip_tags($matches[1]));
                if (strlen($candidate) > 3 && strlen($candidate) < 100) {
                    return $candidate;
                }
            }
        }
        
        return 'Unbekannt';
    }
    
    /**
     * Prüft Cache
     */
    private function checkCache() {
        if (!$this->cacheManager) {
            return null;
        }
        
        $cachedWheelsets = $this->cacheManager->getCachedWheelsets($this->autohausId);
        if ($cachedWheelsets !== null) {
            $this->debugLog("Daten aus Cache geladen: " . count($cachedWheelsets) . " Reifensätze");
            return [
                'success' => true,
                'wheelsets' => $cachedWheelsets,
                'count' => count($cachedWheelsets),
                'cached' => true,
                'cache_time' => 'Aus Cache geladen'
            ];
        }
        
        return null;
    }
    
    /**
     * Validiert Reifensätze
     */
    private function validateWheelsets($wheelsets) {
        $validated = [];
        $seen = [];
        
        $this->debugLog("Validiere " . count($wheelsets) . " Reifensätze");
        
        foreach ($wheelsets as $index => $wheelset) {
            // Validiere erforderliche Felder
            if (empty($wheelset['satznummer']) || empty($wheelset['kennzeichen'])) {
                if ($index < 10) { // Debug nur für erste 10
                    $this->debugLog("Überspringe Reifensatz $index: Satznummer='" . ($wheelset['satznummer'] ?? 'LEER') . "', Kennzeichen='" . ($wheelset['kennzeichen'] ?? 'LEER') . "'");
                }
                continue;
            }
            
            // Bereinige Daten zuerst
            $wheelset['satznummer'] = trim(strip_tags($wheelset['satznummer']));
            $wheelset['kennzeichen'] = trim(strip_tags($wheelset['kennzeichen']));
            $wheelset['halter'] = trim(strip_tags($wheelset['halter'] ?? 'Unbekannt'));
            $wheelset['fahrzeug'] = trim(strip_tags($wheelset['fahrzeug'] ?? ''));
            $wheelset['eingangsdatum'] = trim(strip_tags($wheelset['eingangsdatum'] ?? ''));
            $wheelset['status'] = 'eingelagert';
            
            // Prüfe auf Duplikate (nur basierend auf Satznummer, da Kennzeichen sich wiederholen können)
            $key = $wheelset['satznummer'];
            if (isset($seen[$key])) {
                if ($index < 10) { // Debug nur für erste 10
                    $this->debugLog("Duplikat gefunden: " . $wheelset['satznummer']);
                }
                continue;
            }
            $seen[$key] = true;
            
            $validated[] = $wheelset;
            
            if ($index < 10) { // Debug nur für erste 10
                $this->debugLog("Validiert: " . $wheelset['satznummer'] . " - " . $wheelset['kennzeichen']);
            }
        }
        
        $this->debugLog("Validierung abgeschlossen: " . count($validated) . " von " . count($wheelsets) . " Reifensätzen");
        return $validated;
    }
    
    /**
     * Speichert im Cache
     */
    private function saveToCache($wheelsets) {
        if (!$this->cacheManager) {
            return;
        }
        
        $this->cacheManager->setCachedWheelsets($this->autohausId, 20, $wheelsets);
        $this->debugLog("Daten im Cache gespeichert: " . count($wheelsets) . " Reifensätze");
    }
    
    /**
     * Behandelt Fehler
     */
    private function handleError($exception) {
        $this->debugLog("=== FEHLER BEHANDLUNG ===");
        $this->debugLog("Fehler: " . $exception->getMessage());
        
        // Fallback auf lokale Datenbank
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            $stmt = $conn->prepare("
                SELECT 
                    id,
                    CONCAT(YEAR(created_at), '.', LPAD(MONTH(created_at), 2, '0'), '.', LPAD(DAY(created_at), 2, '0')) as satznummer,
                    COALESCE(license_plate, 'Unbekannt') as kennzeichen,
                    COALESCE(vehicle_id, 'Unbekannt') as fahrzeug,
                    COALESCE(customer_name, 'Unbekannt') as halter,
                    DATE_FORMAT(created_at, '%d.%m.%Y') as eingangsdatum,
                    'eingelagert' as status,
                    NULL as prufdoc_id
                FROM appointments 
                WHERE autohaus_id = ? 
                AND status IN ('confirmed', 'completed')
                ORDER BY created_at DESC
                LIMIT 50
            ");
            $stmt->execute([$this->autohausId]);
            $wheelsets = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($wheelsets)) {
                $this->debugLog("Fallback: Lokale Datenbank verwendet");
                return [
                    'success' => true,
                    'wheelsets' => $wheelsets,
                    'count' => count($wheelsets),
                    'info' => 'Fallback: Lokale Datenbank',
                    'cached' => false,
                    'debug_log' => $this->debugLog
                ];
            }
        } catch (Exception $e) {
            $this->debugLog("Fallback fehlgeschlagen: " . $e->getMessage());
        }
        
        // Letzter Fallback: Mock-Daten
        $this->debugLog("Letzter Fallback: Mock-Daten");
        return [
            'success' => true,
            'wheelsets' => $this->getMockWheelsets(),
            'count' => 3,
            'warning' => 'Demo-Daten (System nicht erreichbar)',
            'cached' => false,
            'debug_log' => $this->debugLog
        ];
    }
    
    /**
     * Generiert Mock-Daten
     */
    private function getMockWheelsets() {
        return [
            [
                'satznummer' => '2025.001.001',
                'kennzeichen' => 'KI-AB 123',
                'fahrzeug' => 'VIN123456789',
                'halter' => 'Max Mustermann GmbH',
                'eingangsdatum' => date('d.m.Y'),
                'status' => 'eingelagert',
                'prufdoc_id' => '12345'
            ],
            [
                'satznummer' => '2025.001.002',
                'kennzeichen' => 'KI-CD 456',
                'fahrzeug' => 'VIN987654321',
                'halter' => 'Anna Schmidt',
                'eingangsdatum' => date('d.m.Y', strtotime('-1 day')),
                'status' => 'eingelagert',
                'prufdoc_id' => '12346'
            ],
            [
                'satznummer' => '2025.001.003',
                'kennzeichen' => 'KI-EF 789',
                'fahrzeug' => 'VIN456789123',
                'halter' => 'Peter Müller & Co. KG',
                'eingangsdatum' => date('d.m.Y', strtotime('-2 days')),
                'status' => 'eingelagert',
                'prufdoc_id' => '12347'
            ]
        ];
    }
    
    /**
     * Debug-Logging
     */
    private function debugLog($message) {
        $timestamp = date('H:i:s');
        $logMessage = "[$timestamp] $message";
        $this->debugLog[] = $logMessage;
        error_log("SmartResyAPI: $logMessage");
    }
}

// API-Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['autohaus_id'])) {
        echo json_encode([
            'success' => false,
            'error' => 'Autohaus-ID ist erforderlich'
        ]);
        exit;
    }
    
    $autohausId = $input['autohaus_id'];
    
    try {
        $api = new SmartResyAPI($autohausId);
        $result = $api->getWheelsets();
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'API-Fehler: ' . $e->getMessage()
        ]);
    }
} else {
    echo json_encode([
        'success' => false,
        'error' => 'Nur POST-Requests erlaubt'
    ]);
}
?>
