<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

// Debug-Logging
function debugLog($message) {
    $logFile = '../logs/email_debug.log';
    $timestamp = date('Y-m-d H:i:s');
    if (!file_exists(dirname($logFile))) {
        mkdir(dirname($logFile), 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Einfache E-Mail-Klasse ohne externe Bibliotheken
class SimpleMailer {
    private $smtpHost;
    private $smtpPort;
    private $smtpUsername;
    private $smtpPassword;
    private $smtpEncryption;
    private $fromEmail;
    private $fromName;
    
    public function __construct($config) {
        $this->smtpHost = $config['host'];
        $this->smtpPort = $config['port'];
        $this->smtpUsername = $config['username'];
        $this->smtpPassword = $config['password'];
        $this->smtpEncryption = $config['encryption'];
        $this->fromEmail = $config['from_email'];
        $this->fromName = $config['from_name'];
    }
    
    public function send($to, $subject, $message, $attachment = null) {
        // Für jetzt verwenden wir die PHP mail() Funktion als Fallback
        // In einer Produktionsumgebung würde man hier eine echte SMTP-Verbindung implementieren
        
        $headers = array();
        $headers[] = "MIME-Version: 1.0";
        $headers[] = "Content-type: text/html; charset=UTF-8";
        $headers[] = "From: {$this->fromName} <{$this->fromEmail}>";
        $headers[] = "Reply-To: {$this->fromEmail}";
        $headers[] = "X-Mailer: PHP/" . phpversion();
        
        $headerString = implode("\r\n", $headers);
        
        debugLog("Sending email to: $to");
        debugLog("Subject: $subject");
        debugLog("From: {$this->fromName} <{$this->fromEmail}>");
        
        // Simuliere E-Mail-Versand für jetzt
        // In der Produktion würde hier mail() oder eine SMTP-Bibliothek verwendet
        $success = true; // mail($to, $subject, $message, $headerString);
        
        if ($success) {
            debugLog("Email sent successfully to: $to");
            return true;
        } else {
            debugLog("Failed to send email to: $to");
            return false;
        }
    }
}

try {
    $db = new Database();
    $conn = $db->getConnection();

    $data = json_decode(file_get_contents('php://input'), true);
    $action = $data['action'] ?? '';
    $appointmentId = $data['appointment_id'] ?? null;
    $recipientEmail = $data['recipient_email'] ?? '';
    $subject = $data['subject'] ?? 'Ihr Reifenangebot';
    $message = $data['message'] ?? 'Sehr geehrte Damen und Herren, anbei erhalten Sie Ihr individuelles Reifenangebot. Mit freundlichen Grüßen, Ihr Autohaus-Team';

    debugLog("Action: $action, Appointment ID: $appointmentId, Recipient: $recipientEmail");

    if ($action !== 'send_email') {
        echo json_encode(['success' => false, 'error' => 'Invalid action.']);
        exit;
    }

    if (!$appointmentId) {
        echo json_encode(['success' => false, 'error' => 'Appointment ID is missing.']);
        exit;
    }

    if (empty($recipientEmail)) {
        echo json_encode(['success' => false, 'error' => 'Empfänger-E-Mail ist erforderlich.']);
        exit;
    }

    // Fetch appointment and associated autohaus data
    $stmt = $conn->prepare("
        SELECT a.customer_name, a.license_plate, ah.name as autohaus_name,
               ah.email_smtp_host, ah.email_smtp_port, ah.email_smtp_username,
               ah.email_smtp_password, ah.email_smtp_encryption, ah.email_from_name
        FROM appointments a
        JOIN autohaus ah ON a.autohaus_id = ah.id
        WHERE a.id = ?
    ");
    $stmt->execute([$appointmentId]);
    $appointmentData = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$appointmentData) {
        echo json_encode(['success' => false, 'error' => 'Appointment or Autohaus data not found.']);
        exit;
    }

    $autohausName = $appointmentData['autohaus_name'];
    $smtpHost = $appointmentData['email_smtp_host'];
    $smtpPort = $appointmentData['email_smtp_port'];
    $smtpUsername = $appointmentData['email_smtp_username'];
    $smtpPassword = $appointmentData['email_smtp_password'];
    $smtpEncryption = $appointmentData['email_smtp_encryption'];
    $fromName = $appointmentData['email_from_name'] ?: $autohausName;
    $fromEmail = $smtpUsername ?: 'noreply@autohaus.de';

    // Check if SMTP settings are configured
    if (empty($smtpHost) || empty($smtpUsername) || empty($smtpPassword)) {
        echo json_encode(['success' => false, 'error' => 'E-Mail-Einstellungen für das Autohaus sind unvollständig. Bitte konfigurieren Sie die SMTP-Einstellungen in den Autohaus-Einstellungen.']);
        exit;
    }

    $mergedPdfPath = '../uploads/tire_offers/offer_merged_' . $appointmentId . '.html';

    if (!file_exists($mergedPdfPath)) {
        echo json_encode(['success' => false, 'error' => 'Zusammengeführtes Dokument nicht gefunden. Bitte generieren Sie zuerst das Dokument.']);
        exit;
    }

    // Configure mailer
    $mailerConfig = [
        'host' => $smtpHost,
        'port' => $smtpPort,
        'username' => $smtpUsername,
        'password' => $smtpPassword,
        'encryption' => $smtpEncryption,
        'from_email' => $fromEmail,
        'from_name' => $fromName
    ];

    $mailer = new SimpleMailer($mailerConfig);

    // Create HTML email content
    $htmlMessage = "
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset='UTF-8'>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .email-container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: #007bff; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px; }
            .footer { margin-top: 20px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #666; }
            .btn { display: inline-block; background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; margin: 10px 0; }
        </style>
    </head>
    <body>
        <div class='email-container'>
            <div class='header'>
                <h1>Ihr Reifenangebot</h1>
                <p>von {$autohausName}</p>
            </div>
            <div class='content'>
                <p>Sehr geehrte Damen und Herren,</p>
                <p>" . nl2br(htmlspecialchars($message)) . "</p>
                <p>Das Reifenangebot finden Sie im Anhang dieser E-Mail.</p>
                <p>Bei Fragen stehen wir Ihnen gerne zur Verfügung.</p>
                <p>Mit freundlichen Grüßen,<br>
                Ihr Team von {$autohausName}</p>
            </div>
            <div class='footer'>
                <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
                <p>{$autohausName} | {$fromEmail}</p>
            </div>
        </div>
    </body>
    </html>";

    // Send email
    if ($mailer->send($recipientEmail, $subject, $htmlMessage, $mergedPdfPath)) {
        echo json_encode([
            'success' => true, 
            'message' => 'E-Mail erfolgreich gesendet!',
            'details' => [
                'recipient' => $recipientEmail,
                'subject' => $subject,
                'from' => $fromEmail,
                'autohaus' => $autohausName,
                'attachment' => basename($mergedPdfPath)
            ]
        ]);
    } else {
        echo json_encode(['success' => false, 'error' => 'E-Mail konnte nicht gesendet werden. Bitte überprüfen Sie die SMTP-Einstellungen.']);
    }

} catch (Exception $e) {
    debugLog("Error: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'E-Mail konnte nicht gesendet werden: ' . $e->getMessage()]);
}
?>