<?php
/**
 * RESY Data Extraction API - Debug Login Version
 * Mit erweitertem Debugging für Login-Probleme
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Debug-Logging-Funktion
function debugLog($message) {
    $logFile = __DIR__ . '/../logs/resy_extraction.log';
    $timestamp = date('Y-m-d H:i:s');
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// HTTP-Helper-Funktion
function httpRequest($url, $options = []) {
    $defaultOptions = [
        'method' => 'GET',
        'headers' => [],
        'data' => null,
        'cookies' => [],
        'follow_redirects' => true,
        'timeout' => 30
    ];
    
    $options = array_merge($defaultOptions, $options);
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, $options['follow_redirects']);
    curl_setopt($ch, CURLOPT_TIMEOUT, $options['timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    if ($options['method'] === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($options['data']) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $options['data']);
        }
    }
    
    if (!empty($options['headers'])) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
    }
    
    if (!empty($options['cookies'])) {
        $cookieString = '';
        foreach ($options['cookies'] as $name => $value) {
            $cookieString .= $name . '=' . $value . '; ';
        }
        curl_setopt($ch, CURLOPT_COOKIE, rtrim($cookieString, '; '));
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception("cURL Error: $error");
    }
    
    return [
        'body' => $response,
        'http_code' => $httpCode,
        'url' => $url
    ];
}

// HTML-Parsing-Helper-Funktionen
function extractFormData($html, $formAction = null) {
    $dom = new DOMDocument();
    @$dom->loadHTML($html);
    $xpath = new DOMXPath($dom);
    
    $forms = $xpath->query('//form');
    debugLog("Gefundene Formulare: " . $forms->length);
    
    foreach ($forms as $form) {
        $action = $form->getAttribute('action');
        debugLog("Formular-Action: $action");
        
        if ($formAction && $action !== $formAction) {
            continue;
        }
        
        $formData = [];
        $inputs = $xpath->query('.//input', $form);
        debugLog("Gefundene Input-Felder: " . $inputs->length);
        
        foreach ($inputs as $input) {
            $name = $input->getAttribute('name');
            $value = $input->getAttribute('value');
            $type = $input->getAttribute('type');
            
            debugLog("Input: name='$name', type='$type', value='$value'");
            
            if ($name && $type !== 'submit') {
                $formData[$name] = $value;
            }
        }
        
        $selects = $xpath->query('.//select', $form);
        foreach ($selects as $select) {
            $name = $select->getAttribute('name');
            $options = $xpath->query('.//option[@selected]', $select);
            if ($options->length > 0) {
                $formData[$name] = $options->item(0)->getAttribute('value');
            }
        }
        
        return [
            'action' => $action,
            'method' => $form->getAttribute('method') ?: 'POST',
            'data' => $formData
        ];
    }
    
    return null;
}

// Verbesserte Login-Funktion mit Debugging
function resyLoginDebug($username, $password, $baseUrl) {
    debugLog("=== DEBUG LOGIN START ===");
    debugLog("Username: $username");
    debugLog("Password: " . str_repeat('*', strlen($password)));
    debugLog("Base URL: $baseUrl");
    
    try {
        // Schritt 1: Login-Seite laden
        $loginUrl = $baseUrl . '/index.php?m=login&a=login';
        debugLog("Lade Login-Seite: $loginUrl");
        
        $response = httpRequest($loginUrl);
        debugLog("Login-Seite Response Code: " . $response['http_code']);
        debugLog("Login-Seite Response Length: " . strlen($response['body']));
        
        if ($response['http_code'] !== 200) {
            throw new Exception("Login-Seite konnte nicht geladen werden. HTTP Code: " . $response['http_code']);
        }
        
        // Prüfe ob Login-Seite korrekt geladen wurde
        if (strpos($response['body'], 'Anmeldename') === false) {
            debugLog("WARNUNG: 'Anmeldename' nicht in Response gefunden");
            debugLog("Response Preview: " . substr($response['body'], 0, 500));
        } else {
            debugLog("Login-Seite korrekt geladen (Anmeldename gefunden)");
        }
        
        // Schritt 2: Formular-Daten extrahieren
        debugLog("Extrahiere Formular-Daten...");
        $formData = extractFormData($response['body']);
        
        if (!$formData) {
            debugLog("FEHLER: Kein Login-Formular gefunden");
            debugLog("Response Preview: " . substr($response['body'], 0, 1000));
            throw new Exception("Login-Formular nicht gefunden");
        }
        
        debugLog("Formular-Daten extrahiert:");
        debugLog("Action: " . $formData['action']);
        debugLog("Method: " . $formData['method']);
        debugLog("Data: " . json_encode($formData['data']));
        
        // Schritt 3: Login-Daten setzen
        $formData['data']['UserName'] = $username;
        $formData['data']['Password'] = $password;
        $formData['data']['button'] = 'submit';
        
        debugLog("Login-Daten gesetzt:");
        debugLog("UserName: " . $formData['data']['UserName']);
        debugLog("Password: " . str_repeat('*', strlen($formData['data']['Password'])));
        debugLog("Button: " . $formData['data']['button']);
        
        // Schritt 4: Login-Request senden
        $loginAction = $baseUrl . '/' . $formData['action'];
        debugLog("Sende Login-Request an: $loginAction");
        
        $loginResponse = httpRequest($loginAction, [
            'method' => 'POST',
            'data' => http_build_query($formData['data']),
            'headers' => [
                'Content-Type: application/x-www-form-urlencoded',
                'Referer: ' . $loginUrl
            ]
        ]);
        
        debugLog("Login-Response Code: " . $loginResponse['http_code']);
        debugLog("Login-Response Length: " . strlen($loginResponse['body']));
        
        // Schritt 5: Erfolg prüfen
        debugLog("Prüfe Login-Erfolg...");
        
        if (strpos($loginResponse['body'], 'Anmeldename') !== false) {
            debugLog("FEHLER: Anmeldeformular noch sichtbar - Login fehlgeschlagen");
            debugLog("Response Preview: " . substr($loginResponse['body'], 0, 1000));
            throw new Exception("Login fehlgeschlagen - Anmeldeformular noch sichtbar");
        }
        
        if (strpos($loginResponse['body'], 'Kennwort') !== false) {
            debugLog("FEHLER: Kennwort-Feld noch sichtbar - Login fehlgeschlagen");
            throw new Exception("Login fehlgeschlagen - Kennwort-Feld noch sichtbar");
        }
        
        debugLog("Login erfolgreich! Anmeldeformular nicht mehr sichtbar");
        
        return [
            'success' => true,
            'sessionData' => [
                'baseUrl' => $baseUrl,
                'loggedIn' => true,
                'timestamp' => time(),
                'username' => $username
            ]
        ];
        
    } catch (Exception $e) {
        debugLog("Login-Fehler: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
    
    debugLog("=== DEBUG LOGIN END ===");
}

// Hauptverarbeitung
try {
    debugLog("=== DEBUG LOGIN API Request Started ===");
    
    // JSON-Input lesen
    $rawInput = file_get_contents('php://input');
    debugLog("Raw Input Length: " . strlen($rawInput));
    
    if (empty($rawInput)) {
        throw new Exception('Keine Eingabedaten empfangen');
    }
    
    $input = json_decode($rawInput, true);
    
    if (!$input) {
        $error = json_last_error_msg();
        debugLog("JSON Decode Error: " . $error);
        throw new Exception('Keine gültigen JSON-Daten empfangen: ' . $error);
    }
    
    $action = $input['action'] ?? '';
    debugLog("Empfangene Aktion: $action");
    
    // Datenbankverbindung
    try {
        require_once __DIR__ . '/../config/database.php';
        $db = new Database();
        $conn = $db->getConnection();
        debugLog("Datenbankverbindung erfolgreich");
    } catch (Exception $e) {
        debugLog("Datenbankfehler: " . $e->getMessage());
        throw new Exception('Datenbankfehler: ' . $e->getMessage());
    }
    
    switch ($action) {
        case 'login':
            $appointmentId = $input['appointment_id'] ?? null;
            debugLog("Debug Login für Appointment ID: $appointmentId");
            
            if (!$appointmentId) {
                throw new Exception('Appointment ID fehlt');
            }
            
            // Autohaus-Daten aus Appointment laden
            $stmt = $conn->prepare("
                SELECT a.resy_username, a.resy_password, a.resy_base_url 
                FROM autohaus a 
                JOIN appointments ap ON a.id = ap.autohaus_id 
                WHERE ap.id = ?
            ");
            $stmt->execute([$appointmentId]);
            $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$autohaus) {
                throw new Exception('Autohaus-Daten nicht gefunden für Appointment ID: ' . $appointmentId);
            }
            
            debugLog("Autohaus gefunden: " . json_encode($autohaus));
            
            $result = resyLoginDebug($autohaus['resy_username'], $autohaus['resy_password'], $autohaus['resy_base_url']);
            echo json_encode($result);
            break;
            
        default:
            throw new Exception("Unbekannte Aktion: $action");
    }
    
} catch (Exception $e) {
    debugLog("API-Fehler: " . $e->getMessage());
    debugLog("Stack Trace: " . $e->getTraceAsString());
    
    // Sicherstellen, dass nur JSON ausgegeben wird
    http_response_code(500);
    $result = [
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'timestamp' => date('Y-m-d H:i:s')
        ]
    ];
    echo json_encode($result);
}

debugLog("=== DEBUG LOGIN API Request Ended ===");
?>
