<?php
/**
 * API zum Laden eingelagerter Reifensätze aus dem RESY-System
 */

require_once 'cache_manager.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

session_start();
require_once dirname(__DIR__) . '/config/database.php';

class ResyWheelsetService {
    private $baseUrl = 'https://resy.dtm-reifen.de';
    private $username;
    private $password;
    private $dealerId;
    private $autohausId;
    private $cacheManager;
    
    public function __construct($autohausId = null) {
        $this->autohausId = $autohausId;
        $this->loadAutohausCredentials();
        
        // Initialisiere CacheManager nur wenn Datenbankverbindung verfügbar ist
        try {
            if (isset($GLOBALS['conn']) && $GLOBALS['conn']) {
                $this->cacheManager = new CacheManager($GLOBALS['conn']);
            } else {
                error_log("Keine Datenbankverbindung für Cache verfügbar");
                $this->cacheManager = null;
            }
        } catch (Exception $e) {
            error_log("Fehler beim Initialisieren des CacheManagers: " . $e->getMessage());
            $this->cacheManager = null;
        }
    }
    
    private function loadAutohausCredentials() {
        try {
            $db = new Database();
            $conn = $db->getConnection();
            
            if ($this->autohausId) {
                $stmt = $conn->prepare("SELECT resy_username, resy_password, resy_dealer_id, resy_base_url, name FROM autohaus WHERE id = ? AND is_active = 1");
                $stmt->execute([$this->autohausId]);
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($autohaus) {
                    $this->username = $autohaus['resy_username'];
                    $this->password = $autohaus['resy_password'];
                    $this->baseUrl = $autohaus['resy_base_url'] ?: 'https://resy.dtm-reifen.de';
                    $this->dealerId = $this->getResyKdId($autohaus['name'], $autohaus['resy_dealer_id']);
                } else {
                    throw new Exception('Autohaus nicht gefunden oder inaktiv');
                }
            } else {
                // Fallback zu Standard-Credentials
                $this->username = 'D.ULUDAG';
                $this->password = 'Ikizler123!';
                $this->dealerId = '453'; // Standard KdId für Kiel
            }
        } catch (Exception $e) {
            error_log("Error loading autohaus credentials: " . $e->getMessage());
            $this->username = 'D.ULUDAG';
            $this->password = 'Ikizler123!';
            $this->dealerId = '453';
        }
    }
    
    /**
     * Konvertiert Autohaus-Namen zur korrekten RESY KdId
     */
    private function getResyKdId($autohausName, $resyDealerId) {
        $mapping = [
            'B & E Volvo Kiel' => '453',
            'B & E Volvo Norderstedt' => '452', 
            'B & E Volvo Glinde' => '454',
            'Kiel' => '453',
            'Norderstedt' => '452',
            'Glinde' => '454'
        ];
        
        // Prüfe zuerst den vollständigen Namen
        if (isset($mapping[$autohausName])) {
            return $mapping[$autohausName];
        }
        
        // Prüfe auch Teilstrings
        foreach ($mapping as $name => $kdid) {
            if (strpos($autohausName, $name) !== false) {
                return $kdid;
            }
        }
        
        // Fallback: Prüfe resy_dealer_id
        if (strpos($resyDealerId, 'Kiel') !== false) {
            return '453';
        } elseif (strpos($resyDealerId, 'Norderstedt') !== false) {
            return '452';
        } elseif (strpos($resyDealerId, 'Glinde') !== false) {
            return '454';
        }
        
        // Letzter Fallback
        return '453';
    }
    
    /**
     * Lädt eingelagerte Reifensätze aus dem RESY-System
     */
    public function getStoredWheelsets() {
        try {
            // Prüfe Cache zuerst
            if ($this->cacheManager && $this->cacheManager->isCacheValid($this->autohausId)) {
                $cachedWheelsets = $this->cacheManager->getCachedWheelsets($this->autohausId);
                if ($cachedWheelsets !== null) {
                    return [
                        'success' => true,
                        'wheelsets' => $cachedWheelsets,
                        'count' => count($cachedWheelsets),
                        'cached' => true,
                        'cache_time' => 'Aus Cache geladen'
                    ];
                }
            }
            
            // Prüfe HTML-Cache
            if ($this->cacheManager) {
                $cachedHtml = $this->cacheManager->getCachedHtml($this->autohausId);
                if ($cachedHtml !== null) {
                    error_log("Verwende gecachte HTML-Response");
                    $wheelsets = $this->parseWheelsetsFromResponse($cachedHtml);
                    
                    // Speichere verarbeitete Daten im Cache
                    $this->cacheManager->setCachedWheelsets($this->autohausId, 20, $wheelsets);
                    
                    return [
                        'success' => true,
                        'wheelsets' => $wheelsets,
                        'count' => count($wheelsets),
                        'cached' => true,
                        'cache_time' => 'Aus HTML-Cache geladen'
                    ];
                }
            }
            
            $ch = curl_init();
            
            // Debug: Logge die verwendeten Credentials
            error_log("RESY Login Debug - Autohaus ID: " . $this->autohausId . ", Username: " . $this->username . ", Base URL: " . $this->baseUrl);
        
        // Schritt 1: Login-Seite laden (wie in tire_order.php)
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1'
        ]);
        
        // Wichtig: Gzip-Dekompression aktivieren
        curl_setopt($ch, CURLOPT_ENCODING, 'gzip,deflate');
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Laden der Login-Seite: ' . curl_error($ch)];
        }
        
        // Schritt 2: Login-Daten senden (wie in tire_order.php)
        $postData = [
            'FN' => 'login',
            'UserName' => $this->username,
            'Password' => $this->password,
            'button' => 'submit'
        ];
        
        curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Referer: ' . $this->baseUrl . '/index.php?m=login&a=login'
        ]);
        
        $loginResponse = curl_exec($ch);
        
        if (curl_error($ch)) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Fehler beim Login: ' . curl_error($ch)];
        }
        
        // Debug: Speichere die Login-Antwort
        file_put_contents(sys_get_temp_dir() . '/resy_login_response_' . $this->autohausId . '.html', $loginResponse);
        
        // Prüfe ob Login erfolgreich war - verbesserte Erkennung
        $loginSuccessful = false;
        
        // Prüfe auf Login-Formular (sollte NICHT vorhanden sein)
        // Nur "Anmeldename" ist ein zuverlässiger Indikator für das Login-Formular
        $hasLoginForm = strpos($loginResponse, 'Anmeldename') !== false;
        
        // Prüfe auf Erfolgs-Indikatoren (sollten vorhanden sein)
        $hasSuccessIndicators = strpos($loginResponse, 'Lagerbestand') !== false ||
                               strpos($loginResponse, 'RESY Websystem') !== false ||
                               strpos($loginResponse, 'Übersicht') !== false ||
                               strpos($loginResponse, 'dashboard') !== false;
        
        if (!$hasLoginForm && $hasSuccessIndicators) {
            $loginSuccessful = true;
            error_log("Login erfolgreich für Autohaus ID: " . $this->autohausId . " - Kein Login-Formular, Erfolgs-Indikatoren gefunden");
        } elseif (!$hasLoginForm) {
            $loginSuccessful = true;
            error_log("Login erfolgreich für Autohaus ID: " . $this->autohausId . " - Kein Login-Formular gefunden");
        } else {
            error_log("Login fehlgeschlagen für Autohaus ID: " . $this->autohausId . " - Login-Formular erkannt");
        }
        
        if (!$loginSuccessful) {
            curl_close($ch);
            return ['success' => false, 'error' => 'Login fehlgeschlagen - ungültige Credentials für Autohaus ID: ' . $this->autohausId . ' (Login-Seite erkannt)'];
        }
        
        // Schritt 2: Lade alle Seiten der Reifensätze
        $allWheelsets = [];
        $page = 1;
        $maxPages = 10; // Sicherheitslimit
        
        while ($page <= $maxPages) {
            error_log("Loading page $page of wheelsets");
            
            // Lade spezifische Seite
            curl_setopt($ch, CURLOPT_URL, $this->baseUrl . '/index.php?m=wheelset&a=search&Page=' . $page);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
                'FN' => 'WheelsetSearch',
                'KD_ID' => $this->dealerId, // Autohaus-spezifische KdId
                'STATUS_ID' => '20', // eingelagert
                'button' => 'submit'
            ]));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/x-www-form-urlencoded',
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Referer: ' . $this->baseUrl . '/index.php?m=wheelset&a=search'
            ]);
            
            $pageResponse = curl_exec($ch);
            
            if (curl_error($ch)) {
                error_log("Error loading page $page: " . curl_error($ch));
                break;
            }
            
            // Speichere HTML im Cache (nur erste Seite)
            if ($page === 1 && $this->cacheManager) {
                $this->cacheManager->setCachedHtml($this->autohausId, $pageResponse);
            }
            
            // Parse die Antwort für diese Seite
            $pageWheelsets = $this->parseWheelsetsFromResponse($pageResponse);
            
            if (empty($pageWheelsets)) {
                error_log("No wheelsets found on page $page, stopping");
                break;
            }
            
            $allWheelsets = array_merge($allWheelsets, $pageWheelsets);
            error_log("Page $page: Found " . count($pageWheelsets) . " wheelsets, total: " . count($allWheelsets));
            
            // Prüfe ob es eine nächste Seite gibt
            if (strpos($pageResponse, 'Page=' . ($page + 1)) === false) {
                error_log("No next page found, stopping at page $page");
                break;
            }
            
            $page++;
        }
        
        curl_close($ch);
        
        // Entferne Duplikate basierend auf Satznummer
        $uniqueWheelsets = [];
        $seenSatznummern = [];
        
        foreach ($allWheelsets as $wheelset) {
            $satznummer = $wheelset['satznummer'] ?? '';
            
            // Nur hinzufügen wenn Satznummer eindeutig und gültig
            if (!empty($satznummer) && 
                !in_array($satznummer, $seenSatznummern) &&
                preg_match('/^[0-9]+\.[0-9]+\.[0-9]+$/', $satznummer)) {
                
                $uniqueWheelsets[] = $wheelset;
                $seenSatznummern[] = $satznummer;
            }
        }
        
        $wheelsets = $uniqueWheelsets;
        error_log("After deduplication: " . count($wheelsets) . " unique wheelsets");
        
        // Speichere im Cache
        if ($this->cacheManager) {
            $this->cacheManager->setCachedWheelsets($this->autohausId, 20, $wheelsets);
        }
        
        return [
            'success' => true,
            'wheelsets' => $wheelsets,
            'count' => count($wheelsets),
            'autohaus_id' => $this->autohausId,
            'dealer_id' => $this->dealerId,
            'cached' => false,
            'cache_time' => 'Neu geladen und gecacht'
        ];
        
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Reifensätze: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'wheelsets' => [],
                'count' => 0,
                'cached' => false
            ];
        }
    }
    
    /**
     * Parst die HTML-Antwort und extrahiert Reifensätze
     */
    private function parseWheelsetsFromResponse($html) {
        $wheelsets = [];
        
        // Debug: Speichere die Antwort für Analyse
        file_put_contents(sys_get_temp_dir() . '/resy_wheelsets_response_' . $this->autohausId . '.html', $html);
        error_log("RESY Search Response saved to: " . sys_get_temp_dir() . '/resy_wheelsets_response_' . $this->autohausId . '.html');
        
        // Debug: Prüfe ob HTML überhaupt Reifensätze enthält
        if (strpos($html, 'LGS') !== false || strpos($html, 'Satznummer') !== false) {
            error_log("HTML contains LGS or Satznummer - should have wheelsets");
        } else {
            error_log("HTML does not contain LGS or Satznummer - might be empty or different format");
        }
        
        // Suche nach verschiedenen Tabellen-Formaten
        $patterns = [
            // Tabelle mit tbody (häufigste) - alle Tabellen
            '/<tbody[^>]*>(.*?)<\/tbody>/s',
            // Standard Tabelle mit class - alle Tabellen
            '/<table[^>]*class="[^"]*table[^"]*"[^>]*>(.*?)<\/table>/s',
            // Einfache Tabelle - alle Tabellen
            '/<table[^>]*>(.*?)<\/table>/s',
            // Div-basierte Tabelle - alle Tabellen
            '/<div[^>]*class="[^"]*table[^"]*"[^>]*>(.*?)<\/div>/s'
        ];
        
        foreach ($patterns as $patternIndex => $pattern) {
            error_log("Trying pattern " . ($patternIndex + 1) . ": " . $pattern);
            
            if (preg_match_all($pattern, $html, $matches)) {
                error_log("Found " . count($matches[1]) . " table(s) with pattern " . ($patternIndex + 1));
                
                foreach ($matches[1] as $tableIndex => $tableHtml) {
                    error_log("Processing table " . ($tableIndex + 1) . " with content length: " . strlen($tableHtml));
                    
                    // Suche nach Zeilen mit Reifensätzen
                    if (preg_match_all('/<tr[^>]*>(.*?)<\/tr>/s', $tableHtml, $rows)) {
                        error_log("Found " . count($rows[1]) . " rows in table " . ($tableIndex + 1));
                        
                        foreach ($rows[1] as $rowIndex => $row) {
                            // Überspringe Header-Zeilen
                            if ($rowIndex === 0 && (strpos($row, 'Satznummer') !== false || strpos($row, 'Kennzeichen') !== false || strpos($row, 'LGS') !== false)) {
                                error_log("Skipping header row " . $rowIndex);
                                continue;
                            }
                            
                            // Suche nach Zellen mit verschiedenen Patterns
                            $cellPatterns = [
                                '/<td[^>]*>(.*?)<\/td>/s',
                                '/<td[^>]*class="[^"]*"[^>]*>(.*?)<\/td>/s',
                                '/<td[^>]*>(.*?)<\/td>/s'
                            ];
                            
                            $cells = [];
                            foreach ($cellPatterns as $cellPattern) {
                                if (preg_match_all($cellPattern, $row, $cellMatches)) {
                                    $cells = $cellMatches[1];
                                    break;
                                }
                            }
                            
                            if (!empty($cells)) {
                                error_log("Row " . $rowIndex . " has " . count($cells) . " cells");
                                
                                if (count($cells) >= 19) {
                                    // RESY-Tabelle hat 19 Spalten, echte Daten beginnen bei Spalte 2
                                    // Spalte 2: Satznummer, 3: Kennzeichen, 4: Fahrzeug, 6: Autohaus-Standort, 9: Datum
                                    // Der echte Kundenname steht wahrscheinlich in einer anderen Spalte
                                    $satznummer = trim(strip_tags($cells[2] ?? ''));
                                    $kennzeichen = trim(strip_tags($cells[3] ?? ''));
                                    $fahrzeug = trim(strip_tags($cells[4] ?? ''));
                                    $autohaus_standort = trim(strip_tags($cells[6] ?? ''));
                                    $eingangsdatum = trim(strip_tags($cells[9] ?? ''));
                                    
                                    // Versuche verschiedene Spalten für den Kundenname
                                    $kunde = '';
                                    $possibleCustomerColumns = [5, 7, 8, 10, 11, 12, 13, 14, 15, 16, 17, 18];
                                    
                                    foreach ($possibleCustomerColumns as $colIndex) {
                                        if (isset($cells[$colIndex])) {
                                            $candidate = trim(strip_tags($cells[$colIndex]));
                                            if ($this->isValidCustomerName($candidate)) {
                                                $kunde = $candidate;
                                                error_log("Kundenname in Spalte $colIndex gefunden: $candidate");
                                                break;
                                            }
                                        }
                                    }
                                    
                                    // Falls kein Kundenname in den Spalten gefunden, suche in der HTML-Struktur
                                    if (empty($kunde)) {
                                        $kunde = $this->findCustomerNameInHtml($html, $satznummer);
                                    }
                                    
                                    // Falls kein Kundenname gefunden, verwende "Unbekannt"
                                    if (empty($kunde)) {
                                        $kunde = 'Kunde unbekannt';
                                    }
                                    
                                    // Überspringe Zeilen mit "Anz=" oder leeren Satznummern
                                    if (empty($satznummer) || strpos($satznummer, 'Anz=') !== false) {
                                        continue;
                                    }
                                    
                                    // Extrahiere PRUFDOC_ID für Prüfbericht-Download
                                    $prufdoc_id = $this->extractPrufdocId($html, $satznummer);
                                    
                                    // Extrahiere relevante Daten
                                    $wheelset = [
                                        'satznummer' => $satznummer,
                                        'kennzeichen' => $kennzeichen,
                                        'fahrzeug' => $fahrzeug,
                                        'halter' => $kunde, // Verwende den echten Kundenname
                                        'eingangsdatum' => $eingangsdatum,
                                        'status' => 'eingelagert',
                                        'prufdoc_id' => $prufdoc_id
                                    ];
                                    
                                    error_log("Extracted wheelset: " . json_encode($wheelset));
                                    
                                    // Nur hinzufügen wenn Satznummer vorhanden und nicht leer
                                    if (!empty($wheelset['satznummer']) && $wheelset['satznummer'] !== '&nbsp;' && $wheelset['satznummer'] !== '') {
                                        $wheelsets[] = $wheelset;
                                        error_log("Added wheelset: " . $wheelset['satznummer']);
                                    } else {
                                        error_log("Skipped wheelset due to empty satznummer: '" . $wheelset['satznummer'] . "'");
                                    }
                                }
                            } else {
                                error_log("No cells found in row " . $rowIndex);
                            }
                        }
                    }
                    
                    // Logge Fortschritt, aber breche nicht ab - verarbeite alle Tabellen
                    $realWheelsets = array_filter($wheelsets, function($wheelset) {
                        return !empty($wheelset['satznummer']) && 
                               $wheelset['satznummer'] !== '&nbsp;' && 
                               strpos($wheelset['satznummer'], 'Anz=') === false;
                    });
                    
                    error_log("Found " . count($realWheelsets) . " real wheelsets in table " . ($tableIndex + 1) . ", continuing...");
                }
            } else {
                error_log("No matches for pattern " . ($patternIndex + 1));
            }
        }
        
        // LGS_ID-basiertes Parsing (findet mehr Reifensätze)
        error_log("Trying LGS_ID-based parsing (finds more wheelsets)");
        $lgsWheelsets = $this->parseLgsIdBasedWheelsets($html);
        if (count($lgsWheelsets) > count($wheelsets)) {
            $wheelsets = $lgsWheelsets;
            error_log("LGS_ID parsing found more wheelsets: " . count($wheelsets));
        }
        
        // Fallback: Suche nach div-basierten Strukturen
        if (empty($wheelsets)) {
            error_log("No wheelsets found with LGS_ID parsing, trying div-based parsing");
            $wheelsets = $this->parseDivBasedWheelsets($html);
        }
        
        // Fallback: Suche nach einzelnen Reifensätzen in der gesamten HTML
        if (empty($wheelsets)) {
            error_log("No wheelsets found with div parsing, trying fallback");
            $wheelsets = $this->fallbackParse($html);
        }
        
        error_log("Final result: " . count($wheelsets) . " wheelsets found");
        return $wheelsets;
    }
    
    /**
     * Parsing für LGS_ID-basierte Strukturen (RESY-spezifisch)
     */
    private function parseLgsIdBasedWheelsets($html) {
        $wheelsets = [];
        
        error_log("Starting LGS_ID-based parsing");
        
        // Suche nach allen LGS_ID-Einträgen
        if (preg_match_all('/LGS_ID[^>]*>([^<]+)</', $html, $lgsMatches)) {
            error_log("Found " . count($lgsMatches[1]) . " LGS_ID entries");
            
            // Gruppiere die Daten in Reifensätze (jeder Reifensatz hat mehrere LGS_ID-Einträge)
            $currentWheelset = null;
            $wheelsetCount = 0;
            
            // Gruppiere die Daten in Blöcke von 5 Einträgen pro Reifensatz
            // Basierend auf der Analyse: Jeder Reifensatz hat 5 LGS_ID-Einträge
            $chunks = array_chunk($lgsMatches[1], 5);
            
            foreach ($chunks as $chunkIndex => $chunk) {
                // Überspringe den ersten Chunk (Header)
                if ($chunkIndex === 0) {
                    continue;
                }
                
                // Erwartete Reihenfolge pro Reifensatz (5 Einträge):
                // 0: Satznummer, 1: Kennzeichen, 2: Fahrzeug-ID, 3: Autohaus-Standort, 4: Datum
                
                if (count($chunk) >= 5) {
                    $satznummer = trim(strip_tags($chunk[0]));
                    $kennzeichen = trim(strip_tags($chunk[1]));
                    $fahrzeug = trim(strip_tags($chunk[2]));
                    $autohaus_standort = trim(strip_tags($chunk[3]));
                    $eingangsdatum = trim(strip_tags($chunk[4]));
                    
                    // Suche nach dem echten Kundenname in der gesamten HTML-Struktur
                    $kunde = $this->findCustomerNameInHtml($html, $satznummer);
                    
                    // Falls kein Kundenname gefunden, verwende "Unbekannt"
                    if (empty($kunde)) {
                        $kunde = 'Kunde unbekannt';
                    }
                    
                    // Überspringe leere oder Header-Einträge
                    if (empty($satznummer) || $satznummer === 'Satznummer' || $satznummer === 'Satznummer:') {
                        continue;
                    }
                    
                    // Prüfe ob es eine echte Satznummer ist und alle Felder gültig sind
                    if (preg_match('/^[0-9]+\.[0-9]+\.[0-9]+$/', $satznummer) &&
                        !empty($kennzeichen) && 
                        !empty($fahrzeug) && 
                        !empty($kunde)) {
                        
                        // Extrahiere PRUFDOC_ID für Prüfbericht-Download
                        $prufdoc_id = $this->extractPrufdocId($html, $satznummer);
                        
                        $wheelset = [
                            'satznummer' => $satznummer,
                            'kennzeichen' => $kennzeichen,
                            'fahrzeug' => $fahrzeug,
                            'halter' => $kunde, // Verwende den echten Kundenname
                            'eingangsdatum' => $eingangsdatum,
                            'status' => 'eingelagert',
                            'prufdoc_id' => $prufdoc_id
                        ];
                        
                        $wheelsets[] = $wheelset;
                        $wheelsetCount++;
                        
                        error_log("Added wheelset: " . $satznummer . " - " . $kennzeichen . " - " . $kunde);
                    }
                }
            }
            
            
            error_log("LGS_ID-based parsing result: " . count($wheelsets) . " wheelsets");
        } else {
            error_log("No LGS_ID entries found");
        }
        
        return $wheelsets;
    }
    
    /**
     * Sucht nach dem echten Kundenname in der HTML-Struktur
     */
    private function findCustomerNameInHtml($html, $satznummer) {
        error_log("Suche Kundenname für Satznummer: $satznummer");
        
        // Suche nach dem Kundenname in verschiedenen HTML-Strukturen
        $patterns = [
            // Suche nach Kundenname in der Nähe der Satznummer (erweiterte Suche)
            '/[^>]*' . preg_quote($satznummer, '/') . '[^<]*<[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            // Suche nach Kundenname in divs oder spans
            '/<div[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            // Suche nach Kundenname in Tabellenzellen
            '/<td[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            // Suche nach Kundenname in spans
            '/<span[^>]*>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i',
            // Suche nach Kundenname in allen Textinhalten
            '/>([^<]*(?:GmbH|AG|UG|eG|Dr\.|Prof\.|Herr|Frau|von|zu)[^<]*)</i'
        ];
        
        foreach ($patterns as $i => $pattern) {
            error_log("Pattern " . ($i + 1) . ": " . $pattern);
            if (preg_match_all($pattern, $html, $matches)) {
                error_log("Pattern " . ($i + 1) . " gefunden: " . count($matches[1]) . " Matches");
                foreach ($matches[1] as $candidate) {
                    $candidate = trim(strip_tags($candidate));
                    error_log("Kandidat: '$candidate'");
                    if ($this->isValidCustomerName($candidate)) {
                        error_log("Gültiger Kundenname gefunden: $candidate");
                        return $candidate;
                    }
                }
            }
        }
        
        // Erweiterte Suche: Alle Textinhalte, die wie Kundennamen aussehen
        error_log("Erweiterte Suche nach Kundennamen...");
        if (preg_match_all('/>([A-Za-zäöüßÄÖÜ][A-Za-zäöüßÄÖÜ\s\.\-\&]{3,50})</', $html, $allMatches)) {
            error_log("Gefundene Textkandidaten: " . count($allMatches[1]));
            foreach ($allMatches[1] as $candidate) {
                $candidate = trim(strip_tags($candidate));
                if ($this->isValidCustomerName($candidate)) {
                    error_log("Gültiger Kundenname in erweiterter Suche: $candidate");
                    return $candidate;
                }
            }
        }
        
        error_log("Kein Kundenname gefunden für Satznummer: $satznummer");
        return '';
    }
    
    /**
     * Prüft ob ein String ein gültiger Kundenname ist
     */
    private function isValidCustomerName($name) {
        if (empty($name) || strlen($name) < 3) {
            return false;
        }
        
        // Filtere bekannte Nicht-Kundennamen aus
        $excludePatterns = [
            '/^(Zur|Sofort|bestellen|vormerken)/i',
            '/^[0-9]+$/',
            '/^[A-Z]{1,3}[^<]*[0-9]+[^<]*$/', // Kennzeichen
            '/^[0-9]+\.[0-9]+\.[0-9]+$/', // Satznummer
            '/^[0-9]{2}\.[0-9]{2}\.[0-9]{4}$/', // Datum
            '/^(113 B & E|Kiel|Autohaus|Standort)/i',
            '/^(Winterreifen|Sommerreifen|Reifen)/i',
            '/^(keine Fehler|leichte Fehler|Austausch notwendig)/i',
            '/^(admin|administrator|user|benutzer|test)/i', // System-Benutzer
            '/^(RESY|Websystem|System)/i' // System-Namen
        ];
        
        foreach ($excludePatterns as $pattern) {
            if (preg_match($pattern, $name)) {
                return false;
            }
        }
        
        // Prüfe ob es wie ein Kundenname aussieht
        return preg_match('/^[A-Za-zäöüßÄÖÜ][A-Za-zäöüßÄÖÜ\s\.\-\&]+$/', $name);
    }
    
    /**
     * Extrahiert die PRUFDOC_ID für Prüfbericht-Download
     */
    private function extractPrufdocId($html, $satznummer) {
        // Suche nach dem Prüfbericht-Link in der Nähe der Satznummer
        $patterns = [
            // Suche nach PRUFDOC_ID in der Nähe der Satznummer
            '/[^>]*' . preg_quote($satznummer, '/') . '[^<]*<[^>]*>.*?PRUFDOC_ID=([0-9]+)/i',
            // Suche nach allen PRUFDOC_ID-Links
            '/PRUFDOC_ID=([0-9]+)/i',
            // Suche nach Prüfbericht-Links
            '/checkreport[^>]*PRUFDOC_ID=([0-9]+)/i'
        ];
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $html, $matches)) {
                $prufdoc_id = trim($matches[1]);
                if (is_numeric($prufdoc_id)) {
                    error_log("PRUFDOC_ID gefunden für Satznummer $satznummer: $prufdoc_id");
                    return $prufdoc_id;
                }
            }
        }
        
        error_log("Keine PRUFDOC_ID gefunden für Satznummer: $satznummer");
        return null;
    }
    
    /**
     * Parsing für div-basierte Strukturen (moderne Web-Apps)
     */
    private function parseDivBasedWheelsets($html) {
        $wheelsets = [];
        
        error_log("Starting div-based parsing");
        
        // Suche nach divs mit wheelset-relevanten Klassen
        $divPatterns = [
            '/<div[^>]*class="[^"]*wheelset[^"]*"[^>]*>(.*?)<\/div>/s',
            '/<div[^>]*class="[^"]*item[^"]*"[^>]*>(.*?)<\/div>/s',
            '/<div[^>]*class="[^"]*row[^"]*"[^>]*>(.*?)<\/div>/s',
            '/<div[^>]*class="[^"]*card[^"]*"[^>]*>(.*?)<\/div>/s'
        ];
        
        foreach ($divPatterns as $patternIndex => $pattern) {
            error_log("Trying div pattern " . ($patternIndex + 1) . ": " . $pattern);
            
            if (preg_match_all($pattern, $html, $divMatches)) {
                error_log("Found " . count($divMatches[1]) . " divs with pattern " . ($patternIndex + 1));
                
                foreach ($divMatches[1] as $divIndex => $divContent) {
                    error_log("Processing div " . ($divIndex + 1) . " with content length: " . strlen($divContent));
                    
                    // Suche nach LGS_ID, Satznummer, Kennzeichen in diesem div
                    $wheelset = $this->extractWheelsetFromDiv($divContent);
                    if ($wheelset) {
                        $wheelsets[] = $wheelset;
                        error_log("Added div-based wheelset: " . json_encode($wheelset));
                    }
                }
                
                if (!empty($wheelsets)) {
                    break;
                }
            }
        }
        
        // Fallback: Suche nach allen divs die LGS_ID enthalten
        if (empty($wheelsets)) {
            error_log("No divs found with specific patterns, searching for LGS_ID in all divs");
            
            if (preg_match_all('/<div[^>]*>(.*?)<\/div>/s', $html, $allDivs)) {
                error_log("Found " . count($allDivs[1]) . " total divs");
                
                foreach ($allDivs[1] as $divIndex => $divContent) {
                    if (strpos($divContent, 'LGS_ID') !== false || 
                        strpos($divContent, 'Satznummer') !== false ||
                        strpos($divContent, 'Kennzeichen') !== false) {
                        
                        error_log("Div " . $divIndex . " contains wheelset keywords");
                        $wheelset = $this->extractWheelsetFromDiv($divContent);
                        if ($wheelset) {
                            $wheelsets[] = $wheelset;
                            error_log("Added keyword-based wheelset: " . json_encode($wheelset));
                        }
                    }
                }
            }
        }
        
        error_log("Div-based parsing result: " . count($wheelsets) . " wheelsets");
        return $wheelsets;
    }
    
    /**
     * Extrahiert Reifensatz-Daten aus einem div-Element
     */
    private function extractWheelsetFromDiv($divContent) {
        $wheelset = [
            'satznummer' => '',
            'kennzeichen' => '',
            'fahrzeug' => '',
            'halter' => '',
            'eingangsdatum' => '',
            'status' => 'eingelagert'
        ];
        
        // Suche nach LGS_ID/Satznummer
        if (preg_match('/LGS_ID[^>]*>([^<]+)</', $divContent, $lgsMatches)) {
            $wheelset['satznummer'] = trim(strip_tags($lgsMatches[1]));
        } elseif (preg_match('/Satznummer[^>]*>([^<]+)</', $divContent, $satzMatches)) {
            $wheelset['satznummer'] = trim(strip_tags($satzMatches[1]));
        } elseif (preg_match('/<span[^>]*>([0-9]+)<\/span>/', $divContent, $numberMatches)) {
            $wheelset['satznummer'] = trim($numberMatches[1]);
        }
        
        // Suche nach Kennzeichen
        if (preg_match('/Kennzeichen[^>]*>([^<]+)</', $divContent, $kennzeichenMatches)) {
            $wheelset['kennzeichen'] = trim(strip_tags($kennzeichenMatches[1]));
        } elseif (preg_match('/<span[^>]*>([A-Z]{1,3}[^<]*[0-9]+[^<]*)<\/span>/', $divContent, $kennzeichenMatches)) {
            $wheelset['kennzeichen'] = trim($kennzeichenMatches[1]);
        }
        
        // Suche nach Fahrzeug
        if (preg_match('/Fahrzeug[^>]*>([^<]+)</', $divContent, $fahrzeugMatches)) {
            $wheelset['fahrzeug'] = trim(strip_tags($fahrzeugMatches[1]));
        }
        
        // Suche nach Halter
        if (preg_match('/Halter[^>]*>([^<]+)</', $divContent, $halterMatches)) {
            $wheelset['halter'] = trim(strip_tags($halterMatches[1]));
        }
        
        // Suche nach Eingangsdatum
        if (preg_match('/Eingang[^>]*>([^<]+)</', $divContent, $datumMatches)) {
            $wheelset['eingangsdatum'] = trim(strip_tags($datumMatches[1]));
        }
        
        // Nur zurückgeben wenn mindestens Satznummer vorhanden
        if (!empty($wheelset['satznummer']) && $wheelset['satznummer'] !== '&nbsp;') {
            return $wheelset;
        }
        
        return null;
    }
    
    /**
     * Fallback-Parsing für verschiedene HTML-Formate
     */
    private function fallbackParse($html) {
        $wheelsets = [];
        
        error_log("Starting fallback parsing");
        
        // Suche nach Satznummern (LGS_ID) in verschiedenen Formaten
        $patterns = [
            '/LGS_ID[^>]*>([^<]+)</',
            '/name="LGS_ID"[^>]*value="([^"]+)"/',
            '/LGS[^>]*>([^<]+)</',
            '/Satznummer[^>]*>([^<]+)</'
        ];
        
        foreach ($patterns as $patternIndex => $pattern) {
            error_log("Trying fallback pattern " . ($patternIndex + 1) . ": " . $pattern);
            
            if (preg_match_all($pattern, $html, $matches)) {
                error_log("Found " . count($matches[1]) . " matches with pattern " . ($patternIndex + 1));
                
                foreach ($matches[1] as $index => $satznummer) {
                    $satznummer = trim(strip_tags($satznummer));
                    if (!empty($satznummer) && $satznummer !== '&nbsp;') {
                        $wheelsets[] = [
                            'satznummer' => $satznummer,
                            'kennzeichen' => '',
                            'fahrzeug' => '',
                            'halter' => '',
                            'eingangsdatum' => '',
                            'status' => 'eingelagert'
                        ];
                        error_log("Added fallback wheelset: " . $satznummer);
                    }
                }
                
                if (!empty($wheelsets)) {
                    break;
                }
            }
        }
        
        // Suche nach Tabellen-Zeilen mit Daten
        if (empty($wheelsets)) {
            error_log("No LGS patterns found, trying table row parsing");
            
            // Suche nach allen Zeilen die Daten enthalten könnten
            if (preg_match_all('/<tr[^>]*>(.*?)<\/tr>/s', $html, $rows)) {
                error_log("Found " . count($rows[1]) . " total rows in HTML");
                
                foreach ($rows[1] as $rowIndex => $row) {
                    // Überspringe Header-Zeilen
                    if (strpos($row, 'Satznummer') !== false || strpos($row, 'Kennzeichen') !== false || strpos($row, 'LGS') !== false) {
                        continue;
                    }
                    
                    // Suche nach Zellen mit Inhalt
                    if (preg_match_all('/<td[^>]*>(.*?)<\/td>/s', $row, $cells)) {
                        $cells = $cells[1];
                        
                        if (count($cells) >= 2) {
                            $firstCell = trim(strip_tags($cells[0]));
                            
                            // Prüfe ob erste Zelle wie eine Satznummer aussieht
                            if (!empty($firstCell) && $firstCell !== '&nbsp;' && (is_numeric($firstCell) || strpos($firstCell, 'LGS') !== false)) {
                                $wheelset = [
                                    'satznummer' => $firstCell,
                                    'kennzeichen' => trim(strip_tags($cells[1] ?? '')),
                                    'fahrzeug' => trim(strip_tags($cells[2] ?? '')),
                                    'halter' => trim(strip_tags($cells[3] ?? '')),
                                    'eingangsdatum' => trim(strip_tags($cells[4] ?? '')),
                                    'status' => 'eingelagert'
                                ];
                                
                                $wheelsets[] = $wheelset;
                                error_log("Added table row wheelset: " . $firstCell);
                            }
                        }
                    }
                }
            }
        }
        
        error_log("Fallback parsing result: " . count($wheelsets) . " wheelsets");
        return $wheelsets;
    }
}

// API-Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['autohaus_id'])) {
        echo json_encode([
            'success' => false,
            'error' => 'Autohaus-ID ist erforderlich'
        ]);
        exit;
    }
    
    $autohausId = $input['autohaus_id'];
    
    // Validiere Autohaus-ID
    try {
        $db = new Database();
        $conn = $db->getConnection();
        
        $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE id = ? AND is_active = 1");
        $stmt->execute([$autohausId]);
        $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$autohaus) {
            echo json_encode([
                'success' => false,
                'error' => 'Ungültige oder inaktive Autohaus-ID'
            ]);
            exit;
        }
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'Fehler bei der Autohaus-Validierung: ' . $e->getMessage()
        ]);
        exit;
    }
    
    $wheelsetService = new ResyWheelsetService($autohausId);
    $result = $wheelsetService->getStoredWheelsets();
    
    echo json_encode($result);
} else {
    echo json_encode([
        'success' => false,
        'error' => 'Nur POST-Requests erlaubt'
    ]);
}
?>
