<?php
/**
 * Professional Wheelsets API - Echte RESY-Integration
 * Lädt echte Daten aus dem RESY-System
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once dirname(__DIR__) . '/config/database.php';

class ProfessionalWheelsetsAPI {
    private $autohausId;
    private $autohausName;
    private $dealerId;
    private $resyUsername;
    private $resyPassword;
    
    public function __construct($autohausId) {
        $this->autohausId = $autohausId;
        $this->loadAutohausData();
    }
    
    /**
     * Lädt Autohaus-Daten aus der Datenbank
     */
    private function loadAutohausData() {
        try {
            $db = new Database();
            $pdo = $db->getConnection();
            $stmt = $pdo->prepare("SELECT name, resy_username, resy_password FROM autohaus WHERE id = ?");
                $stmt->execute([$this->autohausId]);
                $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
                
            if (!$autohaus) {
                throw new Exception("Autohaus nicht gefunden");
            }
            
            $this->autohausName = $autohaus['name'];
            $this->resyUsername = $autohaus['resy_username'];
            $this->resyPassword = $autohaus['resy_password'];
            $this->dealerId = $this->getResyKdId();
            
            error_log("Autohaus geladen: " . $this->autohausName . " (Dealer ID: " . $this->dealerId . ")");
            
        } catch (Exception $e) {
            error_log("Fehler beim Laden der Autohaus-Daten: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Mappt Autohaus-Namen zu RESY Dealer IDs
     */
    private function getResyKdId() {
        $mapping = [
            'B & E Volvo Kiel' => '453',
            'B & E Volvo Norderstedt' => '452', 
            'B & E Volvo Glinde' => '454'
        ];
        
        return $mapping[$this->autohausName] ?? '453';
    }
    
    /**
     * Hauptfunktion: Lädt alle Reifensätze für ein Autohaus
     */
    public function getStoredWheelsets() {
        $startTime = microtime(true);
        error_log("=== START RESY-INTEGRATION ===");
        error_log("Zeit: " . date('Y-m-d H:i:s'));
        error_log("Autohaus ID: " . $this->autohausId);
        error_log("Autohaus Name: " . $this->autohausName);
        error_log("Dealer ID: " . $this->dealerId);
        
        try {
            // OPTION 1: Versuche direkten Zugriff ohne Login
            error_log("OPTION 1: Versuche direkten Zugriff ohne Login...");
            $directResult = $this->tryDirectAccess();
            if ($directResult['success']) {
                error_log("✅ Direkter Zugriff erfolgreich - Login umgangen!");
                return $directResult;
            }
            error_log("❌ Direkter Zugriff fehlgeschlagen: " . $directResult['error']);
            
            // OPTION 2: Kompletter Workflow mit Session-Management
            error_log("OPTION 2: Kompletter Workflow mit Session-Management...");
            $workflowResult = $this->performCompleteWorkflow();
            if ($workflowResult['success']) {
                error_log("✅ Kompletter Workflow erfolgreich!");
                return $workflowResult;
            }
            error_log("❌ Kompletter Workflow fehlgeschlagen: " . $workflowResult['error']);
            
            // OPTION 3: Alle Methoden fehlgeschlagen - KEINE Mock-Daten
            error_log("❌ Alle RESY-Zugriffsmethoden fehlgeschlagen");
            return $this->getNoDataResponse('RESY-System nicht erreichbar - keine Mock-Daten');
            
        } catch (Exception $e) {
            $endTime = microtime(true);
            $duration = round(($endTime - $startTime) * 1000, 2);
            error_log("❌ ProfessionalWheelsetsAPI Fehler: " . $e->getMessage());
            error_log("⏱️ Fehler nach: " . $duration . "ms");
            error_log("=== ENDE RESY-INTEGRATION (FEHLER) ===");
            return $this->getFallbackData('Unerwarteter Fehler: ' . $e->getMessage());
        }
    }
    
    /**
     * Login ins RESY-System
     */
    private function loginToResy() {
        error_log("  🔐 Login: Starte cURL-Session...");
            $ch = curl_init();
            
        // Schritt 1: Login-Seite laden
        error_log("  🔐 Login: Lade Login-Seite...");
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
        ]);
        
        $response = curl_exec($ch);
        
        if (curl_error($ch)) {
            $error = curl_error($ch);
            curl_close($ch);
            error_log("  ❌ Login: cURL-Fehler beim Laden der Login-Seite: " . $error);
            return ['success' => false, 'error' => 'Fehler beim Laden der Login-Seite: ' . $error];
        }
        
        error_log("  ✅ Login: Login-Seite geladen (" . strlen($response) . " Zeichen)");
        
        // Schritt 2: Login-Daten senden
        error_log("  🔐 Login: Sende Login-Daten...");
        error_log("  🔐 Login: Username: " . $this->resyUsername);
        error_log("  🔐 Login: Password: " . str_repeat('*', strlen($this->resyPassword)));
        
        $postData = http_build_query([
            'UserName' => $this->resyUsername,
            'Password' => $this->resyPassword,
            'button' => 'submit'
        ]);
        
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login'
        ]);
        
        $response = curl_exec($ch);
            curl_close($ch);
        
        error_log("  ✅ Login: Login-Response erhalten (" . strlen($response) . " Zeichen)");
        
        // Debug: Speichere Login-Response
        file_put_contents(sys_get_temp_dir() . '/resy_login_response_' . $this->autohausId . '.html', $response);
        error_log("  💾 Login: Response gespeichert in " . sys_get_temp_dir() . '/resy_login_response_' . $this->autohausId . '.html');
        
        // Prüfe Login-Erfolg - Erweiterte Logik
        if (strpos($response, 'technisches Problem') !== false) {
            error_log("  ❌ Login: Technisches Problem erkannt");
            return ['success' => false, 'error' => 'Technisches Problem beim Login'];
        }
        
        // Prüfe auf verschiedene Erfolgs-Indikatoren
        $successIndicators = [
            'Sätze / Rücklieferungen bestellen',
            'RESY Websystem',
            'Lagerbestand',
            'dashboard',
            'wheelset',
            'logout',
            'Abmelden'
        ];
        
        $foundIndicator = false;
        foreach ($successIndicators as $indicator) {
            if (strpos($response, $indicator) !== false) {
                error_log("  ✅ Login: Erfolgs-Indikator gefunden: " . $indicator);
                $foundIndicator = true;
                break;
            }
        }
        
        if ($foundIndicator) {
            error_log("  ✅ Login: Login erfolgreich");
            return ['success' => true];
        }
        
        // Prüfe auf Login-Fehler
        $errorIndicators = [
            'Anmeldename',
            'Kennwort',
            'Anmeldung',
            'InputUser',
            'InputPass',
            'ButtonSubmit'
        ];
        
        $hasLoginForm = false;
        foreach ($errorIndicators as $indicator) {
            if (strpos($response, $indicator) !== false) {
                $hasLoginForm = true;
                    break;
                }
        }
        
        if ($hasLoginForm) {
            error_log("  ❌ Login: Login-Formular noch sichtbar - Login fehlgeschlagen");
            return ['success' => false, 'error' => 'Login fehlgeschlagen - ungültige Credentials oder technisches Problem'];
        }
        
        error_log("  ❌ Login: Unbekannte Response - weder Erfolg noch Fehler erkannt");
        return ['success' => false, 'error' => 'Login fehlgeschlagen - unbekannte Response'];
    }
    
    /**
     * Navigiert zur Suchseite
     */
    private function navigateToSearchPage() {
        error_log("  🔍 Suchseite: Starte cURL-Session...");
        $ch = curl_init();
        
        error_log("  🔍 Suchseite: Lade Suchseite...");
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        if (curl_error($ch)) {
            $error = curl_error($ch);
            error_log("  ❌ Suchseite: cURL-Fehler: " . $error);
            return ['success' => false, 'error' => 'Fehler beim Laden der Suchseite: ' . $error];
        }
        
        error_log("  ✅ Suchseite: Suchseite geladen (" . strlen($response) . " Zeichen)");
        return ['success' => true, 'html' => $response];
    }
    
    /**
     * Führt die Suche durch
     */
    private function performSearch() {
        error_log("  🔎 Suche: Starte cURL-Session...");
        $ch = curl_init();
        
        // Suche mit GET-Parameter
        $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search&FN=WheelsetSearch&KD_ID=' . $this->dealerId . '&STATUS_ID=20';
        error_log("  🔎 Suche: URL: " . $searchUrl);
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIEFILE, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_COOKIEJAR, sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login'
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        if (curl_error($ch)) {
            $error = curl_error($ch);
            error_log("  ❌ Suche: cURL-Fehler: " . $error);
            return ['success' => false, 'error' => 'Fehler bei der Suche: ' . $error];
        }
        
        error_log("  ✅ Suche: Such-Response erhalten (" . strlen($response) . " Zeichen)");
        
        // Debug: Speichere Such-Response
        file_put_contents(sys_get_temp_dir() . '/resy_search_response_' . $this->autohausId . '.html', $response);
        error_log("  💾 Suche: Response gespeichert in " . sys_get_temp_dir() . '/resy_search_response_' . $this->autohausId . '.html');
        
        // Prüfe ob es noch die Login-Seite ist
        if (strpos($response, 'Anmeldename') !== false || strpos($response, 'InputUser') !== false) {
            error_log("  ❌ Suche: Immer noch Login-Seite - Session verloren");
            return ['success' => false, 'error' => 'Session verloren - immer noch Login-Seite'];
        }
        
        return ['success' => true, 'html' => $response];
    }
    
    /**
     * Lädt alle Reifensätze aus der RESY-HTML
     */
    private function loadAllWheelsets($html) {
        error_log("  📄 Parsing: Starte Parsing der RESY-HTML...");
        error_log("  📄 Parsing: HTML-Länge: " . strlen($html) . " Zeichen");
        
        // Speichere HTML für Debugging
        file_put_contents(sys_get_temp_dir() . '/resy_response_' . $this->autohausId . '.html', $html);
        error_log("  💾 Parsing: HTML gespeichert in " . sys_get_temp_dir() . '/resy_response_' . $this->autohausId . '.html');
        
        // Parse die HTML-Antwort
        $wheelsets = $this->parseWheelsetsFromHTML($html);
        
        error_log("  ✅ Parsing: " . count($wheelsets) . " Reifensätze geparst");
        
        return $wheelsets;
    }
    
    /**
     * Parst Reifensätze aus der RESY-HTML
     */
    private function parseWheelsetsFromHTML($html) {
        $wheelsets = [];
        
        // Suche nach der WheelsetSearchResult Tabelle (mit oder ohne ID/Klasse)
        if (preg_match('/<table[^>]*id=["\']WheelsetSearchResult["\'][^>]*>(.*?)<\/table>/s', $html, $matches)) {
            $tableHtml = $matches[1];
            error_log("  📄 Parsing: Tabelle mit ID gefunden");
        } else if (preg_match('/<table[^>]*class=["\'][^"\']*WheelsetSearchResult[^"\']*["\'][^>]*>(.*?)<\/table>/s', $html, $matches)) {
            $tableHtml = $matches[1];
            error_log("  📄 Parsing: Tabelle mit Klasse gefunden");
        } else if (preg_match('/<table[^>]*>(.*?)<\/table>/s', $html, $matches)) {
            $tableHtml = $matches[1];
            error_log("  📄 Parsing: Erste Tabelle gefunden");
        } else {
            error_log("  ❌ Parsing: Keine Tabelle gefunden in HTML");
            return $wheelsets;
        }
        
        // Suche nach tbody oder direkten tr-Elementen
        if (preg_match('/<tbody[^>]*>(.*?)<\/tbody>/s', $tableHtml, $tbodyMatches)) {
            $tbodyHtml = $tbodyMatches[1];
            error_log("  📄 Parsing: tbody gefunden");
        } else {
            $tbodyHtml = $tableHtml;
            error_log("  📄 Parsing: Direkte tr-Elemente verwenden");
        }
        
        // Parse alle Zeilen mit Wheels-Klassen
        if (preg_match_all('/<tr[^>]*class=["\']Wheels[^"\']*["\'][^>]*>(.*?)<\/tr>/s', $tbodyHtml, $rowMatches, PREG_SET_ORDER)) {
            error_log("  📄 Parsing: " . count($rowMatches) . " Reifensätze-Zeilen gefunden");
            foreach ($rowMatches as $rowMatch) {
                $rowHtml = $rowMatch[1];
                $wheelset = $this->parseWheelsetRow($rowHtml);
                if ($wheelset) {
                    $wheelsets[] = $wheelset;
                }
            }
        } else {
            error_log("  ❌ Parsing: Keine Reifensätze-Zeilen gefunden");
        }
        
        // Entferne Duplikate innerhalb dieser Seite
        $uniqueWheelsets = $this->removeDuplicates($wheelsets);
        
        error_log("  ✅ HTML-Parser: " . count($wheelsets) . " Reifensätze aus HTML extrahiert");
        error_log("  🔍 HTML-Parser: " . (count($wheelsets) - count($uniqueWheelsets)) . " Duplikate auf dieser Seite entfernt");
        error_log("  ✅ HTML-Parser: " . count($uniqueWheelsets) . " eindeutige Reifensätze auf dieser Seite");
        
        return $uniqueWheelsets;
    }
    
    /**
     * Parst eine einzelne Reifensatz-Zeile
     */
    private function parseWheelsetRow($rowHtml) {
        try {
            // Parse alle Zellen aus der HTML-Zeile
            if (preg_match_all('/<td[^>]*>(.*?)<\/td>/s', $rowHtml, $cellMatches)) {
                $cells = $cellMatches[1];
                
                // Bereinige Zellen-Inhalt
                $cleanCells = array_map(function($cell) {
                    return trim(strip_tags($cell));
                }, $cells);
                
                // Validiere Satznummer (muss numerisch sein)
                $satznummer = $cleanCells[2] ?? ''; // Index 2 für Satznummer
                if (!is_numeric(str_replace('.', '', $satznummer))) {
                    return null; // Überspringe ungültige Zeilen
                }
                
                return [
                                        'satznummer' => $satznummer,
                    'kennzeichen' => $cleanCells[3] ?? '', // Index 3 für Kennzeichen
                    'fin' => $cleanCells[4] ?? '', // Index 4 für FIN
                    'reifenmodell' => $cleanCells[5] ?? '', // Index 5 für Reifenmodell (leer)
                    'autohaus' => $this->autohausName,
                    'reifenart' => $cleanCells[7] ?? '', // Index 7 für Reifenart
                    'saison' => $cleanCells[8] ?? '', // Index 8 für Saison
                    'eingangsdatum' => $cleanCells[9] ?? '', // Index 9 für Eingangsdatum
                    'haltername' => $cleanCells[14] ?? '', // Index 14 für Haltername
                    'empfehlung' => $cleanCells[15] ?? '', // Index 15 für Empfehlung
                    'profiltiefe' => $cleanCells[17] ?? '', // Index 17 für Profiltiefe
                                        'status' => 'eingelagert',
                    'autohaus_id' => $this->autohausId
                ];
            }
            
            return null;
            
        } catch (Exception $e) {
            error_log("Fehler beim Parsen der Reifensatz-Zeile: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * OPTION 1: Versuche direkten Zugriff ohne Login
     */
    private function tryDirectAccess() {
        error_log("  🚀 Direkter Zugriff: Versuche direkte Suche ohne Login...");
        
        $ch = curl_init();
        
        // Direkte Such-URL mit allen Parametern
        $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search&FN=WheelsetSearch&KD_ID=' . $this->dealerId . '&STATUS_ID=20';
        error_log("  🚀 Direkter Zugriff: URL: " . $searchUrl);
        
        curl_setopt($ch, CURLOPT_URL, $searchUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8'
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        if (curl_error($ch)) {
            return ['success' => false, 'error' => 'cURL-Fehler: ' . curl_error($ch)];
        }
        
        // Prüfe ob Daten vorhanden sind
        if (strpos($response, 'WheelsetSearchResult') !== false || strpos($response, 'Reifensätze') !== false) {
            error_log("  ✅ Direkter Zugriff: Daten gefunden!");
            $wheelsets = $this->loadAllWheelsets($response);
            if (!empty($wheelsets)) {
                return $this->buildSuccessResponse($wheelsets, microtime(true), 'RESY-System (Direkt)');
            }
        }
        
        return ['success' => false, 'error' => 'Keine Daten im direkten Zugriff gefunden'];
    }
    
    /**
     * OPTION 2: Kompletter Workflow mit Session-Management
     */
    private function performCompleteWorkflow() {
        error_log("  🔄 Kompletter Workflow: Starte mit Session-Management...");
        
        $ch = curl_init();
        $cookieFile = sys_get_temp_dir() . '/resy_cookies_' . $this->autohausId . '.txt';
        
        // Cookie-Datei löschen falls vorhanden
        if (file_exists($cookieFile)) {
            unlink($cookieFile);
        }
        
        try {
            // Schritt 1: Login-Seite laden
            error_log("  🔄 Workflow: Lade Login-Seite...");
            curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_COOKIEJAR, $cookieFile);
            curl_setopt($ch, CURLOPT_COOKIEFILE, $cookieFile);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Accept-Encoding: gzip, deflate, br',
                'Connection: keep-alive',
                'Upgrade-Insecure-Requests: 1'
            ]);
            
            $response = curl_exec($ch);
            if (curl_error($ch)) {
                throw new Exception('Login-Seite laden fehlgeschlagen: ' . curl_error($ch));
            }
            error_log("  ✅ Workflow: Login-Seite geladen (" . strlen($response) . " Zeichen)");
            
            // Schritt 2: Login durchführen
            error_log("  🔄 Workflow: Führe Login durch...");
            $postData = http_build_query([
                'UserName' => $this->resyUsername,
                'Password' => $this->resyPassword,
                'button' => 'submit',
                'FN' => 'login'
            ]);
            
            curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Content-Type: application/x-www-form-urlencoded',
                'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login',
                'Origin: https://resy.dtm-reifen.de'
            ]);
            
            $response = curl_exec($ch);
            if (curl_error($ch)) {
                throw new Exception('Login fehlgeschlagen: ' . curl_error($ch));
            }
            
            // Prüfe Login-Erfolg
            if (strpos($response, 'Anmeldename') !== false || strpos($response, 'InputUser') !== false) {
                throw new Exception('Login fehlgeschlagen - immer noch Login-Seite');
            }
            error_log("  ✅ Workflow: Login erfolgreich");
            
            // Schritt 3: Zur Suchseite navigieren
            error_log("  🔄 Workflow: Navigiere zur Suchseite...");
            curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search');
            curl_setopt($ch, CURLOPT_POST, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, null);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login'
            ]);
            
            $response = curl_exec($ch);
            if (curl_error($ch)) {
                throw new Exception('Suchseite laden fehlgeschlagen: ' . curl_error($ch));
            }
            error_log("  ✅ Workflow: Suchseite geladen (" . strlen($response) . " Zeichen)");
            
            // Schritt 4: Suche durchführen (POST-Request mit Formulardaten)
            error_log("  🔄 Workflow: Führe Suche durch...");
            $searchUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search';
            
            // Formulardaten für die Suche
            $searchData = http_build_query([
                'FN' => 'WheelsetSearch',
                'a' => 'search',
                'm' => 'wheelset',
                'KD_ID' => $this->dealerId,
                'STATUS_ID' => '20',
                'button' => 'submit'
            ]);
            
            curl_setopt($ch, CURLOPT_URL, $searchUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $searchData);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Content-Type: application/x-www-form-urlencoded',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Referer: https://resy.dtm-reifen.de/index.php?m=wheelset&a=search',
                'Origin: https://resy.dtm-reifen.de'
            ]);
            
            $response = curl_exec($ch);
            if (curl_error($ch)) {
                throw new Exception('Suche fehlgeschlagen: ' . curl_error($ch));
            }
            error_log("  ✅ Workflow: Suche durchgeführt (" . strlen($response) . " Zeichen)");
            
            // Debug: Speichere Response
            file_put_contents(sys_get_temp_dir() . '/resy_workflow_response_' . $this->autohausId . '.html', $response);
            error_log("  💾 Workflow: Response gespeichert");
            
            // Prüfe ob es noch die Login-Seite ist
            if (strpos($response, 'Anmeldename') !== false || strpos($response, 'InputUser') !== false) {
                throw new Exception('Session verloren - immer noch Login-Seite');
            }
            
            // Parse Reifensätze von allen Seiten
            error_log("  🔄 Workflow: Lade alle Pagination-Seiten...");
            $allWheelsets = $this->loadAllWheelsetsWithPagination($ch, $response, $searchData);
            if (empty($allWheelsets)) {
                throw new Exception('Keine Reifensätze gefunden');
            }
            
            error_log("  ✅ Workflow: " . count($allWheelsets) . " Reifensätze von allen Seiten gefunden");
            return $this->buildSuccessResponse($allWheelsets, microtime(true), 'RESY-System (Workflow mit Pagination)');
            
        } catch (Exception $e) {
            error_log("  ❌ Workflow: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        } finally {
            curl_close($ch);
        }
    }
    
    /**
     * Lädt alle Reifensätze von allen Pagination-Seiten
     */
    private function loadAllWheelsetsWithPagination($ch, $firstPageResponse, $searchData) {
        error_log("  📄 Pagination: Starte mit allen Seiten...");
        
        $allWheelsets = [];
        $currentPage = 1;
        $maxPages = 10; // Sicherheitslimit
        
        // Parse erste Seite
        $firstPageWheelsets = $this->parseWheelsetsFromHTML($firstPageResponse);
        $allWheelsets = array_merge($allWheelsets, $firstPageWheelsets);
        error_log("  📄 Pagination: Seite 1 - " . count($firstPageWheelsets) . " Reifensätze");
        
        // Finde Pagination-Links
        $paginationLinks = $this->extractPaginationLinks($firstPageResponse);
        error_log("  📄 Pagination: Gefundene Pagination-Links: " . implode(', ', array_keys($paginationLinks)));
        
        // Falls keine Pagination-Links gefunden, versuche alternative Methode
        if (empty($paginationLinks)) {
            error_log("  📄 Pagination: Keine Links gefunden, versuche alternative Methode...");
            $paginationLinks = $this->generatePaginationLinks($searchData);
            error_log("  📄 Pagination: Generierte Pagination-Links: " . implode(', ', array_keys($paginationLinks)));
        }
        
        // Lade alle weiteren Seiten
        foreach ($paginationLinks as $pageNum => $pageUrl) {
            if ($pageNum <= 1 || $pageNum > $maxPages) {
                continue; // Überspringe Seite 1 (bereits geladen) und zu hohe Seiten
            }
            
            error_log("  📄 Pagination: Lade Seite $pageNum...");
            
            // Lade Seite
            curl_setopt($ch, CURLOPT_URL, $pageUrl);
            curl_setopt($ch, CURLOPT_POST, false);
            curl_setopt($ch, CURLOPT_POSTFIELDS, null);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
                'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
                'Referer: https://resy.dtm-reifen.de/index.php?m=wheelset&a=search'
            ]);
            
            $pageResponse = curl_exec($ch);
            if (curl_error($ch)) {
                error_log("  ❌ Pagination: Fehler beim Laden von Seite $pageNum: " . curl_error($ch));
                continue;
            }
            
            // Prüfe ob es noch die Login-Seite ist
            if (strpos($pageResponse, 'Anmeldename') !== false || strpos($pageResponse, 'InputUser') !== false) {
                error_log("  ❌ Pagination: Session verloren auf Seite $pageNum");
                break;
            }
            
            // Parse Reifensätze von dieser Seite
            $pageWheelsets = $this->parseWheelsetsFromHTML($pageResponse);
            $allWheelsets = array_merge($allWheelsets, $pageWheelsets);
            error_log("  📄 Pagination: Seite $pageNum - " . count($pageWheelsets) . " Reifensätze (Total: " . count($allWheelsets) . ")");
            
            // Kurze Pause zwischen Requests
            usleep(500000); // 0.5 Sekunden
        }
        
        // Entferne Duplikate basierend auf Satznummer
        $uniqueWheelsets = $this->removeDuplicates($allWheelsets);
        
        error_log("  ✅ Pagination: Alle Seiten geladen - " . count($allWheelsets) . " Reifensätze insgesamt");
        error_log("  🔍 Duplikate: " . (count($allWheelsets) - count($uniqueWheelsets)) . " Duplikate entfernt");
        error_log("  ✅ Eindeutige Reifensätze: " . count($uniqueWheelsets));
        
        return $uniqueWheelsets;
    }
    
    /**
     * Extrahiert Pagination-Links aus der HTML-Response
     */
    private function extractPaginationLinks($html) {
        $links = [];
        
        error_log("  🔍 Pagination: Suche nach Pagination-Links...");
        
        // Methode 1: Suche nach Tablepaginate Container
        if (preg_match('/<ul[^>]*id="Tablepaginate"[^>]*>(.*?)<\/ul>/s', $html, $matches)) {
            error_log("  🔍 Pagination: Tablepaginate Container gefunden");
            $paginationHtml = $matches[1];
            
            // Extrahiere alle Links mit Page-Parameter
            if (preg_match_all('/href="([^"]*Page=(\d+)[^"]*)"/', $paginationHtml, $linkMatches, PREG_SET_ORDER)) {
                foreach ($linkMatches as $match) {
                    $url = html_entity_decode($match[1]);
                    $pageNum = (int)$match[2];
                    
                    // Konvertiere relative URLs zu absoluten URLs
                    if (strpos($url, 'http') !== 0) {
                        $url = 'https://resy.dtm-reifen.de/' . ltrim($url, './');
                    }
                    
                    $links[$pageNum] = $url;
                    error_log("  🔍 Pagination: Link gefunden - Seite $pageNum: $url");
                }
            } else {
                error_log("  ❌ Pagination: Keine Page-Links in Tablepaginate gefunden");
            }
        } else {
            error_log("  ❌ Pagination: Tablepaginate Container nicht gefunden");
            
            // Methode 2: Suche nach allen Page-Parametern im gesamten HTML
            if (preg_match_all('/href="([^"]*Page=(\d+)[^"]*)"/', $html, $linkMatches, PREG_SET_ORDER)) {
                error_log("  🔍 Pagination: Page-Links im gesamten HTML gefunden");
                foreach ($linkMatches as $match) {
                    $url = html_entity_decode($match[1]);
                    $pageNum = (int)$match[2];
                    
                    // Konvertiere relative URLs zu absoluten URLs
                    if (strpos($url, 'http') !== 0) {
                        $url = 'https://resy.dtm-reifen.de/' . ltrim($url, './');
                    }
                    
                    $links[$pageNum] = $url;
                    error_log("  🔍 Pagination: Link gefunden - Seite $pageNum: $url");
                }
            } else {
                error_log("  ❌ Pagination: Keine Page-Links im gesamten HTML gefunden");
            }
        }
        
        // Methode 3: Suche nach numerischen Links (falls andere Methoden fehlschlagen)
        if (empty($links)) {
            error_log("  🔍 Pagination: Versuche alternative Methode...");
            if (preg_match_all('/<a[^>]*href="([^"]*)"[^>]*><span>(\d+)<\/span><\/a>/', $html, $linkMatches, PREG_SET_ORDER)) {
                foreach ($linkMatches as $match) {
                    $url = html_entity_decode($match[1]);
                    $pageNum = (int)$match[2];
                    
                    // Nur URLs mit WheelsetSearch berücksichtigen
                    if (strpos($url, 'WheelsetSearch') !== false) {
                        if (strpos($url, 'http') !== 0) {
                            $url = 'https://resy.dtm-reifen.de/' . ltrim($url, './');
                        }
                        
                        $links[$pageNum] = $url;
                        error_log("  🔍 Pagination: Alternative Link gefunden - Seite $pageNum: $url");
                    }
                }
            }
        }
        
        error_log("  🔍 Pagination: Insgesamt " . count($links) . " Links gefunden");
        return $links;
    }
    
    /**
     * Entfernt Duplikate basierend auf der Satznummer
     */
    private function removeDuplicates($wheelsets) {
        $uniqueWheelsets = [];
        $seenSatznummern = [];
        
        error_log("  🔍 Duplikate: Starte Duplikatsprüfung für " . count($wheelsets) . " Reifensätze...");
        
        foreach ($wheelsets as $wheelset) {
            $satznummer = $wheelset['satznummer'] ?? '';
            
            // Prüfe ob Satznummer leer oder bereits gesehen
            if (empty($satznummer)) {
                error_log("  ⚠️ Duplikate: Reifensatz ohne Satznummer übersprungen");
                continue;
            }
            
            if (in_array($satznummer, $seenSatznummern)) {
                error_log("  🔍 Duplikate: Duplikat gefunden - Satznummer: $satznummer");
                continue;
            }
            
            // Füge zur Liste hinzu
            $seenSatznummern[] = $satznummer;
            $uniqueWheelsets[] = $wheelset;
        }
        
        error_log("  ✅ Duplikate: " . count($uniqueWheelsets) . " eindeutige Reifensätze nach Duplikatsprüfung");
        return $uniqueWheelsets;
    }
    
    /**
     * Generiert Pagination-Links basierend auf der Such-URL
     */
    private function generatePaginationLinks($searchData) {
        $links = [];
        
        error_log("  🔧 Pagination: Generiere Pagination-Links...");
        
        // Extrahiere Parameter aus der Such-URL
        $baseUrl = 'https://resy.dtm-reifen.de/index.php?m=wheelset&a=search';
        
        // Generiere Links für Seiten 2-10 (Seite 1 ist bereits geladen)
        for ($page = 2; $page <= 10; $page++) {
            $url = $baseUrl . '&Page=' . $page . '&' . $searchData;
            $links[$page] = $url;
            error_log("  🔧 Pagination: Generierter Link - Seite $page: $url");
        }
        
        return $links;
    }
    
    /**
     * OPTION 2: Versuche verschiedene Login-Methoden
     */
    private function tryMultipleLoginMethods() {
        error_log("  🔐 Multi-Login: Versuche verschiedene Login-Methoden...");
        
        // Methode 1: Standard Login
        $result1 = $this->loginToResy();
        if ($result1['success']) {
            error_log("  ✅ Multi-Login: Standard-Login erfolgreich");
            return $result1;
        }
        
        // Methode 2: Login mit Session-Cookies
        $result2 = $this->loginWithSessionCookies();
        if ($result2['success']) {
            error_log("  ✅ Multi-Login: Session-Cookie-Login erfolgreich");
            return $result2;
        }
        
        // Methode 3: Login mit verschiedenen Headers
        $result3 = $this->loginWithCustomHeaders();
        if ($result3['success']) {
            error_log("  ✅ Multi-Login: Custom-Header-Login erfolgreich");
            return $result3;
        }
        
        return ['success' => false, 'error' => 'Alle Login-Methoden fehlgeschlagen'];
    }
    
    /**
     * Login mit Session-Cookies und erweiterten Headers
     */
    private function loginWithSessionCookies() {
        error_log("  🍪 Session-Cookie-Login: Starte...");
        
        $ch = curl_init();
        
        // Erstelle Session-Cookie
        $sessionId = 'PHPSESSID=' . uniqid() . '; path=/; domain=resy.dtm-reifen.de';
        
        // Schritt 1: Login-Seite laden
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_COOKIE, $sessionId);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Cookie: ' . $sessionId
        ]);
        
        $response = curl_exec($ch);
        
        // Schritt 2: Login-Daten senden
        $postData = http_build_query([
            'UserName' => $this->resyUsername,
            'Password' => $this->resyPassword,
            'button' => 'submit',
            'FN' => 'login'
        ]);
        
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login',
            'Origin: https://resy.dtm-reifen.de',
            'Cookie: ' . $sessionId
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        if (strpos($response, 'Sätze / Rücklieferungen bestellen') !== false) {
            return ['success' => true];
        }
        
        return ['success' => false, 'error' => 'Session-Cookie-Login fehlgeschlagen'];
    }
    
    /**
     * Login mit Custom Headers
     */
    private function loginWithCustomHeaders() {
        error_log("  📋 Custom-Header-Login: Starte...");
        
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
            'Accept-Language: de-DE,de;q=0.9,en;q=0.8',
            'Accept-Encoding: gzip, deflate, br',
            'Connection: keep-alive',
            'Upgrade-Insecure-Requests: 1',
            'Sec-Fetch-Dest: document',
            'Sec-Fetch-Mode: navigate',
            'Sec-Fetch-Site: none',
            'Cache-Control: max-age=0'
        ]);
        
        $response = curl_exec($ch);
        
        // Login-Daten senden
        $postData = http_build_query([
            'UserName' => $this->resyUsername,
            'Password' => $this->resyPassword,
            'button' => 'submit',
            'FN' => 'login'
        ]);
        
        curl_setopt($ch, CURLOPT_URL, 'https://resy.dtm-reifen.de/index.php?m=login&a=login');
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Content-Type: application/x-www-form-urlencoded',
            'Referer: https://resy.dtm-reifen.de/index.php?m=login&a=login',
            'Origin: https://resy.dtm-reifen.de',
            'Sec-Fetch-Dest: document',
            'Sec-Fetch-Mode: navigate',
            'Sec-Fetch-Site: same-origin'
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        if (strpos($response, 'Sätze / Rücklieferungen bestellen') !== false) {
            return ['success' => true];
        }
        
        return ['success' => false, 'error' => 'Custom-Header-Login fehlgeschlagen'];
    }
    
    /**
     * Erstelle Success-Response
     */
    private function buildSuccessResponse($wheelsets, $startTime, $source) {
        $endTime = microtime(true);
        $duration = round(($endTime - $startTime) * 1000, 2);
        
        return [
            'success' => true,
            'wheelsets' => $wheelsets,
            'count' => count($wheelsets),
            'autohaus' => $this->autohausName,
            'autohaus_id' => $this->autohausId,
            'source' => $source,
            'loaded_at' => date('Y-m-d H:i:s'),
            'duration_ms' => $duration,
            'error' => null
        ];
    }
    
    
    /**
     * Keine Daten - RESY-System nicht erreichbar
     */
    private function getNoDataResponse($error) {
        error_log("❌ KEINE DATEN - RESY-System nicht erreichbar für Autohaus ID: " . $this->autohausId . " - " . $error);
        
        return [
            'success' => false,
            'wheelsets' => [],
            'count' => 0,
            'autohaus' => $this->autohausName,
            'autohaus_id' => $this->autohausId,
            'source' => 'RESY-System (nicht erreichbar)',
            'loaded_at' => date('Y-m-d H:i:s'),
            'error' => $error
        ];
    }
    
    /**
     * Fallback-Daten wenn RESY-System nicht erreichbar ist
     */
    private function getFallbackData($error) {
        error_log("❌ RESY-System nicht erreichbar für Autohaus ID: " . $this->autohausId . " - " . $error);
        
        return [
            'success' => false,
            'wheelsets' => [],
            'count' => 0,
            'autohaus' => $this->autohausName,
            'autohaus_id' => $this->autohausId,
            'source' => 'RESY-System (nicht erreichbar)',
            'loaded_at' => date('Y-m-d H:i:s'),
            'error' => $error
        ];
    }
    
}

// API-Handler
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $autohaus_id = $input['autohaus_id'] ?? null;
} elseif ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $autohaus_id = $_GET['autohaus_id'] ?? null;
}
    
if (!$autohaus_id) {
        echo json_encode([
            'success' => false,
            'error' => 'Autohaus-ID ist erforderlich'
        ]);
        exit;
    }
    
$api = new ProfessionalWheelsetsAPI($autohaus_id);
$result = $api->getStoredWheelsets();
    
    echo json_encode($result);
?>
