<?php
/**
 * E-Mail-Benachrichtigungen für das Boxenstop-System
 */

require_once __DIR__ . '/../config/database.php';

// Verwende vendor-Version (richtige PHPMailer-Implementierung mit Office 365 Fix)
if (file_exists(__DIR__ . '/../vendor/phpmailer/PHPMailer.php')) {
    require_once __DIR__ . '/../vendor/phpmailer/PHPMailer.php';
    require_once __DIR__ . '/../vendor/phpmailer/SMTP.php';
    require_once __DIR__ . '/../vendor/phpmailer/Exception.php';
} else {
    require_once __DIR__ . '/phpmailer/PHPMailer.php';
    require_once __DIR__ . '/phpmailer/SMTP.php';
    require_once __DIR__ . '/phpmailer/Exception.php';
}

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

class EmailNotifications {
    private $db;
    private $conn;
    
    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }
    
    /**
     * Sendet eine E-Mail-Benachrichtigung für Terminstatusänderungen
     */
    public function sendAppointmentNotification($appointmentId, $action) {
        try {
            // Hole Termindetails
            $appointment = $this->getAppointmentDetails($appointmentId);
            if (!$appointment) {
                return ['success' => false, 'error' => 'Termin nicht gefunden'];
            }
            
            // Prüfe ob E-Mail-Adresse vorhanden ist
            if (empty($appointment['email'])) {
                return ['success' => false, 'error' => 'Keine E-Mail-Adresse für Termin vorhanden'];
            }
            
            // Prüfe ob E-Mail-Benachrichtigungen aktiviert sind
            if (!$this->isEmailNotificationsEnabled()) {
                return ['success' => true, 'message' => 'E-Mail-Benachrichtigungen deaktiviert'];
            }
            
            // Hole E-Mail-Einstellungen
            $emailSettings = $this->getEmailSettings();
            
            // Erstelle E-Mail-Inhalt basierend auf Aktion
            $emailContent = $this->createEmailContent($appointment, $action);
            
            // Debug: Log E-Mail-Details
            error_log("Sende E-Mail an: " . $appointment['email']);
            error_log("E-Mail-Betreff: " . $emailContent['subject']);
            error_log("E-Mail-Aktion: " . $action);
            
            // Sende E-Mail
            $result = $this->sendEmail(
                $emailSettings['from_email'],
                $emailSettings['from_name'],
                $appointment['email'],
                $appointment['customer_name'],
                $emailContent['subject'],
                $emailContent['body']
            );
            
            error_log("E-Mail-Ergebnis: " . json_encode($result));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("E-Mail-Fehler: " . $e->getMessage());
            return ['success' => false, 'error' => 'E-Mail konnte nicht gesendet werden: ' . $e->getMessage()];
        }
    }
    
    /**
     * Holt Termindetails aus der Datenbank
     */
    private function getAppointmentDetails($appointmentId) {
        $stmt = $this->conn->prepare("
            SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time
            FROM appointments a
            JOIN events e ON a.event_id = e.id
            WHERE a.id = ?
        ");
        $stmt->execute([$appointmentId]);
        return $stmt->fetch();
    }
    
    /**
     * Prüft ob E-Mail-Benachrichtigungen aktiviert sind
     */
    private function isEmailNotificationsEnabled() {
        $stmt = $this->conn->prepare("
            SELECT setting_value FROM settings 
            WHERE setting_key = 'email_notifications_enabled'
        ");
        $stmt->execute();
        $result = $stmt->fetch();
        return $result && $result['setting_value'] === '1';
    }
    
    /**
     * Holt E-Mail-Einstellungen aus der Datenbank
     */
    private function getEmailSettings() {
        $stmt = $this->conn->prepare("
            SELECT setting_key, setting_value FROM settings 
            WHERE setting_key IN ('company_name', 'mail_from_email', 'mail_from_name', 'smtp_server', 'smtp_port', 'smtp_username', 'smtp_password', 'smtp_use_tls', 'smtp_use_ssl')
        ");
        $stmt->execute();
        $settings = [];
        while ($row = $stmt->fetch()) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        return [
            'from_email' => $settings['mail_from_email'] ?? 'rentacar-kiel@be-automobile.de',
            'from_name' => $settings['mail_from_name'] ?? 'Boxenstop Volvo Service',
            'smtp_host' => $settings['smtp_server'] ?? 'smtp-mail.outlook.com',
            'smtp_port' => (int)($settings['smtp_port'] ?? 587),
            'smtp_username' => $settings['smtp_username'] ?? 'rentacar-kiel@be-automobile.de',
            'smtp_password' => $settings['smtp_password'] ?? 'W@050373011206af',
            'smtp_use_tls' => $settings['smtp_use_tls'] === '1',
            'smtp_use_ssl' => $settings['smtp_use_ssl'] === '1'
        ];
    }
    
    /**
     * Erstellt E-Mail-Inhalt basierend auf Aktion
     */
    private function createEmailContent($appointment, $action) {
        $companyName = $this->getEmailSettings()['from_name'];
        $eventDate = date('d.m.Y', strtotime($appointment['event_date']));
        $slotTime = date('H:i', strtotime($appointment['slot_time']));
        
        switch ($action) {
            case 'approved':
                return [
                    'subject' => "✅ Terminbestätigung - {$appointment['event_name']} am {$eventDate}",
                    'body' => $this->getApprovalEmailBody($appointment, $eventDate, $slotTime, $companyName)
                ];
                
            case 'rejected':
                return [
                    'subject' => "❌ Terminabschluss - {$appointment['event_name']} am {$eventDate}",
                    'body' => $this->getRejectionEmailBody($appointment, $eventDate, $slotTime, $companyName)
                ];
                
            case 'rescheduled':
                return [
                    'subject' => "📅 Terminverschiebung - {$appointment['event_name']}",
                    'body' => $this->getRescheduleEmailBody($appointment, $eventDate, $slotTime, $companyName)
                ];
                
            default:
                return [
                    'subject' => "Terminupdate - {$appointment['event_name']}",
                    'body' => "Ihr Termin wurde aktualisiert."
                ];
        }
    }
    
    /**
     * E-Mail-Body für Terminbestätigung
     */
    private function getApprovalEmailBody($appointment, $eventDate, $slotTime, $companyName) {
        return "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .header { background: #1976d2; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; }
                .appointment-details { background: #f5f5f5; padding: 15px; border-radius: 8px; margin: 15px 0; }
                .footer { background: #f8f9fa; padding: 15px; text-align: center; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='header'>
                <h1>✅ Termin bestätigt!</h1>
                <p>Ihre Terminanfrage wurde genehmigt</p>
            </div>
            
            <div class='content'>
                <p>Liebe/r {$appointment['customer_name']},</p>
                
                <p>vielen Dank für Ihre Terminanfrage! Wir freuen uns, Ihnen mitteilen zu können, dass Ihr Termin bestätigt wurde.</p>
                
                <div class='appointment-details'>
                    <h3>📅 Termindetails</h3>
                    <p><strong>Event:</strong> {$appointment['event_name']}</p>
                    <p><strong>Datum:</strong> {$eventDate}</p>
                    <p><strong>Zeit:</strong> {$slotTime} Uhr</p>
                    <p><strong>Fahrzeug:</strong> {$appointment['license_plate']}</p>
                    " . ($appointment['storage_number'] ? "<p><strong>Einlagerungsnummer:</strong> {$appointment['storage_number']}</p>" : "") . "
                </div>
                
                <p>Bitte erscheinen Sie pünktlich zu Ihrem Termin. Bei Fragen stehen wir Ihnen gerne zur Verfügung.</p>
                
                <p>Mit freundlichen Grüßen<br>
                Ihr {$companyName} Team</p>
            </div>
            
            <div class='footer'>
                <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
            </div>
        </body>
        </html>";
    }
    
    /**
     * E-Mail-Body für Terminablehnung
     */
    private function getRejectionEmailBody($appointment, $eventDate, $slotTime, $companyName) {
        $rejectionReason = $appointment['rejection_reason'] ?? 'Leider ist der gewünschte Termin nicht verfügbar.';
        
        return "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .header { background: #dc3545; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; }
                .appointment-details { background: #f5f5f5; padding: 15px; border-radius: 8px; margin: 15px 0; }
                .footer { background: #f8f9fa; padding: 15px; text-align: center; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='header'>
                <h1>❌ Termin nicht verfügbar</h1>
                <p>Ihre Terminanfrage konnte leider nicht bestätigt werden</p>
            </div>
            
            <div class='content'>
                <p>Liebe/r {$appointment['customer_name']},</p>
                
                <p>vielen Dank für Ihre Terminanfrage. Leider können wir Ihren gewünschten Termin nicht bestätigen.</p>
                
                <div class='appointment-details'>
                    <h3>📅 Angefragter Termin</h3>
                    <p><strong>Event:</strong> {$appointment['event_name']}</p>
                    <p><strong>Datum:</strong> {$eventDate}</p>
                    <p><strong>Zeit:</strong> {$slotTime} Uhr</p>
                    <p><strong>Grund:</strong> {$rejectionReason}</p>
                </div>
                
                <p>Bitte kontaktieren Sie uns telefonisch, um einen alternativen Termin zu vereinbaren.</p>
                
                <p>Mit freundlichen Grüßen<br>
                Ihr {$companyName} Team</p>
            </div>
            
            <div class='footer'>
                <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
            </div>
        </body>
        </html>";
    }
    
    /**
     * E-Mail-Body für Terminverschiebung
     */
    private function getRescheduleEmailBody($appointment, $eventDate, $slotTime, $companyName) {
        $rescheduleReason = $appointment['reschedule_reason'] ?? 'Der Termin wurde verschoben.';
        
        return "
        <html>
        <head>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .header { background: #ffc107; color: #212529; padding: 20px; text-align: center; }
                .content { padding: 20px; }
                .appointment-details { background: #f5f5f5; padding: 15px; border-radius: 8px; margin: 15px 0; }
                .footer { background: #f8f9fa; padding: 15px; text-align: center; font-size: 12px; color: #666; }
            </style>
        </head>
        <body>
            <div class='header'>
                <h1>📅 Termin verschoben</h1>
                <p>Ihr Termin wurde auf einen neuen Zeitpunkt verschoben</p>
            </div>
            
            <div class='content'>
                <p>Liebe/r {$appointment['customer_name']},</p>
                
                <p>Ihr Termin wurde verschoben. Hier sind die neuen Details:</p>
                
                <div class='appointment-details'>
                    <h3>📅 Neuer Termin</h3>
                    <p><strong>Event:</strong> {$appointment['event_name']}</p>
                    <p><strong>Datum:</strong> {$eventDate}</p>
                    <p><strong>Zeit:</strong> {$slotTime} Uhr</p>
                    <p><strong>Grund:</strong> {$rescheduleReason}</p>
                </div>
                
                <p>Bitte erscheinen Sie pünktlich zu Ihrem neuen Termin. Bei Fragen stehen wir Ihnen gerne zur Verfügung.</p>
                
                <p>Mit freundlichen Grüßen<br>
                Ihr {$companyName} Team</p>
            </div>
            
            <div class='footer'>
                <p>Diese E-Mail wurde automatisch generiert. Bitte antworten Sie nicht direkt auf diese E-Mail.</p>
            </div>
        </body>
        </html>";
    }
    
    /**
     * Sendet E-Mail über PHPMailer
     */
    private function sendEmail($fromEmail, $fromName, $toEmail, $toName, $subject, $body) {
        $mail = new PHPMailer(true);
        $emailSettings = $this->getEmailSettings();
        
        try {
            // Server settings
            $mail->isSMTP();
            $mail->Host = $emailSettings['smtp_host'];
            $mail->SMTPAuth = true;
            $mail->Username = $emailSettings['smtp_username'];
            $mail->Password = $emailSettings['smtp_password'];
            
            // Verschlüsselung basierend auf Einstellungen
            if ($emailSettings['smtp_use_ssl']) {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_SMTPS;
            } elseif ($emailSettings['smtp_use_tls']) {
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
            }
            
            $mail->Port = $emailSettings['smtp_port'];
            $mail->CharSet = 'UTF-8';
            $mail->SMTPDebug = 2; // Debug aktiviert für Termin-E-Mails
            $mail->Debugoutput = function($str, $level) {
                $logFile = __DIR__ . '/../logs/appointment_email_debug.log';
                file_put_contents($logFile, date('Y-m-d H:i:s') . " Level $level: $str\n", FILE_APPEND | LOCK_EX);
            };
            
            // Recipients
            $mail->setFrom($fromEmail, $fromName);
            $mail->addAddress($toEmail, $toName); // An Kunden-E-Mail senden
            
            // Content
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $body;
            
            $mail->send();
            error_log("E-Mail erfolgreich gesendet an: " . $toEmail);
            return ['success' => true, 'message' => 'E-Mail erfolgreich gesendet an ' . $toEmail];
            
        } catch (Exception $e) {
            $errorMsg = 'E-Mail-Fehler: ' . $e->getMessage();
            error_log("E-Mail-Fehler für " . $toEmail . ": " . $errorMsg);
            error_log("PHPMailer ErrorInfo: " . $mail->ErrorInfo);
            return ['success' => false, 'error' => $errorMsg . " (PHPMailer: " . $mail->ErrorInfo . ")"];
        }
    }
}
?>
