<?php
/**
 * Test-Datei für Autohaus-spezifische WhatsApp-Benachrichtigungen
 * Verwendet HelloMateo API für WhatsApp-Nachrichten
 */

header('Content-Type: application/json');
require_once '../config/database.php';
require_once '../api/whatsapp_notifications.php';

session_start();

// Prüfe ob Admin eingeloggt ist
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['action']) || $input['action'] !== 'test_autohaus_whatsapp') {
        throw new Exception('Ungültige Anfrage');
    }
    
    $autohausId = $input['autohaus_id'] ?? null;
    if (!$autohausId) {
        throw new Exception('Autohaus-ID fehlt');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Hole Autohaus-Details
    $stmt = $conn->prepare("SELECT * FROM autohaus WHERE id = ?");
    $stmt->execute([$autohausId]);
    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$autohaus) {
        throw new Exception('Autohaus nicht gefunden');
    }
    
    // Prüfe WhatsApp-Einstellungen für dieses Autohaus
    $whatsappEnabled = $autohaus['whatsapp_notifications_enabled'] ?? false;
    $whatsappPhone = $autohaus['whatsapp_phone_number'] ?? '';
    $whatsappApiKey = $autohaus['whatsapp_api_key'] ?? '';
    $whatsappBusinessName = $autohaus['whatsapp_business_name'] ?? $autohaus['name'];
    $whatsappTestMode = $autohaus['whatsapp_test_mode'] ?? true;
    
    if (!$whatsappEnabled) {
        throw new Exception('WhatsApp-Benachrichtigungen für dieses Autohaus sind deaktiviert');
    }
    
    if (empty($whatsappPhone)) {
        throw new Exception('Keine WhatsApp-Nummer für dieses Autohaus konfiguriert');
    }
    
    // Erstelle Test-Nachricht
    $testMessage = "🧪 *WhatsApp-Test von " . $autohaus['name'] . "*\n\n" .
                   "Hallo! Dies ist eine Test-Nachricht vom Boxenstop-System.\n\n" .
                   "✅ WhatsApp-Integration funktioniert korrekt\n" .
                   "🏢 Autohaus: " . $autohaus['name'] . "\n" .
                   "📱 Nummer: " . $whatsappPhone . "\n" .
                   "🕐 Zeit: " . date('d.m.Y H:i:s') . "\n\n" .
                   "Mit freundlichen Grüßen\n" .
                   "Ihr " . $whatsappBusinessName . " Team";
    
    // Test-Modus oder echter Versand
    if ($whatsappTestMode || empty($whatsappApiKey)) {
        // Simuliere WhatsApp-Versendung
        error_log("WhatsApp Test (Autohaus) - Nachricht an " . $whatsappPhone . ": " . $testMessage);
        error_log("WhatsApp Test (Autohaus) - Autohaus: " . $autohaus['name']);
        
        $mode = $whatsappTestMode ? 'Test-Modus' : 'Kein API-Key';
        echo json_encode([
            'success' => true,
            'message' => 'WhatsApp-Test erfolgreich (' . $mode . ')',
            'autohaus_name' => $autohaus['name'],
            'recipient' => $whatsappPhone,
            'test_mode' => $whatsappTestMode,
            'mode' => $mode
        ]);
    } else {
        // Echter WhatsApp-Versand über HelloMateo API
        $whatsapp = new WhatsAppNotifications();
        
        // Temporäre Einstellungen für diesen Test setzen
        $originalSettings = [
            'api_key' => $whatsappApiKey,
            'phone_number' => $whatsappPhone,
            'business_name' => $whatsappBusinessName,
            'test_mode' => false
        ];
        
        // Sende Test-Nachricht
        $result = $whatsapp->sendWhatsAppMessage($whatsappPhone, $testMessage);
        
        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => 'WhatsApp-Test erfolgreich gesendet',
                'autohaus_name' => $autohaus['name'],
                'recipient' => $whatsappPhone,
                'test_mode' => false,
                'mode' => 'Live-Versand'
            ]);
        } else {
            throw new Exception($result['error'] ?? 'WhatsApp-Nachricht konnte nicht gesendet werden');
        }
    }
    
} catch (Exception $e) {
    error_log("WhatsApp Test (Autohaus) - Fehler: " . $e->getMessage());
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
