<?php
/**
 * Admin User Management Dashboard - Komplett neue Version
 * Moderne UI basierend auf appointments.php Design
 * Gleiches Backend-Logic wie vorher
 */

session_start();
require_once '../config/database.php';

// Check if user is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Get current user data with autohaus information
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Administrator'; // Default name
$current_user_role = 'admin';
$current_user_autohaus_id = null;
$current_user_autohaus_name = null;
$can_switch_autohaus = false;

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.role, u.autohaus_id, u.can_switch_autohaus, a.name as autohaus_name, u.username
        FROM admin_users u
        LEFT JOIN autohaus a ON u.autohaus_id = a.id
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    $current_user_role = $user_data['role'] ?? 'admin';
    $current_user_autohaus_id = $user_data['autohaus_id'];
    $current_user_autohaus_name = $user_data['autohaus_name'];
    $current_user_name = $user_data['username'] ?? 'Administrator';
    $can_switch_autohaus = $user_data['can_switch_autohaus'] == 1;
}

// Get user theme preference
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

$action = $_GET['action'] ?? 'list';
$user_id = $_GET['id'] ?? null;
$status_filter = $_GET['status'] ?? '';

// Handle form submissions
if ($_POST) {
    $postAction = $_POST['action'] ?? $action;
    $postId = $_POST['id'] ?? $user_id;
    if ($postAction === 'approve' && $postId) {
        handleUserApprove($conn, $postId);
    } elseif ($postAction === 'reject' && $postId) {
        handleUserReject($conn, $postId);
    } elseif ($postAction === 'bulk_approve') {
        handleBulkApprove($conn);
    } elseif ($postAction === 'bulk_reject') {
        handleBulkReject($conn);
    } elseif ($postAction === 'update_theme') {
        handleThemeUpdate($conn, $current_user_id);
    }
}

// Get users (filtered by status)
$users = getUsers($conn, $status_filter);

// Get statistics
$stats = getUserStats($conn);

function getUsers($conn, $status_filter = '') {
    $sql = "
        SELECT 
            u.id, u.first_name, u.last_name, u.email, u.phone_number, 
            u.license_plate, u.status, u.created_at, u.admin_notes,
            u.registration_source, u.registration_device, u.registration_ip,
            u.last_login_source, u.last_login_device,
            u.verification_status, u.verification_document, u.verification_reason,
            u.verified_by, u.verified_at, u.profile_image,
            a.name as dealership_name,
            COUNT(vd.id) as document_count,
            MAX(vd.upload_date) as last_document_upload,
            au.username as approved_by_name,
            ru.username as rejected_by_name
        FROM users u
        LEFT JOIN autohaus a ON u.preferred_dealership_id = a.id
        LEFT JOIN verification_documents vd ON u.id = vd.user_id
        LEFT JOIN admin_users au ON u.approved_by = au.id
        LEFT JOIN admin_users ru ON u.rejected_by = ru.id
    ";
    
    $where_conditions = [];
    $params = [];
    
    if ($status_filter) {
        $where_conditions[] = "u.status = ?";
        $params[] = $status_filter;
    }
    
    if (!empty($where_conditions)) {
        $sql .= " WHERE " . implode(" AND ", $where_conditions);
    }
    
    $sql .= " GROUP BY u.id ORDER BY u.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getUserStats($conn) {
    $stmt = $conn->prepare("
        SELECT 
            COUNT(*) as total,
            COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending,
            COUNT(CASE WHEN status = 'approved' THEN 1 END) as approved,
            COUNT(CASE WHEN status = 'rejected' THEN 1 END) as rejected,
            COUNT(CASE WHEN status = 'suspended' THEN 1 END) as suspended
        FROM users
    ");
    $stmt->execute();
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function handleUserApprove($conn, $user_id) {
    $admin_notes = $_POST['admin_notes'] ?? '';
    $admin_id = $_SESSION['admin_user_id'];
    
    $conn->beginTransaction();
    
    try {
        // Approve user
        $stmt = $conn->prepare("
            UPDATE users 
            SET status = 'approved', 
                approved_by = ?, 
                approved_at = NOW(),
                admin_notes = ?,
                rejected_by = NULL,
                rejected_at = NULL,
                rejection_reason = NULL
            WHERE id = ?
        ");
        $stmt->execute([$admin_id, $admin_notes, $user_id]);
        
        // Create notification
        $stmt = $conn->prepare("
            INSERT INTO user_notifications (user_id, type, title, message)
            VALUES (?, 'account_approved', 'Account freigeschaltet', 'Ihr Account wurde erfolgreich freigeschaltet. Sie können sich jetzt anmelden.')
        ");
        $stmt->execute([$user_id]);
        
        $conn->commit();
        $_SESSION['success_message'] = 'Benutzer erfolgreich freigeschaltet';
    } catch (Exception $e) {
        $conn->rollBack();
        $_SESSION['error_message'] = 'Fehler beim Freischalten: ' . $e->getMessage();
    }
}

function handleUserReject($conn, $user_id) {
    $rejection_reason = $_POST['rejection_reason'] ?? '';
    $admin_notes = $_POST['admin_notes'] ?? '';
    
    if (empty($rejection_reason)) {
        $_SESSION['error_message'] = 'Ablehnungsgrund ist erforderlich';
        return;
    }
    
    $admin_id = $_SESSION['admin_user_id'];
    
    $conn->beginTransaction();
    
    try {
        // Reject user
        $stmt = $conn->prepare("
            UPDATE users 
            SET status = 'rejected', 
                rejected_by = ?, 
                rejected_at = NOW(),
                rejection_reason = ?,
                admin_notes = ?,
                approved_by = NULL,
                approved_at = NULL
            WHERE id = ?
        ");
        $stmt->execute([$admin_id, $rejection_reason, $admin_notes, $user_id]);
        
        // Create notification
        $stmt = $conn->prepare("
            INSERT INTO user_notifications (user_id, type, title, message)
            VALUES (?, 'account_rejected', 'Account abgelehnt', ?)
        ");
        $stmt->execute([$user_id, "Ihr Account wurde abgelehnt. Grund: " . $rejection_reason]);
        
        $conn->commit();
        $_SESSION['success_message'] = 'Benutzer abgelehnt';
    } catch (Exception $e) {
        $conn->rollBack();
        $_SESSION['error_message'] = 'Fehler beim Ablehnen: ' . $e->getMessage();
    }
}

function handleBulkApprove($conn) {
    $user_ids = $_POST['user_ids'] ?? [];
    if (empty($user_ids)) {
        $_SESSION['error_message'] = 'Keine Benutzer ausgewählt';
        return;
    }
    
    $admin_id = $_SESSION['admin_user_id'];
    $admin_notes = $_POST['bulk_admin_notes'] ?? '';
    
    $conn->beginTransaction();
    
    try {
        foreach ($user_ids as $user_id) {
            // Approve user
            $stmt = $conn->prepare("
                UPDATE users 
                SET status = 'approved', 
                    approved_by = ?, 
                    approved_at = NOW(),
                    admin_notes = ?
                WHERE id = ?
            ");
            $stmt->execute([$admin_id, $admin_notes, $user_id]);
            
            // Create notification
            $stmt = $conn->prepare("
                INSERT INTO user_notifications (user_id, type, title, message)
                VALUES (?, 'account_approved', 'Account freigeschaltet', 'Ihr Account wurde erfolgreich freigeschaltet.')
            ");
            $stmt->execute([$user_id]);
        }
        
        $conn->commit();
        $_SESSION['success_message'] = count($user_ids) . ' Benutzer erfolgreich freigeschaltet';
    } catch (Exception $e) {
        $conn->rollBack();
        $_SESSION['error_message'] = 'Fehler beim Freischalten: ' . $e->getMessage();
    }
}

function handleBulkReject($conn) {
    $user_ids = $_POST['user_ids'] ?? [];
    if (empty($user_ids)) {
        $_SESSION['error_message'] = 'Keine Benutzer ausgewählt';
        return;
    }
    
    $admin_id = $_SESSION['admin_user_id'];
    $rejection_reason = $_POST['bulk_rejection_reason'] ?? '';
    $admin_notes = $_POST['bulk_admin_notes'] ?? '';
    
    if (empty($rejection_reason)) {
        $_SESSION['error_message'] = 'Ablehnungsgrund ist erforderlich';
        return;
    }
    
    $conn->beginTransaction();
    
    try {
        foreach ($user_ids as $user_id) {
            // Reject user
            $stmt = $conn->prepare("
                UPDATE users 
                SET status = 'rejected', 
                    rejected_by = ?, 
                    rejected_at = NOW(),
                    rejection_reason = ?,
                    admin_notes = ?
                WHERE id = ?
            ");
            $stmt->execute([$admin_id, $rejection_reason, $admin_notes, $user_id]);
            
            // Create notification
            $stmt = $conn->prepare("
                INSERT INTO user_notifications (user_id, type, title, message)
                VALUES (?, 'account_rejected', 'Account abgelehnt', ?)
            ");
            $stmt->execute([$user_id, "Ihr Account wurde abgelehnt. Grund: " . $rejection_reason]);
        }
        
        $conn->commit();
        $_SESSION['success_message'] = count($user_ids) . ' Benutzer abgelehnt';
    } catch (Exception $e) {
        $conn->rollBack();
        $_SESSION['error_message'] = 'Fehler beim Ablehnen: ' . $e->getMessage();
    }
}

function handleThemeUpdate($conn, $user_id) {
    $theme = $_POST['theme'] ?? 'light';
    
    if ($user_id) {
        try {
            $stmt = $conn->prepare("UPDATE admin_users SET theme = ? WHERE id = ?");
            $stmt->execute([$theme, $user_id]);
        } catch (Exception $e) {
            error_log('Theme update error: ' . $e->getMessage());
        }
    }
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Benutzerverwaltung - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/global-dark-theme.css">
    <link rel="stylesheet" href="assets/css/unified-sidebar.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&family=SF+Pro+Display:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="admin-body" data-theme="<?php echo htmlspecialchars($current_theme); ?>" data-user-id="<?php echo htmlspecialchars($current_user_id); ?>">
    <div class="admin-container">
        <!-- Include Unified Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <main class="admin-main">
            <!-- Modern Header Section - Based on appointments.php -->
            <header class="modern-header">
                <div class="header-content">
                    <div class="header-left">
                        <div class="page-title-section">
                            <h1 class="page-title">
                                <i class="fas fa-user-shield"></i>
                                Benutzerverwaltung
                            </h1>
                            <p class="page-subtitle">Verwalten Sie alle Benutzerregistrierungen und Verifikationen</p>
                        </div>
                        <div class="stats-overview">
                            <div class="stat-card">
                                <div class="stat-icon">
                                    <i class="fas fa-users"></i>
                                </div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $stats['total']; ?></div>
                                    <div class="stat-label">Gesamt Benutzer</div>
                                </div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-icon pending">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $stats['pending']; ?></div>
                                    <div class="stat-label">Ausstehend</div>
                                </div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-icon approved">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $stats['approved']; ?></div>
                                    <div class="stat-label">Freigeschaltet</div>
                                </div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-icon rejected">
                                    <i class="fas fa-times-circle"></i>
                                </div>
                                <div class="stat-content">
                                    <div class="stat-number"><?php echo $stats['rejected']; ?></div>
                                    <div class="stat-label">Abgelehnt</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="header-right">
                        <div class="header-actions">
                            <div class="filter-section">
                                <select id="status-filter" class="modern-select" onchange="filterByStatus()">
                                    <option value="">Alle Status</option>
                                    <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Ausstehend</option>
                                    <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>Freigeschaltet</option>
                                    <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Abgelehnt</option>
                                    <option value="suspended" <?php echo $status_filter === 'suspended' ? 'selected' : ''; ?>>Gesperrt</option>
                                </select>
                            </div>
                            <div class="action-buttons">
                                <button class="btn btn-primary" onclick="exportUsers()">
                                    <i class="fas fa-download"></i>
                                    <span>Exportieren</span>
                                </button>
                                <button class="btn btn-secondary" onclick="manualRefresh()" title="Daten aktualisieren">
                                    <i class="fas fa-sync-alt"></i>
                                    <span>Aktualisieren</span>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </header>

            <!-- Content Area -->
            <div class="content-area">
                <!-- Messages -->
                <?php if (isset($_SESSION['success_message'])): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
                    </div>
                <?php endif; ?>

                <?php if (isset($_SESSION['error_message'])): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
                    </div>
                <?php endif; ?>

                <!-- Users Management Card -->
                <div class="content-card">
                    <div class="card-header">
                        <div class="card-title-section">
                            <h3>Benutzer Übersicht</h3>
                            <div class="card-subtitle">
                                <span class="user-count"><?php echo count($users); ?> Benutzer</span>
                                <?php 
                                $pendingCount = count(array_filter($users, function($u) { return $u['status'] === 'pending'; }));
                                $approvedCount = count(array_filter($users, function($u) { return $u['status'] === 'approved'; }));
                                $rejectedCount = count(array_filter($users, function($u) { return $u['status'] === 'rejected'; }));
                                ?>
                                <?php if ($pendingCount > 0): ?>
                                    <span class="status-indicator pending">
                                        <i class="fas fa-clock"></i>
                                        <?php echo $pendingCount; ?> ausstehend
                                    </span>
                                <?php endif; ?>
                                <?php if ($approvedCount > 0): ?>
                                    <span class="status-indicator approved">
                                        <i class="fas fa-check-circle"></i>
                                        <?php echo $approvedCount; ?> freigeschaltet
                                    </span>
                                <?php endif; ?>
                                <?php if ($rejectedCount > 0): ?>
                                    <span class="status-indicator rejected">
                                        <i class="fas fa-times-circle"></i>
                                        <?php echo $rejectedCount; ?> abgelehnt
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="card-actions">
                            <button class="btn btn-outline" onclick="toggleBulkMode()">
                                <i class="fas fa-check-square"></i>
                                <span>Bulk-Modus</span>
                            </button>
                        </div>
                    </div>
                    
                    <div class="card-content">
                        <?php if (empty($users)): ?>
                            <div class="empty-state">
                                <div class="empty-icon">
                                    <i class="fas fa-user-times"></i>
                                </div>
                                <h4>Keine Benutzer vorhanden</h4>
                                <p>Es wurden keine Benutzer mit den aktuellen Filtereinstellungen gefunden.</p>
                            </div>
                        <?php else: ?>
                            <!-- Bulk Actions Toolbar -->
                            <div id="bulk-actions-toolbar" class="bulk-actions-toolbar" style="display: none;">
                                <div class="bulk-actions-content">
                                    <div class="bulk-actions-left">
                                        <button class="btn btn-sm btn-outline" onclick="toggleAllUsers()">
                                            <i class="fas fa-check-square"></i>
                                            <span id="select-all-text">Alle auswählen</span>
                                        </button>
                                        <div class="selected-count">
                                            <i class="fas fa-users"></i>
                                            <span id="selected-count-text">0 Benutzer ausgewählt</span>
                                        </div>
                                    </div>
                                    <div class="bulk-actions-right">
                                        <button class="btn btn-sm btn-success" onclick="bulkApproveUsers()">
                                            <i class="fas fa-check"></i>
                                            <span>Freischalten</span>
                                        </button>
                                        <button class="btn btn-sm btn-danger" onclick="bulkRejectUsers()">
                                            <i class="fas fa-times"></i>
                                            <span>Ablehnen</span>
                                        </button>
                                        <button class="btn btn-sm btn-primary" onclick="bulkExportUsers()">
                                            <i class="fas fa-download"></i>
                                            <span>Exportieren</span>
                                        </button>
                                    </div>
                                </div>
                            </div>

                            <!-- Users Table -->
                            <div class="table-container">
                                <table class="modern-table">
                                    <thead>
                                        <tr>
                                            <th width="50">
                                                <input type="checkbox" id="select-all-checkbox" onchange="toggleAllUsers()">
                                            </th>
                                            <th data-sort="user_name" width="200">Benutzer</th>
                                            <th data-sort="email" width="220">Kontakt</th>
                                            <th data-sort="license_plate" width="120">Kennzeichen</th>
                                            <th data-sort="registration_source" width="120">Registriert via</th>
                                            <th data-sort="dealership_name" width="150">Autohaus</th>
                                            <th data-sort="status" width="120">Status</th>
                                            <th width="100">Dokumente</th>
                                            <th data-sort="created_at" width="140">Registriert</th>
                                            <th width="140">Aktionen</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($users as $user): ?>
                                            <tr class="user-row" data-user-id="<?php echo $user['id']; ?>" data-status="<?php echo $user['status']; ?>">
                                                <td>
                                                    <input type="checkbox" class="user-checkbox" value="<?php echo $user['id']; ?>" onchange="updateBulkActions()">
                                                </td>
                                                <td>
                                                    <div class="user-info">
                                                    <div class="user-avatar">
                                                        <?php if (!empty($user['profile_image'])): ?>
                                                            <img src="../<?php echo htmlspecialchars($user['profile_image']); ?>" alt="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>" style="width: 100%; height: 100%; object-fit: cover; border-radius: 50%;">
                                                        <?php else: ?>
                                                            <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                                                        <?php endif; ?>
                                                    </div>
                                                    <div class="user-details">
                                                        <div class="user-name"><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></div>
                                                        <div class="user-id">ID: <?php echo $user['id']; ?></div>
                                                    </div>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="contact-info">
                                                        <div class="contact-item">
                                                            <i class="fas fa-envelope"></i>
                                                            <span><?php echo htmlspecialchars($user['email']); ?></span>
                                                        </div>
                                                        <div class="contact-item">
                                                            <i class="fas fa-phone"></i>
                                                            <span><?php echo htmlspecialchars($user['phone_number']); ?></span>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="license-plate">
                                                        <?php echo htmlspecialchars($user['license_plate']); ?>
                                                    </div>
                                                </td>
                                                <td>
                                                    <?php
                                                    $source = $user['registration_source'] ?? 'unknown';
                                                    $source_icons = [
                                                        'web' => 'fas fa-globe',
                                                        'ios' => 'fab fa-apple',
                                                        'android' => 'fab fa-android',
                                                        'apple_signin' => 'fab fa-apple',
                                                        'google_signin' => 'fab fa-google',
                                                        'unknown' => 'fas fa-question-circle'
                                                    ];
                                                    $source_colors = [
                                                        'web' => '#007bff',
                                                        'ios' => '#000000',
                                                        'android' => '#3DDC84',
                                                        'apple_signin' => '#A8B1B9',
                                                        'google_signin' => '#4285f4',
                                                        'unknown' => '#6c757d'
                                                    ];
                                                    $source_labels = [
                                                        'web' => 'Web',
                                                        'ios' => 'iOS App',
                                                        'android' => 'Android App',
                                                        'apple_signin' => 'Apple Sign-In',
                                                        'google_signin' => 'Google Sign-In',
                                                        'unknown' => 'Unbekannt'
                                                    ];
                                                    $icon = $source_icons[$source] ?? $source_icons['unknown'];
                                                    $color = $source_colors[$source] ?? $source_colors['unknown'];
                                                    $label = $source_labels[$source] ?? 'Unbekannt';
                                                    ?>
                                                    <div class="registration-source" style="display: flex; align-items: center; gap: 0.5rem;">
                                                        <i class="<?php echo $icon; ?>" style="color: <?php echo $color; ?>; font-size: 1.2rem;"></i>
                                                        <span style="color: #495057; font-weight: 500;"><?php echo $label; ?></span>
                                                    </div>
                                                    <?php if (!empty($user['registration_device'])): ?>
                                                        <small style="color: #6c757d; font-size: 0.85rem;">
                                                            <?php echo htmlspecialchars($user['registration_device']); ?>
                                                        </small>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="dealership">
                                                        <?php echo htmlspecialchars($user['dealership_name'] ?? 'Nicht gewählt'); ?>
                                                    </div>
                                                </td>
                                                <td>
                                                    <span class="status-badge status-<?php echo $user['status']; ?>">
                                                        <?php 
                                                        $status_labels = [
                                                            'pending' => 'Ausstehend',
                                                            'approved' => 'Freigeschaltet', 
                                                            'rejected' => 'Abgelehnt',
                                                            'suspended' => 'Gesperrt'
                                                        ]; 
                                                        echo $status_labels[$user['status']] ?? $user['status']; 
                                                        ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <div class="document-info">
                                                        <?php if (!empty($user['verification_document'])): ?>
                                                            <div class="document-view">
                                                                <a href="../<?php echo htmlspecialchars($user['verification_document']); ?>" 
                                                                   target="_blank" 
                                                                   class="btn btn-sm" 
                                                                   style="background: #28a745; color: white; padding: 0.4rem 0.8rem; border-radius: 6px; text-decoration: none; display: inline-flex; align-items: center; gap: 0.4rem; font-size: 0.85rem; font-weight: 600;">
                                                                    <i class="fas fa-file-pdf"></i>
                                                                    <span>Ansehen</span>
                                                                </a>
                                                            </div>
                                                            <?php if ($user['verification_status'] === 'rejected' && !empty($user['verification_reason'])): ?>
                                                                <div style="color: #dc3545; font-size: 0.85rem; margin-top: 0.25rem;">
                                                                    <i class="fas fa-times-circle"></i>
                                                                    <strong>Abgelehnt</strong>
                                                                </div>
                                                                <div style="color: #6c757d; font-size: 0.8rem;">
                                                                    <?php echo htmlspecialchars(substr($user['verification_reason'], 0, 40)) . (strlen($user['verification_reason']) > 40 ? '...' : ''); ?>
                                                                </div>
                                                            <?php elseif ($user['verification_status'] === 'pending'): ?>
                                                                <div style="color: #ffc107; font-size: 0.85rem; margin-top: 0.25rem;">
                                                                    <i class="fas fa-hourglass-half"></i>
                                                                    <strong>Wartend</strong>
                                                                </div>
                                                            <?php elseif ($user['verification_status'] === 'approved'): ?>
                                                                <div style="color: #28a745; font-size: 0.85rem; margin-top: 0.25rem;">
                                                                    <i class="fas fa-check-circle"></i>
                                                                    <strong>Freigeschaltet</strong>
                                                                </div>
                                                            <?php endif; ?>
                                                        <?php else: ?>
                                                            <div style="color: #6c757d; font-size: 0.9rem;">
                                                                <i class="fas fa-times"></i>
                                                                Kein Dokument
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="registration-date">
                                                        <?php echo date('d.m.Y H:i', strtotime($user['created_at'])); ?>
                                                    </div>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <button class="action-btn btn-view" onclick="showUserDetails(<?php echo htmlspecialchars(json_encode($user)); ?>)" title="Details anzeigen">
                                                            <i class="fas fa-eye"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Auto-refresh indicator -->
    <div id="auto-refresh-indicator" class="auto-refresh-indicator">
        <i class="fas fa-sync-alt"></i>
        <span>Daten werden aktualisiert...</span>
    </div>

    <!-- NEUES PROFESSIONELLES USER MODAL -->
    <div id="user-modal" class="new-professional-modal" style="display: none;">
        <div class="new-modal-content">
            <div id="user-details">
                <!-- Content wird hier geladen -->
            </div>
        </div>
    </div>

    <!-- Email Send Modal -->
    <div id="email-send-modal" class="modal" style="display: none;">
        <div class="modal-content" style="max-width: 700px;">
            <div class="modal-header">
                <h3><i class="fas fa-envelope"></i> E-Mail senden</h3>
                <button class="modal-close" onclick="closeEmailSendModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body" style="padding: 2rem;">
                <div id="email-preview" style="background: #f8f9fa; padding: 1.5rem; border-radius: 12px; margin-bottom: 1.5rem; border: 2px solid #dee2e6; min-height: 200px; max-height: 400px; overflow-y: auto;">
                    <!-- E-Mail-Vorschau -->
                </div>
                
                <div id="email-result" style="display: none; padding: 1rem; border-radius: 8px; margin-bottom: 1rem;"></div>
                
                <div style="text-align: center;">
                    <button id="send-email-btn" class="btn btn-primary" onclick="sendEmailConfirm()" style="padding: 1rem 2rem; font-size: 1.1rem;">
                        <i class="fas fa-paper-plane"></i> E-Mail senden
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Approve User Modal -->
    <div id="approve-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Benutzer freischalten</h3>
                <button class="modal-close" onclick="closeApproveModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <form id="approve-form" method="POST">
                    <input type="hidden" name="action" value="approve">
                    <input type="hidden" name="id" id="approve-user-id">
                    <div class="form-group">
                        <label for="admin_notes">Admin-Notizen (optional)</label>
                        <textarea name="admin_notes" id="admin_notes" rows="3" placeholder="Notizen zur Freischaltung..."></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeApproveModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-success">Freischalten</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Suspend User Modal -->
    <div id="suspend-user-modal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fas fa-pause-circle"></i> Benutzer sperren</h2>
                <button class="close-btn" onclick="closeSuspendUserModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="suspend-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <p>Sie sind dabei, diesen Benutzer zu sperren. Der Benutzer wird eine Benachrichtigung erhalten und kann sich nicht mehr anmelden.</p>
                </div>
                <form id="suspend-form">
                    <div class="form-group">
                        <label for="suspend-reason">Grund für die Sperrung (optional):</label>
                        <textarea id="suspend-reason" name="reason" rows="4" placeholder="Geben Sie den Grund für die Sperrung an..."></textarea>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeSuspendUserModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-danger">
                            <i class="fas fa-pause-circle"></i>
                            Benutzer sperren
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Reject User Modal -->
    <div id="reject-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Benutzer ablehnen</h3>
                <button class="modal-close" onclick="closeRejectModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <form id="reject-form" method="POST">
                    <input type="hidden" name="action" value="reject">
                    <input type="hidden" name="id" id="reject-user-id">
                    <div class="form-group">
                        <label for="rejection_reason">Ablehnungsgrund *</label>
                        <textarea name="rejection_reason" id="rejection_reason" rows="3" placeholder="Grund für die Ablehnung..." required></textarea>
                    </div>
                    <div class="form-group">
                        <label for="reject_admin_notes">Admin-Notizen (optional)</label>
                        <textarea name="admin_notes" id="reject_admin_notes" rows="3" placeholder="Zusätzliche Notizen..."></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeRejectModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-danger">Ablehnen</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Bulk Actions Modal -->
    <div id="bulk-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Bulk-Aktionen</h3>
                <button class="modal-close" onclick="closeBulkModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body">
                <div id="bulk-modal-content"></div>
            </div>
        </div>
    </div>

    <!-- Document Verification Modal -->
    <div id="document-verification-modal" class="modal verification-modal">
        <div class="modal-content verification-modal-content">
            <div class="modal-header verification-header">
                <h3>Dokumentenverifikation</h3>
                <button class="modal-close" onclick="closeDocumentVerificationModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body verification-body">
                <div id="document-verification-content"></div>
            </div>
        </div>
    </div>

    <!-- Document Approval Modal -->
    <div id="document-approval-modal" class="modal approval-modal">
        <div class="modal-content approval-modal-content">
            <div class="modal-header approval-header">
                <h3>Dokument genehmigen</h3>
                <button class="modal-close" onclick="closeDocumentApprovalModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body approval-body">
                <form id="document-approval-form">
                    <input type="hidden" id="approval-document-id">
                    <input type="hidden" id="approval-user-id">
                    <div class="form-group">
                        <label for="approval-notes">Genehmigungsnotizen (optional)</label>
                        <textarea id="approval-notes" rows="4" placeholder="Notizen zur Genehmigung..."></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeDocumentApprovalModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-success">Dokument genehmigen</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Document Rejection Modal -->
    <div id="document-rejection-modal" class="modal rejection-modal">
        <div class="modal-content rejection-modal-content">
            <div class="modal-header rejection-header">
                <h3>Dokument ablehnen</h3>
                <button class="modal-close" onclick="closeDocumentRejectionModal()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-body rejection-body">
                <form id="document-rejection-form">
                    <input type="hidden" id="rejection-document-id">
                    <input type="hidden" id="rejection-user-id">
                    <div class="form-group">
                        <label for="rejection-reason">Ablehnungsgrund *</label>
                        <textarea id="rejection-reason" rows="4" placeholder="Grund für die Ablehnung..." required></textarea>
                    </div>
                    <div class="form-group">
                        <label for="rejection-notes">Zusätzliche Notizen (optional)</label>
                        <textarea id="rejection-notes" rows="3" placeholder="Weitere Notizen..."></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeDocumentRejectionModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-danger">Dokument ablehnen</button>
                    </div>
                </form>
            </div>
        </div>
    </div>


    <script>
        // Status filter
        function filterByStatus() {
            const status = document.getElementById('status-filter').value;
            if (status) {
                window.location.href = 'admin_dashboard.php?status=' + status;
            } else {
                window.location.href = 'admin_dashboard.php';
            }
        }

        // Toggle bulk mode
        function toggleBulkMode() {
            const toolbar = document.getElementById('bulk-actions-toolbar');
            const isVisible = toolbar.style.display === 'block';
            toolbar.style.display = isVisible ? 'none' : 'block';
            
            if (!isVisible) {
                // Clear all selections when entering bulk mode
                document.querySelectorAll('.user-checkbox').forEach(cb => cb.checked = false);
                updateBulkActions();
            }
        }

        // Toggle all users checkbox
        function toggleAllUsers() {
            const userCheckboxes = document.querySelectorAll('.user-checkbox');
            const allChecked = Array.from(userCheckboxes).every(cb => cb.checked);
            const selectAllText = document.getElementById('select-all-text');
            
            userCheckboxes.forEach(cb => {
                cb.checked = !allChecked;
            });
            
            selectAllText.textContent = allChecked ? 'Alle auswählen' : 'Alle abwählen';
            updateBulkActions();
        }

        // Update bulk actions toolbar
        function updateBulkActions() {
            const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
            const toolbar = document.getElementById('bulk-actions-toolbar');
            const countText = document.getElementById('selected-count-text');
            const selectAllText = document.getElementById('select-all-text');
            
            if (checkedBoxes.length > 0) {
                toolbar.style.display = 'block';
                countText.textContent = `${checkedBoxes.length} Benutzer ausgewählt`;
                selectAllText.textContent = checkedBoxes.length === document.querySelectorAll('.user-checkbox').length ? 'Alle abwählen' : 'Alle auswählen';
            } else {
                toolbar.style.display = 'none';
                selectAllText.textContent = 'Alle auswählen';
            }
        }

        // NEUES PROFESSIONELLES USER MODAL - KOMPLETT NEU
        function showUserDetails(user) {
            const modal = document.getElementById('user-modal');
            const details = document.getElementById('user-details');
            
            details.innerHTML = `
                <div class="new-user-dashboard">
                    <!-- Header -->
                    <div class="new-header">
                        <div class="new-user-info">
                            <div class="new-avatar">
                                ${user.profile_image ? 
                                    `<img src="../${user.profile_image}" alt="${user.first_name} ${user.last_name}" style="width: 100%; height: 100%; object-fit: cover; border-radius: 50%;">` :
                                    `<i class="fas fa-user-circle"></i>`
                                }
                            </div>
                            <div class="new-user-details">
                                <h1>${user.first_name} ${user.last_name}</h1>
                                <div class="new-user-meta">
                                    <span>ID: ${user.id}</span>
                                    <span class="new-status-badge ${user.status}">
                                        <i class="fas ${getUserStatusIcon(user.status)}"></i>
                                        ${getUserStatusLabel(user.status)}
                                    </span>
                                    <span>Registriert: ${formatDate(user.created_at)}</span>
                                </div>
                            </div>
                        </div>
                        <button class="new-close-btn" onclick="closeNewUserModal()">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    
                    <!-- Main Content -->
                    <div class="new-main-content">
                        <!-- Left Column -->
                        <div class="new-left-column">
                            <!-- Personal Info Card -->
                            <div class="new-card">
                                <div class="new-card-header">
                                    <i class="fas fa-user"></i>
                                    <h3>Persönliche Daten</h3>
                                    <div style="margin-left: auto;">
                                        <button id="edit-personal-btn-${user.id}" onclick="toggleEditPersonal(${user.id})" class="new-edit-btn">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button id="save-personal-btn-${user.id}" onclick="savePersonalData(${user.id})" class="new-save-btn" style="display: none;">
                                            <i class="fas fa-save"></i>
                                        </button>
                                        <button id="cancel-personal-btn-${user.id}" onclick="cancelEditPersonal(${user.id})" class="new-cancel-btn" style="display: none;">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="new-card-content">
                                    <div class="new-info-grid">
                                        <div class="new-info-field">
                                            <label>Vorname</label>
                                            <div class="new-info-value">
                                                <input type="text" id="edit-first-name-${user.id}" value="${user.first_name || ''}" class="new-edit-input" style="display: none;">
                                                <span id="display-first-name-${user.id}">${user.first_name || 'Nicht angegeben'}</span>
                                            </div>
                                        </div>
                                        <div class="new-info-field">
                                            <label>Nachname</label>
                                            <div class="new-info-value">
                                                <input type="text" id="edit-last-name-${user.id}" value="${user.last_name || ''}" class="new-edit-input" style="display: none;">
                                                <span id="display-last-name-${user.id}">${user.last_name || 'Nicht angegeben'}</span>
                                            </div>
                                        </div>
                                        <div class="new-info-field">
                                            <label>E-Mail</label>
                                            <div class="new-info-value email">
                                                <input type="email" id="edit-email-${user.id}" value="${user.email}" class="new-edit-input" style="display: none;">
                                                <span id="display-email-${user.id}">${user.email}</span>
                                            </div>
                                        </div>
                                        <div class="new-info-field">
                                            <label>Telefon</label>
                                            <div class="new-info-value">
                                                <input type="tel" id="edit-phone-${user.id}" value="${user.phone_number || ''}" class="new-edit-input" style="display: none;">
                                                <span id="display-phone-${user.id}">${user.phone_number || 'Nicht angegeben'}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Vehicle Card -->
                            <div class="new-card">
                                <div class="new-card-header">
                                    <i class="fas fa-car"></i>
                                    <h3>Fahrzeugdaten</h3>
                                    <div style="margin-left: auto;">
                                        <button id="edit-vehicle-btn-${user.id}" onclick="toggleEditVehicle(${user.id})" class="new-edit-btn">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button id="save-vehicle-btn-${user.id}" onclick="saveVehicleData(${user.id})" class="new-save-btn" style="display: none;">
                                            <i class="fas fa-save"></i>
                                        </button>
                                        <button id="cancel-vehicle-btn-${user.id}" onclick="cancelEditVehicle(${user.id})" class="new-cancel-btn" style="display: none;">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="new-card-content">
                                    <div class="new-info-grid">
                                        <div class="new-info-field">
                                            <label>Kennzeichen</label>
                                            <div class="new-info-value license">
                                                <input type="text" id="edit-license-${user.id}" value="${user.license_plate || ''}" class="new-edit-input" style="display: none;">
                                                <span id="display-license-${user.id}">${user.license_plate || 'Nicht angegeben'}</span>
                                            </div>
                                        </div>
                                        <div class="new-info-field">
                                            <label>Autohaus</label>
                                            <div class="new-info-value">
                                                <input type="text" id="edit-dealership-${user.id}" value="${user.dealership_name || ''}" class="new-edit-input" style="display: none;">
                                                <span id="display-dealership-${user.id}">${user.dealership_name || 'Nicht gewählt'}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Right Column -->
                        <div class="new-right-column">
                            <!-- Actions Card -->
                            <div class="new-card">
                                <div class="new-card-header">
                                    <i class="fas fa-bolt"></i>
                                    <h3>Schnellaktionen</h3>
                                </div>
                                <div class="new-card-content">
                                    <div class="new-actions-grid">
                                        ${user.status === 'pending' || user.status === 'rejected' ? `
                                            <button class="new-action-btn success" onclick="approveUser(${user.id})" id="approve-btn-${user.id}" disabled>
                                                <i class="fas fa-check-circle"></i>
                                                <span>Freischalten</span>
                                            </button>
                                            <button class="new-action-btn danger" onclick="rejectUser(${user.id})" id="reject-btn-${user.id}" disabled>
                                                <i class="fas fa-times-circle"></i>
                                                <span>Ablehnen</span>
                                            </button>
                                            <div class="new-action-info" id="action-info-${user.id}" style="grid-column: 1 / -1; text-align: center; padding: 10px; background: #fff3cd; border-radius: 6px; color: #856404; font-size: 12px;">
                                                <i class="fas fa-info-circle"></i>
                                                <span>Warten auf Dokumente...</span>
                                            </div>
                                        ` : `
                                            <button class="new-action-btn danger" onclick="suspendUser(${user.id})">
                                                <i class="fas fa-pause-circle"></i>
                                                <span>Sperren</span>
                                            </button>
                                        `}
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Documents Card -->
                            <div class="new-card">
                                <div class="new-card-header">
                                    <i class="fas fa-file-alt"></i>
                                    <h3>Verifikationsdokumente</h3>
                                </div>
                                <div class="new-card-content">
                                    ${user.verification_document ? `
                                        <div style="padding: 1rem; background: #f8f9fa; border-radius: 8px; margin-bottom: 1rem;">
                                            <div style="display: flex; align-items: center; gap: 1rem; margin-bottom: 1rem;">
                                                <div style="flex: 1;">
                                                    <div style="font-weight: 700; color: #495057; margin-bottom: 0.5rem;">Hochgeladenes Dokument</div>
                                                    <div style="color: #6c757d; font-size: 0.9rem;">
                                                        <strong>Status:</strong> 
                                                        <span class="status-badge status-${user.verification_status || 'pending'}" style="padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.85rem; display: inline-block; margin-left: 0.5rem;">
                                                            ${user.verification_status === 'pending' ? '<i class="fas fa-hourglass-half"></i> Wartend' : 
                                                              user.verification_status === 'approved' ? '<i class="fas fa-check-circle"></i> Freigeschaltet' : 
                                                              user.verification_status === 'rejected' ? '<i class="fas fa-times-circle"></i> Abgelehnt' : 
                                                              'Unbekannt'}
                                                        </span>
                                                    </div>
                                                    ${user.verification_reason ? `
                                                        <div style="margin-top: 0.5rem; color: #856404; font-size: 0.9rem;">
                                                            <strong>Ablehnungsgrund:</strong> ${user.verification_reason}
                                                        </div>
                                                    ` : ''}
                                                    ${user.verified_at ? `
                                                        <div style="margin-top: 0.5rem; color: #6c757d; font-size: 0.85rem;">
                                                            <i class="fas fa-calendar"></i> Verifiziert am: ${new Date(user.verified_at).toLocaleDateString('de-DE')}
                                                        </div>
                                                    ` : ''}
                                                </div>
                                            </div>
                                            <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                                <a href="${user.verification_document.startsWith('http') ? user.verification_document : '../' + user.verification_document}" 
                                                   target="_blank" 
                                                   class="new-action-btn success" 
                                                   style="background: #28a745; color: white; padding: 0.75rem 1.5rem; border-radius: 8px; text-decoration: none; display: inline-flex; align-items: center; gap: 0.5rem; font-weight: 600;">
                                                    <i class="fas fa-eye"></i>
                                                    <span>Dokument ansehen</span>
                                                </a>
                                                <button class="new-action-btn primary" onclick="approveUserFromModal(${user.id}, '${user.verification_document || ''}')" 
                                                        style="background: #28a745; color: white; padding: 0.75rem 1.5rem; border-radius: 8px; border: none; font-weight: 600; cursor: pointer;">
                                                    <i class="fas fa-check"></i>
                                                    <span>Freischalten</span>
                                                </button>
                                                <button class="new-action-btn danger" onclick="rejectUserFromModal(${user.id}, '${user.verification_document || ''}')" 
                                                        style="background: #dc3545; color: white; padding: 0.75rem 1.5rem; border-radius: 8px; border: none; font-weight: 600; cursor: pointer;">
                                                    <i class="fas fa-times"></i>
                                                    <span>Ablehnen</span>
                                                </button>
                                            </div>
                                        </div>
                                    ` : `
                                        <div style="padding: 2rem; text-align: center; color: #6c757d;">
                                            <i class="fas fa-file-alt" style="font-size: 3rem; margin-bottom: 1rem; color: #dee2e6;"></i>
                                            <div style="font-weight: 600; margin-bottom: 0.5rem;">Kein Dokument hochgeladen</div>
                                            <div style="font-size: 0.9rem;">Dieser Benutzer hat noch kein Verifikationsdokument hochgeladen.</div>
                                        </div>
                                    `}
                                </div>
                            </div>
                            
                            <!-- Notes Card -->
                            <div class="new-card">
                                <div class="new-card-header">
                                    <i class="fas fa-sticky-note"></i>
                                    <h3>Admin-Notizen</h3>
                                </div>
                                <div class="new-card-content">
                                    <textarea class="new-notes-textarea" id="new-admin-notes-${user.id}" placeholder="Admin-Notizen hier eingeben...">${user.admin_notes || ''}</textarea>
                                    <div style="margin-top: 15px; text-align: right;">
                                        <button class="new-save-btn" onclick="saveNewAdminNotes(${user.id})">
                                            <i class="fas fa-save"></i>
                                            <span>Speichern</span>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            modal.style.display = 'flex';
            
            // Documents are already loaded in the modal HTML
        }

        // Approve user
        function approveUser(userId) {
            document.getElementById('approve-user-id').value = userId;
            document.getElementById('approve-modal').style.display = 'block';
        }
        
        async function approveUserDirectly(userId) {
            try {
                const response = await fetch('../api/admin_approve_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        user_id: userId
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showNotification('Benutzer wurde erfolgreich freigeschaltet!', 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message || 'Fehler beim Freischalten', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        }

        // Reject user
        function rejectUser(userId) {
            // Check if user has documents before allowing rejection
            const rejectBtn = document.getElementById(`reject-btn-${userId}`);
            if (rejectBtn && rejectBtn.disabled) {
                showNotification('Benutzer kann erst abgelehnt werden, wenn Dokumente hochgeladen wurden', 'warning');
                return;
            }
            
            document.getElementById('reject-user-id').value = userId;
            document.getElementById('reject-modal').style.display = 'block';
        }

        // Store current action for email modal
        let currentEmailAction = {
            type: null, // 'approve' or 'reject'
            userId: null,
            verificationDocument: null,
            userData: null // Store user data here
        };

        // Approve user from modal (with document check and email preview)
        function approveUserFromModal(userId, verificationDocument) {
            if (!verificationDocument || verificationDocument === '' || verificationDocument === 'null') {
                showNotification('Benutzer hat noch kein Verifikationsdokument hochgeladen. Bitte warten Sie, bis der Benutzer ein Dokument hochgeladen hat.', 'error');
                return;
            }
            
            // Get user data from modal content
            const modalContent = document.getElementById('user-details');
            if (!modalContent) {
                showNotification('User-Daten nicht gefunden', 'error');
                return;
            }
            
            // Extract user data from modal HTML
            // Find email from display-email span
            const emailSpan = modalContent.querySelector(`#display-email-${userId}`);
            const userEmail = emailSpan ? emailSpan.textContent.trim() : '';
            
            // Find name from header h1
            const nameH1 = modalContent.querySelector('.new-user-details h1');
            const userName = nameH1 ? nameH1.textContent.trim() : 'Unknown User';
            const nameParts = userName.split(' ');
            const userData = {
                id: userId,
                email: userEmail,
                first_name: nameParts[0] || '',
                last_name: nameParts.slice(1).join(' ') || ''
            };
            
            currentEmailAction = {
                type: 'approve',
                userId: userId,
                verificationDocument: verificationDocument,
                userData: userData
            };
            
            openEmailPreviewModal(userData, 'approve');
        }

        // Reject user from modal (with document check and email preview)
        function rejectUserFromModal(userId, verificationDocument) {
            if (!verificationDocument || verificationDocument === '' || verificationDocument === 'null') {
                showNotification('Benutzer hat noch kein Verifikationsdokument hochgeladen. Bitte warten Sie, bis der Benutzer ein Dokument hochgeladen hat.', 'error');
                return;
            }
            
            // Get user data from modal content
            const modalContent = document.getElementById('user-details');
            if (!modalContent) {
                showNotification('User-Daten nicht gefunden', 'error');
                return;
            }
            
            // Extract user data from modal HTML
            // Find email from display-email span
            const emailSpan = modalContent.querySelector(`#display-email-${userId}`);
            const userEmail = emailSpan ? emailSpan.textContent.trim() : '';
            
            // Find name from header h1
            const nameH1 = modalContent.querySelector('.new-user-details h1');
            const userName = nameH1 ? nameH1.textContent.trim() : 'Unknown User';
            const nameParts = userName.split(' ');
            const userData = {
                id: userId,
                email: userEmail,
                first_name: nameParts[0] || '',
                last_name: nameParts.slice(1).join(' ') || ''
            };
            
            const reason = prompt('Bitte geben Sie einen Ablehnungsgrund an:');
            if (!reason || !reason.trim()) {
                return;
            }
            
            currentEmailAction = {
                type: 'reject',
                userId: userId,
                verificationDocument: verificationDocument,
                reason: reason.trim(),
                userData: userData
            };
            
            openEmailPreviewModal(userData, 'reject', reason.trim());
        }

        // Open email preview modal
        function openEmailPreviewModal(user, action, reason = '') {
            const modal = document.getElementById('email-send-modal');
            const preview = document.getElementById('email-preview');
            const resultDiv = document.getElementById('email-result');
            const sendBtn = document.getElementById('send-email-btn');
            
            resultDiv.style.display = 'none';
            resultDiv.innerHTML = '';
            sendBtn.disabled = false;
            
            if (action === 'approve') {
                preview.innerHTML = `
                    <h4 style="margin-top: 0; color: #28a745;">
                        <i class="fas fa-check-circle"></i> Freischaltungs-E-Mail
                    </h4>
                    <hr style="margin: 1rem 0;">
                    <p><strong>An:</strong> ${user.email}</p>
                    <p><strong>Betreff:</strong> Ihr Konto wurde freigeschaltet - Boxenstop</p>
                    <hr style="margin: 1rem 0;">
                    <div style="background: white; padding: 1rem; border-radius: 8px;">
                        <h5 style="color: #28a745; margin-top: 0;">✅ Ihr Konto wurde freigeschaltet!</h5>
                        <p><strong>Hallo ${user.first_name} ${user.last_name},</strong></p>
                        <p>Ihr Konto bei Boxenstop wurde erfolgreich verifiziert und freigeschaltet.</p>
                        <p><strong>Sie können jetzt:</strong></p>
                        <ul>
                            <li>✅ Termine buchen</li>
                            <li>✅ Fahrzeuge verwalten</li>
                            <li>✅ Alle Services nutzen</li>
                        </ul>
                    </div>
                `;
            } else {
                preview.innerHTML = `
                    <h4 style="margin-top: 0; color: #dc3545;">
                        <i class="fas fa-times-circle"></i> Ablehnungs-E-Mail
                    </h4>
                    <hr style="margin: 1rem 0;">
                    <p><strong>An:</strong> ${user.email}</p>
                    <p><strong>Betreff:</strong> Ihr Antrag wurde abgelehnt - Boxenstop</p>
                    <p><strong>Ablehnungsgrund:</strong> ${reason}</p>
                    <hr style="margin: 1rem 0;">
                    <div style="background: white; padding: 1rem; border-radius: 8px; border-left: 4px solid #ffc107;">
                        <h5 style="color: #dc3545; margin-top: 0;">❌ Ihr Antrag wurde abgelehnt</h5>
                        <p><strong>Hallo ${user.first_name} ${user.last_name},</strong></p>
                        <p>Ihr Verifikationsantrag wurde abgelehnt.</p>
                        <p><strong>Ablehnungsgrund:</strong> ${reason}</p>
                        <p><strong>Was können Sie tun?</strong></p>
                        <ul>
                            <li>📄 Überprüfen Sie Ihr Dokument</li>
                            <li>📤 Laden Sie ein neues Verifikationsdokument hoch</li>
                            <li>📞 Kontaktieren Sie uns bei Fragen</li>
                        </ul>
                    </div>
                `;
            }
            
            modal.style.display = 'flex';
        }

        // Close email send modal
        function closeEmailSendModal() {
            document.getElementById('email-send-modal').style.display = 'none';
        }

        // Send email and process action
        async function sendEmailConfirm() {
            const sendBtn = document.getElementById('send-email-btn');
            const resultDiv = document.getElementById('email-result');
            
            sendBtn.disabled = true;
            sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird gesendet...';
            
            try {
                if (currentEmailAction.type === 'approve') {
                    const response = await fetch('../api/admin_approve_user.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            user_id: currentEmailAction.userId
                        })
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        resultDiv.style.display = 'block';
                        resultDiv.style.background = '#d4edda';
                        resultDiv.style.color = '#155724';
                        resultDiv.style.border = '1px solid #c3e6cb';
                        resultDiv.innerHTML = '<i class="fas fa-check-circle"></i> E-Mail wurde erfolgreich gesendet und Benutzer wurde freigeschaltet!';
                        
                        setTimeout(() => {
                            closeEmailSendModal();
                            location.reload();
                        }, 2000);
                    } else {
                        throw new Error(data.message || 'Fehler beim Versenden');
                    }
                } else {
                    const response = await fetch('../api/admin_reject_user.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            user_id: currentEmailAction.userId,
                            reason: currentEmailAction.reason
                        })
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        resultDiv.style.display = 'block';
                        resultDiv.style.background = '#d4edda';
                        resultDiv.style.color = '#155724';
                        resultDiv.style.border = '1px solid #c3e6cb';
                        resultDiv.innerHTML = '<i class="fas fa-check-circle"></i> E-Mail wurde erfolgreich gesendet und Benutzer wurde abgelehnt!';
                        
                        setTimeout(() => {
                            closeEmailSendModal();
                            location.reload();
                        }, 2000);
                    } else {
                        throw new Error(data.message || 'Fehler beim Versenden');
                    }
                }
            } catch (error) {
                resultDiv.style.display = 'block';
                resultDiv.className = 'alert alert-error';
                resultDiv.style.background = '#f8d7da';
                resultDiv.style.color = '#721c24';
                resultDiv.style.border = '1px solid #f5c6cb';
                resultDiv.innerHTML = '<i class="fas fa-exclamation-circle"></i> Fehler: ' + error.message;
                
                sendBtn.disabled = false;
                sendBtn.innerHTML = '<i class="fas fa-paper-plane"></i> E-Mail senden';
            }
        }

        // Reject user with reason
        async function rejectUserWithReason(userId, reason) {
            try {
                const response = await fetch('../api/admin_reject_user.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        user_id: userId,
                        reason: reason
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showNotification('Benutzer wurde abgelehnt', 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message || 'Fehler beim Ablehnen', 'error');
                }
            } catch (error) {
                showNotification('Verbindungsfehler', 'error');
            }
        }

        // Bulk approve users
        function bulkApproveUsers() {
            const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
            if (checkedBoxes.length === 0) {
                alert('Bitte wählen Sie mindestens einen Benutzer aus.');
                return;
            }
            
            const userIds = Array.from(checkedBoxes).map(cb => cb.value);
            const modal = document.getElementById('bulk-modal');
            const content = document.getElementById('bulk-modal-content');
            
            content.innerHTML = `
                <form method="POST">
                    <input type="hidden" name="action" value="bulk_approve">
                    ${userIds.map(id => `<input type="hidden" name="user_ids[]" value="${id}">`).join('')}
                    <div class="form-group">
                        <label for="bulk_admin_notes">Admin-Notizen (optional)</label>
                        <textarea name="bulk_admin_notes" id="bulk_admin_notes" rows="3" placeholder="Notizen für alle ausgewählten Benutzer..."></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeBulkModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-success">${userIds.length} Benutzer freischalten</button>
                    </div>
                </form>
            `;
            
            modal.style.display = 'block';
        }

        // Bulk reject users
        function bulkRejectUsers() {
            const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
            if (checkedBoxes.length === 0) {
                alert('Bitte wählen Sie mindestens einen Benutzer aus.');
                return;
            }
            
            const userIds = Array.from(checkedBoxes).map(cb => cb.value);
            const modal = document.getElementById('bulk-modal');
            const content = document.getElementById('bulk-modal-content');
            
            content.innerHTML = `
                <form method="POST">
                    <input type="hidden" name="action" value="bulk_reject">
                    ${userIds.map(id => `<input type="hidden" name="user_ids[]" value="${id}">`).join('')}
                    <div class="form-group">
                        <label for="bulk_rejection_reason">Ablehnungsgrund *</label>
                        <textarea name="bulk_rejection_reason" id="bulk_rejection_reason" rows="3" placeholder="Grund für die Ablehnung..." required></textarea>
                    </div>
                    <div class="form-group">
                        <label for="bulk_admin_notes">Admin-Notizen (optional)</label>
                        <textarea name="bulk_admin_notes" id="bulk_admin_notes" rows="3" placeholder="Zusätzliche Notizen..."></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn btn-secondary" onclick="closeBulkModal()">Abbrechen</button>
                        <button type="submit" class="btn btn-danger">${userIds.length} Benutzer ablehnen</button>
                    </div>
                </form>
            `;
            
            modal.style.display = 'block';
        }

        // Export users
        function exportUsers() {
            window.location.href = 'admin_dashboard.php?action=export';
        }

        // Bulk export users
        function bulkExportUsers() {
            const checkedBoxes = document.querySelectorAll('.user-checkbox:checked');
            if (checkedBoxes.length === 0) {
                alert('Bitte wählen Sie mindestens einen Benutzer aus.');
                return;
            }
            
            const userIds = Array.from(checkedBoxes).map(cb => cb.value).join(',');
            window.location.href = `admin_dashboard.php?action=export&user_ids=${userIds}`;
        }

        // Close modals
        function closeUserModal() {
            document.getElementById('user-modal').style.display = 'none';
        }

        function closeApproveModal() {
            document.getElementById('approve-modal').style.display = 'none';
        }

        function closeRejectModal() {
            document.getElementById('reject-modal').style.display = 'none';
        }

        function closeBulkModal() {
            document.getElementById('bulk-modal').style.display = 'none';
        }

        // New verification modal functions
        function closeDocumentVerificationModal() {
            document.getElementById('document-verification-modal').style.display = 'none';
        }

        function closeDocumentApprovalModal() {
            document.getElementById('document-approval-modal').style.display = 'none';
        }

        function closeDocumentRejectionModal() {
            document.getElementById('document-rejection-modal').style.display = 'none';
        }

        // Update verification buttons based on checkbox
        function updateVerificationButtons(userId) {
            const checkbox = document.getElementById(`document-reviewed-${userId}`);
            const approveBtn = document.getElementById(`approve-btn-${userId}`);
            const rejectBtn = document.getElementById(`reject-btn-${userId}`);
            
            if (checkbox && approveBtn && rejectBtn) {
                const isChecked = checkbox.checked;
                approveBtn.disabled = !isChecked;
                rejectBtn.disabled = !isChecked;
                
                if (isChecked) {
                    approveBtn.classList.remove('disabled');
                    rejectBtn.classList.remove('disabled');
                } else {
                    approveBtn.classList.add('disabled');
                    rejectBtn.classList.add('disabled');
                }
            }
        }

        // Open document approval modal
        function openDocumentApprovalModal(userId) {
            const checkbox = document.getElementById(`document-reviewed-${userId}`);
            if (!checkbox || !checkbox.checked) {
                alert('Bitte bestätigen Sie zuerst, dass Sie das Dokument geprüft haben.');
                return;
            }
            
            // Get current document info
            if (currentReviewDocuments[userId]) {
                document.getElementById('approval-document-id').value = currentReviewDocuments[userId].id;
                document.getElementById('approval-user-id').value = userId;
                
                // Pre-fill notes if available
                const notes = document.getElementById(`review-notes-${userId}`).value;
                if (notes) {
                    document.getElementById('approval-notes').value = notes;
                }
                
                document.getElementById('document-approval-modal').style.display = 'block';
            }
        }

        // Open document rejection modal
        function openDocumentRejectionModal(userId) {
            const checkbox = document.getElementById(`document-reviewed-${userId}`);
            if (!checkbox || !checkbox.checked) {
                alert('Bitte bestätigen Sie zuerst, dass Sie das Dokument geprüft haben.');
                return;
            }
            
            // Get current document info
            if (currentReviewDocuments[userId]) {
                document.getElementById('rejection-document-id').value = currentReviewDocuments[userId].id;
                document.getElementById('rejection-user-id').value = userId;
                
                // Pre-fill notes if available
                const notes = document.getElementById(`review-notes-${userId}`).value;
                if (notes) {
                    document.getElementById('rejection-reason').value = notes;
                }
                
                document.getElementById('document-rejection-modal').style.display = 'block';
            }
        }

        // Close modals when clicking outside
        window.onclick = function(event) {
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                if (event.target === modal) {
                    modal.style.display = 'none';
                }
            });
        }

        // Helper functions
        function formatDate(dateString) {
            if (!dateString) return 'Nicht angegeben';
            const date = new Date(dateString);
            return date.toLocaleDateString('de-DE', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        }

        // Get user status icon
        function getUserStatusIcon(status) {
            const icons = {
                'pending': 'fa-clock',
                'approved': 'fa-check-circle',
                'rejected': 'fa-times-circle',
                'suspended': 'fa-ban'
            };
            return icons[status] || 'fa-question';
        }

        // Get user status label
        function getUserStatusLabel(status) {
            const labels = {
                'pending': 'Ausstehend',
                'approved': 'Freigeschaltet',
                'rejected': 'Abgelehnt',
                'suspended': 'Gesperrt'
            };
            return labels[status] || status;
        }

        // Save admin notes
        async function saveAdminNotes(userId) {
            const notes = document.getElementById(`admin-notes-${userId}`).value;
            
            try {
                const response = await fetch('api/save_admin_notes.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        user_id: userId,
                        notes: notes
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showNotification('Notizen erfolgreich gespeichert', 'success');
                } else {
                    showNotification('Fehler beim Speichern der Notizen', 'error');
                }
            } catch (error) {
                console.error('Error saving notes:', error);
                showNotification('Fehler beim Speichern der Notizen', 'error');
            }
        }

        // Neue Verifikations-Funktionen
        function closeDocumentVerificationModal() {
            document.getElementById('document-verification-modal').style.display = 'none';
        }

        function closeDocumentApprovalModal() {
            document.getElementById('document-approval-modal').style.display = 'none';
        }

        function closeDocumentRejectionModal() {
            document.getElementById('document-rejection-modal').style.display = 'none';
        }

        function updateVerificationButtons(userId) {
            const checkbox = document.getElementById(`document-reviewed-${userId}`);
            const approveBtn = document.getElementById(`approve-btn-${userId}`);
            const rejectBtn = document.getElementById(`reject-btn-${userId}`);
            
            if (checkbox && approveBtn && rejectBtn) {
                const isChecked = checkbox.checked;
                approveBtn.disabled = !isChecked;
                rejectBtn.disabled = !isChecked;
                
                if (isChecked) {
                    approveBtn.classList.remove('disabled');
                    rejectBtn.classList.remove('disabled');
                } else {
                    approveBtn.classList.add('disabled');
                    rejectBtn.classList.add('disabled');
                }
            }
        }

        function openDocumentApprovalModal(userId) {
            const checkbox = document.getElementById(`document-reviewed-${userId}`);
            if (!checkbox || !checkbox.checked) {
                alert('Bitte bestätigen Sie zuerst, dass Sie das Dokument geprüft haben.');
                return;
            }
            
            const notes = document.getElementById(`review-notes-${userId}`).value;
            document.getElementById('approval-notes').value = notes;
            document.getElementById('document-approval-modal').style.display = 'block';
        }

        function openDocumentRejectionModal(userId) {
            const checkbox = document.getElementById(`document-reviewed-${userId}`);
            if (!checkbox || !checkbox.checked) {
                alert('Bitte bestätigen Sie zuerst, dass Sie das Dokument geprüft haben.');
                return;
            }
            
            const notes = document.getElementById(`review-notes-${userId}`).value;
            document.getElementById('rejection-notes').value = notes;
            document.getElementById('document-rejection-modal').style.display = 'block';
        }

        function closeDocumentReview(userId) {
            const reviewSection = document.getElementById(`document-review-section-${userId}`);
            if (reviewSection) {
                reviewSection.style.display = 'none';
            }
        }

        function downloadCurrentDocument(userId) {
            const currentDoc = window.currentDocument;
            if (currentDoc) {
                downloadDocument(`api/view_document.php?file=${encodeURIComponent(currentDoc.filePath)}`, currentDoc.fileName);
            }
        }

        function openDocumentInNewTab(userId) {
            const currentDoc = window.currentDocument;
            if (currentDoc) {
                window.open(`api/view_document.php?file=${encodeURIComponent(currentDoc.filePath)}`, '_blank');
            }
        }

        // NEUE HILFSFUNKTIONEN FÜR DAS NEUE MODAL
        function closeNewUserModal() {
            document.getElementById('user-modal').style.display = 'none';
        }

        function loadNewUserDocuments(userId) {
            // Diese Funktion wird nicht mehr benötigt, 
            // da Dokumente direkt im Modal HTML angezeigt werden
            console.log('Documents are displayed inline in modal for user:', userId);
            return;
            
            try {
                container.innerHTML = `
                    <div class="new-documents-loading">
                        <i class="fas fa-spinner fa-spin"></i>
                        <span>Dokumente werden geladen...</span>
                    </div>
                `;
                
                fetch(`api/get_user_documents.php?user_id=${userId}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success && data.documents && data.documents.length > 0) {
                            const documentsHtml = data.documents.map(doc => `
                                <div style="background: #f8f9fa; border-radius: 10px; padding: 15px; margin-bottom: 15px; border: 1px solid #e9ecef;">
                                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                                        <div>
                                            <strong>${doc.file_name}</strong>
                                            <div style="font-size: 12px; color: #6c757d; margin-top: 5px;">
                                                ${getDocumentTypeLabel(doc.document_type)} • ${formatFileSize(doc.file_size)} • ${formatDate(doc.upload_date)}
                                            </div>
                                        </div>
                                        <div style="display: flex; gap: 8px;">
                                            <button onclick="viewDocument('${doc.file_path}', '${doc.mime_type}')" style="background: #007bff; color: white; border: none; padding: 8px 12px; border-radius: 6px; cursor: pointer;" title="Anzeigen">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                            <button onclick="downloadDocument('api/view_document.php?file=${encodeURIComponent(doc.file_path)}', '${doc.file_name}')" style="background: #28a745; color: white; border: none; padding: 8px 12px; border-radius: 6px; cursor: pointer;" title="Download">
                                                <i class="fas fa-download"></i>
                                            </button>
                                        </div>
                                    </div>
                                    ${doc.status ? `
                                        <div style="padding: 8px; background: ${doc.status === 'approved' ? 'rgba(40, 167, 69, 0.1)' : 'rgba(220, 53, 69, 0.1)'}; border-radius: 6px; font-size: 12px;">
                                            <i class="fas ${getStatusIcon(doc.status)}"></i>
                                            ${getStatusLabel(doc.status)}
                                            ${doc.reviewed_by_name ? ` • Von: ${doc.reviewed_by_name}` : ''}
                                            ${doc.admin_notes ? `<br><strong>Notizen:</strong> ${doc.admin_notes}` : ''}
                                        </div>
                                    ` : ''}
                                </div>
                            `).join('');
                            
                            container.innerHTML = documentsHtml;
                            
                            // Enable action buttons since documents exist
                            enableActionButtons(userId);
                        } else {
                            container.innerHTML = `
                                <div style="text-align: center; padding: 40px; color: #6c757d;">
                                    <i class="fas fa-file-alt" style="font-size: 48px; margin-bottom: 15px; opacity: 0.5;"></i>
                                    <h4 style="margin: 0 0 10px 0;">Keine Dokumente vorhanden</h4>
                                    <p style="margin: 0;">Dieser Benutzer hat noch keine Verifikationsdokumente hochgeladen.</p>
                                </div>
                            `;
                            
                            // Keep action buttons disabled since no documents exist
                            disableActionButtons(userId);
                        }
                    })
                    .catch(error => {
                        console.error('Error loading documents:', error);
                        container.innerHTML = `
                            <div style="text-align: center; padding: 40px; color: #dc3545;">
                                <i class="fas fa-exclamation-triangle" style="font-size: 48px; margin-bottom: 15px;"></i>
                                <h4 style="margin: 0 0 10px 0;">Fehler beim Laden der Dokumente</h4>
                                <p style="margin: 0 0 15px 0;">Die Dokumente konnten nicht geladen werden.</p>
                                <button onclick="loadNewUserDocuments(${userId})" style="background: #007bff; color: white; border: none; padding: 10px 20px; border-radius: 6px; cursor: pointer;">
                                    <i class="fas fa-retry"></i> Erneut versuchen
                                </button>
                            </div>
                        `;
                    });
            } catch (error) {
                console.error('Error loading documents:', error);
            }
        }

        function saveNewAdminNotes(userId) {
            const notes = document.getElementById(`new-admin-notes-${userId}`).value;
            
            fetch('api/save_admin_notes.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    user_id: userId,
                    notes: notes
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Notizen erfolgreich gespeichert', 'success');
                } else {
                    showNotification('Fehler beim Speichern der Notizen', 'error');
                }
            })
            .catch(error => {
                console.error('Error saving notes:', error);
                showNotification('Fehler beim Speichern der Notizen', 'error');
            });
        }

        // Edit Functions
        function toggleEditPersonal(userId) {
            const editBtn = document.getElementById(`edit-personal-btn-${userId}`);
            const saveBtn = document.getElementById(`save-personal-btn-${userId}`);
            const cancelBtn = document.getElementById(`cancel-personal-btn-${userId}`);
            
            // Hide display spans and show inputs
            document.getElementById(`display-first-name-${userId}`).style.display = 'none';
            document.getElementById(`display-last-name-${userId}`).style.display = 'none';
            document.getElementById(`display-email-${userId}`).style.display = 'none';
            document.getElementById(`display-phone-${userId}`).style.display = 'none';
            
            document.getElementById(`edit-first-name-${userId}`).style.display = 'block';
            document.getElementById(`edit-last-name-${userId}`).style.display = 'block';
            document.getElementById(`edit-email-${userId}`).style.display = 'block';
            document.getElementById(`edit-phone-${userId}`).style.display = 'block';
            
            // Show save/cancel buttons
            editBtn.style.display = 'none';
            saveBtn.style.display = 'flex';
            cancelBtn.style.display = 'flex';
        }

        function cancelEditPersonal(userId) {
            const editBtn = document.getElementById(`edit-personal-btn-${userId}`);
            const saveBtn = document.getElementById(`save-personal-btn-${userId}`);
            const cancelBtn = document.getElementById(`cancel-personal-btn-${userId}`);
            
            // Show display spans and hide inputs
            document.getElementById(`display-first-name-${userId}`).style.display = 'block';
            document.getElementById(`display-last-name-${userId}`).style.display = 'block';
            document.getElementById(`display-email-${userId}`).style.display = 'block';
            document.getElementById(`display-phone-${userId}`).style.display = 'block';
            
            document.getElementById(`edit-first-name-${userId}`).style.display = 'none';
            document.getElementById(`edit-last-name-${userId}`).style.display = 'none';
            document.getElementById(`edit-email-${userId}`).style.display = 'none';
            document.getElementById(`edit-phone-${userId}`).style.display = 'none';
            
            // Show edit button
            editBtn.style.display = 'flex';
            saveBtn.style.display = 'none';
            cancelBtn.style.display = 'none';
        }

        function savePersonalData(userId) {
            const data = {
                user_id: userId,
                first_name: document.getElementById(`edit-first-name-${userId}`).value,
                last_name: document.getElementById(`edit-last-name-${userId}`).value,
                email: document.getElementById(`edit-email-${userId}`).value,
                phone_number: document.getElementById(`edit-phone-${userId}`).value
            };
            
            fetch('api/update_user_data.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update display values
                    document.getElementById(`display-first-name-${userId}`).textContent = data.first_name || 'Nicht angegeben';
                    document.getElementById(`display-last-name-${userId}`).textContent = data.last_name || 'Nicht angegeben';
                    document.getElementById(`display-email-${userId}`).textContent = data.email;
                    document.getElementById(`display-phone-${userId}`).textContent = data.phone_number || 'Nicht angegeben';
                    
                    cancelEditPersonal(userId);
                    showNotification('Persönliche Daten erfolgreich aktualisiert', 'success');
                } else {
                    showNotification('Fehler beim Aktualisieren der Daten', 'error');
                }
            })
            .catch(error => {
                console.error('Error updating personal data:', error);
                showNotification('Fehler beim Aktualisieren der Daten', 'error');
            });
        }

        function toggleEditVehicle(userId) {
            const editBtn = document.getElementById(`edit-vehicle-btn-${userId}`);
            const saveBtn = document.getElementById(`save-vehicle-btn-${userId}`);
            const cancelBtn = document.getElementById(`cancel-vehicle-btn-${userId}`);
            
            // Hide display spans and show inputs
            document.getElementById(`display-license-${userId}`).style.display = 'none';
            document.getElementById(`display-dealership-${userId}`).style.display = 'none';
            
            document.getElementById(`edit-license-${userId}`).style.display = 'block';
            document.getElementById(`edit-dealership-${userId}`).style.display = 'block';
            
            // Show save/cancel buttons
            editBtn.style.display = 'none';
            saveBtn.style.display = 'flex';
            cancelBtn.style.display = 'flex';
        }

        function cancelEditVehicle(userId) {
            const editBtn = document.getElementById(`edit-vehicle-btn-${userId}`);
            const saveBtn = document.getElementById(`save-vehicle-btn-${userId}`);
            const cancelBtn = document.getElementById(`cancel-vehicle-btn-${userId}`);
            
            // Show display spans and hide inputs
            document.getElementById(`display-license-${userId}`).style.display = 'block';
            document.getElementById(`display-dealership-${userId}`).style.display = 'block';
            
            document.getElementById(`edit-license-${userId}`).style.display = 'none';
            document.getElementById(`edit-dealership-${userId}`).style.display = 'none';
            
            // Show edit button
            editBtn.style.display = 'flex';
            saveBtn.style.display = 'none';
            cancelBtn.style.display = 'none';
        }

        function saveVehicleData(userId) {
            const data = {
                user_id: userId,
                license_plate: document.getElementById(`edit-license-${userId}`).value,
                dealership_name: document.getElementById(`edit-dealership-${userId}`).value
            };
            
            fetch('api/update_user_data.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update display values
                    document.getElementById(`display-license-${userId}`).textContent = data.license_plate || 'Nicht angegeben';
                    document.getElementById(`display-dealership-${userId}`).textContent = data.dealership_name || 'Nicht gewählt';
                    
                    cancelEditVehicle(userId);
                    showNotification('Fahrzeugdaten erfolgreich aktualisiert', 'success');
                } else {
                    showNotification('Fehler beim Aktualisieren der Fahrzeugdaten', 'error');
                }
            })
            .catch(error => {
                console.error('Error updating vehicle data:', error);
                showNotification('Fehler beim Aktualisieren der Fahrzeugdaten', 'error');
            });
        }

        // Enable/Disable Action Buttons based on document availability
        function enableActionButtons(userId) {
            const approveBtn = document.getElementById(`approve-btn-${userId}`);
            const rejectBtn = document.getElementById(`reject-btn-${userId}`);
            const actionInfo = document.getElementById(`action-info-${userId}`);
            
            if (approveBtn) {
                approveBtn.disabled = false;
                approveBtn.style.opacity = '1';
                approveBtn.style.cursor = 'pointer';
            }
            
            if (rejectBtn) {
                rejectBtn.disabled = false;
                rejectBtn.style.opacity = '1';
                rejectBtn.style.cursor = 'pointer';
            }
            
            if (actionInfo) {
                actionInfo.innerHTML = `
                    <i class="fas fa-check-circle"></i>
                    <span>Dokumente vorhanden - Aktionen verfügbar</span>
                `;
                actionInfo.style.background = 'rgba(40, 167, 69, 0.1)';
                actionInfo.style.color = '#28a745';
            }
        }

        function disableActionButtons(userId) {
            const approveBtn = document.getElementById(`approve-btn-${userId}`);
            const rejectBtn = document.getElementById(`reject-btn-${userId}`);
            const actionInfo = document.getElementById(`action-info-${userId}`);
            
            if (approveBtn) {
                approveBtn.disabled = true;
                approveBtn.style.opacity = '0.5';
                approveBtn.style.cursor = 'not-allowed';
            }
            
            if (rejectBtn) {
                rejectBtn.disabled = true;
                rejectBtn.style.opacity = '0.5';
                rejectBtn.style.cursor = 'not-allowed';
            }
            
            if (actionInfo) {
                actionInfo.innerHTML = `
                    <i class="fas fa-info-circle"></i>
                    <span>Warten auf Dokumente...</span>
                `;
                actionInfo.style.background = '#fff3cd';
                actionInfo.style.color = '#856404';
            }
        }

        // Suspend User Function
        function suspendUser(userId) {
            // Store user ID for the form
            window.suspendUserId = userId;
            
            // Show the modal
            document.getElementById('suspend-user-modal').style.display = 'flex';
            
            // Clear previous reason
            document.getElementById('suspend-reason').value = '';
            
            // Focus on reason textarea
            setTimeout(() => {
                document.getElementById('suspend-reason').focus();
            }, 100);
        }

        // Close Suspend Modal
        function closeSuspendUserModal() {
            document.getElementById('suspend-user-modal').style.display = 'none';
            window.suspendUserId = null;
        }

        // Handle Suspend Form Submission
        document.addEventListener('DOMContentLoaded', function() {
            const suspendForm = document.getElementById('suspend-form');
            if (suspendForm) {
                suspendForm.addEventListener('submit', function(e) {
                    e.preventDefault();
                    
                    if (!window.suspendUserId) {
                        showNotification('Fehler: Benutzer-ID nicht gefunden', 'error');
                        return;
                    }
                    
                    const reason = document.getElementById('suspend-reason').value;
                    
                    const data = {
                        user_id: window.suspendUserId,
                        action: 'suspend',
                        reason: reason
                    };
                    
                    fetch('api/suspend_user.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        credentials: 'same-origin', // Include cookies/session
                        body: JSON.stringify(data)
                    })
                    .then(response => {
                        console.log('Response status:', response.status);
                        console.log('Response headers:', response.headers);
                        
                        if (!response.ok) {
                            throw new Error(`HTTP error! status: ${response.status}`);
                        }
                        
                        return response.text(); // Get as text first
                    })
                    .then(text => {
                        console.log('Raw response:', text);
                        
                        try {
                            const data = JSON.parse(text);
                            if (data.success) {
                                showNotification('Benutzer erfolgreich gesperrt', 'success');
                                closeSuspendUserModal();
                                // Refresh the page to update the status
                                setTimeout(() => {
                                    window.location.reload();
                                }, 1500);
                            } else {
                                showNotification('Fehler beim Sperren des Benutzers: ' + (data.message || 'Unbekannter Fehler'), 'error');
                            }
                        } catch (parseError) {
                            console.error('JSON parse error:', parseError);
                            console.error('Response text:', text);
                            showNotification('Fehler beim Verarbeiten der Antwort vom Server', 'error');
                        }
                    })
                    .catch(error => {
                        console.error('Error suspending user:', error);
                        showNotification('Fehler beim Sperren des Benutzers: ' + error.message, 'error');
                    });
                });
            }
        });

        // Review document function
        function reviewDocument(userId, docId, fileName, filePath, mimeType, docType, fileSize, uploadDate, status) {
            // Store current document info globally
            window.currentDocument = {
                id: docId,
                fileName: fileName,
                filePath: filePath,
                mimeType: mimeType,
                docType: docType,
                fileSize: fileSize,
                uploadDate: uploadDate,
                status: status
            };

            // Show the document review section
            const reviewSection = document.getElementById(`document-review-section-${userId}`);
            if (reviewSection) {
                reviewSection.style.display = 'block';
                
                // Update document info
                document.getElementById(`review-document-name-${userId}`).textContent = fileName;
                document.getElementById(`review-document-type-${userId}`).textContent = getDocumentTypeLabel(docType);
                document.getElementById(`review-document-size-${userId}`).textContent = formatFileSize(fileSize);
                document.getElementById(`review-document-date-${userId}`).textContent = formatDate(uploadDate);
                
                // Load document content
                loadDocumentContent(userId, filePath, mimeType);
                
                // Reset checkbox and buttons
                const checkbox = document.getElementById(`document-reviewed-${userId}`);
                if (checkbox) {
                    checkbox.checked = false;
                    updateVerificationButtons(userId);
                }
                
                // Clear notes
                const notesField = document.getElementById(`review-notes-${userId}`);
                if (notesField) {
                    notesField.value = '';
                }
            }
        }

        // Load document content in viewer
        function loadDocumentContent(userId, filePath, mimeType) {
            const viewerContent = document.getElementById(`document-viewer-content-${userId}`);
            if (!viewerContent) return;

            viewerContent.innerHTML = `
                <div class="viewer-loading-large">
                    <i class="fas fa-spinner fa-spin"></i>
                    <span>Dokument wird geladen...</span>
                </div>
            `;

            // Determine how to display the document
            if (mimeType.startsWith('image/')) {
                // Display image directly
                viewerContent.innerHTML = `
                    <img src="api/view_document.php?file=${encodeURIComponent(filePath)}" 
                         alt="Dokument" 
                         style="max-width: 100%; max-height: 100%; object-fit: contain;">
                `;
            } else if (mimeType === 'application/pdf') {
                // Display PDF in iframe
                viewerContent.innerHTML = `
                    <iframe src="api/view_document.php?file=${encodeURIComponent(filePath)}" 
                            width="100%" 
                            height="100%" 
                            style="border: none;">
                    </iframe>
                `;
            } else if (mimeType === 'text/plain') {
                // Display text content
                fetch(`api/view_document.php?file=${encodeURIComponent(filePath)}`)
                    .then(response => response.text())
                    .then(text => {
                        viewerContent.innerHTML = `
                            <div style="padding: 20px; white-space: pre-wrap; font-family: monospace; background: #f8f9fa; border-radius: 8px; max-height: 100%; overflow-y: auto;">
                                ${text}
                            </div>
                        `;
                    })
                    .catch(error => {
                        viewerContent.innerHTML = `
                            <div style="padding: 20px; text-align: center; color: #dc3545;">
                                <i class="fas fa-exclamation-triangle"></i>
                                <p>Fehler beim Laden des Dokuments</p>
                            </div>
                        `;
                    });
            } else {
                // Unsupported file type
                viewerContent.innerHTML = `
                    <div style="padding: 20px; text-align: center; color: #6c757d;">
                        <i class="fas fa-file"></i>
                        <p>Dateityp wird nicht unterstützt</p>
                        <p><small>Bitte laden Sie das Dokument herunter, um es anzuzeigen</small></p>
                    </div>
                `;
            }
        }

        // Auto-refresh functionality
        let autoRefreshInterval;
        let isRefreshing = false;

        function startAutoRefresh() {
            // Refresh every 30 seconds
            autoRefreshInterval = setInterval(refreshUserData, 30000);
        }

        function stopAutoRefresh() {
            if (autoRefreshInterval) {
                clearInterval(autoRefreshInterval);
            }
        }

        async function refreshUserData() {
            if (isRefreshing) return;
            
            isRefreshing = true;
            const indicator = document.getElementById('auto-refresh-indicator');
            
            try {
                // Show refresh indicator
                indicator.classList.add('show');
                indicator.innerHTML = '<i class="fas fa-sync-alt fa-spin"></i><span>Daten werden aktualisiert...</span>';
                
                // Get current filter
                const statusFilter = document.getElementById('status-filter').value;
                const url = statusFilter ? `admin_dashboard.php?status=${statusFilter}&refresh=1` : 'admin_dashboard.php?refresh=1';
                
                // Fetch updated data
                const response = await fetch(url);
                if (!response.ok) throw new Error('Network response was not ok');
                
                const html = await response.text();
                const parser = new DOMParser();
                const doc = parser.parseFromString(html, 'text/html');
                
                // Update the table content
                const newTable = doc.querySelector('.modern-table');
                const currentTable = document.querySelector('.modern-table');
                
                if (newTable && currentTable) {
                    currentTable.innerHTML = newTable.innerHTML;
                    
                    // Reattach event listeners
                    attachEventListeners();
                    
                    // Update statistics
                    updateStatistics(doc);
                    
                    // Show success indicator
                    indicator.classList.add('success');
                    indicator.innerHTML = '<i class="fas fa-check"></i><span>Daten aktualisiert</span>';
                    
                    setTimeout(() => {
                        indicator.classList.remove('show', 'success');
                    }, 2000);
                }
                
            } catch (error) {
                console.error('Auto-refresh error:', error);
                
                // Show error indicator
                indicator.classList.add('error');
                indicator.innerHTML = '<i class="fas fa-exclamation-triangle"></i><span>Fehler beim Aktualisieren</span>';
                
                setTimeout(() => {
                    indicator.classList.remove('show', 'error');
                }, 3000);
            } finally {
                isRefreshing = false;
            }
        }

        function updateStatistics(doc) {
            // Update header statistics
            const newStatCards = doc.querySelectorAll('.stat-card');
            const currentStatCards = document.querySelectorAll('.stat-card');
            
            if (newStatCards.length === currentStatCards.length) {
                newStatCards.forEach((newCard, index) => {
                    if (currentStatCards[index]) {
                        currentStatCards[index].innerHTML = newCard.innerHTML;
                    }
                });
            }
            
            // Update card subtitle
            const newCardSubtitle = doc.querySelector('.card-subtitle');
            const currentCardSubtitle = document.querySelector('.card-subtitle');
            
            if (newCardSubtitle && currentCardSubtitle) {
                currentCardSubtitle.innerHTML = newCardSubtitle.innerHTML;
            }
        }

        function attachEventListeners() {
            // Reattach checkbox event listeners
            const checkboxes = document.querySelectorAll('.user-checkbox');
            checkboxes.forEach(checkbox => {
                checkbox.addEventListener('change', updateBulkActions);
            });
            
            // Reattach select all checkbox
            const selectAllCheckbox = document.getElementById('select-all-checkbox');
            if (selectAllCheckbox) {
                selectAllCheckbox.addEventListener('change', toggleAllUsers);
            }
        }

        // Initialize auto-refresh when page loads
        document.addEventListener('DOMContentLoaded', function() {
            startAutoRefresh();
            
            // Stop auto-refresh when user is interacting with modals
            const modals = document.querySelectorAll('.modal');
            modals.forEach(modal => {
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        stopAutoRefresh();
                        setTimeout(startAutoRefresh, 5000); // Restart after 5 seconds
                    }
                });
            });
            
            // Stop auto-refresh when user is typing in forms
            const textareas = document.querySelectorAll('textarea');
            textareas.forEach(textarea => {
                textarea.addEventListener('focus', stopAutoRefresh);
                textarea.addEventListener('blur', function() {
                    setTimeout(startAutoRefresh, 2000); // Restart after 2 seconds
                });
            });
        });

        // Manual refresh function
        function manualRefresh() {
            refreshUserData();
        }

        // Load user documents - Updated for new workflow
        async function loadUserDocuments(userId) {
            const container = document.getElementById(`documents-container-${userId}`);
            if (!container) {
                console.error('Container nicht gefunden für User ID:', userId);
                return;
            }
            
            try {
                container.innerHTML = `
                    <div class="documents-loading-large">
                        <i class="fas fa-spinner fa-spin"></i>
                        <span>Dokumente werden geladen...</span>
                    </div>
                `;
                
                const response = await fetch(`api/get_user_documents.php?user_id=${userId}`);
                const data = await response.json();
                
                if (data.success && data.documents && data.documents.length > 0) {
                    const documentsHtml = data.documents.map(doc => `
                        <div class="document-card" data-doc-id="${doc.id}">
                            <div class="document-preview">
                                ${getDocumentPreview(doc)}
                            </div>
                            <div class="document-info">
                                <div class="document-name">${doc.file_name}</div>
                                <div class="document-meta">
                                    <span class="document-type">${getDocumentTypeLabel(doc.document_type)}</span>
                                    <span class="document-size">${formatFileSize(doc.file_size)}</span>
                                </div>
                                <div class="document-date">${formatDate(doc.upload_date)}</div>
                                ${doc.status ? `
                                    <div class="document-status status-${doc.status}">
                                        <i class="fas ${getStatusIcon(doc.status)}"></i>
                                        <span>${getStatusLabel(doc.status)}</span>
                                    </div>
                                    ${doc.reviewed_by_name ? `
                                        <div class="document-review-info">
                                            <small>
                                                ${doc.status === 'approved' ? 'Genehmigt' : 'Abgelehnt'} von: ${doc.reviewed_by_name}<br>
                                                ${doc.reviewed_at ? formatDate(doc.reviewed_at) : ''}
                                            </small>
                                        </div>
                                    ` : ''}
                                    ${doc.admin_notes ? `
                                        <div class="document-review-notes">
                                            <small><strong>Notizen:</strong> ${doc.admin_notes}</small>
                                        </div>
                                    ` : ''}
                                ` : ''}
                            </div>
                            <div class="document-actions">
                                <button class="action-btn btn-view" onclick="viewDocument('${doc.file_path}', '${doc.mime_type}')" title="Dokument anzeigen">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="action-btn btn-download" onclick="downloadDocument('api/view_document.php?file=${encodeURIComponent(doc.file_path)}', '${doc.file_name}')" title="Dokument herunterladen">
                                    <i class="fas fa-download"></i>
                                </button>
                                <button class="action-btn btn-review" onclick="reviewDocument(${userId}, ${doc.id}, '${doc.file_name}', '${doc.file_path}', '${doc.mime_type}', '${doc.document_type}', ${doc.file_size}, '${doc.upload_date}', '${doc.status || 'pending'}')" title="Dokument prüfen">
                                    <i class="fas fa-search"></i>
                                </button>
                            </div>
                        </div>
                    `).join('');
                    
                    container.innerHTML = `
                        <div class="documents-grid">
                            ${documentsHtml}
                        </div>
                    `;
                } else {
                    console.log('Keine Dokumente gefunden oder API-Fehler');
                    container.innerHTML = `
                        <div class="documents-empty">
                            <i class="fas fa-file-alt"></i>
                            <h4>Keine Dokumente vorhanden</h4>
                            <p>Dieser Benutzer hat noch keine Verifikationsdokumente hochgeladen.</p>
                            <p><small>Debug: API Response - ${JSON.stringify(data)}</small></p>
                        </div>
                    `;
                }
            } catch (error) {
                console.error('Error loading documents:', error);
                container.innerHTML = `
                    <div class="documents-error">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h4>Fehler beim Laden der Dokumente</h4>
                        <p>Die Dokumente konnten nicht geladen werden. Bitte versuchen Sie es erneut.</p>
                        <button class="btn btn-sm btn-primary" onclick="loadUserDocuments(${userId})">
                            <i class="fas fa-retry"></i>
                            <span>Erneut versuchen</span>
                        </button>
                    </div>
                `;
            }
        }

        // Get document preview HTML
        function getDocumentPreview(doc) {
            if (doc.mime_type.startsWith('image/')) {
                return `<img src="api/view_document.php?file=${encodeURIComponent(doc.file_path)}" alt="${doc.file_name}" class="document-image">`;
            } else if (doc.mime_type === 'application/pdf') {
                return `<div class="document-pdf-icon"><i class="fas fa-file-pdf"></i></div>`;
            } else {
                return `<div class="document-file-icon"><i class="fas fa-file"></i></div>`;
            }
        }

        // Get document type label
        function getDocumentTypeLabel(type) {
            const labels = {
                'invoice': 'Rechnung',
                'id_card': 'Ausweis',
                'license': 'Führerschein',
                'registration': 'Fahrzeugbrief',
                'insurance': 'Versicherung',
                'other': 'Sonstiges'
            };
            return labels[type] || type;
        }

        // Get status icon
        function getStatusIcon(status) {
            const icons = {
                'pending': 'fa-clock',
                'approved': 'fa-check-circle',
                'rejected': 'fa-times-circle'
            };
            return icons[status] || 'fa-question';
        }

        // Get status label
        function getStatusLabel(status) {
            const labels = {
                'pending': 'Ausstehend',
                'approved': 'Genehmigt',
                'rejected': 'Abgelehnt'
            };
            return labels[status] || status;
        }

        // Format file size
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        // View document
        function viewDocument(filePath, mimeType) {
            const secureUrl = `api/view_document.php?file=${encodeURIComponent(filePath)}`;
            
            if (mimeType.startsWith('image/')) {
                // Open image in new tab
                window.open(secureUrl, '_blank');
            } else if (mimeType === 'application/pdf') {
                // Open PDF in new tab
                window.open(secureUrl, '_blank');
            } else {
                // Download for other file types
                downloadDocument(secureUrl, filePath.split('/').pop());
            }
        }

        // Download document
        function downloadDocument(filePath, fileName) {
            const link = document.createElement('a');
            link.href = filePath;
            link.download = fileName;
            link.target = '_blank';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        }

        // Approve document
        async function approveDocument(docId) {
            if (!confirm('Möchten Sie dieses Dokument genehmigen?')) return;
            
            try {
                const response = await fetch('api/admin_document_action.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'approve',
                        document_id: docId
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Reload documents
                    const userId = document.querySelector('.document-card[data-doc-id="' + docId + '"]').closest('.documents-container').id.replace('documents-container-', '');
                    loadUserDocuments(userId);
                    
                    // Show success message
                    showNotification('Dokument erfolgreich genehmigt', 'success');
                } else {
                    showNotification('Fehler beim Genehmigen: ' + data.message, 'error');
                }
            } catch (error) {
                console.error('Error approving document:', error);
                showNotification('Fehler beim Genehmigen des Dokuments', 'error');
            }
        }

        // Reject document
        async function rejectDocument(docId) {
            const reason = prompt('Grund für die Ablehnung:');
            if (!reason) return;
            
            try {
                const response = await fetch('api/admin_document_action.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'reject',
                        document_id: docId,
                        reason: reason
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Reload documents
                    const userId = document.querySelector('.document-card[data-doc-id="' + docId + '"]').closest('.documents-container').id.replace('documents-container-', '');
                    loadUserDocuments(userId);
                    
                    // Show success message
                    showNotification('Dokument abgelehnt', 'success');
                } else {
                    showNotification('Fehler beim Ablehnen: ' + data.message, 'error');
                }
            } catch (error) {
                console.error('Error rejecting document:', error);
                showNotification('Fehler beim Ablehnen des Dokuments', 'error');
            }
        }

        // Show notification
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notification notification-${type}`;
            notification.innerHTML = `
                <i class="fas ${type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-exclamation-circle' : 'fa-info-circle'}"></i>
                <span>${message}</span>
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.classList.add('show');
            }, 100);
            
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => {
                    document.body.removeChild(notification);
                }, 300);
            }, 3000);
        }

        // Document Review Functions
        let currentReviewDocuments = {};

        // Review document
        function reviewDocument(userId, docId, fileName, filePath, mimeType, docType, fileSize, uploadDate, reviewStatus) {
            currentReviewDocuments[userId] = {
                id: docId,
                fileName: fileName,
                filePath: filePath,
                mimeType: mimeType,
                docType: docType,
                fileSize: fileSize,
                uploadDate: uploadDate,
                reviewStatus: reviewStatus
            };

            // Update modal header
            document.getElementById(`review-document-name-${userId}`).textContent = fileName;
            document.getElementById(`review-document-type-${userId}`).textContent = getDocumentTypeLabel(docType);
            document.getElementById(`review-document-size-${userId}`).textContent = formatFileSize(fileSize);
            document.getElementById(`review-document-date-${userId}`).textContent = formatDate(uploadDate);

            // Load document content
            loadDocumentForReview(userId, filePath, mimeType);

            // Clear previous notes
            document.getElementById(`review-notes-${userId}`).value = '';

            // Show review section
            document.getElementById(`document-review-section-${userId}`).style.display = 'block';
            
            // Scroll to review section
            document.getElementById(`document-review-section-${userId}`).scrollIntoView({ behavior: 'smooth' });
        }

        // Load document for review with professional viewer
        function loadDocumentForReview(userId, filePath, mimeType) {
            const viewerContent = document.getElementById(`document-viewer-content-${userId}`);
            const secureUrl = `api/view_document.php?file=${encodeURIComponent(filePath)}`;

            if (mimeType.startsWith('image/')) {
                viewerContent.innerHTML = `
                    <div class="image-viewer">
                        <img src="${secureUrl}" alt="Dokument" class="viewer-image" onload="hideDocumentLoading(${userId})" onerror="showDocumentError(${userId})">
                    </div>
                `;
            } else if (mimeType === 'application/pdf') {
                viewerContent.innerHTML = `
                    <div class="pdf-viewer">
                        <iframe src="${secureUrl}" class="viewer-pdf" onload="hideDocumentLoading(${userId})" onerror="showDocumentError(${userId})"></iframe>
                    </div>
                `;
            } else if (mimeType === 'text/plain') {
                viewerContent.innerHTML = `
                    <div class="text-viewer">
                        <iframe src="${secureUrl}" class="viewer-text" onload="hideDocumentLoading(${userId})" onerror="showDocumentError(${userId})"></iframe>
                    </div>
                `;
            } else {
                viewerContent.innerHTML = `
                    <div class="unsupported-viewer">
                        <div class="unsupported-icon">
                            <i class="fas fa-file"></i>
                        </div>
                        <h4>Dateityp nicht unterstützt</h4>
                        <p>Dieser Dateityp kann nicht direkt angezeigt werden.</p>
                        <button class="btn btn-primary" onclick="downloadCurrentDocument(${userId})">
                            <i class="fas fa-download"></i>
                            <span>Herunterladen</span>
                        </button>
                    </div>
                `;
                hideDocumentLoading(userId);
            }
        }

        // Hide document loading
        function hideDocumentLoading(userId) {
            const loading = document.querySelector(`#document-viewer-content-${userId} .viewer-loading`);
            if (loading) {
                loading.style.display = 'none';
            }
        }

        // Show document error
        function showDocumentError(userId) {
            const viewerContent = document.getElementById(`document-viewer-content-${userId}`);
            viewerContent.innerHTML = `
                <div class="document-error-viewer">
                    <div class="error-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <h4>Fehler beim Laden des Dokuments</h4>
                    <p>Das Dokument konnte nicht geladen werden.</p>
                    <button class="btn btn-primary" onclick="downloadCurrentDocument(${userId})">
                        <i class="fas fa-download"></i>
                        <span>Herunterladen</span>
                    </button>
                </div>
            `;
        }

        // Download current document
        function downloadCurrentDocument(userId) {
            if (currentReviewDocuments[userId]) {
                const secureUrl = `api/view_document.php?file=${encodeURIComponent(currentReviewDocuments[userId].filePath)}`;
                downloadDocument(secureUrl, currentReviewDocuments[userId].fileName);
            }
        }

        // Open document in new tab
        function openDocumentInNewTab(userId) {
            if (currentReviewDocuments[userId]) {
                const secureUrl = `api/view_document.php?file=${encodeURIComponent(currentReviewDocuments[userId].filePath)}`;
                window.open(secureUrl, '_blank');
            }
        }

        // Close document review
        function closeDocumentReview(userId) {
            document.getElementById(`document-review-section-${userId}`).style.display = 'none';
            delete currentReviewDocuments[userId];
        }

        // Approve document from review
        async function approveDocumentFromReview(userId) {
            if (!currentReviewDocuments[userId]) return;

            const notes = document.getElementById(`review-notes-${userId}`).value.trim();
            
            try {
                const response = await fetch('api/admin_document_action.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'approve',
                        document_id: currentReviewDocuments[userId].id,
                        notes: notes
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Close review section
                    closeDocumentReview(userId);
                    
                    // Reload documents
                    loadUserDocuments(userId);
                    
                    // Show success message
                    showNotification('Dokument erfolgreich genehmigt', 'success');
                } else {
                    showNotification('Fehler beim Genehmigen: ' + data.message, 'error');
                }
            } catch (error) {
                console.error('Error approving document:', error);
                showNotification('Fehler beim Genehmigen des Dokuments', 'error');
            }
        }

        // Reject document from review
        async function rejectDocumentFromReview(userId) {
            if (!currentReviewDocuments[userId]) return;

            const notes = document.getElementById(`review-notes-${userId}`).value.trim();
            
            if (!notes) {
                const reason = prompt('Grund für die Ablehnung:');
                if (!reason) return;
                document.getElementById(`review-notes-${userId}`).value = reason;
            }
            
            try {
                const response = await fetch('api/admin_document_action.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'reject',
                        document_id: currentReviewDocuments[userId].id,
                        reason: notes || 'Kein Grund angegeben'
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Close review section
                    closeDocumentReview(userId);
                    
                    // Reload documents
                    loadUserDocuments(userId);
                    
                    // Show success message
                    showNotification('Dokument abgelehnt', 'success');
                } else {
                    showNotification('Fehler beim Ablehnen: ' + data.message, 'error');
                }
            } catch (error) {
                console.error('Error rejecting document:', error);
                showNotification('Fehler beim Ablehnen des Dokuments', 'error');
            }
        }
    </script>

    <style>
        /* Modern CSS Variables */
        :root {
            --primary-color: #007AFF;
            --primary-color-dark: #0056CC;
            --secondary-color: #5856D6;
            --success-color: #34C759;
            --warning-color: #FF9500;
            --danger-color: #FF3B30;
            --info-color: #5AC8FA;
            
            --surface-primary: #FFFFFF;
            --surface-secondary: #F2F2F7;
            --surface-tertiary: #FFFFFF;
            --surface-quaternary: #F2F2F7;
            
            --text-primary: #000000;
            --text-secondary: #3C3C43;
            --text-tertiary: #3C3C4399;
            --text-quaternary: #3C3C4366;
            
            --border-color: #C6C6C8;
            --border-color-light: #E5E5EA;
            --border-color-dark: #AEAEB2;
            
            --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.1);
            --shadow-md: 0 4px 6px rgba(0, 0, 0, 0.1);
            --shadow-lg: 0 10px 15px rgba(0, 0, 0, 0.1);
            --shadow-xl: 0 20px 25px rgba(0, 0, 0, 0.1);
            
            --border-radius: 8px;
            --border-radius-lg: 12px;
            --border-radius-xl: 16px;
            
            --spacing-xs: 4px;
            --spacing-sm: 8px;
            --spacing-md: 12px;
            --spacing-lg: 16px;
            --spacing-xl: 20px;
            --spacing-2xl: 24px;
            --spacing-3xl: 32px;
            --spacing-4xl: 40px;
            
            --font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            --font-size-xs: 11px;
            --font-size-sm: 13px;
            --font-size-md: 15px;
            --font-size-lg: 17px;
            --font-size-xl: 19px;
            --font-size-2xl: 22px;
            --font-size-3xl: 28px;
            
            --font-weight-normal: 400;
            --font-weight-medium: 500;
            --font-weight-semibold: 600;
            --font-weight-bold: 700;
        }

        /* Dark theme variables */
        [data-theme="dark"] {
            --surface-primary: #1C1C1E;
            --surface-secondary: #2C2C2E;
            --surface-tertiary: #3A3A3C;
            --surface-quaternary: #48484A;
            
            --text-primary: #FFFFFF;
            --text-secondary: #EBEBF5;
            --text-tertiary: #EBEBF599;
            --text-quaternary: #EBEBF566;
            
            --border-color: #38383A;
            --border-color-light: #2C2C2E;
            --border-color-dark: #48484A;
        }

        /* Modern Header Styles */
        .modern-header {
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color-light);
            padding: var(--spacing-2xl) var(--spacing-3xl);
            margin-bottom: var(--spacing-2xl);
        }

        .header-content {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: var(--spacing-2xl);
        }

        .header-left {
            flex: 1;
        }

        .page-title-section {
            margin-bottom: var(--spacing-2xl);
        }

        .page-title {
            font-size: var(--font-size-3xl);
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
            margin: 0 0 var(--spacing-sm) 0;
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        .page-title i {
            color: var(--primary-color);
        }

        .page-subtitle {
            font-size: var(--font-size-lg);
            color: var(--text-secondary);
            margin: 0;
        }

        .stats-overview {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--spacing-lg);
        }

        .stat-card {
            background: var(--surface-secondary);
            border: 1px solid var(--border-color-light);
            border-radius: var(--border-radius-lg);
            padding: var(--spacing-lg);
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            transition: all 0.2s ease;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .stat-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--border-radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: var(--font-size-xl);
            color: white;
        }

        .stat-icon.pending {
            background: var(--warning-color);
        }

        .stat-icon.approved {
            background: var(--success-color);
        }

        .stat-icon.rejected {
            background: var(--danger-color);
        }

        .stat-icon:not(.pending):not(.approved):not(.rejected) {
            background: var(--primary-color);
        }

        .stat-content {
            flex: 1;
        }

        .stat-number {
            font-size: var(--font-size-2xl);
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
            line-height: 1;
        }

        .stat-label {
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
            margin-top: var(--spacing-xs);
        }

        .header-right {
            flex-shrink: 0;
        }

        .header-actions {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-lg);
            align-items: flex-end;
        }

        .filter-section {
            display: flex;
            gap: var(--spacing-md);
        }

        .modern-select {
            padding: var(--spacing-md) var(--spacing-lg);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius-lg);
            background: var(--surface-primary);
            color: var(--text-primary);
            font-size: var(--font-size-md);
            font-family: var(--font-family);
            min-width: 200px;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .modern-select:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(0, 122, 255, 0.1);
        }

        .action-buttons {
            display: flex;
            gap: var(--spacing-md);
        }

        /* Content Area */
        .content-area {
            padding: 0 var(--spacing-3xl) var(--spacing-3xl);
        }

        /* Content Card */
        .content-card {
            background: var(--surface-primary);
            border: 1px solid var(--border-color-light);
            border-radius: var(--border-radius-xl);
            box-shadow: var(--shadow-sm);
            overflow: hidden;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-xl) var(--spacing-2xl);
            border-bottom: 1px solid var(--border-color-light);
            background: var(--surface-secondary);
        }

        .card-title-section h3 {
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin: 0 0 var(--spacing-sm) 0;
        }

        .card-subtitle {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }

        .user-count {
            font-weight: var(--font-weight-medium);
        }

        .status-indicator {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius-lg);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-medium);
        }

        .status-indicator.pending {
            background: rgba(255, 149, 0, 0.1);
            color: var(--warning-color);
        }

        .status-indicator.approved {
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
        }

        .status-indicator.rejected {
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
        }

        .card-actions {
            display: flex;
            gap: var(--spacing-md);
        }

        .card-content {
            padding: var(--spacing-2xl);
        }

        /* Bulk Actions Toolbar */
        .bulk-actions-toolbar {
            background: var(--surface-secondary);
            border: 1px solid var(--border-color-light);
            border-radius: var(--border-radius-lg);
            padding: var(--spacing-md) var(--spacing-lg);
            margin-bottom: var(--spacing-lg);
            box-shadow: var(--shadow-sm);
        }

        .bulk-actions-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .bulk-actions-left {
            display: flex;
            align-items: center;
            gap: var(--spacing-lg);
        }

        .selected-count {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-weight: var(--font-weight-medium);
            color: var(--text-primary);
        }

        .bulk-actions-right {
            display: flex;
            gap: var(--spacing-sm);
        }

        /* Modern Table */
        .table-container {
            overflow-x: auto;
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
        }

        .modern-table {
            width: 100%;
            border-collapse: collapse;
            background: var(--surface-primary);
        }

        .modern-table thead {
            background: var(--surface-secondary);
        }

        .modern-table th {
            padding: var(--spacing-lg) var(--spacing-md);
            text-align: left;
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-size: var(--font-size-sm);
            border-bottom: 1px solid var(--border-color-light);
            white-space: nowrap;
        }

        .modern-table td {
            padding: var(--spacing-lg) var(--spacing-md);
            border-bottom: 1px solid var(--border-color-light);
            vertical-align: middle;
        }

        .modern-table tbody tr:hover {
            background: var(--surface-secondary);
        }

        .modern-table tbody tr:last-child td {
            border-bottom: none;
        }

        /* User Info Styles */
        .user-info {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: var(--primary-color);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: var(--font-size-md);
        }

        .user-details {
            flex: 1;
        }

        .user-name {
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-size: var(--font-size-md);
        }

        .user-id {
            font-size: var(--font-size-xs);
            color: var(--text-secondary);
            margin-top: var(--spacing-xs);
        }

        /* Contact Info */
        .contact-info {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
        }

        .contact-item {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }

        .contact-item i {
            color: var(--primary-color);
            width: 14px;
        }

        /* License Plate */
        .license-plate {
            font-family: 'Courier New', monospace;
            font-weight: var(--font-weight-bold);
            color: var(--text-primary);
            background: var(--surface-secondary);
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            text-align: center;
        }

        /* Dealership */
        .dealership {
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }

        /* Status Badges */
        .status-badge {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-xs);
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius-lg);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-medium);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .status-badge.status-pending {
            background: rgba(255, 149, 0, 0.1);
            color: var(--warning-color);
        }

        .status-badge.status-approved {
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
        }

        .status-badge.status-rejected {
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
        }

        .status-badge.status-suspended {
            background: rgba(142, 142, 147, 0.1);
            color: #8E8E93;
        }

        /* Document Info */
        .document-info {
            text-align: center;
        }

        .document-count {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: var(--spacing-xs);
            font-weight: var(--font-weight-medium);
            color: var(--text-primary);
        }

        .last-upload {
            font-size: var(--font-size-xs);
            color: var(--text-secondary);
            margin-top: var(--spacing-xs);
        }

        /* Registration Date */
        .registration-date {
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }

        /* Action Buttons */
        .action-buttons {
            display: flex;
            gap: var(--spacing-xs);
        }

        .action-btn {
            width: 32px;
            height: 32px;
            border: none;
            border-radius: var(--border-radius);
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            font-size: var(--font-size-sm);
        }

        .action-btn.btn-view {
            background: rgba(142, 142, 147, 0.1);
            color: #8E8E93;
        }

        .action-btn.btn-view:hover {
            background: rgba(142, 142, 147, 0.2);
        }

        .action-btn.btn-approve {
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
        }

        .action-btn.btn-approve:hover {
            background: rgba(52, 199, 89, 0.2);
        }

        .action-btn.btn-reject {
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
        }

        .action-btn.btn-reject:hover {
            background: rgba(255, 59, 48, 0.2);
        }

        /* Button Styles */
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: var(--spacing-sm);
            padding: var(--spacing-md) var(--spacing-lg);
            border: none;
            border-radius: var(--border-radius-lg);
            font-size: var(--font-size-md);
            font-weight: var(--font-weight-medium);
            text-decoration: none;
            cursor: pointer;
            transition: all 0.2s ease;
            font-family: var(--font-family);
            min-height: 44px;
        }

        .btn-sm {
            padding: var(--spacing-sm) var(--spacing-md);
            font-size: var(--font-size-sm);
            min-height: 36px;
        }

        .btn-primary {
            background: var(--primary-color);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-color-dark);
            transform: translateY(-1px);
        }

        .btn-secondary {
            background: var(--surface-secondary);
            color: var(--text-primary);
            border: 1px solid var(--border-color);
        }

        .btn-secondary:hover {
            background: var(--surface-tertiary);
        }

        .btn-outline {
            background: transparent;
            color: var(--text-primary);
            border: 1px solid var(--border-color);
        }

        .btn-outline:hover {
            background: var(--surface-secondary);
        }

        .btn-success {
            background: var(--success-color);
            color: white;
        }

        .btn-success:hover {
            background: #2FB344;
            transform: translateY(-1px);
        }

        .btn-danger {
            background: var(--danger-color);
            color: white;
        }

        .btn-danger:hover {
            background: #E6342A;
            transform: translateY(-1px);
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: var(--spacing-4xl) var(--spacing-2xl);
            color: var(--text-secondary);
        }

        .empty-icon {
            font-size: 64px;
            margin-bottom: var(--spacing-lg);
            color: var(--text-tertiary);
        }

        .empty-state h4 {
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin: 0 0 var(--spacing-sm) 0;
        }

        .empty-state p {
            font-size: var(--font-size-md);
            margin: 0;
        }

        /* Alert Styles */
        .alert {
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
            padding: var(--spacing-lg) var(--spacing-xl);
            border-radius: var(--border-radius-lg);
            margin-bottom: var(--spacing-lg);
            font-weight: var(--font-weight-medium);
        }

        .alert-success {
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
            border: 1px solid rgba(52, 199, 89, 0.2);
        }

        .alert-error {
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
            border: 1px solid rgba(255, 59, 48, 0.2);
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 10000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(4px);
        }

        .modal-content {
            background-color: var(--surface-primary);
            margin: 5% auto;
            padding: 0;
            border-radius: var(--border-radius-xl);
            width: 90%;
            max-width: 800px;
            box-shadow: var(--shadow-xl);
            border: 1px solid var(--border-color-light);
            opacity: 1;
            transform: scale(1);
            transition: all 0.3s ease;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-xl) var(--spacing-2xl);
            border-bottom: 1px solid var(--border-color-light);
            background: var(--surface-secondary);
            border-radius: var(--border-radius-xl) var(--border-radius-xl) 0 0;
        }

        .modal-header h3 {
            margin: 0;
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
        }

        .modal-close {
            background: none;
            border: none;
            color: var(--text-secondary);
            font-size: 24px;
            cursor: pointer;
            transition: color 0.2s ease;
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: var(--border-radius);
        }

        .modal-close:hover {
            color: var(--text-primary);
            background: var(--surface-tertiary);
        }

        .modal-body {
            padding: var(--spacing-2xl);
        }

        .form-group {
            margin-bottom: var(--spacing-lg);
        }

        .form-group label {
            display: block;
            margin-bottom: var(--spacing-sm);
            font-weight: var(--font-weight-medium);
            color: var(--text-primary);
        }

        .form-group textarea {
            width: 100%;
            padding: var(--spacing-md);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius-lg);
            background: var(--surface-primary);
            color: var(--text-primary);
            font-family: var(--font-family);
            resize: vertical;
            min-height: 80px;
            font-size: var(--font-size-md);
        }

        .form-group textarea:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(0, 122, 255, 0.1);
        }

        .modal-actions {
            display: flex;
            gap: var(--spacing-md);
            justify-content: flex-end;
            margin-top: var(--spacing-xl);
        }

        /* User Details Styles */
        .user-details-content {
            font-family: var(--font-family);
        }

        .detail-header {
            display: flex;
            align-items: center;
            gap: var(--spacing-lg);
            margin-bottom: var(--spacing-xl);
            padding-bottom: var(--spacing-lg);
            border-bottom: 1px solid var(--border-color-light);
        }

        .user-avatar-large {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: var(--primary-color);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 32px;
        }

        .user-info h3 {
            margin: 0 0 var(--spacing-sm) 0;
            font-size: var(--font-size-xl);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
        }

        .user-subtitle {
            margin: 0;
            font-size: var(--font-size-sm);
            color: var(--text-secondary);
        }

        .status-section {
            margin-left: auto;
        }

        .status-badge-large {
            display: inline-flex;
            align-items: center;
            gap: var(--spacing-sm);
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--border-radius-lg);
            font-weight: var(--font-weight-medium);
            font-size: var(--font-size-sm);
        }

        .status-badge-large.status-success {
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
        }

        .status-badge-large.status-warning {
            background: rgba(255, 149, 0, 0.1);
            color: var(--warning-color);
        }

        .status-badge-large.status-danger {
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
        }

        .status-badge-large.status-secondary {
            background: rgba(142, 142, 147, 0.1);
            color: #8E8E93;
        }

        .detail-content-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: var(--spacing-xl);
        }

        .detail-section {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            padding: var(--spacing-lg);
            border: 1px solid var(--border-color-light);
        }

        .section-header {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            margin-bottom: var(--spacing-lg);
            padding-bottom: var(--spacing-md);
            border-bottom: 1px solid var(--border-color-light);
        }

        .section-header i {
            color: var(--primary-color);
            font-size: var(--font-size-lg);
        }

        .section-header h4 {
            margin: 0;
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
        }

        .info-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-sm) 0;
            border-bottom: 1px solid var(--border-color-light);
        }

        .info-item:last-child {
            border-bottom: none;
        }

        .info-label {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-weight: var(--font-weight-medium);
            color: var(--text-secondary);
            font-size: var(--font-size-sm);
        }

        .info-label i {
            color: var(--primary-color);
            width: 16px;
        }

        .info-value {
            font-weight: var(--font-weight-medium);
            color: var(--text-primary);
            text-align: right;
        }

        /* Auto-refresh indicator */
        .auto-refresh-indicator {
            position: fixed;
            top: 20px;
            right: 20px;
            background: var(--primary-color);
            color: white;
            padding: var(--spacing-sm) var(--spacing-md);
            border-radius: var(--border-radius-lg);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-medium);
            z-index: 1000;
            opacity: 0;
            transform: translateY(-20px);
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
        }

        .auto-refresh-indicator.show {
            opacity: 1;
            transform: translateY(0);
        }

        .auto-refresh-indicator.success {
            background: var(--success-color);
        }

        .auto-refresh-indicator.error {
            background: var(--danger-color);
        }

        /* Documents Section Styles */
        .documents-section {
            grid-column: 1 / -1;
        }

        .documents-container {
            min-height: 200px;
        }

        .documents-loading {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: var(--spacing-2xl);
            color: var(--text-secondary);
        }

        .documents-loading i {
            font-size: var(--font-size-xl);
            margin-bottom: var(--spacing-md);
        }

        .documents-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: var(--spacing-lg);
        }

        .document-card {
            background: var(--surface-primary);
            border: 1px solid var(--border-color-light);
            border-radius: var(--border-radius-lg);
            overflow: hidden;
            transition: all 0.2s ease;
            box-shadow: var(--shadow-sm);
        }

        .document-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .document-preview {
            height: 120px;
            background: var(--surface-secondary);
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
            position: relative;
        }

        .document-image {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .document-pdf-icon,
        .document-file-icon {
            font-size: 48px;
            color: var(--text-tertiary);
        }

        .document-pdf-icon {
            color: #DC2626;
        }

        .document-info {
            padding: var(--spacing-md);
        }

        .document-name {
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            font-size: var(--font-size-sm);
            margin-bottom: var(--spacing-xs);
            word-break: break-word;
        }

        .document-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: var(--spacing-xs);
            font-size: var(--font-size-xs);
            color: var(--text-secondary);
        }

        .document-type {
            background: var(--primary-color);
            color: white;
            padding: 2px var(--spacing-xs);
            border-radius: var(--border-radius);
            font-weight: var(--font-weight-medium);
        }

        .document-date {
            font-size: var(--font-size-xs);
            color: var(--text-tertiary);
            margin-bottom: var(--spacing-sm);
        }

        .document-status {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-medium);
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            margin-bottom: var(--spacing-sm);
        }

        .document-status.status-pending {
            background: rgba(255, 149, 0, 0.1);
            color: var(--warning-color);
        }

        .document-status.status-approved {
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
        }

        .document-status.status-rejected {
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
        }

        .document-review-info,
        .document-review-notes {
            margin-top: var(--spacing-xs);
            padding: var(--spacing-xs);
            background: var(--surface-tertiary);
            border-radius: var(--border-radius);
            font-size: var(--font-size-xs);
            color: var(--text-secondary);
        }

        .document-review-notes {
            background: rgba(90, 200, 250, 0.1);
            border-left: 3px solid var(--info-color);
        }

        .document-actions {
            display: flex;
            gap: var(--spacing-xs);
            padding: var(--spacing-sm) var(--spacing-md);
            background: var(--surface-secondary);
            border-top: 1px solid var(--border-color-light);
        }

        .action-btn.btn-download {
            background: rgba(0, 122, 255, 0.1);
            color: var(--primary-color);
        }

        .action-btn.btn-download:hover {
            background: rgba(0, 122, 255, 0.2);
        }

        .action-btn.btn-review {
            background: rgba(255, 149, 0, 0.1);
            color: var(--warning-color);
        }

        .action-btn.btn-review:hover {
            background: rgba(255, 149, 0, 0.2);
        }

        .documents-empty,
        .documents-error {
            text-align: center;
            padding: var(--spacing-2xl);
            color: var(--text-secondary);
        }

        .documents-empty i,
        .documents-error i {
            font-size: 48px;
            margin-bottom: var(--spacing-md);
            color: var(--text-tertiary);
        }

        .documents-empty h4,
        .documents-error h4 {
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin: 0 0 var(--spacing-sm) 0;
        }

        .documents-empty p,
        .documents-error p {
            font-size: var(--font-size-md);
            margin: 0 0 var(--spacing-lg) 0;
        }

        .documents-error i {
            color: var(--danger-color);
        }

        /* Notification Styles */
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            background: var(--surface-primary);
            border: 1px solid var(--border-color-light);
            border-radius: var(--border-radius-lg);
            padding: var(--spacing-md) var(--spacing-lg);
            box-shadow: var(--shadow-lg);
            z-index: 10001;
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            font-weight: var(--font-weight-medium);
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            max-width: 400px;
        }

        .notification.show {
            opacity: 1;
            transform: translateX(0);
        }

        .notification-success {
            border-color: var(--success-color);
            background: rgba(52, 199, 89, 0.1);
            color: var(--success-color);
        }

        .notification-error {
            border-color: var(--danger-color);
            background: rgba(255, 59, 48, 0.1);
            color: var(--danger-color);
        }

        .notification-info {
            border-color: var(--info-color);
            background: rgba(90, 200, 250, 0.1);
            color: var(--info-color);
        }

        /* NEUES PROFESSIONELLES USER MODAL - KOMPLETT NEU */
        .new-professional-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100vw;
            height: 100vh;
            background: rgba(0, 0, 0, 0.8);
            z-index: 10000;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .new-modal-content {
            width: 95vw;
            height: 95vh;
            background: white;
            border-radius: 20px;
            box-shadow: 0 25px 50px rgba(0, 0, 0, 0.3);
            overflow: hidden;
            display: flex;
            flex-direction: column;
            position: relative;
        }

        #user-details {
            overflow-y: auto;
            flex: 1;
            max-height: 100%;
        }

        .new-user-dashboard {
            width: 100%;
            min-height: 100%;
            display: flex;
            flex-direction: column;
            background: #f8f9fa;
        }

        /* Header */
        .new-header {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            padding: 30px 40px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 4px 20px rgba(0, 123, 255, 0.3);
        }

        .new-user-info {
            display: flex;
            align-items: center;
            gap: 25px;
        }

        .new-avatar {
            width: 80px;
            height: 80px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 40px;
            backdrop-filter: blur(10px);
        }

        .new-user-details h1 {
            font-size: 28px;
            font-weight: bold;
            margin: 0 0 8px 0;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        .new-user-meta {
            display: flex;
            gap: 20px;
            align-items: center;
            font-size: 14px;
            opacity: 0.9;
        }

        .new-status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 6px;
            backdrop-filter: blur(10px);
        }

        .new-status-badge.pending {
            background: rgba(255, 193, 7, 0.3);
            color: #ffc107;
        }

        .new-status-badge.approved {
            background: rgba(40, 167, 69, 0.3);
            color: #28a745;
        }

        .new-status-badge.rejected {
            background: rgba(220, 53, 69, 0.3);
            color: #dc3545;
        }

        .new-close-btn {
            background: rgba(255, 255, 255, 0.2);
            border: none;
            color: white;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            font-size: 18px;
            transition: all 0.2s ease;
        }

        .new-close-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: scale(1.1);
        }

        /* Main Content */
        .new-main-content {
            flex: 1;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
            padding: 30px;
            overflow-y: auto;
        }

        /* Left Column */
        .new-left-column {
            display: flex;
            flex-direction: column;
            gap: 25px;
        }

        /* Right Column */
        .new-right-column {
            display: flex;
            flex-direction: column;
            gap: 25px;
        }

        /* Cards */
        .new-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            transition: all 0.3s ease;
        }

        .new-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }

        .new-card-header {
            background: #f8f9fa;
            padding: 20px 25px;
            border-bottom: 1px solid #e9ecef;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .new-card-header i {
            color: #007bff;
            font-size: 18px;
        }

        .new-card-header h3 {
            margin: 0;
            font-size: 18px;
            font-weight: 600;
            color: #212529;
        }

        .new-card-content {
            padding: 25px;
        }

        /* Info Grid */
        .new-info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }

        .new-info-field {
            display: flex;
            flex-direction: column;
            gap: 6px;
        }

        .new-info-field.full-width {
            grid-column: 1 / -1;
        }

        .new-info-field label {
            font-size: 12px;
            font-weight: 600;
            color: #6c757d;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .new-info-value {
            font-size: 14px;
            color: #212529;
            font-weight: 500;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 1px solid #e9ecef;
        }

        .new-info-value.email {
            color: #007bff;
            word-break: break-all;
        }

        .new-info-value.license {
            font-family: 'Courier New', monospace;
            font-weight: bold;
            text-align: center;
            background: #e9ecef;
        }

        /* Actions */
        .new-actions-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .new-action-btn {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 20px;
            border: none;
            border-radius: 12px;
            font-size: 13px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s ease;
            min-height: 80px;
        }

        .new-action-btn i {
            font-size: 24px;
        }

        .new-action-btn.success {
            background: rgba(40, 167, 69, 0.1);
            color: #28a745;
            border: 2px solid rgba(40, 167, 69, 0.2);
        }

        .new-action-btn.success:hover {
            background: rgba(40, 167, 69, 0.2);
            transform: translateY(-2px);
        }

        .new-action-btn.danger {
            background: rgba(220, 53, 69, 0.1);
            color: #dc3545;
            border: 2px solid rgba(220, 53, 69, 0.2);
        }

        .new-action-btn.danger:hover {
            background: rgba(220, 53, 69, 0.2);
            transform: translateY(-2px);
        }

        .new-action-btn.info {
            background: rgba(0, 123, 255, 0.1);
            color: #007bff;
            border: 2px solid rgba(0, 123, 255, 0.2);
        }

        .new-action-btn.info:hover {
            background: rgba(0, 123, 255, 0.2);
            transform: translateY(-2px);
        }

        .new-action-btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
            transform: none !important;
            background: rgba(108, 117, 125, 0.1) !important;
            color: #6c757d !important;
            border-color: rgba(108, 117, 125, 0.2) !important;
        }

        .new-action-btn:disabled:hover {
            background: rgba(108, 117, 125, 0.1) !important;
            transform: none !important;
        }

        /* Documents */
        .new-documents-container {
            min-height: 300px;
            padding: 20px;
        }

        .new-documents-loading {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: #6c757d;
            gap: 15px;
        }

        .new-documents-loading i {
            font-size: 32px;
        }

        /* Notes */
        .new-notes-textarea {
            width: 100%;
            padding: 15px;
            border: 1px solid #dee2e6;
            border-radius: 10px;
            background: #f8f9fa;
            color: #212529;
            font-family: inherit;
            font-size: 14px;
            resize: vertical;
            min-height: 120px;
        }

        .new-notes-textarea:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
        }

        .new-save-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.2s ease;
        }

        .new-save-btn:hover {
            background: #0056b3;
            transform: translateY(-1px);
        }

        /* Edit Controls */
        .new-edit-btn {
            background: linear-gradient(135deg, #6c757d, #5a6268);
            color: white;
            border: none;
            padding: 10px 16px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            font-size: 13px;
            box-shadow: 0 2px 4px rgba(108, 117, 125, 0.2);
            position: relative;
            overflow: hidden;
        }

        .new-edit-btn:hover {
            background: linear-gradient(135deg, #5a6268, #495057);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(108, 117, 125, 0.3);
        }

        .new-edit-btn:active {
            transform: translateY(0);
            box-shadow: 0 2px 4px rgba(108, 117, 125, 0.2);
        }

        .new-save-btn {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border: none;
            padding: 10px 16px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            font-size: 13px;
            box-shadow: 0 2px 4px rgba(40, 167, 69, 0.2);
            position: relative;
            overflow: hidden;
        }

        .new-save-btn:hover {
            background: linear-gradient(135deg, #20c997, #17a2b8);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(40, 167, 69, 0.3);
        }

        .new-save-btn:active {
            transform: translateY(0);
            box-shadow: 0 2px 4px rgba(40, 167, 69, 0.2);
        }

        .new-cancel-btn {
            background: linear-gradient(135deg, #dc3545, #c82333);
            color: white;
            border: none;
            padding: 10px 16px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            font-size: 13px;
            box-shadow: 0 2px 4px rgba(220, 53, 69, 0.2);
            position: relative;
            overflow: hidden;
        }

        .new-cancel-btn:hover {
            background: linear-gradient(135deg, #c82333, #bd2130);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(220, 53, 69, 0.3);
        }

        .new-cancel-btn:active {
            transform: translateY(0);
            box-shadow: 0 2px 4px rgba(220, 53, 69, 0.2);
        }

        .new-edit-input {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #007bff;
            border-radius: 6px;
            background: white;
            color: #212529;
            font-family: inherit;
            font-size: 14px;
        }

        .new-edit-input:focus {
            outline: none;
            border-color: #0056b3;
            box-shadow: 0 0 0 2px rgba(0, 123, 255, 0.1);
        }

        /* Suspend Modal Styles */
        .suspend-warning {
            background: linear-gradient(135deg, #fff3cd, #ffeaa7);
            border: 1px solid #ffeaa7;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .suspend-warning i {
            color: #856404;
            font-size: 20px;
        }

        .suspend-warning p {
            margin: 0;
            color: #856404;
            font-weight: 500;
        }

        .form-actions {
            display: flex;
            gap: 12px;
            justify-content: flex-end;
            margin-top: 20px;
        }

        .form-actions .btn {
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .form-actions .btn-secondary {
            background: #6c757d;
            color: white;
            border: none;
        }

        .form-actions .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-2px);
        }

        .form-actions .btn-danger {
            background: linear-gradient(135deg, #dc3545, #c82333);
            color: white;
            border: none;
        }

        .form-actions .btn-danger:hover {
            background: linear-gradient(135deg, #c82333, #bd2130);
            transform: translateY(-2px);
        }

        /* Responsive */
        @media (max-width: 1400px) {
            .new-main-content {
                grid-template-columns: 1fr;
                gap: 25px;
            }
        }

        @media (max-width: 768px) {
            .new-modal-content {
                width: 100vw;
                height: 100vh;
                border-radius: 0;
            }
            
            .new-header {
                flex-direction: column;
                gap: 20px;
                text-align: center;
            }
            
            .new-user-info {
                flex-direction: column;
                gap: 15px;
            }
            
            .new-user-meta {
                flex-direction: column;
                gap: 10px;
            }
            
            .new-info-grid {
                grid-template-columns: 1fr;
            }
            
            .new-actions-grid {
                grid-template-columns: 1fr;
            }
        }

        .modal-header-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-xl);
            background: linear-gradient(135deg, var(--primary-color), #4a90e2);
            color: white;
            border-radius: var(--border-radius-lg) var(--border-radius-lg) 0 0;
        }

        .user-header-info {
            display: flex;
            align-items: center;
            gap: var(--spacing-lg);
        }

        .user-avatar {
            font-size: 3rem;
            opacity: 0.9;
        }

        .user-basic-info h2 {
            margin: 0 0 var(--spacing-sm) 0;
            font-size: 1.8rem;
            font-weight: 600;
        }

        .user-meta {
            display: flex;
            gap: var(--spacing-md);
            align-items: center;
            font-size: var(--font-size-sm);
            opacity: 0.9;
        }

        .user-status {
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
        }

        .user-status.status-pending {
            background: rgba(255, 193, 7, 0.2);
            color: #ffc107;
        }

        .user-status.status-approved {
            background: rgba(40, 167, 69, 0.2);
            color: #28a745;
        }

        .user-status.status-rejected {
            background: rgba(220, 53, 69, 0.2);
            color: #dc3545;
        }

        .modal-header-actions {
            display: flex;
            gap: var(--spacing-sm);
        }

        .modal-main-content {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: var(--spacing-xl);
            padding: var(--spacing-xl);
            flex: 1;
            overflow-y: auto;
        }

        .user-info-column {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-lg);
        }

        .actions-column {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-lg);
        }

        .info-section {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
            overflow: hidden;
        }

        .section-header {
            display: flex;
            align-items: center;
            gap: var(--spacing-sm);
            padding: var(--spacing-lg);
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color-light);
        }

        .section-header i {
            color: var(--primary-color);
            font-size: 1.1rem;
        }

        .section-header h3 {
            margin: 0;
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--text-primary);
        }

        .info-grid {
            padding: var(--spacing-lg);
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: var(--spacing-md);
        }

        .info-item {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-xs);
        }

        .info-item.full-width {
            grid-column: 1 / -1;
        }

        .info-item label {
            font-size: var(--font-size-sm);
            font-weight: 500;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .info-item span {
            font-size: var(--font-size-md);
            color: var(--text-primary);
            font-weight: 500;
        }

        .info-item .email {
            color: var(--primary-color);
            word-break: break-all;
        }

        /* Action Section */
        .action-section {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
            overflow: hidden;
        }

        .action-buttons {
            padding: var(--spacing-lg);
            display: flex;
            flex-direction: column;
            gap: var(--spacing-md);
        }

        .btn-large {
            padding: var(--spacing-md) var(--spacing-lg);
            font-size: var(--font-size-md);
            font-weight: 500;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: var(--spacing-sm);
            border-radius: var(--border-radius);
            transition: all 0.2s ease;
        }

        .btn-large:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }

        /* Documents Section */
        .documents-section {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
            overflow: hidden;
            display: flex;
            flex-direction: column;
        }

        .documents-section .section-header {
            justify-content: space-between;
        }

        .documents-container {
            padding: var(--spacing-lg);
            flex: 1;
            min-height: 200px;
        }

        .documents-loading {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            color: var(--text-secondary);
            gap: var(--spacing-sm);
        }

        .documents-loading i {
            font-size: 2rem;
        }

        /* Notes Section */
        .notes-section {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
            overflow: hidden;
        }

        .notes-content {
            padding: var(--spacing-lg);
            display: flex;
            flex-direction: column;
            gap: var(--spacing-md);
        }

        .notes-content textarea {
            width: 100%;
            padding: var(--spacing-sm);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius);
            font-family: inherit;
            font-size: var(--font-size-sm);
            resize: vertical;
            min-height: 100px;
        }

        .notes-content textarea:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 2px rgba(52, 144, 220, 0.2);
        }

        /* Document Review Section */
        .document-review-section {
            margin-top: var(--spacing-xl);
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
            overflow: hidden;
        }

        .review-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-lg);
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color-light);
        }

        .review-header h3 {
            margin: 0;
            color: var(--text-primary);
        }

        .review-content {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: var(--spacing-xl);
            padding: var(--spacing-lg);
        }

        .document-viewer-panel {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            overflow: hidden;
            border: 1px solid var(--border-color-light);
            display: flex;
            flex-direction: column;
        }

        .viewer-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-lg);
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color-light);
        }

        .document-info h5 {
            margin: 0 0 var(--spacing-xs) 0;
            color: var(--text-primary);
            font-size: var(--font-size-lg);
        }

        .document-meta {
            display: flex;
            gap: var(--spacing-sm);
            align-items: center;
        }

        .meta-badge {
            background: var(--primary-color);
            color: white;
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            font-size: var(--font-size-xs);
            font-weight: 500;
        }

        .meta-info {
            color: var(--text-secondary);
            font-size: var(--font-size-sm);
        }

        .viewer-controls {
            display: flex;
            gap: var(--spacing-sm);
        }

        .btn-outline {
            background: transparent;
            border: 1px solid var(--border-color);
            color: var(--text-secondary);
        }

        .btn-outline:hover {
            background: var(--surface-tertiary);
            border-color: var(--primary-color);
            color: var(--primary-color);
        }

        .document-viewer-frame {
            flex: 1;
            background: white;
            position: relative;
            min-height: 500px;
        }

        .viewer-loading {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            text-align: center;
            color: var(--text-secondary);
        }

        .viewer-loading i {
            font-size: 2rem;
            margin-bottom: var(--spacing-sm);
        }

        /* Document Viewer Types */
        .image-viewer {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #f8f9fa;
        }

        .viewer-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: var(--border-radius);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }

        .pdf-viewer, .text-viewer {
            width: 100%;
            height: 100%;
        }

        .viewer-pdf, .viewer-text {
            width: 100%;
            height: 100%;
            border: none;
            border-radius: var(--border-radius);
        }

        .unsupported-viewer {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            height: 100%;
            text-align: center;
            padding: var(--spacing-xl);
        }

        .unsupported-icon {
            font-size: 3rem;
            color: var(--text-secondary);
            margin-bottom: var(--spacing-md);
        }

        /* Approval Panel */
        .approval-panel {
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            border: 1px solid var(--border-color-light);
            display: flex;
            flex-direction: column;
        }

        .approval-header {
            padding: var(--spacing-lg);
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color-light);
        }

        .approval-header h5 {
            margin: 0;
            color: var(--text-primary);
            font-size: var(--font-size-lg);
        }

        .approval-form {
            padding: var(--spacing-lg);
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        .approval-form .form-group {
            margin-bottom: var(--spacing-lg);
        }

        .approval-form label {
            display: block;
            margin-bottom: var(--spacing-sm);
            font-weight: 500;
            color: var(--text-primary);
        }

        .approval-form textarea {
            width: 100%;
            padding: var(--spacing-sm);
            border: 1px solid var(--border-color);
            border-radius: var(--border-radius);
            font-family: inherit;
            font-size: var(--font-size-sm);
            resize: vertical;
            min-height: 100px;
        }

        .approval-form textarea:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 2px rgba(52, 144, 220, 0.2);
        }

        .approval-actions {
            display: flex;
            flex-direction: column;
            gap: var(--spacing-md);
            margin-top: auto;
        }

        .approval-actions .btn {
            padding: var(--spacing-md) var(--spacing-lg);
            font-size: var(--font-size-md);
            font-weight: 500;
            border-radius: var(--border-radius);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: var(--spacing-sm);
            transition: all 0.2s ease;
        }

        .approval-actions .btn-success {
            background: var(--success-color);
            border: 1px solid var(--success-color);
            color: white;
        }

        .approval-actions .btn-success:hover {
            background: #28a745;
            border-color: #28a745;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.3);
        }

        .approval-actions .btn-danger {
            background: var(--danger-color);
            border: 1px solid var(--danger-color);
            color: white;
        }

        .approval-actions .btn-danger:hover {
            background: #dc3545;
            border-color: #dc3545;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
        }

        .document-viewer-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: var(--spacing-lg);
            background: var(--surface-primary);
            border-bottom: 1px solid var(--border-color-light);
        }

        .document-info-header h4 {
            margin: 0 0 var(--spacing-xs) 0;
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
        }

        .document-meta-header {
            display: flex;
            gap: var(--spacing-md);
            align-items: center;
        }

        .document-type-badge {
            background: var(--primary-color);
            color: white;
            padding: var(--spacing-xs) var(--spacing-sm);
            border-radius: var(--border-radius);
            font-size: var(--font-size-xs);
            font-weight: var(--font-weight-medium);
        }

        .document-size-info,
        .document-date-info {
            font-size: var(--font-size-xs);
            color: var(--text-secondary);
        }

        .document-viewer-controls {
            display: flex;
            gap: var(--spacing-sm);
        }

        .document-viewer-container {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            background: var(--surface-primary);
            position: relative;
        }

        .document-viewer-content {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .document-loading {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            color: var(--text-secondary);
        }

        .document-loading i {
            font-size: var(--font-size-2xl);
            margin-bottom: var(--spacing-md);
        }

        .document-image-viewer {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: var(--spacing-lg);
        }

        .review-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: var(--border-radius-lg);
            box-shadow: var(--shadow-lg);
        }

        .document-pdf-viewer {
            width: 100%;
            height: 100%;
            padding: var(--spacing-lg);
        }

        .review-pdf {
            width: 100%;
            height: 100%;
            border: none;
            border-radius: var(--border-radius-lg);
            box-shadow: var(--shadow-lg);
        }

        .document-unsupported-viewer,
        .document-error-viewer {
            text-align: center;
            padding: var(--spacing-2xl);
            color: var(--text-secondary);
        }

        .unsupported-icon,
        .error-icon {
            font-size: 64px;
            margin-bottom: var(--spacing-lg);
            color: var(--text-tertiary);
        }

        .error-icon {
            color: var(--danger-color);
        }

        .document-unsupported-viewer h4,
        .document-error-viewer h4 {
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
            color: var(--text-primary);
            margin: 0 0 var(--spacing-sm) 0;
        }

        .document-unsupported-viewer p,
        .document-error-viewer p {
            font-size: var(--font-size-md);
            margin: 0 0 var(--spacing-lg) 0;
        }

        .document-review-section {
            display: flex;
            flex-direction: column;
            background: var(--surface-secondary);
            border-radius: var(--border-radius-lg);
            padding: var(--spacing-lg);
        }

        .review-form {
            height: 100%;
            display: flex;
            flex-direction: column;
        }

        .review-form .form-group {
            margin-bottom: var(--spacing-xl);
        }

        .review-form textarea {
            min-height: 120px;
        }

        .review-actions {
            margin-top: auto;
            display: flex;
            flex-direction: column;
            gap: var(--spacing-lg);
        }

        .review-action-buttons {
            display: flex;
            gap: var(--spacing-md);
        }

        .review-action-buttons .btn {
            flex: 1;
            padding: var(--spacing-lg);
            font-size: var(--font-size-lg);
            font-weight: var(--font-weight-semibold);
        }

        .review-secondary-actions {
            display: flex;
            justify-content: center;
        }

        .review-secondary-actions .btn {
            padding: var(--spacing-md) var(--spacing-xl);
        }

        /* Responsive Design */
        @media (max-width: 1200px) {
            .header-content {
                flex-direction: column;
                align-items: stretch;
            }
            
            .header-right {
                align-self: stretch;
            }
            
            .header-actions {
                flex-direction: row;
                justify-content: space-between;
                align-items: center;
            }
            
            .stats-overview {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .detail-content-grid {
                grid-template-columns: 1fr;
            }
            
            .documents-grid {
                grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            }
            
            .document-review-content {
                grid-template-columns: 1fr;
                height: auto;
            }
            
            .document-review-modal .modal-content {
                height: 95vh;
                margin: 2.5vh auto;
            }
        }

        @media (max-width: 768px) {
            .modern-header {
                padding: var(--spacing-lg) var(--spacing-xl);
            }
            
            .content-area {
                padding: 0 var(--spacing-xl) var(--spacing-xl);
            }
            
            .stats-overview {
                grid-template-columns: 1fr;
            }
            
            .header-actions {
                flex-direction: column;
                gap: var(--spacing-md);
            }
            
            .action-buttons {
                width: 100%;
                justify-content: stretch;
            }
            
            .action-buttons .btn {
                flex: 1;
            }
            
            .modern-select {
                min-width: auto;
                width: 100%;
            }
            
            .modal-content {
                width: 95%;
                margin: 2% auto;
            }
            
            .detail-header {
                flex-direction: column;
                text-align: center;
                gap: var(--spacing-md);
            }
            
            .status-section {
                margin-left: 0;
            }
            
            .table-container {
                overflow-x: auto;
            }
            
            .modern-table {
                min-width: 800px;
            }
            
            .action-buttons {
                flex-direction: column;
                gap: var(--spacing-xs);
            }
            
            .action-btn {
                width: 28px;
                height: 28px;
                font-size: var(--font-size-xs);
            }
            
            .card-header {
                flex-direction: column;
                align-items: stretch;
                gap: var(--spacing-md);
            }
            
            .card-actions {
                justify-content: center;
            }
            
            .documents-grid {
                grid-template-columns: 1fr;
            }
            
            .document-card {
                margin-bottom: var(--spacing-md);
            }
            
            .notification {
                right: 10px;
                left: 10px;
                max-width: none;
            }
            
            .modal-main-content {
                grid-template-columns: 1fr;
                gap: var(--spacing-lg);
                padding: var(--spacing-lg);
            }
            
            .user-header-info {
                flex-direction: column;
                text-align: center;
                gap: var(--spacing-md);
            }
            
            .user-meta {
                flex-direction: column;
                gap: var(--spacing-sm);
            }
            
            .info-grid {
                grid-template-columns: 1fr;
            }
            
            .action-buttons {
                flex-direction: row;
                gap: var(--spacing-sm);
            }
            
            .btn-large {
                flex: 1;
                padding: var(--spacing-sm) var(--spacing-md);
                font-size: var(--font-size-sm);
            }
            
            .review-content {
                grid-template-columns: 1fr;
                gap: var(--spacing-lg);
            }
            
            .document-viewer-frame {
                min-height: 400px;
            }
            
            .approval-actions {
                flex-direction: row;
                gap: var(--spacing-sm);
            }
            
            .approval-actions .btn {
                flex: 1;
                padding: var(--spacing-sm) var(--spacing-md);
                font-size: var(--font-size-sm);
            }
        }

        @media (max-width: 480px) {
            .modal-body {
                padding: var(--spacing-lg);
            }
            
            .modal-header {
                padding: var(--spacing-md) var(--spacing-lg);
            }
            
            .detail-section {
                padding: var(--spacing-md);
            }
            
            .bulk-actions-toolbar {
                padding: var(--spacing-sm) var(--spacing-md);
            }
            
            .bulk-actions-content {
                flex-direction: column;
                gap: var(--spacing-md);
                align-items: stretch;
            }
            
            .bulk-actions-left {
                flex-direction: column;
                gap: var(--spacing-sm);
                align-items: center;
            }
            
            .bulk-actions-right {
                justify-content: center;
            }
            
            .selected-count {
                font-size: var(--font-size-xs);
            }
            
            .user-name {
                font-size: var(--font-size-sm);
            }
            
            .contact-item {
                font-size: var(--font-size-xs);
            }
            
            .status-badge {
                font-size: var(--font-size-xs);
                padding: 2px var(--spacing-xs);
            }
        }
    </style>
</body>
</html>