# Profile API Dokumentation

## Übersicht

Die Profile-API bietet Endpunkte für:

- Benachrichtigungen abrufen und als gelesen markieren
- Activity Log (Aktivitätsprotokoll) abrufen
- Termine stornieren

---

## 1. Benachrichtigungen

### GET /api/get_user_notifications.php

**Beschreibung:** Ruft alle Benachrichtigungen für den eingeloggten Benutzer ab.

**Anfrage:**

- Session-basiert (benötigt `$_SESSION['user_logged_in']`)

**Response:**

```json
{
  "success": true,
  "notifications": [
    {
      "id": 1,
      "user_id": 5,
      "type": "appointment",
      "title": "Termin bestätigt",
      "message": "Ihr Termin am 15.03.2024 wurde bestätigt.",
      "is_read": 0,
      "created_at": "2024-03-10 10:00:00"
    }
  ],
  "unread_count": 3
}
```

**Fehler:**

- `401 Unauthorized` - Benutzer nicht eingeloggt
- `500 Internal Server Error` - Datenbankfehler

---

### POST /api/mark_notification_read.php

**Beschreibung:** Markiert eine Benachrichtigung als gelesen.

**Anfrage:**

```json
{
  "notification_id": 1
}
```

**Response:**

```json
{
  "success": true,
  "message": "Benachrichtigung als gelesen markiert"
}
```

**Fehler:**

- `401 Unauthorized` - Benutzer nicht eingeloggt
- `400 Bad Request` - Ungültige ID oder Benachrichtigung gehört nicht dem Benutzer

---

## 2. Activity Log

### GET /api/get_activity_log.php

**Beschreibung:** Ruft das Aktivitätsprotokoll des Benutzers ab.

**Anfrage-Parameter:**

- `limit` (optional, default: 20) - Anzahl der Einträge
- `offset` (optional, default: 0) - Offset für Pagination

**Response:**

```json
{
  "success": true,
  "activities": [
    {
      "id": 1,
      "user_id": 5,
      "action_type": "vehicle_added",
      "description": "Fahrzeug M-AB 123 hinzugefügt",
      "metadata": null,
      "created_at": "2024-03-10 10:00:00"
    }
  ],
  "total_count": 25,
  "stats": [
    {
      "action_type": "vehicle_added",
      "count": 5
    }
  ],
  "limit": 20,
  "offset": 0
}
```

**Action Types:**

- `profile_updated` - Profil aktualisiert
- `vehicle_added` - Fahrzeug hinzugefügt
- `vehicle_updated` - Fahrzeug bearbeitet
- `vehicle_deleted` - Fahrzeug gelöscht
- `appointment_created` - Termin erstellt
- `appointment_cancelled` - Termin storniert
- `appointment_rescheduled` - Termin verschoben
- `document_uploaded` - Dokument hochgeladen
- `password_changed` - Passwort geändert
- `login` - Einloggen
- `logout` - Ausloggen

**Fehler:**

- `401 Unauthorized` - Benutzer nicht eingeloggt
- `500 Internal Server Error` - Datenbankfehler

---

## 3. Termin stornieren

### POST /api/cancel_appointment.php

**Beschreibung:** Storniert einen Termin des Benutzers.

**Anfrage:**

```json
{
  "appointment_id": 10
}
```

**Response:**

```json
{
  "success": true,
  "message": "Termin erfolgreich storniert"
}
```

**Validierungen:**

- Termin muss dem Benutzer gehören
- Termin darf nicht in der Vergangenheit liegen
- Termin wird als `is_active = 0` und `status = 'cancelled'` markiert

**Automatisch erstellt:**

- Activity Log Eintrag (`appointment_cancelled`)
- Benachrichtigung für den Benutzer

**Fehler:**

- `401 Unauthorized` - Benutzer nicht eingeloggt
- `400 Bad Request` - Termin nicht gefunden, bereits vergangen, oder nicht autorisiert

---

## Datenbank-Tabellen

### user_notifications

```sql
CREATE TABLE user_notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    type ENUM('system', 'appointment', 'verification', 'warning', 'success'),
    title VARCHAR(255),
    message TEXT,
    is_read BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

### user_activity_log

```sql
CREATE TABLE user_activity_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    action_type ENUM(
        'profile_updated',
        'vehicle_added',
        'vehicle_updated',
        'vehicle_deleted',
        'appointment_created',
        'appointment_cancelled',
        'appointment_rescheduled',
        'document_uploaded',
        'password_changed',
        'login',
        'logout'
    ) NOT NULL,
    description TEXT NOT NULL,
    metadata JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

---

## Beispiel-Integration

### Benachrichtigungen laden (JavaScript)

```javascript
async function loadNotifications() {
  const response = await fetch("api/get_user_notifications.php");
  const data = await response.json();

  if (data.success) {
    console.log("Ungelesene Benachrichtigungen:", data.unread_count);
    console.log("Alle Benachrichtigungen:", data.notifications);
  }
}
```

### Benachrichtigung als gelesen markieren

```javascript
async function markAsRead(notificationId) {
  const response = await fetch("api/mark_notification_read.php", {
    method: "POST",
    headers: { "Content-Type": "application/json" },
    body: JSON.stringify({ notification_id: notificationId }),
  });

  const data = await response.json();
  if (data.success) {
    console.log("Benachrichtigung als gelesen markiert");
  }
}
```

### Activity Log laden

```javascript
async function loadActivityLog(limit = 20, offset = 0) {
  const response = await fetch(
    `api/get_activity_log.php?limit=${limit}&offset=${offset}`
  );
  const data = await response.json();

  if (data.success) {
    console.log("Aktivitäten:", data.activities);
    console.log("Gesamt:", data.total_count);
    console.log("Statistiken:", data.stats);
  }
}
```

---

## Sicherheit

Alle Endpunkte erfordern:

- Gültige Session (`$_SESSION['user_logged_in']`)
- Jeder Benutzer kann nur seine eigenen Daten abrufen
- Validierung gegen SQL-Injection durch prepared statements
- CORS-Headers für API-Zugriff
