<?php
/**
 * Volvo ID OAuth 2.0 / OpenID Connect Utility
 * PKCE, Token-Management, API-Calls
 */

// ENV-Loader: Lade .env falls vorhanden, sonst getenv()
function volvoLoadEnv($key, $default = null) {
    static $envLoaded = false;
    static $envData = [];
    
    if (!$envLoaded) {
        $envFile = __DIR__ . '/.env';
        if (file_exists($envFile)) {
            $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            foreach ($lines as $line) {
                if (strpos(trim($line), '#') === 0) continue; // Skip comments
                if (strpos($line, '=') === false) continue;
                [$key, $value] = explode('=', $line, 2);
                $envData[trim($key)] = trim($value, " \t\n\r\0\x0B\"'");
            }
        }
        $envLoaded = true;
    }
    
    return $envData[$key] ?? getenv($key) ?: $default;
}

// Volvo OAuth Konfiguration
function volvoGetConfig() {
    return [
        'client_id' => volvoLoadEnv('VOLVO_CLIENT_ID'),
        'client_secret' => volvoLoadEnv('VOLVO_CLIENT_SECRET'),
        'redirect_uri' => volvoLoadEnv('VOLVO_REDIRECT_URI', 'https://be-automobileapps.de/auth/volvo/callback.php'),
        'scopes' => volvoLoadEnv('VOLVO_SCOPES', 'openid offline_access connected_vehicle:read'),
        'authorize_url' => 'https://volvoid.eu.volvocars.com/as/authorization.oauth2',
        'token_url' => 'https://volvoid.eu.volvocars.com/as/token.oauth2',
        'vcc_api_key' => volvoLoadEnv('VOLVO_VCC_API_KEY'),
        'vcc_api_base' => 'https://api.volvocars.com',
        'demo_enabled' => volvoLoadEnv('VOLVO_DEMO_ENABLED', 'false') === 'true',
        'demo_access_token' => volvoLoadEnv('VOLVO_DEMO_ACCESS_TOKEN'),
        'refresh_skew' => (int)volvoLoadEnv('VOLVO_REFRESH_SKEW', '90')
    ];
}

// Sichere Session-Konfiguration
function volvoStartSession() {
    if (session_status() === PHP_SESSION_NONE) {
        ini_set('session.cookie_httponly', '1');
        ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) ? '1' : '0');
        ini_set('session.cookie_samesite', 'Lax');
        session_start();
    }
}

// PKCE: Code Verifier generieren
function volvoGenerateCodeVerifier() {
    return bin2hex(random_bytes(32));
}

// PKCE: Code Challenge aus Verifier (S256)
function volvoGenerateCodeChallenge($verifier) {
    return rtrim(strtr(base64_encode(hash('sha256', $verifier, true)), '+/', '-_'), '=');
}

// State-Token für CSRF-Schutz
function volvoGenerateState() {
    return bin2hex(random_bytes(16));
}

// State validieren
function volvoValidateState($state) {
    volvoStartSession();
    $storedState = $_SESSION['volvo_oauth_state'] ?? null;
    unset($_SESSION['volvo_oauth_state']);
    return $state && $storedState && hash_equals($storedState, $state);
}

// Volvo Authorization URL bauen
function volvoBuildAuthorizeUrl($demo = false) {
    $config = volvoGetConfig();
    
    // Demo-Modus: Verwende Demo-Token direkt
    if ($demo && $config['demo_enabled'] && $config['demo_access_token']) {
        volvoStartSession();
        $_SESSION['volvo_access'] = $config['demo_access_token'];
        $_SESSION['volvo_exp'] = time() + 3600; // 1h Demo
        $_SESSION['volvo_refresh'] = null;
        return null; // Redirect nicht nötig
    }
    
    volvoStartSession();
    
    $codeVerifier = volvoGenerateCodeVerifier();
    $codeChallenge = volvoGenerateCodeChallenge($codeVerifier);
    $state = volvoGenerateState();
    
    $_SESSION['volvo_code_verifier'] = $codeVerifier;
    $_SESSION['volvo_oauth_state'] = $state;
    
    $params = [
        'client_id' => $config['client_id'],
        'response_type' => 'code',
        'redirect_uri' => $config['redirect_uri'],
        'scope' => $config['scopes'],
        'code_challenge' => $codeChallenge,
        'code_challenge_method' => 'S256',
        'state' => $state
    ];
    
    return $config['authorize_url'] . '?' . http_build_query($params);
}

// Token-Tausch (Authorization Code → Access Token)
function volvoExchangeCodeForTokens($code, $codeVerifier) {
    $config = volvoGetConfig();
    
    $data = [
        'grant_type' => 'authorization_code',
        'code' => $code,
        'redirect_uri' => $config['redirect_uri'],
        'client_id' => $config['client_id'],
        'client_secret' => $config['client_secret'],
        'code_verifier' => $codeVerifier
    ];
    
    $ch = curl_init($config['token_url']);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception('cURL Fehler: ' . $error);
    }
    
    if ($httpCode !== 200) {
        error_log("Volvo Token Exchange failed: HTTP $httpCode - $response");
        throw new Exception('Token-Tausch fehlgeschlagen');
    }
    
    $tokens = json_decode($response, true);
    if (!$tokens || !isset($tokens['access_token'])) {
        throw new Exception('Ungültige Token-Antwort');
    }
    
    return $tokens;
}

// Token-Refresh mit Rotation
function volvoRefreshTokens($refreshToken) {
    $config = volvoGetConfig();
    
    $data = [
        'grant_type' => 'refresh_token',
        'refresh_token' => $refreshToken,
        'client_id' => $config['client_id'],
        'client_secret' => $config['client_secret']
    ];
    
    $ch = curl_init($config['token_url']);
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => ['Content-Type: application/x-www-form-urlencoded'],
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception('cURL Fehler: ' . $error);
    }
    
    if ($httpCode !== 200) {
        error_log("Volvo Refresh failed: HTTP $httpCode - $response");
        throw new Exception('Token-Refresh fehlgeschlagen');
    }
    
    $tokens = json_decode($response, true);
    if (!$tokens || !isset($tokens['access_token'])) {
        throw new Exception('Ungültige Refresh-Antwort');
    }
    
    return $tokens;
}

// Prüft ob Access Token gültig ist und refreshed falls nötig
function volvoEnsureAccess() {
    volvoStartSession();
    $config = volvoGetConfig();
    
    $accessToken = $_SESSION['volvo_access'] ?? null;
    $refreshToken = $_SESSION['volvo_refresh'] ?? null;
    $expiresAt = $_SESSION['volvo_exp'] ?? 0;
    
    $now = time();
    $refreshBefore = $expiresAt - $config['refresh_skew'];
    
    // Token abgelaufen oder läuft bald ab → Refresh
    if (!$accessToken || $now >= $refreshBefore) {
        if (!$refreshToken) {
            return null; // Kein Refresh-Token → Neu-Login nötig
        }
        
        try {
            $tokens = volvoRefreshTokens($refreshToken);
            
            // Refresh-Rotation: Neues refresh_token speichern falls vorhanden
            $_SESSION['volvo_access'] = $tokens['access_token'];
            $_SESSION['volvo_exp'] = $now + ($tokens['expires_in'] ?? 3600);
            if (isset($tokens['refresh_token'])) {
                $_SESSION['volvo_refresh'] = $tokens['refresh_token']; // Rotation
            }
            
            return $tokens['access_token'];
        } catch (Exception $e) {
            error_log('Volvo Refresh Error: ' . $e->getMessage());
            // Session invalidieren
            unset($_SESSION['volvo_access'], $_SESSION['volvo_refresh'], $_SESSION['volvo_exp']);
            return null;
        }
    }
    
    return $accessToken;
}

// Volvo Connected Vehicle API Aufruf
function volvoApiGet($endpoint, $accessToken = null) {
    $config = volvoGetConfig();
    
    if (!$accessToken) {
        $accessToken = volvoEnsureAccess();
        if (!$accessToken) {
            throw new Exception('Kein gültiger Access Token');
        }
    }
    
    $url = $config['vcc_api_base'] . '/' . ltrim($endpoint, '/');
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'Authorization: Bearer ' . $accessToken,
            'vcc-api-key: ' . $config['vcc_api_key'],
            'Content-Type: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => true
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception('cURL Fehler: ' . $error);
    }
    
    if ($httpCode !== 200) {
        error_log("Volvo API failed: HTTP $httpCode - $response");
        throw new Exception('API-Aufruf fehlgeschlagen: HTTP ' . $httpCode);
    }
    
    $data = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Ungültige JSON-Antwort');
    }
    
    return $data;
}

