"""
Performance-Test für Kunden-Endpoints
Testet die Laufzeit für "Deniz Uludag"
"""

import requests
import time
import json
from datetime import datetime

# API-Konfiguration
API_BASE_URL = "http://localhost:5000/api"  # Anpassen falls nötig
API_KEY = "bungert-p2-api-key-2024-default"  # Falls erforderlich

def test_endpoint(name, url, params=None, headers=None):
    """Testet einen Endpoint und misst die Zeit"""
    start_time = time.time()
    try:
        if headers is None:
            headers = {}
        if API_KEY:
            headers['X-API-Key'] = API_KEY
        
        response = requests.get(url, params=params, headers=headers, timeout=30)
        elapsed = time.time() - start_time
        
        status = response.status_code
        try:
            data = response.json()
            count = data.get('count', len(data.get('data', [])) if isinstance(data.get('data'), list) else 1)
        except:
            data = None
            count = 0
        
        return {
            'name': name,
            'url': url,
            'status': status,
            'time': elapsed,
            'count': count,
            'success': status == 200
        }
    except Exception as e:
        elapsed = time.time() - start_time
        return {
            'name': name,
            'url': url,
            'status': 'ERROR',
            'time': elapsed,
            'error': str(e),
            'success': False
        }

def main():
    print("=" * 80)
    print("PERFORMANCE-TEST: Kunden-Daten für 'Deniz Uludag'")
    print("=" * 80)
    print(f"Startzeit: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
    
    results = []
    
    # Test 1: Suche nach "Deniz Uludag"
    print("🔍 Test 1: Suche nach 'Deniz Uludag'...")
    result1 = test_endpoint(
        "Kunden-Suche (Deniz Uludag)",
        f"{API_BASE_URL}/kunden",
        params={'search': 'Deniz Uludag', 'limit': 100}
    )
    results.append(result1)
    print(f"   ⏱️  Zeit: {result1['time']:.3f}s")
    print(f"   📊 Status: {result1['status']}")
    print(f"   📈 Gefundene Kunden: {result1.get('count', 0)}")
    if result1.get('error'):
        print(f"   ❌ Fehler: {result1['error']}")
    print()
    
    # Wenn Suche erfolgreich, hole Kunden-ID und teste weitere Endpoints
    if result1['success']:
        try:
            # Hole die Kunden-ID aus dem ersten Ergebnis
            search_url = f"{API_BASE_URL}/kunden"
            headers = {'X-API-Key': API_KEY} if API_KEY else {}
            response = requests.get(
                search_url, 
                params={'search': 'Deniz Uludag', 'limit': 1},
                headers=headers,
                timeout=30
            )
            
            if response.status_code == 200:
                data = response.json()
                if data.get('data') and len(data['data']) > 0:
                    kunde = data['data'][0]
                    kunden_adr_id = kunde.get('KundenAdrID')
                    
                    if kunden_adr_id:
                        print(f"✅ Kunden-ID gefunden: {kunden_adr_id}\n")
                        
                        # Test 2: Kunde nach ID abrufen
                        print("🔍 Test 2: Kunde nach ID abrufen...")
                        result2 = test_endpoint(
                            f"Kunde nach ID ({kunden_adr_id})",
                            f"{API_BASE_URL}/kunden/by-id/{kunden_adr_id}"
                        )
                        results.append(result2)
                        print(f"   ⏱️  Zeit: {result2['time']:.3f}s")
                        print(f"   📊 Status: {result2['status']}")
                        if result2.get('error'):
                            print(f"   ❌ Fehler: {result2['error']}")
                        print()
                        
                        # Test 3: Suche nach Aufträgen über Kennzeichen (falls vorhanden)
                        # Dafür müssten wir erst die Aufträge des Kunden finden
                        print("🔍 Test 3: Suche nach Aufträgen des Kunden...")
                        # Wir müssen erst alle Aufträge durchsuchen - das ist langsam!
                        # Besser: Endpoint für Aufträge nach KundenID erstellen
                        print("   ⚠️  Hinweis: Kein direkter Endpoint für Aufträge nach KundenID verfügbar")
                        print("   💡 Empfehlung: Endpoint /api/auftraege/by-customer/<kunden_adr_id> hinzufügen")
                        print()
        except Exception as e:
            print(f"❌ Fehler beim Abrufen der Kunden-ID: {e}\n")
    
    # Test 4: Alle Kunden (ohne Suche) - Performance-Baseline
    print("🔍 Test 4: Alle Kunden abrufen (Limit 10, Baseline)...")
    result4 = test_endpoint(
        "Alle Kunden (Limit 10)",
        f"{API_BASE_URL}/kunden",
        params={'limit': 10}
    )
    results.append(result4)
    print(f"   ⏱️  Zeit: {result4['time']:.3f}s")
    print(f"   📊 Status: {result4['status']}")
    print(f"   📈 Gefundene Kunden: {result4.get('count', 0)}")
    if result4.get('error'):
        print(f"   ❌ Fehler: {result4['error']}")
    print()
    
    # Zusammenfassung
    print("=" * 80)
    print("ZUSAMMENFASSUNG")
    print("=" * 80)
    
    total_time = sum(r['time'] for r in results)
    successful_tests = sum(1 for r in results if r['success'])
    
    print(f"\n✅ Erfolgreiche Tests: {successful_tests}/{len(results)}")
    print(f"⏱️  Gesamtzeit: {total_time:.3f}s")
    print(f"⏱️  Durchschnittliche Zeit pro Test: {total_time/len(results):.3f}s\n")
    
    print("Detaillierte Ergebnisse:")
    print("-" * 80)
    for r in results:
        status_icon = "✅" if r['success'] else "❌"
        print(f"{status_icon} {r['name']}")
        print(f"   Zeit: {r['time']:.3f}s | Status: {r['status']}")
        if r.get('count'):
            print(f"   Ergebnisse: {r['count']}")
        if r.get('error'):
            print(f"   Fehler: {r['error']}")
        print()
    
    # Performance-Bewertung
    print("=" * 80)
    print("PERFORMANCE-BEWERTUNG")
    print("=" * 80)
    
    search_time = results[0]['time'] if results else 0
    
    if search_time < 0.5:
        print("✅ AUSGEZEICHNET: Suche < 0.5s")
    elif search_time < 1.0:
        print("✅ GUT: Suche < 1.0s")
    elif search_time < 2.0:
        print("⚠️  AKZEPTABEL: Suche < 2.0s")
    else:
        print("❌ LANGSAM: Suche > 2.0s - Optimierung empfohlen!")
    
    print(f"\n💡 Empfehlungen:")
    print("   1. Index auf KundenName1 und KundenVorname prüfen")
    print("   2. Query-Optimierung: DISTINCT kann langsam sein")
    print("   3. Caching für häufige Suchen implementieren")
    print("   4. Endpoint für Aufträge nach KundenID hinzufügen")
    print("   5. Pagination für große Ergebnislisten")
    
    print("\n" + "=" * 80)
    print(f"Test beendet: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print("=" * 80)

if __name__ == '__main__':
    main()
















