"""
Performance-Test für ALLE Endpoints in app.py_backendp2
Testet jeden Endpoint und misst die Laufzeit
"""

import requests
import time
import json
from datetime import datetime
from typing import Dict, List

# API-Konfiguration
API_BASE_URL = "http://localhost:5000/api"  # Anpassen falls nötig
API_KEY = "bungert-p2-api-key-2024-default"  # Falls erforderlich

def test_endpoint(name: str, method: str, url: str, params: dict = None, headers: dict = None, data: dict = None) -> Dict:
    """Testet einen Endpoint und misst die Zeit"""
    start_time = time.time()
    try:
        if headers is None:
            headers = {}
        if API_KEY:
            headers['X-API-Key'] = API_KEY
        headers['Content-Type'] = 'application/json'
        
        if method == 'GET':
            response = requests.get(url, params=params, headers=headers, timeout=30)
        elif method == 'POST':
            response = requests.post(url, json=data, headers=headers, timeout=30)
        elif method == 'PUT':
            response = requests.put(url, json=data, headers=headers, timeout=30)
        else:
            return {'name': name, 'error': f'Unbekannte Methode: {method}', 'success': False}
        
        elapsed = time.time() - start_time
        
        status = response.status_code
        try:
            response_data = response.json()
            count = response_data.get('count', len(response_data.get('data', [])) if isinstance(response_data.get('data'), list) else 1)
            data_size = len(json.dumps(response_data))
        except:
            response_data = None
            count = 0
            data_size = 0
        
        return {
            'name': name,
            'url': url,
            'method': method,
            'status': status,
            'time': elapsed,
            'count': count,
            'data_size_kb': round(data_size / 1024, 2),
            'success': status == 200,
            'response_preview': str(response_data)[:200] if response_data else None
        }
    except requests.exceptions.Timeout:
        elapsed = time.time() - start_time
        return {
            'name': name,
            'url': url,
            'method': method,
            'status': 'TIMEOUT',
            'time': elapsed,
            'error': 'Request timeout after 30s',
            'success': False
        }
    except Exception as e:
        elapsed = time.time() - start_time
        return {
            'name': name,
            'url': url,
            'method': method,
            'status': 'ERROR',
            'time': elapsed,
            'error': str(e),
            'success': False
        }

def main():
    print("=" * 80)
    print("PERFORMANCE-TEST: ALLE Backend-API Endpoints")
    print("=" * 80)
    print(f"Startzeit: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
    print(f"API Base URL: {API_BASE_URL}\n")
    
    results = []
    
    # ============================================================================
    # SYSTEM ENDPOINTS
    # ============================================================================
    print("🔧 SYSTEM ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Health Check",
        "GET",
        f"{API_BASE_URL}/system/health"
    ))
    
    results.append(test_endpoint(
        "API Key Info",
        "GET",
        f"{API_BASE_URL}/system/apikey"
    ))
    
    # ============================================================================
    # TERMINE ENDPOINTS
    # ============================================================================
    print("\n📅 TERMINE ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Termine Liste (30 Tage)",
        "GET",
        f"{API_BASE_URL}/termine",
        params={'days': 30}
    ))
    
    results.append(test_endpoint(
        "Termine Liste (7 Tage)",
        "GET",
        f"{API_BASE_URL}/termine",
        params={'days': 7}
    ))
    
    # Test mit Beispiel-Auftragsnummer (muss angepasst werden)
    results.append(test_endpoint(
        "Termin Details (Beispiel)",
        "GET",
        f"{API_BASE_URL}/termine/12345"  # Beispiel - wird wahrscheinlich 404 geben
    ))
    
    # ============================================================================
    # AUFTRÄGE ENDPOINTS
    # ============================================================================
    print("\n📋 AUFTRÄGE ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Aufträge Liste (30 Tage)",
        "GET",
        f"{API_BASE_URL}/auftraege",
        params={'days': 30}
    ))
    
    results.append(test_endpoint(
        "Aufträge Liste (7 Tage)",
        "GET",
        f"{API_BASE_URL}/auftraege",
        params={'days': 7}
    ))
    
    results.append(test_endpoint(
        "Auftrag nach ID (Beispiel)",
        "GET",
        f"{API_BASE_URL}/auftraege/12345"  # Beispiel
    ))
    
    results.append(test_endpoint(
        "Aufträge nach Kennzeichen (Beispiel)",
        "GET",
        f"{API_BASE_URL}/auftraege/by-license/M-AB%20123",  # Beispiel
        params={'days': 2000}
    ))
    
    results.append(test_endpoint(
        "Auftragspositionen (Beispiel)",
        "GET",
        f"{API_BASE_URL}/auftraege/positionen/12345"  # Beispiel
    ))
    
    # ============================================================================
    # KUNDEN ENDPOINTS
    # ============================================================================
    print("\n👤 KUNDEN ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Kunden Liste (Limit 10)",
        "GET",
        f"{API_BASE_URL}/kunden",
        params={'limit': 10}
    ))
    
    results.append(test_endpoint(
        "Kunden Suche: 'Deniz Uludag'",
        "GET",
        f"{API_BASE_URL}/kunden",
        params={'search': 'Deniz Uludag', 'limit': 10}
    ))
    
    results.append(test_endpoint(
        "Kunden Suche: 'Uludag'",
        "GET",
        f"{API_BASE_URL}/kunden",
        params={'search': 'Uludag', 'limit': 10}
    ))
    
    results.append(test_endpoint(
        "Kunde nach ID (Beispiel)",
        "GET",
        f"{API_BASE_URL}/kunden/by-id/12345"  # Beispiel
    ))
    
    # ============================================================================
    # STEMPELZEITEN ENDPOINTS
    # ============================================================================
    print("\n⏰ STEMPELZEITEN ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Stempelzeiten (7 Tage)",
        "GET",
        f"{API_BASE_URL}/stempelzeiten",
        params={'days': 7}
    ))
    
    results.append(test_endpoint(
        "Stempelzeiten für Auftrag (Beispiel)",
        "GET",
        f"{API_BASE_URL}/stempelzeiten",
        params={'auftrag_nr': 12345}
    ))
    
    # ============================================================================
    # FINANZEN ENDPOINTS
    # ============================================================================
    print("\n💰 FINANZEN ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Umsatz (30 Tage)",
        "GET",
        f"{API_BASE_URL}/finanzen/umsatz",
        params={'days': 30}
    ))
    
    results.append(test_endpoint(
        "Umsatz (7 Tage)",
        "GET",
        f"{API_BASE_URL}/finanzen/umsatz",
        params={'days': 7}
    ))
    
    # ============================================================================
    # FORMULARE ENDPOINTS
    # ============================================================================
    print("\n📄 FORMULARE ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Belegarten Liste",
        "GET",
        f"{API_BASE_URL}/formulare/belegarten"
    ))
    
    results.append(test_endpoint(
        "Belegarten für Auftrag (Beispiel)",
        "GET",
        f"{API_BASE_URL}/formulare/belegarten/auftrag/12345"  # Beispiel
    ))
    
    # ============================================================================
    # TEILE ENDPOINTS
    # ============================================================================
    print("\n🔧 TEILE ENDPOINTS")
    print("-" * 80)
    
    results.append(test_endpoint(
        "Teil nach Nummer (Beispiel)",
        "GET",
        f"{API_BASE_URL}/teile/RE123456"  # Beispiel
    ))
    
    # ============================================================================
    # ZUSAMMENFASSUNG
    # ============================================================================
    print("\n" + "=" * 80)
    print("ZUSAMMENFASSUNG")
    print("=" * 80)
    
    successful = [r for r in results if r['success']]
    failed = [r for r in results if not r['success']]
    
    print(f"\n✅ Erfolgreiche Tests: {len(successful)}/{len(results)}")
    print(f"❌ Fehlgeschlagene Tests: {len(failed)}/{len(results)}")
    
    if successful:
        total_time = sum(r['time'] for r in successful)
        avg_time = total_time / len(successful)
        max_time = max(r['time'] for r in successful)
        min_time = min(r['time'] for r in successful)
        
        print(f"\n⏱️  Performance-Statistiken (nur erfolgreiche):")
        print(f"   Gesamtzeit: {total_time:.3f}s")
        print(f"   Durchschnitt: {avg_time:.3f}s")
        print(f"   Schnellster: {min_time:.3f}s")
        print(f"   Langsamster: {max_time:.3f}s")
        
        # Top 5 langsamste
        slowest = sorted(successful, key=lambda x: x['time'], reverse=True)[:5]
        print(f"\n🐌 Top 5 langsamste Endpoints:")
        for i, r in enumerate(slowest, 1):
            print(f"   {i}. {r['name']}: {r['time']:.3f}s ({r.get('count', 0)} Ergebnisse, {r.get('data_size_kb', 0)} KB)")
        
        # Schnellste
        fastest = sorted(successful, key=lambda x: x['time'])[:5]
        print(f"\n⚡ Top 5 schnellste Endpoints:")
        for i, r in enumerate(fastest, 1):
            print(f"   {i}. {r['name']}: {r['time']:.3f}s ({r.get('count', 0)} Ergebnisse)")
    
    if failed:
        print(f"\n❌ Fehlgeschlagene Endpoints:")
        for r in failed:
            print(f"   - {r['name']}: {r.get('error', r.get('status', 'Unknown'))}")
    
    # Performance-Bewertung
    print("\n" + "=" * 80)
    print("PERFORMANCE-BEWERTUNG")
    print("=" * 80)
    
    if successful:
        slow_endpoints = [r for r in successful if r['time'] > 2.0]
        medium_endpoints = [r for r in successful if 0.5 < r['time'] <= 2.0]
        fast_endpoints = [r for r in successful if r['time'] <= 0.5]
        
        print(f"\n⚡ Sehr schnell (< 0.5s): {len(fast_endpoints)}")
        print(f"✅ Gut (0.5-2.0s): {len(medium_endpoints)}")
        print(f"⚠️  Langsam (> 2.0s): {len(slow_endpoints)}")
        
        if slow_endpoints:
            print(f"\n💡 Optimierungsempfehlungen für langsame Endpoints:")
            for r in slow_endpoints:
                print(f"   - {r['name']}: {r['time']:.3f}s")
                print(f"     → Query-Optimierung, Caching, oder Limit reduzieren")
    
    print("\n" + "=" * 80)
    print(f"Test beendet: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print("=" * 80)
    
    # Speichere Ergebnisse als JSON
    with open('backend_performance_results.json', 'w', encoding='utf-8') as f:
        json.dump({
            'timestamp': datetime.now().isoformat(),
            'results': results,
            'summary': {
                'total': len(results),
                'successful': len(successful),
                'failed': len(failed)
            }
        }, f, indent=2, ensure_ascii=False)
    
    print("\n📊 Ergebnisse gespeichert in: backend_performance_results.json")

if __name__ == '__main__':
    main()
















