<?php
/**
 * Einrichtungsassistent - Boxenstop
 */

session_start();
require_once '../config/database.php';

// Check if setup is already completed
$db = new Database();
$conn = $db->getConnection();

$stmt = $conn->prepare("SELECT setting_value FROM settings WHERE setting_key = 'setup_completed'");
$stmt->execute();
$setup_completed = $stmt->fetchColumn();

if ($setup_completed) {
    header('Location: ../index.php');
    exit;
}

$step = $_GET['step'] ?? 1;
$step = max(1, min(5, (int)$step)); // Ensure step is between 1 and 5

$error_message = '';
$success_message = '';

// Handle form submissions
if ($_POST) {
    switch ($step) {
        case 1:
            handleDatabaseSetup($conn);
            break;
        case 2:
            handleCompanySettings($conn);
            break;
        case 3:
            handleAdminAccount($conn);
            break;
        case 4:
            handleFirstEvent($conn);
            break;
        case 5:
            completeSetup($conn);
            break;
    }
}

function handleDatabaseSetup($conn) {
    global $error_message, $success_message;
    
    try {
        // Test database connection
        $conn->query("SELECT 1");
        $success_message = 'Datenbankverbindung erfolgreich!';
        header('Location: ?step=2');
        exit;
    } catch (Exception $e) {
        $error_message = 'Datenbankverbindung fehlgeschlagen: ' . $e->getMessage();
    }
}

function handleCompanySettings($conn) {
    global $error_message, $success_message;
    
    try {
        $settings = [
            'company_name' => $_POST['company_name'],
            'contact_phone' => $_POST['contact_phone'],
            'contact_email' => $_POST['contact_email']
        ];
        
        foreach ($settings as $key => $value) {
            $stmt = $conn->prepare("UPDATE settings SET setting_value = ? WHERE setting_key = ?");
            $stmt->execute([$value, $key]);
        }
        
        $success_message = 'Firmeneinstellungen gespeichert!';
        header('Location: ?step=3');
        exit;
    } catch (Exception $e) {
        $error_message = 'Fehler beim Speichern: ' . $e->getMessage();
    }
}

function handleAdminAccount($conn) {
    global $error_message, $success_message;
    
    try {
        $username = $_POST['username'];
        $email = $_POST['email'];
        $password = $_POST['password'];
        $confirm_password = $_POST['confirm_password'];
        
        if ($password !== $confirm_password) {
            throw new Exception('Passwörter stimmen nicht überein');
        }
        
        if (strlen($password) < 6) {
            throw new Exception('Passwort muss mindestens 6 Zeichen lang sein');
        }
        
        $password_hash = password_hash($password, PASSWORD_DEFAULT);
        
        $stmt = $conn->prepare("UPDATE admin_users SET username = ?, email = ?, password_hash = ? WHERE id = 1");
        $stmt->execute([$username, $email, $password_hash]);
        
        $success_message = 'Admin-Konto erstellt!';
        header('Location: ?step=4');
        exit;
    } catch (Exception $e) {
        $error_message = 'Fehler beim Erstellen des Admin-Kontos: ' . $e->getMessage();
    }
}

function handleFirstEvent($conn) {
    global $error_message, $success_message;
    
    try {
        $event_data = [
            'name' => $_POST['event_name'],
            'event_date' => $_POST['event_date'],
            'start_time' => $_POST['start_time'],
            'end_time' => $_POST['end_time'],
            'slot_duration' => $_POST['slot_duration'],
            'max_monteure' => $_POST['max_monteure']
        ];
        
        $stmt = $conn->prepare("
            INSERT INTO events (name, event_date, start_time, end_time, slot_duration, max_monteure, is_active)
            VALUES (?, ?, ?, ?, ?, ?, 1)
        ");
        $stmt->execute(array_values($event_data));
        
        $success_message = 'Erstes Event erstellt!';
        header('Location: ?step=5');
        exit;
    } catch (Exception $e) {
        $error_message = 'Fehler beim Erstellen des Events: ' . $e->getMessage();
    }
}

function completeSetup($conn) {
    try {
        $stmt = $conn->prepare("UPDATE settings SET setting_value = '1' WHERE setting_key = 'setup_completed'");
        $stmt->execute();
        
        header('Location: ../index.php?setup=completed');
        exit;
    } catch (Exception $e) {
        $error_message = 'Fehler beim Abschließen der Einrichtung: ' . $e->getMessage();
    }
}

// Get current settings
$current_settings = [];
$stmt = $conn->prepare("SELECT setting_key, setting_value FROM settings");
$stmt->execute();
while ($row = $stmt->fetch()) {
    $current_settings[$row['setting_key']] = $row['setting_value'];
}
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="googlebot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="bingbot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <title>Einrichtung - Boxenstop</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="setup-body">
    <div class="setup-container">
        <div class="setup-card">
            <!-- Header -->
            <div class="setup-header">
                <img src="../assets/images/volvo-logo.png" alt="Volvo" class="logo-img">
                <h1>Boxenstop Einrichtung</h1>
                <p>Willkommen! Lassen Sie uns Ihr Boxenstop-System einrichten.</p>
            </div>

            <!-- Progress Bar -->
            <div class="progress-bar">
                <div class="progress-steps">
                    <?php for ($i = 1; $i <= 5; $i++): ?>
                        <div class="step <?php echo $i <= $step ? 'completed' : ''; ?> <?php echo $i == $step ? 'current' : ''; ?>">
                            <div class="step-number"><?php echo $i; ?></div>
                            <div class="step-label">
                                <?php
                                $labels = [
                                    1 => 'Datenbank',
                                    2 => 'Firma',
                                    3 => 'Admin',
                                    4 => 'Event',
                                    5 => 'Fertig'
                                ];
                                echo $labels[$i];
                                ?>
                            </div>
                        </div>
                    <?php endfor; ?>
                </div>
                <div class="progress-line">
                    <div class="progress-fill" style="width: <?php echo (($step - 1) / 4) * 100; ?>%"></div>
                </div>
            </div>

            <!-- Messages -->
            <?php if ($error_message): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo $error_message; ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo $success_message; ?>
                </div>
            <?php endif; ?>

            <!-- Step Content -->
            <div class="step-content">
                <?php switch ($step): case 1: ?>
                    <!-- Step 1: Database Setup -->
                    <div class="step-form">
                        <h2>Datenbankverbindung prüfen</h2>
                        <p>Wir prüfen die Verbindung zu Ihrer MySQL-Datenbank.</p>
                        
                        <div class="info-box">
                            <i class="fas fa-info-circle"></i>
                            <div>
                                <strong>Hinweis:</strong> Stellen Sie sicher, dass die Datenbank "boxxenstopp" existiert und die Verbindungsparameter in der Datei <code>config/database.php</code> korrekt sind.
                            </div>
                        </div>
                        
                        <form method="POST">
                            <div class="form-actions">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-database"></i>
                                    Verbindung testen
                                </button>
                            </div>
                        </form>
                    </div>
                <?php break; case 2: ?>
                    <!-- Step 2: Company Settings -->
                    <div class="step-form">
                        <h2>Firmeneinstellungen</h2>
                        <p>Geben Sie die Informationen zu Ihrem Autohaus ein.</p>
                        
                        <form method="POST">
                            <div class="form-group">
                                <label for="company_name">Firmenname *</label>
                                <input type="text" id="company_name" name="company_name" required 
                                       value="<?php echo htmlspecialchars($current_settings['company_name'] ?? 'Volvo Autohaus'); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="contact_phone">Telefonnummer</label>
                                <input type="tel" id="contact_phone" name="contact_phone" 
                                       value="<?php echo htmlspecialchars($current_settings['contact_phone'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="contact_email">E-Mail-Adresse</label>
                                <input type="email" id="contact_email" name="contact_email" 
                                       value="<?php echo htmlspecialchars($current_settings['contact_email'] ?? ''); ?>">
                            </div>
                            
                            <div class="form-actions">
                                <a href="?step=1" class="btn btn-secondary">Zurück</a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i>
                                    Speichern & Weiter
                                </button>
                            </div>
                        </form>
                    </div>
                <?php break; case 3: ?>
                    <!-- Step 3: Admin Account -->
                    <div class="step-form">
                        <h2>Admin-Konto erstellen</h2>
                        <p>Erstellen Sie Ihr Administrator-Konto für den Zugang zum Admin-Bereich.</p>
                        
                        <form method="POST">
                            <div class="form-group">
                                <label for="username">Benutzername *</label>
                                <input type="text" id="username" name="username" required 
                                       value="admin">
                            </div>
                            
                            <div class="form-group">
                                <label for="email">E-Mail-Adresse *</label>
                                <input type="email" id="email" name="email" required 
                                       value="admin@volvo-autohaus.de">
                            </div>
                            
                            <div class="form-group">
                                <label for="password">Passwort *</label>
                                <input type="password" id="password" name="password" required 
                                       minlength="6">
                            </div>
                            
                            <div class="form-group">
                                <label for="confirm_password">Passwort bestätigen *</label>
                                <input type="password" id="confirm_password" name="confirm_password" required 
                                       minlength="6">
                            </div>
                            
                            <div class="form-actions">
                                <a href="?step=2" class="btn btn-secondary">Zurück</a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-user-plus"></i>
                                    Konto erstellen & Weiter
                                </button>
                            </div>
                        </form>
                    </div>
                <?php break; case 4: ?>
                    <!-- Step 4: First Event -->
                    <div class="step-form">
                        <h2>Erstes Boxenstop Event</h2>
                        <p>Erstellen Sie Ihr erstes Boxenstop Event.</p>
                        
                        <form method="POST">
                            <div class="form-grid">
                                <div class="form-group">
                                    <label for="event_name">Event Name *</label>
                                    <input type="text" id="event_name" name="event_name" required 
                                           value="Frühjahres Boxenstop 2024">
                                </div>
                                
                                <div class="form-group">
                                    <label for="event_date">Datum *</label>
                                    <input type="date" id="event_date" name="event_date" required 
                                           value="<?php echo date('Y-m-d', strtotime('+1 month')); ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="start_time">Startzeit *</label>
                                    <input type="time" id="start_time" name="start_time" required 
                                           value="08:00">
                                </div>
                                
                                <div class="form-group">
                                    <label for="end_time">Endzeit *</label>
                                    <input type="time" id="end_time" name="end_time" required 
                                           value="17:00">
                                </div>
                                
                                <div class="form-group">
                                    <label for="slot_duration">Slot-Dauer (Minuten) *</label>
                                    <input type="number" id="slot_duration" name="slot_duration" required 
                                           min="15" max="120" value="30">
                                </div>
                                
                                <div class="form-group">
                                    <label for="max_monteure">Max. Monteure *</label>
                                    <input type="number" id="max_monteure" name="max_monteure" required 
                                           min="1" max="20" value="5">
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <a href="?step=3" class="btn btn-secondary">Zurück</a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-calendar-plus"></i>
                                    Event erstellen & Weiter
                                </button>
                            </div>
                        </form>
                    </div>
                <?php break; case 5: ?>
                    <!-- Step 5: Complete -->
                    <div class="step-form">
                        <h2>Einrichtung abgeschlossen!</h2>
                        <p>Ihr Boxenstop-System ist jetzt einsatzbereit.</p>
                        
                        <div class="completion-info">
                            <div class="info-item">
                                <i class="fas fa-check-circle"></i>
                                <div>
                                    <h4>Datenbank</h4>
                                    <p>Erfolgreich verbunden</p>
                                </div>
                            </div>
                            
                            <div class="info-item">
                                <i class="fas fa-check-circle"></i>
                                <div>
                                    <h4>Firmeneinstellungen</h4>
                                    <p>Konfiguriert</p>
                                </div>
                            </div>
                            
                            <div class="info-item">
                                <i class="fas fa-check-circle"></i>
                                <div>
                                    <h4>Admin-Konto</h4>
                                    <p>Erstellt</p>
                                </div>
                            </div>
                            
                            <div class="info-item">
                                <i class="fas fa-check-circle"></i>
                                <div>
                                    <h4>Erstes Event</h4>
                                    <p>Angelegt</p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="next-steps">
                            <h3>Nächste Schritte:</h3>
                            <ul>
                                <li>Melden Sie sich im Admin-Bereich an</li>
                                <li>Erstellen Sie weitere Events</li>
                                <li>Konfigurieren Sie die Einstellungen</li>
                                <li>Testen Sie die Terminbuchung</li>
                            </ul>
                        </div>
                        
                        <div class="form-actions">
                            <a href="../admin/login.php" class="btn btn-primary">
                                <i class="fas fa-sign-in-alt"></i>
                                Zum Admin-Bereich
                            </a>
                            <a href="../index.php" class="btn btn-secondary">
                                <i class="fas fa-home"></i>
                                Zur Hauptseite
                            </a>
                        </div>
                    </div>
                <?php break; endswitch; ?>
            </div>
        </div>
    </div>

    <style>
        .setup-body {
            background: var(--racing-gradient);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }
        
        .setup-container {
            width: 100%;
            max-width: 800px;
        }
        
        .setup-card {
            background: white;
            border-radius: 20px;
            padding: 3rem;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.2);
            position: relative;
            overflow: hidden;
        }
        
        .setup-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 6px;
            background: var(--f1-gradient);
        }
        
        .setup-header {
            text-align: center;
            margin-bottom: 3rem;
        }
        
        .setup-header .logo-img {
            height: 80px;
            width: auto;
            margin-bottom: 1rem;
        }
        
        .setup-header h1 {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--volvo-blue);
            margin: 0 0 0.5rem 0;
        }
        
        .setup-header p {
            color: #666;
            font-size: 1.1rem;
            margin: 0;
        }
        
        .progress-bar {
            margin-bottom: 3rem;
        }
        
        .progress-steps {
            display: flex;
            justify-content: space-between;
            margin-bottom: 1rem;
        }
        
        .step {
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 0.5rem;
            position: relative;
        }
        
        .step-number {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: #e9ecef;
            color: #6c757d;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .step.completed .step-number {
            background: var(--f1-red);
            color: white;
        }
        
        .step.current .step-number {
            background: var(--volvo-blue);
            color: white;
        }
        
        .step-label {
            font-size: 0.9rem;
            font-weight: 500;
            color: #666;
            text-align: center;
        }
        
        .step.completed .step-label {
            color: var(--f1-red);
        }
        
        .step.current .step-label {
            color: var(--volvo-blue);
        }
        
        .progress-line {
            height: 4px;
            background: #e9ecef;
            border-radius: 2px;
            position: relative;
            margin-top: -20px;
            z-index: 1;
        }
        
        .progress-fill {
            height: 100%;
            background: var(--f1-gradient);
            border-radius: 2px;
            transition: width 0.3s ease;
        }
        
        .step-content {
            min-height: 400px;
        }
        
        .step-form h2 {
            font-size: 1.8rem;
            font-weight: 600;
            color: var(--volvo-blue);
            margin: 0 0 1rem 0;
        }
        
        .step-form p {
            color: #666;
            margin-bottom: 2rem;
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
        }
        
        .form-group label {
            font-weight: 600;
            color: var(--volvo-blue);
            margin-bottom: 0.5rem;
        }
        
        .form-group input {
            padding: 1rem;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.3s ease;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: var(--f1-red);
            box-shadow: 0 0 0 3px rgba(225, 6, 0, 0.1);
        }
        
        .form-actions {
            display: flex;
            gap: 1rem;
            justify-content: center;
            margin-top: 2rem;
        }
        
        .info-box {
            background: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 2rem;
            display: flex;
            align-items: flex-start;
            gap: 1rem;
        }
        
        .info-box i {
            color: #1976d2;
            margin-top: 0.25rem;
        }
        
        .completion-info {
            margin: 2rem 0;
        }
        
        .info-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 8px;
            margin-bottom: 1rem;
        }
        
        .info-item i {
            color: #28a745;
            font-size: 1.5rem;
        }
        
        .info-item h4 {
            margin: 0 0 0.25rem 0;
            color: var(--volvo-blue);
        }
        
        .info-item p {
            margin: 0;
            color: #666;
        }
        
        .next-steps {
            background: #f8f9fa;
            padding: 1.5rem;
            border-radius: 8px;
            margin: 2rem 0;
        }
        
        .next-steps h3 {
            color: var(--volvo-blue);
            margin: 0 0 1rem 0;
        }
        
        .next-steps ul {
            margin: 0;
            padding-left: 1.5rem;
        }
        
        .next-steps li {
            margin-bottom: 0.5rem;
            color: #666;
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 1rem 2rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            text-decoration: none;
            cursor: pointer;
            transition: all 0.3s ease;
            font-size: 1rem;
        }
        
        .btn-primary {
            background: var(--f1-gradient);
            color: white;
        }
        
        .btn-primary:hover {
            background: var(--volvo-gradient);
            transform: translateY(-2px);
            box-shadow: 0 8px 24px rgba(30, 60, 114, 0.3);
        }
        
        .btn-secondary {
            background: #f8f9fa;
            color: #6c757d;
            border: 1px solid #dee2e6;
        }
        
        .btn-secondary:hover {
            background: #e9ecef;
            color: #495057;
        }
        
        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        @media (max-width: 768px) {
            .setup-card {
                padding: 2rem;
            }
            
            .progress-steps {
                flex-direction: column;
                gap: 1rem;
            }
            
            .step {
                flex-direction: row;
                justify-content: flex-start;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .form-actions {
                flex-direction: column;
            }
        }
    </style>
</body>
</html>
