// Service Worker für B&E Boxenstop PWA
const CACHE_NAME = 'boxenstop-v2'; // Version erhöht um Cache zu leeren
const RUNTIME_CACHE = 'boxenstop-runtime-v2';

// Dateien, die beim Installieren gecacht werden sollen
const PRECACHE_URLS = [
  '/',
  '/index.php',
  '/assets/css/style.css',
  '/assets/css/volvo-theme.css',
  '/assets/images/new-pwa_icon.png'
];

// Install Event - Cache wichtige Dateien
self.addEventListener('install', (event) => {
  console.log('[Service Worker] Install Event');
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then((cache) => {
        console.log('[Service Worker] Caching wichtige Dateien');
        return cache.addAll(PRECACHE_URLS);
      })
      .then(() => self.skipWaiting())
      .catch((error) => {
        console.error('[Service Worker] Cache Fehler:', error);
      })
  );
});

// Activate Event - Alte Caches löschen
self.addEventListener('activate', (event) => {
  console.log('[Service Worker] Activate Event');
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== CACHE_NAME && cacheName !== RUNTIME_CACHE) {
            console.log('[Service Worker] Lösche alten Cache:', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    })
    .then(() => self.clients.claim())
  );
});

// Fetch Event - Network First Strategy mit Fallback
self.addEventListener('fetch', (event) => {
  // Nur GET-Requests cachen
  if (event.request.method !== 'GET') {
    return;
  }

  // API-Requests und dynamische Inhalte nicht cachen
  if (event.request.url.includes('/api/') || 
      event.request.url.includes('?') ||
      event.request.url.includes('.php') && !event.request.url.includes('index.php')) {
    // Network only für API-Requests
    event.respondWith(fetch(event.request));
    return;
  }
  
  // pwa-install.js immer vom Netzwerk laden (für Debug-Zwecke)
  if (event.request.url.includes('pwa-install.js')) {
    event.respondWith(fetch(event.request));
    return;
  }

  // Network First Strategy für andere Requests
  // Prüfe ob Request gecacht werden kann (nur http/https)
  const url = new URL(event.request.url);
  const canCache = url.protocol === 'http:' || url.protocol === 'https:';
  
  event.respondWith(
    fetch(event.request)
      .then((response) => {
        // Nur erfolgreiche Responses cachen, wenn Request gecacht werden kann
        if (response.status === 200 && canCache) {
          const responseToCache = response.clone();
          caches.open(RUNTIME_CACHE).then((cache) => {
            try {
              cache.put(event.request, responseToCache);
            } catch (e) {
              // Ignoriere Cache-Fehler (z.B. für chrome-extension://)
              console.log('[Service Worker] Cache-Fehler ignoriert:', e);
            }
          });
        }
        return response;
      })
      .catch(() => {
        // Fallback: Aus Cache laden
        return caches.match(event.request).then((cachedResponse) => {
          if (cachedResponse) {
            return cachedResponse;
          }
          // Fallback für Navigation Requests
          if (event.request.mode === 'navigate') {
            return caches.match('/index.php');
          }
        });
      })
  );
});

// Push Event - Empfange Push-Benachrichtigungen
self.addEventListener('push', (event) => {
  console.log('[Service Worker] Push Event empfangen');
  
  let notificationData = {
    title: 'B&E Boxenstop',
    body: 'Sie haben eine neue Benachrichtigung',
    icon: '/assets/images/new-pwa_icon.png',
    badge: '/assets/images/new-pwa_icon.png',
    tag: 'notification',
    requireInteraction: false,
    data: {
      url: '/profile.php'
    }
  };
  
  // Wenn Daten mit dem Push-Event gesendet wurden
  if (event.data) {
    try {
      const data = event.data.json();
      notificationData = {
        title: data.title || notificationData.title,
        body: data.message || data.body || notificationData.body,
        icon: data.icon || notificationData.icon,
        badge: data.badge || notificationData.badge,
        tag: data.tag || notificationData.tag,
        requireInteraction: data.requireInteraction || false,
        data: {
          url: data.url || '/profile.php',
          notificationId: data.notificationId || null,
          type: data.type || 'info'
        }
      };
    } catch (e) {
      console.error('[Service Worker] Fehler beim Parsen der Push-Daten:', e);
    }
  }
  
  // Benachrichtigung anzeigen mit verbessertem Design
  event.waitUntil(
    self.registration.showNotification(notificationData.title, {
      body: notificationData.body,
      icon: notificationData.icon,
      badge: notificationData.badge,
      tag: notificationData.tag,
      requireInteraction: notificationData.requireInteraction,
      data: notificationData.data,
      vibrate: [200, 100, 200],
      image: notificationData.data.image || null, // Optional: Großes Bild
      timestamp: Date.now(),
      silent: false,
      renotify: true,
      actions: [
        {
          action: 'open',
          title: 'Öffnen',
          icon: '/assets/images/new-pwa_icon.png'
        },
        {
          action: 'close',
          title: 'Schließen'
        }
      ],
      // Zusätzliche Optionen für besseres Aussehen
      dir: 'ltr',
      lang: 'de',
      // Farben basierend auf Typ
      ...(notificationData.data.type === 'success' && {
        badge: '/assets/images/new-pwa_icon.png',
        icon: '/assets/images/new-pwa_icon.png'
      }),
      ...(notificationData.data.type === 'error' && {
        badge: '/assets/images/new-pwa_icon.png',
        icon: '/assets/images/new-pwa_icon.png'
      }),
      ...(notificationData.data.type === 'warning' && {
        badge: '/assets/images/new-pwa_icon.png',
        icon: '/assets/images/new-pwa_icon.png'
      })
    })
  );
});

// Notification Click Event - Reaktion auf Klick auf Benachrichtigung
self.addEventListener('notificationclick', (event) => {
  console.log('[Service Worker] Notification Click Event');
  
  event.notification.close();
  
  // Standard-Aktion: Öffne die App/Seite
  let urlToOpen = '/profile.php';
  
  if (event.notification.data && event.notification.data.url) {
    urlToOpen = event.notification.data.url;
  }
  
  // Wenn eine spezifische Aktion gewählt wurde
  if (event.action === 'open') {
    event.waitUntil(
      clients.matchAll({ type: 'window', includeUncontrolled: true })
        .then((clientList) => {
          // Prüfe ob bereits ein Fenster/Tab geöffnet ist
          for (let i = 0; i < clientList.length; i++) {
            const client = clientList[i];
            if (client.url === urlToOpen && 'focus' in client) {
              return client.focus();
            }
          }
          // Öffne neues Fenster/Tab
          if (clients.openWindow) {
            return clients.openWindow(urlToOpen);
          }
        })
    );
  } else if (event.action === 'close') {
    // Benachrichtigung wurde geschlossen, nichts tun
    return;
  } else {
    // Standard-Klick: Öffne die App/Seite
    event.waitUntil(
      clients.matchAll({ type: 'window', includeUncontrolled: true })
        .then((clientList) => {
          // Prüfe ob bereits ein Fenster/Tab geöffnet ist
          for (let i = 0; i < clientList.length; i++) {
            const client = clientList[i];
            if ('focus' in client) {
              return client.focus();
            }
          }
          // Öffne neues Fenster/Tab
          if (clients.openWindow) {
            return clients.openWindow(urlToOpen);
          }
        })
    );
  }
  
  // Markiere Benachrichtigung als gelesen wenn notificationId vorhanden
  if (event.notification.data && event.notification.data.notificationId) {
    // Sende Request an Backend um Benachrichtigung als gelesen zu markieren
    fetch('/api/mark_notification_read.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      credentials: 'same-origin', // Wichtig für Session-Cookies
      body: JSON.stringify({
        notification_id: event.notification.data.notificationId // API erwartet notification_id
      })
    }).then(response => {
      if (response.ok) {
        console.log('[Service Worker] Benachrichtigung als gelesen markiert');
      } else {
        console.error('[Service Worker] Fehler beim Markieren als gelesen:', response.status);
      }
    }).catch(err => {
      console.error('[Service Worker] Fehler beim Markieren als gelesen:', err);
    });
  }
});

