<?php
/**
 * SQL Injection Security Check Script
 * 
 * Dieses Skript prüft die Frontend-Dateien auf SQL Injection-Schwachstellen.
 * 
 * Verwendung:
 * 1. Im Browser öffnen: http://localhost/boxenstop/security_check_sql_injection.php
 * 2. Oder via CLI: php security_check_sql_injection.php
 */

// Dateien die geprüft werden sollen
$files_to_check = [
    'index.php',
    'booking.php',
    'login.php',
    'profile.php'
];

// Gefährliche Muster die auf SQL Injection hindeuten
$dangerous_patterns = [
    // Direkte String-Konkatenation in SQL
    [
        'pattern' => '/\$sql\s*=\s*["\'].*\$.*["\']/s',
        'description' => 'SQL-String mit direkter Variablen-Konkatenation',
        'severity' => 'HIGH'
    ],
    [
        'pattern' => '/\$conn->(query|exec)\s*\(\s*["\'].*\$[a-zA-Z_][a-zA-Z0-9_]*.*["\']/s',
        'description' => 'Direkte SQL-Abfrage mit Benutzereingabe (unsicher!)',
        'severity' => 'HIGH',
        'note' => 'DDL-Statements (ALTER TABLE, CREATE TABLE) mit statischen Strings sind OK'
    ],
    [
        'pattern' => '/mysqli_query\s*\([^,]+,\s*["\'].*\$.*["\']/s',
        'description' => 'mysqli_query mit String-Konkatenation',
        'severity' => 'HIGH'
    ],
    [
        'pattern' => '/mysql_query\s*\(["\'].*\$.*["\']/s',
        'description' => 'Veraltete mysql_query Funktion',
        'severity' => 'CRITICAL'
    ],
    [
        'pattern' => '/\$conn->prepare\s*\(\s*\$[a-zA-Z_][a-zA-Z0-9_]*\s*\./s',
        'description' => 'Prepared Statement mit String-Konkatenation (unsicher!)',
        'severity' => 'HIGH',
        'note' => 'Nur wenn Variable konkateniert wird, nicht bei statischen Strings'
    ],
    // Benutzereingaben die direkt verwendet werden könnten
    [
        'pattern' => '/\$_GET\[[^\]]+\]\s*\.\s*["\']|["\']\s*\.\s*\$_GET\[[^\]]+\]/',
        'description' => '$_GET direkt in String-Konkatenation',
        'severity' => 'MEDIUM'
    ],
    [
        'pattern' => '/\$_POST\[[^\]]+\]\s*\.\s*["\']|["\']\s*\.\s*\$_POST\[[^\]]+\]/',
        'description' => '$_POST direkt in String-Konkatenation',
        'severity' => 'MEDIUM'
    ],
    [
        'pattern' => '/\$_REQUEST\[[^\]]+\]\s*\.\s*["\']|["\']\s*\.\s*\$_REQUEST\[[^\]]+\]/',
        'description' => '$_REQUEST direkt in String-Konkatenation',
        'severity' => 'MEDIUM'
    ],
    // Gefährliche Funktionen
    [
        'pattern' => '/eval\s*\(/',
        'description' => 'eval() Funktion gefunden (extrem gefährlich!)',
        'severity' => 'CRITICAL'
    ],
    [
        'pattern' => '/\bexec\s*\(\s*["\']|exec\s*\(\s*\$|exec\s*\(\s*_/',
        'description' => 'PHP exec() Funktion gefunden (nicht PDO exec!)',
        'severity' => 'HIGH',
        'note' => 'Unterscheidet zwischen PHP exec() und PDO $conn->exec()'
    ],
    [
        'pattern' => '/system\s*\(/',
        'description' => 'system() Funktion gefunden',
        'severity' => 'HIGH'
    ],
    [
        'pattern' => '/shell_exec\s*\(/',
        'description' => 'shell_exec() Funktion gefunden',
        'severity' => 'HIGH'
    ],
    // Prepared Statements ohne Parameter
    [
        'pattern' => '/->prepare\s*\([^)]+\)\s*;\s*->execute\s*\(\s*\)\s*;/s',
        'description' => 'Prepared Statement ohne Parameter (könnte unsicher sein)',
        'severity' => 'LOW'
    ]
];

// Sichere Muster (Prepared Statements)
$safe_patterns = [
    '/->prepare\s*\([^)]+\)\s*;\s*->execute\s*\([^)]+\)/s' => 'Prepared Statement mit Parametern',
    '/->prepare\s*\([^)]+\)\s*;\s*->bindParam\s*\(/s' => 'Prepared Statement mit bindParam',
    '/->prepare\s*\([^)]+\)\s*;\s*->bindValue\s*\(/s' => 'Prepared Statement mit bindValue'
];

$results = [];
$total_issues = 0;
$critical_issues = 0;
$high_issues = 0;
$medium_issues = 0;
$low_issues = 0;

foreach ($files_to_check as $file) {
    if (!file_exists($file)) {
        $results[$file] = [
            'exists' => false,
            'error' => 'Datei nicht gefunden'
        ];
        continue;
    }
    
    $content = file_get_contents($file);
    $file_results = [
        'exists' => true,
        'lines' => substr_count($content, "\n") + 1,
        'issues' => [],
        'safe_patterns' => [],
        'summary' => [
            'critical' => 0,
            'high' => 0,
            'medium' => 0,
            'low' => 0
        ]
    ];
    
    // Prüfe auf gefährliche Muster
    foreach ($dangerous_patterns as $pattern_info) {
        $pattern = $pattern_info['pattern'];
        $description = $pattern_info['description'];
        $severity = $pattern_info['severity'];
        
        if (preg_match_all($pattern, $content, $matches, PREG_OFFSET_CAPTURE | PREG_SET_ORDER)) {
            foreach ($matches as $match) {
                $offset = $match[0][1];
                $line_number = substr_count(substr($content, 0, $offset), "\n") + 1;
                $matched_text = substr($match[0][0], 0, 200); // Erste 200 Zeichen für bessere Analyse
                
                // False Positive Filter
                $is_false_positive = false;
                
                // Filter 1: Statische SQL-Strings in Prepared Statements (keine Variablen)
                if (preg_match('/->prepare\s*\(/', $matched_text)) {
                    // Prüfe ob wirklich Variablen konkateniert werden
                    $context = substr($content, max(0, $offset - 200), 400);
                    // Wenn nur statische Strings, kein Problem
                    if (!preg_match('/\$[a-zA-Z_][a-zA-Z0-9_]*\s*\.|\.\s*\$[a-zA-Z_][a-zA-Z0-9_]*/', $context)) {
                        $is_false_positive = true;
                    }
                }
                
                // Filter 2: DDL-Statements (ALTER TABLE, CREATE TABLE) mit statischen Strings
                if (preg_match('/->exec\s*\(/', $matched_text)) {
                    $context = substr($content, max(0, $offset - 200), 400);
                    // DDL-Statements mit statischen Strings sind OK
                    if (preg_match('/ALTER\s+TABLE|CREATE\s+TABLE|DROP\s+TABLE/i', $context) && 
                        !preg_match('/\$[a-zA-Z_][a-zA-Z0-9_]*\s*\.|\.\s*\$[a-zA-Z_][a-zA-Z0-9_]*/', $context)) {
                        $is_false_positive = true;
                    }
                }
                
                // Filter 3: PDO exec() vs PHP exec()
                if (preg_match('/exec\s*\(/', $matched_text)) {
                    $context = substr($content, max(0, $offset - 100), 200);
                    // Wenn $conn->exec() oder $pdo->exec(), ist es PDO, nicht PHP exec()
                    if (preg_match('/\$[a-zA-Z_][a-zA-Z0-9_]*\s*->\s*exec\s*\(/', $context)) {
                        $is_false_positive = true;
                    }
                }
                
                // Filter 4: $_GET in Redirect-URLs (nicht SQL, aber sollte validiert werden)
                if (preg_match('/\$_GET\[.*\]\s*\.\s*["\']/', $matched_text)) {
                    $context = substr($content, max(0, $offset - 100), 200);
                    // Wenn in header('Location: ...) oder ähnlichem, ist es XSS-Risiko, nicht SQL Injection
                    if (preg_match('/header\s*\(\s*["\']Location|Location\s*:/i', $context)) {
                        // Behalte es, aber markiere als XSS-Risiko, nicht SQL Injection
                        $description = '$_GET in Redirect-URL (XSS-Risiko, sollte validiert werden)';
                        $severity = 'MEDIUM';
                    }
                }
                
                if (!$is_false_positive) {
                    $file_results['issues'][] = [
                        'line' => $line_number,
                        'severity' => $severity,
                        'description' => $description,
                        'matched_text' => $matched_text,
                        'pattern' => $pattern
                    ];
                    
                    $file_results['summary'][strtolower($severity)]++;
                    $total_issues++;
                    
                    switch ($severity) {
                        case 'CRITICAL':
                            $critical_issues++;
                            break;
                        case 'HIGH':
                            $high_issues++;
                            break;
                        case 'MEDIUM':
                            $medium_issues++;
                            break;
                        case 'LOW':
                            $low_issues++;
                            break;
                    }
                }
            }
        }
    }
    
    // Prüfe auf sichere Muster
    foreach ($safe_patterns as $pattern => $description) {
        if (preg_match_all($pattern, $content, $matches)) {
            $file_results['safe_patterns'][] = [
                'description' => $description,
                'count' => count($matches[0])
            ];
        }
    }
    
    // Zähle Prepared Statements
    preg_match_all('/->prepare\s*\(/s', $content, $prepare_matches);
    $file_results['prepared_statements_count'] = count($prepare_matches[0]);
    
    // Zähle execute() Aufrufe
    preg_match_all('/->execute\s*\(/s', $content, $execute_matches);
    $file_results['execute_count'] = count($execute_matches[0]);
    
    $results[$file] = $file_results;
}

// HTML-Output
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SQL Injection Security Check</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
            padding: 2rem;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 2rem;
        }
        
        h1 {
            color: #333;
            margin-bottom: 1rem;
            border-bottom: 3px solid #0056b3;
            padding-bottom: 0.5rem;
        }
        
        .summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .summary-card {
            padding: 1.5rem;
            border-radius: 8px;
            text-align: center;
        }
        
        .summary-card.total {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .summary-card.critical {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            color: white;
        }
        
        .summary-card.high {
            background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);
            color: #333;
        }
        
        .summary-card.medium {
            background: linear-gradient(135deg, #ffecd2 0%, #fcb69f 100%);
            color: #333;
        }
        
        .summary-card.low {
            background: linear-gradient(135deg, #a8edea 0%, #fed6e3 100%);
            color: #333;
        }
        
        .summary-card h3 {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }
        
        .summary-card p {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        
        .file-section {
            margin-bottom: 2rem;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
        }
        
        .file-header {
            background: #f8f9fa;
            padding: 1rem;
            border-bottom: 1px solid #e0e0e0;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .file-header h2 {
            color: #333;
            font-size: 1.2rem;
        }
        
        .file-info {
            font-size: 0.9rem;
            color: #666;
        }
        
        .issues-list {
            padding: 1rem;
        }
        
        .issue {
            padding: 1rem;
            margin-bottom: 0.5rem;
            border-left: 4px solid;
            border-radius: 4px;
            background: #f8f9fa;
        }
        
        .issue.critical {
            border-left-color: #dc3545;
            background: #fff5f5;
        }
        
        .issue.high {
            border-left-color: #fd7e14;
            background: #fff8f0;
        }
        
        .issue.medium {
            border-left-color: #ffc107;
            background: #fffef0;
        }
        
        .issue.low {
            border-left-color: #17a2b8;
            background: #f0f9ff;
        }
        
        .issue-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
        }
        
        .issue-severity {
            padding: 0.25rem 0.75rem;
            border-radius: 4px;
            font-size: 0.8rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .issue-severity.critical {
            background: #dc3545;
            color: white;
        }
        
        .issue-severity.high {
            background: #fd7e14;
            color: white;
        }
        
        .issue-severity.medium {
            background: #ffc107;
            color: #333;
        }
        
        .issue-severity.low {
            background: #17a2b8;
            color: white;
        }
        
        .issue-line {
            font-family: 'Courier New', monospace;
            font-size: 0.85rem;
            color: #666;
            margin-top: 0.5rem;
        }
        
        .issue-code {
            font-family: 'Courier New', monospace;
            font-size: 0.85rem;
            background: #e9ecef;
            padding: 0.5rem;
            border-radius: 4px;
            margin-top: 0.5rem;
            overflow-x: auto;
        }
        
        .safe-patterns {
            padding: 1rem;
            background: #d4edda;
            border-left: 4px solid #28a745;
            border-radius: 4px;
            margin-top: 1rem;
        }
        
        .safe-patterns h3 {
            color: #155724;
            margin-bottom: 0.5rem;
        }
        
        .safe-patterns ul {
            list-style: none;
            padding-left: 0;
        }
        
        .safe-patterns li {
            padding: 0.25rem 0;
            color: #155724;
        }
        
        .no-issues {
            padding: 2rem;
            text-align: center;
            color: #28a745;
            font-size: 1.1rem;
        }
        
        .recommendations {
            background: #e3f2fd;
            border-left: 4px solid #2196f3;
            padding: 1.5rem;
            border-radius: 4px;
            margin-top: 2rem;
        }
        
        .recommendations h2 {
            color: #1976d2;
            margin-bottom: 1rem;
        }
        
        .recommendations ul {
            margin-left: 1.5rem;
        }
        
        .recommendations li {
            margin-bottom: 0.5rem;
            color: #333;
        }
        
        .timestamp {
            text-align: center;
            color: #666;
            font-size: 0.9rem;
            margin-top: 2rem;
            padding-top: 2rem;
            border-top: 1px solid #e0e0e0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔒 SQL Injection Security Check</h1>
        
        <div class="summary">
            <div class="summary-card total">
                <h3><?php echo $total_issues; ?></h3>
                <p>Gesamt gefundene Probleme</p>
            </div>
            <div class="summary-card critical">
                <h3><?php echo $critical_issues; ?></h3>
                <p>Kritische Probleme</p>
            </div>
            <div class="summary-card high">
                <h3><?php echo $high_issues; ?></h3>
                <p>Hohe Priorität</p>
            </div>
            <div class="summary-card medium">
                <h3><?php echo $medium_issues; ?></h3>
                <p>Mittlere Priorität</p>
            </div>
            <div class="summary-card low">
                <h3><?php echo $low_issues; ?></h3>
                <p>Niedrige Priorität</p>
            </div>
        </div>
        
        <?php foreach ($results as $file => $file_data): ?>
            <?php if (!$file_data['exists']): ?>
                <div class="file-section">
                    <div class="file-header">
                        <h2><?php echo htmlspecialchars($file); ?></h2>
                        <span class="file-info">❌ Datei nicht gefunden</span>
                    </div>
                </div>
                <?php continue; ?>
            <?php endif; ?>
            
            <div class="file-section">
                <div class="file-header">
                    <h2><?php echo htmlspecialchars($file); ?></h2>
                    <div class="file-info">
                        <?php echo $file_data['lines']; ?> Zeilen | 
                        <?php echo $file_data['prepared_statements_count']; ?> Prepared Statements | 
                        <?php echo count($file_data['issues']); ?> Probleme
                    </div>
                </div>
                
                <?php if (empty($file_data['issues'])): ?>
                    <div class="no-issues">
                        ✅ Keine SQL Injection-Schwachstellen gefunden!
                    </div>
                <?php else: ?>
                    <div class="issues-list">
                        <?php foreach ($file_data['issues'] as $issue): ?>
                            <div class="issue <?php echo strtolower($issue['severity']); ?>">
                                <div class="issue-header">
                                    <strong><?php echo htmlspecialchars($issue['description']); ?></strong>
                                    <span class="issue-severity <?php echo strtolower($issue['severity']); ?>">
                                        <?php echo $issue['severity']; ?>
                                    </span>
                                </div>
                                <div class="issue-line">
                                    Zeile <?php echo $issue['line']; ?>
                                </div>
                                <div class="issue-code">
                                    <?php echo htmlspecialchars($issue['matched_text']); ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
                
                <?php if (!empty($file_data['safe_patterns'])): ?>
                    <div class="safe-patterns">
                        <h3>✅ Sichere Muster gefunden:</h3>
                        <ul>
                            <?php foreach ($file_data['safe_patterns'] as $safe): ?>
                                <li>• <?php echo htmlspecialchars($safe['description']); ?> (<?php echo $safe['count']; ?>x)</li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
        
        <div class="recommendations">
            <h2>📋 Empfehlungen</h2>
            <ul>
                <li><strong>Immer Prepared Statements verwenden:</strong> Alle SQL-Abfragen sollten mit <code>$conn->prepare()</code> und <code>->execute([$params])</code> erstellt werden.</li>
                <li><strong>Benutzereingaben validieren:</strong> Prüfen Sie alle Eingaben auf Typ, Länge und Format bevor Sie sie verwenden.</li>
                <li><strong>Whitelist-Ansatz:</strong> Erlauben Sie nur bekannte, sichere Werte (z.B. bei Dropdown-Auswahlen).</li>
                <li><strong>Keine String-Konkatenation:</strong> Verwenden Sie niemals <code>$sql = "SELECT * FROM users WHERE id = " . $_GET['id'];</code></li>
                <li><strong>PDO verwenden:</strong> PDO mit Prepared Statements ist sicherer als mysqli oder die veraltete mysql_* API.</li>
                <li><strong>Fehlerbehandlung:</strong> Zeigen Sie keine SQL-Fehlermeldungen direkt an Benutzer an.</li>
                <li><strong>Regelmäßige Prüfungen:</strong> Führen Sie diesen Check regelmäßig durch, besonders nach Code-Änderungen.</li>
            </ul>
        </div>
        
        <div class="timestamp">
            Prüfung durchgeführt am: <?php echo date('d.m.Y H:i:s'); ?>
        </div>
    </div>
</body>
</html>

