const { chromium } = require('playwright');
const activateAllColumns = require('./activate_resy_columns');

// Parameter: Entweder aus JSON-Datei (wenn als Argument übergeben) oder Command-Line/ENV
const fs = require('fs');
let autohausId, username, password, dealerId;

if (process.argv[2] && fs.existsSync(process.argv[2])) {
    // Parameter aus JSON-Datei (verhindert "!" Problem in Windows CMD!)
    const paramsFile = process.argv[2];
    const params = JSON.parse(fs.readFileSync(paramsFile, 'utf8'));
    autohausId = params.autohausId || '1';
    username = params.username || 'D.ULUDAG';
    password = params.password || 'Ikizler123!';
    dealerId = params.dealerId || '453';
    
    // Lösche JSON-Datei nach dem Lesen (Sicherheit)
    try {
        fs.unlinkSync(paramsFile);
    } catch (e) {
        // Ignoriere Fehler beim Löschen
    }
} else {
    // Fallback: Command-Line Parameter oder ENV-Variablen
    autohausId = process.argv[2] || process.env.AUTOHAUS_ID || '1';
    username = process.argv[3] || process.env.RESY_USERNAME || 'D.ULUDAG';
    password = process.argv[4] || process.env.RESY_PASSWORD || 'Ikizler123!';
    dealerId = process.argv[5] || process.env.RESY_DEALER_ID || '453';
}

// DEBUG: Zeige alle Parameter
console.log('🔍 DEBUG - Empfangene Parameter:');
console.log(`   autohausId: "${autohausId}"`);
console.log(`   username: "${username}"`);
console.log(`   password: "${password}" (Länge: ${password.length})`);
console.log(`   dealerId: "${dealerId}"`);
console.log('');

async function fetchWheelsets() {
    console.log(`🚀 Starte RESY-Import mit Playwright für Autohaus ${autohausId}...`);
    console.log(`📋 Parameter:`);
    console.log(`   - autohausId: ${autohausId}`);
    console.log(`   - username: ${username}`);
    console.log(`   - password: [HIDDEN]`);
    console.log(`   - dealerId: ${dealerId}`);
    console.log(``);
    
    // WICHTIG: headless: true für viel schnellere Ausführung!
    const browser = await chromium.launch({
        headless: true, // Schneller! (Kein Browser-Fenster)
        channel: 'chrome'
    });
    
    const page = await browser.newPage();
    
    try {
        console.log('[1/5] Navigiere zur Login-Seite...');
        await page.goto('https://resy.dtm-reifen.de/index.php?m=login&a=login', {
            waitUntil: 'domcontentloaded', // Schneller als 'networkidle'
            timeout: 20000
        });
        
        console.log('[1b/5] Warte auf Login-Formular...');
        await page.waitForSelector('#InputUser', { timeout: 5000 });
        await page.waitForSelector('#InputPass', { timeout: 5000 });
        await page.waitForTimeout(300); // Reduziert von 1000ms
        
        console.log('[2/5] Fülle Login-Felder...');
        
        // WICHTIG: Verwende type() für zuverlässige Eingabe, aber schneller!
        // Username mit minimalem Delay
        await page.click('#InputUser', { timeout: 5000 });
        await page.waitForTimeout(200); // Reduziert
        await page.type('#InputUser', username, { delay: 50 }); // Reduziert von 100ms
        await page.waitForTimeout(200);
        
        // Password mit minimalem Delay
        await page.click('#InputPass', { timeout: 5000 });
        await page.waitForTimeout(200);
        await page.type('#InputPass', password, { delay: 50 }); // Reduziert von 100ms
        await page.waitForTimeout(200);
        
        // Verifiziere dass beide Felder gefüllt sind
        const usernameValue = await page.$eval('#InputUser', el => el.value);
        const passwordValue = await page.$eval('#InputPass', el => el.value);
        const passwordLength = passwordValue.length;
        
        console.log(`[2b/5] Verifizierung:`);
        console.log(`   Username - Erwartet: "${username}", Gefunden: "${usernameValue}"`);
        console.log(`   Password - Länge: Erwartet ${password.length}, Gefunden ${passwordLength}`);
        console.log(`   Password - Erste/Letzte: "${passwordValue.substring(0, 1)}...${passwordValue.substring(passwordLength - 1)}"`);
        console.log(`   Password - Vollständig: "${passwordValue}"`);
        console.log(`   Password - Match: ${passwordValue === password}`);
        
        if (usernameValue !== username) {
            throw new Error(`Username nicht korrekt eingegeben! Erwartet: "${username}", Gefunden: "${usernameValue}"`);
        }
        
        if (passwordValue !== password) {
            throw new Error(`Password stimmt nicht überein! Erwartet: "${password}", Gefunden: "${passwordValue}"`);
        }
        
        if (passwordLength !== password.length) {
            throw new Error(`Password-Länge stimmt nicht! Erwartet: ${password.length} Zeichen, Gefunden: ${passwordLength} Zeichen`);
        }
        
        console.log('[3/5] Klicke Login-Button...');
        await page.waitForSelector('button[type="submit"]', { timeout: 5000 });
        await page.waitForTimeout(200);
        await page.click('button[type="submit"]');
        
        console.log('[3b/5] Warte auf Login-Verarbeitung...');
        // Warte auf Navigation weg von Login-Seite (schneller als fixed timeout!)
        try {
            await page.waitForURL(/dashboard|wheelset|search/, { timeout: 10000 });
        } catch (e) {
            await page.waitForTimeout(3000); // Fallback
        }
        
        // Prüfe ob Login erfolgreich war (sollte zur Dashboard oder Suche weiterleiten)
        const currentUrl = page.url();
        console.log(`[3c/5] Aktuelle URL nach Login: ${currentUrl}`);
        
        // Wenn noch auf Login-Seite: Prüfe auf Fehlermeldung
        if (currentUrl.includes('login')) {
            const errorText = await page.evaluate(() => {
                const errorElement = document.querySelector('.error, .alert-danger, [class*="error"], [class*="Error"]');
                return errorElement ? errorElement.textContent.trim() : '';
            });
            
            if (errorText) {
                throw new Error(`Login fehlgeschlagen: ${errorText}`);
            }
            
            // Zusätzliche Wartezeit falls Login noch verarbeitet wird
            await page.waitForTimeout(3000);
        }
        
        console.log('[4/5] Navigiere zur Suche...');
        await page.goto('https://resy.dtm-reifen.de/resyweb/index.php?m=wheelset&a=search', {
            waitUntil: 'domcontentloaded', // Schneller
            timeout: 20000
        });
        
        console.log('[4b/5] Warte auf Suchformular...');
        await page.waitForSelector('#WheelKD_ID', { timeout: 10000 });
        await page.waitForTimeout(500); // Reduziert
        
        console.log(`[4c/5] Filtere nach Autohaus - RESY Dealer ID: ${dealerId}...`);
        try {
            // Versuche zuerst per value zu selecten
            await page.selectOption('#WheelKD_ID', dealerId);
            console.log(`   ✅ Autohaus ${dealerId} per value ausgewählt`);
        } catch (error) {
            console.log(`   ⚠️  Wert ${dealerId} nicht gefunden, versuche per Text...`);
            // Fallback: Suche per Text
            try {
                await page.selectOption('#WheelKD_ID', { label: dealerId });
            } catch (e) {
                // Letzter Fallback: Suche nach Autohaus-Name basierend auf dealerId
                let autohausName = '';
                if (dealerId === '452') autohausName = 'Norderstedt';
                else if (dealerId === '453') autohausName = 'Kiel';
                else if (dealerId === '454') autohausName = 'Glinde';
                
                if (autohausName) {
                    await page.selectOption('#WheelKD_ID', { label: new RegExp(autohausName, 'i') });
                } else {
                    throw new Error(`Konnte Autohaus ${dealerId} nicht auswählen`);
                }
            }
        }
        
        // Verifiziere Auswahl
        const selectedValue = await page.$eval('#WheelKD_ID', el => el.value);
        const selectedText = await page.$eval('#WheelKD_ID', el => {
            const option = el.options[el.selectedIndex];
            return option ? option.text : '';
        });
        console.log(`   ✅ Verifizierung - Ausgewähltes Autohaus: ${selectedValue} (${selectedText})`);
        
        await page.waitForTimeout(300); // Reduziert
        
        console.log('[4d/5] Setze Status auf "eingelagert"...');
        await page.selectOption('#WheelSTATUS_ID', '20');
        await page.waitForTimeout(300); // Reduziert
        
        console.log('[4e/5] Klicke Suchen (1. Mal)...');
        await page.click('button[type="submit"]');
        // Warte auf Tabelle statt fixed timeout
        await page.waitForSelector('table.ResultTable', { timeout: 15000 });
        await page.waitForTimeout(1000); // Minimal für JavaScript-Ausführung
        
        // ✅ AKTIVIERE ALLE WICHTIGEN SPALTEN
        console.log('[4f/5] Aktiviere alle wichtigen Spalten...');
        try {
            await activateAllColumns(page);
            console.log('   ✅ Spalten aktiviert!');
            
            // WARTE auf Seitenreload (Die Spaltenaktivierung triggert automatisch einen Reload)
            console.log('[4g/5] Warte auf Seitenreload nach Aktivierung...');
            // Warte auf Tabelle statt fixed timeout
            await page.waitForSelector('table.ResultTable', { timeout: 15000 });
            await page.waitForTimeout(2000); // Reduziert für Spalten-Aktivierung
        } catch (error) {
            console.log('⚠️  Spalten-Aktivierung fehlgeschlagen, fahre ohne fort...');
        }
        
        // Finde Gesamtzahl der Sätze aus der Überschrift
        const expectedCount = await page.evaluate(() => {
            const title = document.querySelector('p.tableTitle');
            if (title) {
                const text = title.textContent;
                const match = text.match(/(\d+)/);
                if (match) return parseInt(match[1]);
            }
            return 0;
        });
        
        console.log(`📊 Erwartete Anzahl: ${expectedCount} Reifensätze\n`);
        
        // Finde Gesamtzahl der Seiten
        const totalPages = await page.evaluate(() => {
            const links = document.querySelectorAll('ul#Tablepaginate li.PageLink a');
            let maxPage = 1;
            links.forEach(link => {
                const text = link.querySelector('span')?.textContent;
                const num = parseInt(text);
                if (num && num > maxPage) maxPage = num;
            });
            return maxPage;
        });
        
        console.log(`📄 ${totalPages} Seiten gefunden\n`);
        
        // Lade ALLE Seiten
        let allWheelsets = [];
        
        for (let p = 1; p <= totalPages; p++) {
            if (p > 1) {
                console.log(`📄 Lade Seite ${p}/${totalPages}...`);
                // WICHTIG: Verwende den dynamischen dealerId statt hardcodiert 453!
                const url = `https://resy.dtm-reifen.de/resyweb/index.php?Page=${p}&FN=WheelsetSearch&a=search&m=wheelset&KD_ID=${dealerId}&STATUS_ID=20&button=submit`;
                await page.goto(url, { waitUntil: 'domcontentloaded', timeout: 20000 });
                await page.waitForSelector('table.ResultTable', { timeout: 10000 });
                await page.waitForTimeout(1000); // Reduziert
            }
            
                           const wheelsets = await page.evaluate(() => {
                               const rows = document.querySelectorAll('table.ResultTable tbody tr');
                               const data = [];
                               
                               // DEBUG: Logge erste Zeile
                               let rowCount = 0;
                               
                               rows.forEach(row => {
                                   const cells = row.querySelectorAll('td');
                                   if (cells.length >= 12) {
                                       rowCount++;
                                       
                                       // DEBUG für erste 2 Zeilen
                                       if (rowCount <= 2) {
                                           console.log(`\n🔍 DEBUG Zeile ${rowCount}:`);
                                           console.log(`   Zelle 11: "${cells[11]?.textContent.trim()}"`);
                                           console.log(`   Zelle 12: "${cells[12]?.textContent.trim()}"`);
                                           console.log(`   Zelle 15: "${cells[15]?.textContent.trim()}"`);
                                           console.log(`   Zelle 17: "${cells[17]?.textContent.trim()}"`);
                                       }
                             
                             const getCell = (idx) => {
                                 if (idx < cells.length) {
                                     const text = cells[idx]?.textContent.trim() || '';
                                     return text;
                                 }
                                 return '';
                             };
                             
                             // Helper für Datumsfeld
                             const getDateCell = (idx) => {
                                 const text = getCell(idx);
                                 // Nur echtes Datum zurückgeben (dd.mm.yyyy)
                                 if (text.match(/^\d{2}\.\d{2}\.\d{4}$/)) return text;
                                 return '';
                             };
                             
                                   // DEBUG für erste 2 Zeilen - zeige ALLE Zellen 0-30!
                                   if (data.length <= 3) {
                                       console.log(`\n🔍 DEBUG für Kennzeichen: ${getCell(5)}`);
                                       console.log(`   Gesamt Zellen: ${cells.length}`);
                                       // Zeige die wichtigen Zellen 10-20
                                       for (let i = 10; i < Math.min(20, cells.length); i++) {
                                           console.log(`   Zelle ${i}: "${getCell(i)}"`);
                                       }
                                   }
                                   
                                   data.push({
                                       // Basis-Daten basierend auf deiner Liste!
                                       satznummer: getCell(2) || '',       // [2] Satznummer
                                       haltername: getCell(3) || '',       // [3] Haltername
                                       kennzeichen: getCell(4) || '',     // [4] Kennzeichen
                                       fin: getCell(5) || '',              // [5] FIN
                                       
                                       // Reifen-Info - LEER in aktueller RESY-Ansicht!
                                       reifenmodell: '',                   // [NICHT VERFÜGBAR in Haupttabelle]
                                       reifenart: getCell(11) || '',       // [11] Reifenart
                                       saison: getCell(12) || '',          // [12] Saison
                                       reifengroesse: getCell(16) || '',   // [16] Reifengröße
                                       profiltiefe: getCell(18) || '',     // [18] Profiltiefe
                                       
                                       // Autohaus-Info - UM 1 VERSCHOBEN!
                                       autohaus: getCell(7) || '',         // [7] Autohaus (war [6])
                                       ziel_autohaus: getCell(8) || '',    // [8] Ziel-Autohaus (war [7])
                                       
                                       // Datumsfelder - UM 1 VERSCHOBEN!
                                       eingangsdatum: getDateCell(9) || '',  // [9] Eingangsdatum (war [8])
                                       rueckliefertermin: getDateCell(10) || '', // [10] Rückliefertermin (war [9])
                                       
                                       // Andere
                                       lagerplatz: getCell(17) || '',      // [17] Lagerplatz
                                       empfehlung: getCell(14) || '',      // [14] Empfehlung
                                       pruefbericht: getCell(19) || '',    // [19] Prüfbericht
                                       ausgangsdatum: '',
                                       status: 'gelagert',
                                       rueckgabe_bemerkung: getCell(13) || ''  // [13] Rückgabe Bemerkung
                                   });
                            }
                        });
                        return data;
                    });
            
                    // Filtere Header-Zeilen und leere Kennzeichen
                    const filtered = wheelsets.filter(w => {
                        const k = w.kennzeichen.trim();
                        return k && k !== 'Kennzeichen' && !k.includes('Satznummer') && k.length > 0;
                    });
                    
                    console.log(`   ✅ Seite ${p}: ${filtered.length}/${wheelsets.length} Reifensätze (${wheelsets.length - filtered.length} gefiltert)`);
                    
                    // Debug: Zeige erste 5 Zeilen mit allen Spalten
                    if (p === 1) {
                        console.log(`   🔍 Debug: Erste 3 Zeilen der Seite 1:`);
                        wheelsets.slice(0, 3).forEach((w, i) => {
                            console.log(`      Zeile ${i+1}: ${w.allCells}`);
                        });
                    }
                    
                    allWheelsets = allWheelsets.concat(filtered);
        }
        
        console.log(`\n✅ INSGESAMT ${allWheelsets.length} Reifensätze gefunden!`);
        console.log(`📊 Erwartet waren: ${expectedCount}`);
        
        if (allWheelsets.length < expectedCount) {
            console.log(`⚠️  WARNUNG: ${expectedCount - allWheelsets.length} Reifensätze fehlen!`);
        }
        
        console.log('\n');
        
        // Entferne debug-Feld vor Export - INKLUDIERE ALLE FELDER!
        const cleanWheelsets = allWheelsets.map(w => ({
            satznummer: w.satznummer,
            haltername: w.haltername,
            kennzeichen: w.kennzeichen,
            fin: w.fin,
            autohaus: w.autohaus,
            ziel_autohaus: w.ziel_autohaus,
            eingangsdatum: w.eingangsdatum,
            rueckliefertermin: w.rueckliefertermin,
            pruefbericht: w.pruefbericht,
            reifenmodell: w.reifenmodell,
            reifenart: w.reifenart,
            saison: w.saison,
            reifengroesse: w.reifengroesse,
            profiltiefe: w.profiltiefe,
            lagerplatz: w.lagerplatz,
            empfehlung: w.empfehlung,
            rueckgabe_bemerkung: w.rueckgabe_bemerkung || ''
        }));
        
        console.log(JSON.stringify({
            success: true,
            count: cleanWheelsets.length,
            expected: expectedCount,
            wheelsets: cleanWheelsets
        }));
        
    } catch (error) {
        console.error('❌ Fehler:', error.message);
        console.log(JSON.stringify({
            success: false,
            error: error.message,
            wheelsets: []
        }));
    } finally {
        await browser.close();
    }
}

fetchWheelsets();

