#!/usr/bin/env node
/**
 * GLB-Modell-Optimierung mit lokaler Installation
 * 
 * Verwendung:
 *   1. npm install @gltf-transform/cli (lokal, ohne -g)
 *   2. node optimize_models_local.js input.glb output.glb
 */

const { execSync } = require('child_process');
const fs = require('fs');
const path = require('path');

// Prüfe ob gltf-transform lokal installiert ist
function checkLocalGltfTransform() {
    const localPath = path.join(__dirname, 'node_modules', '@gltf-transform', 'cli', 'bin', 'gltf-transform.js');
    if (fs.existsSync(localPath)) {
        return localPath;
    }
    
    // Prüfe ob es als npx verfügbar ist
    try {
        execSync('npx --version', { stdio: 'ignore' });
        return 'npx gltf-transform';
    } catch (error) {
        return null;
    }
}

// Optimiere eine einzelne GLB-Datei
function optimizeModel(inputPath, outputPath) {
    console.log(`\n🔧 Optimiere: ${inputPath}`);
    console.log(`   → ${outputPath}`);
    
    const gltfTransform = checkLocalGltfTransform();
    
    if (!gltfTransform) {
        console.error('❌ gltf-transform ist nicht verfügbar!');
        console.log('\n📦 Installiere lokal mit:');
        console.log('   npm install @gltf-transform/cli');
        return false;
    }
    
    try {
        let command;
        if (gltfTransform === 'npx gltf-transform') {
            command = `npx gltf-transform optimize "${inputPath}" "${outputPath}" ` +
                `--compress draco ` +
                `--texture-compress webp ` +
                `--texture-size 2048 ` +
                `--texture-format ktx2 ` +
                `--simplify 0.02`;
        } else {
            command = `node "${gltfTransform}" optimize "${inputPath}" "${outputPath}" ` +
                `--compress draco ` +
                `--texture-compress webp ` +
                `--texture-size 2048 ` +
                `--texture-format ktx2 ` +
                `--simplify 0.02`;
        }
        
        console.log(`   Ausführung: ${command}`);
        execSync(command, { stdio: 'inherit' });
        
        // Dateigrößen vergleichen
        const inputSize = fs.statSync(inputPath).size;
        const outputSize = fs.statSync(outputPath).size;
        const reduction = ((1 - outputSize / inputSize) * 100).toFixed(1);
        
        console.log(`\n✅ Optimierung abgeschlossen!`);
        console.log(`   Original: ${(inputSize / 1024 / 1024).toFixed(2)} MB`);
        console.log(`   Optimiert: ${(outputSize / 1024 / 1024).toFixed(2)} MB`);
        console.log(`   Reduktion: ${reduction}%`);
        
        return true;
    } catch (error) {
        console.error(`❌ Fehler beim Optimieren von ${inputPath}:`, error.message);
        return false;
    }
}

// Optimiere alle GLB-Dateien in einem Ordner
function optimizeAllInFolder(folderPath) {
    const files = fs.readdirSync(folderPath);
    const glbFiles = files.filter(f => f.endsWith('.glb') && !f.includes('_optimized'));
    
    if (glbFiles.length === 0) {
        console.log('⚠️ Keine GLB-Dateien gefunden!');
        return;
    }
    
    console.log(`\n📁 Gefundene GLB-Dateien: ${glbFiles.length}`);
    
    glbFiles.forEach(file => {
        const inputPath = path.join(folderPath, file);
        const outputPath = path.join(folderPath, file.replace('.glb', '_optimized.glb'));
        optimizeModel(inputPath, outputPath);
    });
}

// Hauptfunktion
function main() {
    console.log('🚀 GLB-Modell-Optimierung (Lokale Installation)');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    
    const args = process.argv.slice(2);
    
    if (args.length === 0) {
        // Keine Argumente: Optimiere alle Dateien im assets/3dmodel Ordner
        const defaultFolder = path.join(__dirname, 'assets', '3dmodel');
        if (fs.existsSync(defaultFolder)) {
            optimizeAllInFolder(defaultFolder);
        } else {
            console.error('❌ Ordner assets/3dmodel nicht gefunden!');
            console.log('\nVerwendung:');
            console.log('  node optimize_models_local.js input.glb output.glb');
            console.log('  node optimize_models_local.js (optimiert alle .glb im assets/3dmodel Ordner)');
            console.log('\n📦 Installiere zuerst lokal:');
            console.log('  npm install @gltf-transform/cli');
        }
    } else if (args.length === 2) {
        // Zwei Argumente: Optimiere einzelne Datei
        const inputPath = path.resolve(args[0]);
        const outputPath = path.resolve(args[1]);
        
        if (!fs.existsSync(inputPath)) {
            console.error(`❌ Datei nicht gefunden: ${inputPath}`);
            process.exit(1);
        }
        
        optimizeModel(inputPath, outputPath);
    } else {
        console.error('❌ Ungültige Argumente!');
        console.log('\nVerwendung:');
        console.log('  node optimize_models_local.js input.glb output.glb');
        console.log('  node optimize_models_local.js (optimiert alle .glb im assets/3dmodel Ordner)');
        console.log('\n📦 Installiere zuerst lokal:');
        console.log('  npm install @gltf-transform/cli');
        process.exit(1);
    }
    
    console.log('\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log('✅ Alle Optimierungen abgeschlossen!');
}

main();

