import { LiteralKeys, Nullable, RefCollectionValue, RefKeys, RefListKeys, RefMapKeys, RefMapValue, RefSetKeys } from './constants.js';
import { BaseEvent, EventDispatcher, GraphNodeEvent } from './event-dispatcher.js';
import { Graph } from './graph.js';
import { GraphEdge } from './graph-edge.js';
type GraphNodeAttributesInternal<Parent extends GraphNode, Attributes extends object> = {
    [Key in keyof Attributes]: Attributes[Key] extends GraphNode ? GraphEdge<Parent, Attributes[Key]> : Attributes[Key] extends GraphNode[] ? GraphEdge<Parent, Attributes[Key][number]>[] : Attributes[Key] extends {
        [key: string]: GraphNode;
    } ? Record<string, GraphEdge<Parent, Attributes[Key][string]>> : Attributes[Key];
};
export declare const $attributes: unique symbol;
export declare const $immutableKeys: unique symbol;
/**
 * Represents a node in a {@link Graph}.
 */
export declare abstract class GraphNode<Attributes extends object = object> extends EventDispatcher<GraphNodeEvent> {
    private _disposed;
    /**
     * Internal graph used to search and maintain references.
     * @hidden
     */
    protected readonly graph: Graph<GraphNode>;
    /**
     * Attributes (literal values and GraphNode references) associated with this instance. For each
     * GraphNode reference, the attributes stores a {@link GraphEdge}. List and Map references are
     * stored as arrays and dictionaries of edges.
     * @internal
     */
    protected readonly [$attributes]: GraphNodeAttributesInternal<this, Attributes>;
    /**
     * Attributes included with `getDefaultAttributes` are considered immutable, and cannot be
     * modifed by `.setRef()`, `.copy()`, or other GraphNode methods. Both the edges and the
     * properties will be disposed with the parent GraphNode.
     *
     * Currently, only single-edge references (getRef/setRef) are supported as immutables.
     *
     * @internal
     */
    protected readonly [$immutableKeys]: Set<string>;
    constructor(graph: Graph<GraphNode>);
    /**
     * Returns default attributes for the graph node. Subclasses having any attributes (either
     * literal values or references to other graph nodes) must override this method. Literal
     * attributes should be given their default values, if any. References should generally be
     * initialized as empty (Ref → null, RefList → [], RefMap → {}) and then modified by setters.
     *
     * Any single-edge references (setRef) returned by this method will be considered immutable,
     * to be owned by and disposed with the parent node. Multi-edge references (addRef, removeRef,
     * setRefMap) cannot be returned as default attributes.
     */
    protected getDefaults(): Nullable<Attributes>;
    /**
     * Constructs and returns an object used to store a graph nodes attributes. Compared to the
     * default Attributes interface, this has two distinctions:
     *
     * 1. Slots for GraphNode<T> objects are replaced with slots for GraphEdge<this, GraphNode<T>>
     * 2. GraphNode<T> objects provided as defaults are considered immutable
     *
     * @internal
     */
    private _createAttributes;
    /** @internal Returns true if two nodes are on the same {@link Graph}. */
    isOnGraph(other: GraphNode): boolean;
    /** Returns true if the node has been permanently removed from the graph. */
    isDisposed(): boolean;
    /**
     * Removes both inbound references to and outbound references from this object. At the end
     * of the process the object holds no references, and nothing holds references to it. A
     * disposed object is not reusable.
     */
    dispose(): void;
    /**
     * Removes all inbound references to this object. At the end of the process the object is
     * considered 'detached': it may hold references to child resources, but nothing holds
     * references to it. A detached object may be re-attached.
     */
    detach(): this;
    /**
     * Transfers this object's references from the old node to the new one. The old node is fully
     * detached from this parent at the end of the process.
     *
     * @hidden
     */
    swap(prevValue: GraphNode, nextValue: GraphNode): this;
    /**********************************************************************************************
     * Literal attributes.
     */
    /** @hidden */
    protected get<K extends LiteralKeys<Attributes>>(attribute: K): Attributes[K];
    /** @hidden */
    protected set<K extends LiteralKeys<Attributes>>(attribute: K, value: Attributes[K]): this;
    /**********************************************************************************************
     * Ref: 1:1 graph node references.
     */
    /** @hidden */
    protected getRef<K extends RefKeys<Attributes>>(attribute: K): (GraphNode & Attributes[K]) | null;
    /** @hidden */
    protected setRef<K extends RefKeys<Attributes>>(attribute: K, value: (GraphNode & Attributes[K]) | null, attributes?: Record<string, unknown>): this;
    /**********************************************************************************************
     * RefList: 1:many graph node references.
     */
    /** @hidden */
    protected listRefs<K extends RefListKeys<Attributes> | RefSetKeys<Attributes>>(attribute: K): RefCollectionValue<Attributes[K]>[];
    /** @hidden */
    protected addRef<K extends RefListKeys<Attributes> | RefSetKeys<Attributes>>(attribute: K, value: RefCollectionValue<Attributes[K]>, attributes?: Record<string, unknown>): this;
    /** @hidden */
    protected removeRef<K extends RefListKeys<Attributes> | RefSetKeys<Attributes>>(attribute: K, value: RefCollectionValue<Attributes[K]>): this;
    /** @hidden */
    private assertRefList;
    /**********************************************************************************************
     * RefMap: Named 1:many (map) graph node references.
     */
    /** @hidden */
    protected listRefMapKeys<K extends RefMapKeys<Attributes>>(attribute: K): string[];
    /** @hidden */
    protected listRefMapValues<K extends RefMapKeys<Attributes>, V extends RefMapValue<Attributes[K]>>(attribute: K): V[];
    /** @hidden */
    protected getRefMap<K extends RefMapKeys<Attributes>, V extends RefMapValue<Attributes[K]>>(attribute: K, key: string): V | null;
    /** @hidden */
    protected setRefMap<K extends RefMapKeys<Attributes>, V extends RefMapValue<Attributes[K]>>(attribute: K, key: string, value: V | null, metadata?: Record<string, unknown>): this;
    /** @hidden */
    private assertRefMap;
    /**********************************************************************************************
     * Events.
     */
    /**
     * Dispatches an event on the GraphNode, and on the associated
     * Graph. Event types on the graph are prefixed, `"node:[type]"`.
     */
    dispatchEvent(event: BaseEvent): this;
    /**********************************************************************************************
     * Internal.
     */
    /** @hidden */
    _destroyRef<K extends RefKeys<Attributes> | RefListKeys<Attributes> | RefSetKeys<Attributes> | RefMapKeys<Attributes>>(ref: GraphEdge<this, GraphNode & Attributes[K]>): void;
}
export {};
