import { type Primitive, type Transform } from '@gltf-transform/core';
/** Options for the {@link unwrap} transform. */
export interface UnwrapOptions {
    /** watlas instance. */
    watlas: unknown;
    /**
     * Target texture coordinate index (0, 1, 2, ...) for generated unwrapping.
     * Default: 0.
     */
    texcoord?: number;
    /**
     * Whether to overwrite existing attributes at the target texCoord index, if
     * any. Default: false.
     */
    overwrite?: boolean;
    /**
     * Methods of grouping texcoords with the {@link unwrap} function.
     * Default: 'mesh'.
     */
    groupBy?: 'primitive' | 'mesh' | 'scene';
}
/** Options for the {@link unwrapPrimitives} function. */
export interface UnwrapPrimitivesOptions {
    /** watlas instance. */
    watlas: unknown;
    /**
     * Target texture coordinate index (0, 1, 2, ...) for generated unwrapping.
     * Default: 0.
     */
    texcoord?: number;
    /**
     * Whether to overwrite existing attributes at the target texCoord index, if
     * any. Default: false.
     */
    overwrite?: boolean;
    /**
     * Per-primitive texel density weights. Texel space in the atlas is allocated
     * proportionally with geometry dimensions in local space. If specified,
     * weights scale the allocation. Default: [1, 1, 1, ...].
     */
    weights?: number[];
}
export declare const UNWRAP_DEFAULTS: Required<Omit<UnwrapOptions, 'watlas'>>;
/**
 * Generate new texture coordinates (“UV mappings”) for {@link Primitive Primitives}.
 * Useful for adding texture coordinates in scenes without existing UVs, or for
 * creating a second set of texture coordinates for baked textures such as ambient
 * occlusion maps and lightmaps. Operation may increase vertex count to
 * accommodate UV seams.
 *
 * UV layouts may be grouped, reducing the number of textures required. Available
 * groupings:
 *
 * - `"primitive"`: Each primitive is given it's own texcoord atlas.
 * - `"mesh"`: All primitives in a mesh share a texcoord atlas. (default)
 * - `"scene"`: All primitives in the scene share a texcoord atlas.
 *
 * Example:
 *
 * ```ts
 * import * as watlas from 'watlas';
 * import { unwrap } from '@gltf-transform/functions';
 *
 * // Generate a TEXCOORD_1 attribute for all primitives.
 * await document.transform(
 *   unwrap({ watlas, texcoord: 1, overwrite: true, groupBy: 'scene' })
 * );
 * ```
 *
 * For more control and customization, see {@link unwrapPrimitives}.
 *
 * @experimental
 * @category Transforms
 */
export declare function unwrap(_options: UnwrapOptions): Transform;
/**
 * Generate new texture coordinates (“UV mappings”) for {@link Primitive Primitives}.
 * Useful for adding texture coordinates in scenes without existing UVs, or for
 * creating a second set of texture coordinates for baked textures such as ambient
 * occlusion maps and lightmaps. Operation may increase vertex count to
 * accommodate UV seams.
 *
 * UV layouts may be grouped, reducing the number of textures required. Available
 * groupings:
 *
 * - `"primitive"`: Each primitive is given it's own texcoord atlas.
 * - `"mesh"`: All primitives in a mesh share a texcoord atlas. (default)
 * - `"scene"`: All primitives in the scene share a texcoord atlas.
 *
 * watlas must be initialized before calling this function.
 *
 * Example:
 *
 * ```ts
 * import * as watlas from 'watlas';
 * import { unwrapPrimitives } from '@gltf-transform/functions';
 *
 * // Initialize watlas.
 * await watlas.Initialize();
 *
 * // Generate a TEXCOORD_1 attribute for the specified primitives.
 * unwrapPrimitives(mesh.listPrimitives(), {
 *   watlas,
 *   texcoord: 1,
 *   overwrite: true
 * });
 * ```
 *
 * To create texture coordinates for an entire Document, see {@link unwrap}.
 *
 * @experimental
 */
export declare function unwrapPrimitives(primitives: Primitive[], options: UnwrapPrimitivesOptions): void;
