import { type Node, type Transform } from '@gltf-transform/core';
export interface UninstanceOptions {
}
/**
 * Removes extension {@link EXTMeshGPUInstancing}, reversing the effects of the
 * {@link instance} transform or similar instancing operations. For each {@link Node}
 * associated with an {@link InstancedMesh}, the Node's {@link Mesh} and InstancedMesh will
 * be detached. In their place, one Node per instance will be attached to the original
 * Node as children, associated with the same Mesh. The extension, `EXT_mesh_gpu_instancing`,
 * will be removed from the {@link Document}.
 *
 * In applications that support `EXT_mesh_gpu_instancing`, removing the extension
 * is likely to substantially increase draw calls and reduce performance. Removing
 * the extension may be helpful for compatibility in applications without such support.
 *
 * Example:
 *
 * ```ts
 * import { uninstance } from '@gltf-transform/functions';
 *
 * document.getRoot().listNodes(); // → [ Node x 10 ]
 *
 * await document.transform(uninstance());
 *
 * document.getRoot().listNodes(); // → [ Node x 1000 ]
 * ```
 *
 * @category Transforms
 */
export declare function uninstance(_options?: UninstanceOptions): Transform;
/**
 * Given a {@link Node} with an {@link InstancedMesh} extension, returns a list
 * containing one Node per instance in the InstancedMesh. Each Node will have
 * the transform (translation/rotation/scale) of the corresponding instance,
 * and will be assigned to the same {@link Mesh}.
 *
 * May be used to unpack instancing previously applied with {@link instance}
 * and {@link EXTMeshGPUInstancing}. For a transform that applies this operation
 * to the entire {@link Document}, see {@link uninstance}.
 *
 * Example:
 * ```javascript
 * import { createInstanceNodes } from '@gltf-transform/functions';
 *
 * for (const instanceNode of createInstanceNodes(batchNode)) {
 *  batchNode.addChild(instanceNode);
 * }
 *
 * batchNode.setMesh(null).setExtension('EXTMeshGPUInstancing', null);
 * ```
 */
export declare function createInstanceNodes(batchNode: Node): Node[];
