import { type Node, type Transform } from '@gltf-transform/core';
/** Options for the {@link join} function. */
export interface JoinOptions {
    /**
     * Prevents joining distinct {@link Mesh Meshes} and {@link Node Nodes}.
     * Joins only Primitives found within the same parent Mesh. To preserve
     * only _named_ Nodes and Meshes, use
     * {@link JoinOptions.keepNamed keepNamed} instead. Default: false.
     */
    keepMeshes?: boolean;
    /**
     * Prevents joining _named_ {@link Mesh Meshes} and {@link Node Nodes}.
     * If {@link JoinOptions.keepMeshes keepMeshes} is enabled, keepNamed will
     * have no effect. Default: false.
     */
    keepNamed?: boolean;
    /**
     * Whether to perform cleanup steps after completing the operation. Recommended, and enabled by
     * default. Cleanup removes temporary resources created during the operation, but may also remove
     * pre-existing unused or duplicate resources in the {@link Document}. Applications that require
     * keeping these resources may need to disable cleanup, instead calling {@link dedup} and
     * {@link prune} manually (with customized options) later in the processing pipeline.
     * @experimental
     */
    cleanup?: boolean;
    /**
     * A filter function used to evaluate a condition on a given {@link Node Node}.
     * This function should return a boolean indicating whether the node
     * satisfies the provided condition.
     *
     * @param {Node} node - The node instance to be evaluated.
     * @returns {boolean} - The result of the evaluation; `true` if the condition is met, otherwise `false`.
     */
    filter?: (node: Node) => boolean;
}
export declare const JOIN_DEFAULTS: Required<JoinOptions>;
/**
 * Joins compatible {@link Primitive Primitives} and reduces draw calls.
 * Primitives are eligible for joining if they are members of the same
 * {@link Mesh} or, optionally, attached to sibling {@link Node Nodes}
 * in the scene hierarchy. For best results, apply {@link dedup} and
 * {@link flatten} first to maximize the number of Primitives that
 * can be joined.
 *
 * NOTE: In a Scene that heavily reuses the same Mesh data, joining may
 * increase vertex count. Consider alternatives, like
 * {@link instance instancing} with {@link EXTMeshGPUInstancing}.
 *
 * Example:
 *
 * ```ts
 * import { PropertyType } from '@gltf-transform/core';
 * import { join, flatten, dedup } from '@gltf-transform/functions';
 *
 * await document.transform(
 * 	dedup({ propertyTypes: [PropertyType.MATERIAL] }),
 * 	flatten(),
 * 	join({ keepNamed: false }),
 * );
 * ```
 *
 * @category Transforms
 */
export declare function join(_options?: JoinOptions): Transform;
