import type { Mesh, Node, Primitive, Scene } from '@gltf-transform/core';
/**
 * Various methods of estimating a vertex count. For some background on why
 * multiple definitions of a vertex count should exist, see [_Vertex Count
 * Higher in Engine than in 3D Software_](https://shahriyarshahrabi.medium.com/vertex-count-higher-in-engine-than-in-3d-software-badc348ada66).
 * Totals for a {@link Scene}, {@link Node}, or {@link Mesh} will not
 * necessarily match the sum of the totals for each {@link Primitive}. Choose
 * the appropriate method for a relevant total or estimate:
 *
 * - {@link getSceneVertexCount}
 * - {@link getNodeVertexCount}
 * - {@link getMeshVertexCount}
 * - {@link getPrimitiveVertexCount}
 *
 * Many rendering features, such as volumetric transmission, may lead
 * to additional passes over some or all vertices. These tradeoffs are
 * implementation-dependent, and not considered here.
 */
export declare enum VertexCountMethod {
    /**
     * Expected number of vertices processed by the vertex shader for one render
     * pass, without considering the vertex cache.
     */
    RENDER = "render",
    /**
     * Expected number of vertices processed by the vertex shader for one render
     * pass, assuming an Average Transform to Vertex Ratio (ATVR) of 1. Approaching
     * this result requires optimizing for locality of vertex references (see
     * {@link reorder}).
     *
     * References:
     * - [ACMR and ATVR](https://www.realtimerendering.com/blog/acmr-and-atvr/), Real-Time Rendering
     */
    RENDER_CACHED = "render-cached",
    /**
     * Expected number of vertices uploaded to the GPU, assuming that a client
     * uploads each unique {@link Accessor} only once. Unless glTF vertex
     * attributes are pre-processed to a known buffer layout, and the client is
     * optimized for that buffer layout, this total will be optimistic.
     */
    UPLOAD = "upload",
    /**
     * Expected number of vertices uploaded to the GPU, assuming that a client
     * uploads each unique {@link Primitive} individually, duplicating vertex
     * attribute {@link Accessor Accessors} shared by multiple primitives, but
     * never uploading the same Mesh or Primitive to GPU memory more than once.
     */
    UPLOAD_NAIVE = "upload-naive",
    /**
     * Number of vertex positions never used by any {@link Primitive}. If all
     * vertices are unused, this total will match `UPLOAD`.
     */
    UNUSED = "unused"
}
/**
 * Computes total number of vertices in a {@link Scene}, by the
 * specified method. Totals for the Scene will not necessarily match the sum
 * of the totals for each {@link Mesh} or {@link Primitive} within it. See
 * {@link VertexCountMethod} for available methods.
 */
export declare function getSceneVertexCount(scene: Scene, method: VertexCountMethod): number;
/**
 * Computes total number of vertices in a {@link Node}, by the
 * specified method. Totals for the node will not necessarily match the sum
 * of the totals for each {@link Mesh} or {@link Primitive} within it. See
 * {@link VertexCountMethod} for available methods.
 */
export declare function getNodeVertexCount(node: Node | Scene, method: VertexCountMethod): number;
/**
 * Computes total number of vertices in a {@link Mesh}, by the
 * specified method. Totals for the Mesh will not necessarily match the sum
 * of the totals for each {@link Primitive} within it. See
 * {@link VertexCountMethod} for available methods.
 */
export declare function getMeshVertexCount(mesh: Mesh, method: VertexCountMethod): number;
/**
 * Computes total number of vertices in a {@link Primitive}, by the
 * specified method. See {@link VertexCountMethod} for available methods.
 */
export declare function getPrimitiveVertexCount(prim: Primitive, method: VertexCountMethod): number;
