import { Extension, PropertyType, type ReaderContext, type WriterContext } from '@gltf-transform/core';
import { KHR_MATERIALS_TRANSMISSION } from '../constants.js';
import { Transmission } from './transmission.js';
/**
 * [`KHR_materials_transmission`](https://github.com/KhronosGroup/gltf/blob/main/extensions/2.0/Khronos/KHR_materials_transmission/)
 * provides a common type of optical transparency: infinitely-thin materials with no refraction,
 * scattering, or dispersion.
 *
 * ![Illustration](/media/extensions/khr-materials-transmission.png)
 *
 * > _**Figure:** Sphere using `KHR_materials_transmission` with varying roughness (0.0, 0.2, 0.4).
 * > Source: Khronos Group._
 *
 * While default PBR materials using alpha blending become invisible as their opacity approaches
 * zero, a transmissive material continues to reflect light in a glass-like manner, even at low
 * transmission values. When combined with {@link KHRMaterialsVolume}, transmission may be used for
 * thicker materials and refractive effects.
 *
 * Properties:
 * - {@link Transmission}
 *
 * ### Example
 *
 * The `KHRMaterialsTransmission` class provides a single {@link ExtensionProperty} type,
 * `Transmission`, which may be attached to any {@link Material} instance. For example:
 *
 * ```typescript
 * import { KHRMaterialsTransmission, Transmission } from '@gltf-transform/extensions';
 *
 * // Create an Extension attached to the Document.
 * const transmissionExtension = document.createExtension(KHRMaterialsTransmission);
 *
 * // Create a Transmission property.
 * const transmission = transmissionExtension.createTransmission()
 * 	.setTransmissionFactor(1.0);
 *
 * // Attach the property to a Material.
 * material.setExtension('KHR_materials_transmission', transmission);
 * ```
 */
export declare class KHRMaterialsTransmission extends Extension {
    static readonly EXTENSION_NAME: typeof KHR_MATERIALS_TRANSMISSION;
    readonly extensionName: typeof KHR_MATERIALS_TRANSMISSION;
    readonly prereadTypes: PropertyType[];
    readonly prewriteTypes: PropertyType[];
    /** Creates a new Transmission property for use on a {@link Material}. */
    createTransmission(): Transmission;
    /** @hidden */
    read(_context: ReaderContext): this;
    /** @hidden */
    write(_context: WriterContext): this;
    /** @hidden */
    preread(context: ReaderContext): this;
    /** @hidden */
    prewrite(context: WriterContext): this;
}
