import { BufferViewUsage } from '../constants.js';
import type { Document } from '../document.js';
import type { JSONDocument } from '../json-document.js';
import type { Accessor, Animation, Buffer, Camera, Material, Mesh, Node, Property, Scene, Skin, Texture, TextureInfo } from '../properties/index.js';
import type { GLTF } from '../types/gltf.js';
import { type ILogger } from '../utils/index.js';
import type { WriterOptions } from './writer.js';
type PropertyDef = GLTF.IScene | GLTF.INode | GLTF.IMaterial | GLTF.ISkin | GLTF.ITexture;
declare enum BufferViewTarget {
    ARRAY_BUFFER = 34962,
    ELEMENT_ARRAY_BUFFER = 34963
}
/**
 * Model class providing writing state to a {@link GLTFWriter} and its {@link Extension}
 * implementations.
 *
 * @hidden
 */
export declare class WriterContext {
    private readonly _doc;
    readonly jsonDoc: JSONDocument;
    readonly options: Required<WriterOptions>;
    /** Explicit buffer view targets defined by glTF specification. */
    static readonly BufferViewTarget: typeof BufferViewTarget;
    /**
     * Implicit buffer view usage, not required by glTF specification, but nonetheless useful for
     * proper grouping of accessors into buffer views. Additional usages are defined by extensions,
     * like `EXT_mesh_gpu_instancing`.
     */
    static readonly BufferViewUsage: typeof BufferViewUsage;
    /** Maps usage type to buffer view target. Usages not mapped have undefined targets. */
    static readonly USAGE_TO_TARGET: {
        [key: string]: BufferViewTarget | undefined;
    };
    readonly accessorIndexMap: Map<Accessor, number>;
    readonly animationIndexMap: Map<Animation, number>;
    readonly bufferIndexMap: Map<Buffer, number>;
    readonly cameraIndexMap: Map<Camera, number>;
    readonly skinIndexMap: Map<Skin, number>;
    readonly materialIndexMap: Map<Material, number>;
    readonly meshIndexMap: Map<Mesh, number>;
    readonly nodeIndexMap: Map<Node, number>;
    readonly imageIndexMap: Map<Texture, number>;
    readonly textureDefIndexMap: Map<string, number>;
    readonly textureInfoDefMap: Map<TextureInfo, GLTF.ITextureInfo>;
    readonly samplerDefIndexMap: Map<string, number>;
    readonly sceneIndexMap: Map<Scene, number>;
    readonly imageBufferViews: Uint8Array[];
    readonly otherBufferViews: Map<Buffer, Uint8Array[]>;
    readonly otherBufferViewsIndexMap: Map<Uint8Array, number>;
    readonly extensionData: {
        [key: string]: unknown;
    };
    bufferURIGenerator: UniqueURIGenerator<Buffer>;
    imageURIGenerator: UniqueURIGenerator<Texture>;
    logger: ILogger;
    private readonly _accessorUsageMap;
    readonly accessorUsageGroupedByParent: Set<string>;
    readonly accessorParents: Map<Accessor, Property>;
    constructor(_doc: Document, jsonDoc: JSONDocument, options: Required<WriterOptions>);
    /**
     * Creates a TextureInfo definition, and any Texture or Sampler definitions it requires. If
     * possible, Texture and Sampler definitions are shared.
     */
    createTextureInfoDef(texture: Texture, textureInfo: TextureInfo): GLTF.ITextureInfo;
    createPropertyDef(property: Property): PropertyDef;
    createAccessorDef(accessor: Accessor): GLTF.IAccessor;
    createImageData(imageDef: GLTF.IImage, data: Uint8Array, texture: Texture): void;
    assignResourceURI(uri: string, data: Uint8Array, throwOnConflict: boolean): void;
    /**
     * Returns implicit usage type of the given accessor, related to grouping accessors into
     * buffer views. Usage is a superset of buffer view target, including ARRAY_BUFFER and
     * ELEMENT_ARRAY_BUFFER, but also usages that do not match GPU buffer view targets such as
     * IBMs. Additional usages are defined by extensions, like `EXT_mesh_gpu_instancing`.
     */
    getAccessorUsage(accessor: Accessor): BufferViewUsage | string;
    /**
     * Sets usage for the given accessor. Some accessor types must be grouped into
     * buffer views with like accessors. This includes the specified buffer view "targets", but
     * also implicit usage like IBMs or instanced mesh attributes. If unspecified, an accessor
     * will be grouped with other accessors of unspecified usage.
     */
    addAccessorToUsageGroup(accessor: Accessor, usage: BufferViewUsage | string): this;
}
export declare class UniqueURIGenerator<T extends Texture | Buffer> {
    private readonly multiple;
    private readonly basename;
    private counter;
    constructor(multiple: boolean, basename: (t: T) => string);
    createURI(object: T, extension: string): string;
}
export {};
