import { EventEmitter } from 'events';
import fs from 'fs';
import path from 'path';
import kebabCase from 'lodash/kebabCase.js';
import mapKeys from 'lodash/mapKeys.js';
import reduce from 'lodash/reduce.js';
import { format, createLogger, transports } from 'winston';
import { inspect, format as format$1 } from 'util';
import c from 'chalk';
export { default as chalk } from 'chalk';
import replace from 'lodash/replace.js';
import { EOL } from 'os';
import camelCase from 'lodash/camelCase.js';
import isNumber from 'lodash/isNumber.js';
import { table, getBorderCharacters } from 'table';
import filter from 'lodash/filter.js';
import sortBy from 'lodash/sortBy.js';
import filter$1 from 'lodash/fp/filter.js';
import map from 'lodash/fp/map.js';
import flatMap from 'lodash/flatMap.js';
import wrap from 'wrap-ansi';
import map$1 from 'lodash/map.js';
import zipObject from 'lodash/zipObject.js';
import invert from 'lodash/invert.js';
import pickBy from 'lodash/pickBy.js';
import { glob } from 'glob';
import findIndex from 'lodash/findIndex.js';

function _extends() {
  _extends = Object.assign ? Object.assign.bind() : function (target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];
      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }
    return target;
  };
  return _extends.apply(this, arguments);
}
function _objectWithoutPropertiesLoose(source, excluded) {
  if (source == null) return {};
  var target = {};
  var sourceKeys = Object.keys(source);
  var key, i;
  for (i = 0; i < sourceKeys.length; i++) {
    key = sourceKeys[i];
    if (excluded.indexOf(key) >= 0) continue;
    target[key] = source[key];
  }
  return target;
}

/**
 * @packageDocumentation
 * @internal
 */
class BaseError extends Error {
  constructor(message, meta = {}) {
    super(message);
    this.meta = void 0;
    Object.setPrototypeOf(this, new.target.prototype);
    this.name = this.constructor.name;
    this.meta = meta;
    Error.captureStackTrace(this, this.constructor);
  }
}

/**
 * @packageDocumentation
 * @internal
 */
class ActionError extends BaseError {
  constructor(error) {
    const message = typeof error === "string" ? error : error.message;
    super(message, {
      error
    });
  }
}

const _excluded = ["level"];
const caporalFormat = format.printf(data => {
  const {
      level
    } = data,
    meta = _objectWithoutPropertiesLoose(data, _excluded);
  let {
    message
  } = data;
  let prefix = "";
  const levelStr = getLevelString(level);
  const metaStr = formatMeta(meta);
  if (metaStr !== "") {
    message += `${EOL}${levelStr}: ${metaStr}`;
  }
  if (level === "error") {
    // TODO(cleanup)
    const paddingLeft = meta.paddingLeft;
    const spaces = " ".repeat(paddingLeft || 7);
    prefix = EOL;
    message = `${replace(message, new RegExp(EOL, "g"), EOL + spaces)}${EOL}`;
  }
  return `${prefix}${levelStr}: ${message}`;
});
function formatMeta(meta) {
  delete meta.message;
  delete meta[Symbol.for("level")];
  delete meta[Symbol.for("message")];
  delete meta[Symbol.for("splat")];
  if (Object.keys(meta).length) {
    return inspect(meta, {
      showHidden: false,
      colors: logger.colorsEnabled
    });
  }
  return "";
}
function getLevelString(level) {
  if (!logger.colorsEnabled) {
    return level;
  }
  let levelStr = level;
  switch (level) {
    case "error":
      levelStr = c.bold.redBright(level);
      break;
    case "warn":
      levelStr = c.hex("#FF9900")(level);
      break;
    case "info":
      levelStr = c.hex("#569cd6")(level);
      break;
    case "debug":
    case "silly":
      levelStr = c.dim(level);
      break;
  }
  return levelStr;
}
let logger = createDefaultLogger();
function setLogger(loggerObj) {
  logger = loggerObj;
}
function createDefaultLogger() {
  const logger = createLogger({
    transports: [new transports.Console({
      format: format.combine(format.splat(), caporalFormat)
    })]
  });
  // disableColors() disable on the logger level,
  // while chalk supports the --color/--no-color flag
  // as well as the FORCE_COLOR env var
  logger.disableColors = () => {
    logger.transports[0].format = caporalFormat;
    logger.colorsEnabled = false;
  };
  logger.colorsEnabled = c.supportsColor !== false;
  return logger;
}

/**
 * @param err - Error object
 */
function fatalError(error) {
  if (logger.level == "debug") {
    logger.log(_extends({
      level: "error"
    }, error, {
      message: error.message + "\n\n" + error.stack,
      stack: error.stack,
      name: error.name
    }));
  } else {
    logger.error(error.message);
  }
  process.exitCode = 1;
}

/**
 * @packageDocumentation
 * @internal
 */
class InvalidValidatorError extends BaseError {
  constructor(validator) {
    super("Caporal setup error: Invalid flag validator setup.", {
      validator
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
class MissingArgumentError extends BaseError {
  constructor(argument, command) {
    const msg = `Missing required argument ${c.bold(argument.name)}.`;
    super(msg, {
      argument,
      command
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
class MissingFlagError extends BaseError {
  constructor(flag, command) {
    const msg = `Missing required flag ${c.bold(flag.allNotations.join(" | "))}.`;
    super(msg, {
      flag,
      command
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
function colorize(text) {
  return text.replace(/<([^>]+)>/gi, match => {
    return c.hex("#569cd6")(match);
  }).replace(/<command>/gi, match => {
    return c.keyword("orange")(match);
  }).replace(/\[([^[\]]+)\]/gi, match => {
    return c.hex("#aaa")(match);
  }).replace(/ --?([^\s,]+)/gi, match => {
    return c.green(match);
  });
}

/**
 * @packageDocumentation
 * @internal
 */
class ValidationSummaryError extends BaseError {
  constructor(cmd, errors) {
    const plural = errors.length > 1 ? "s" : "";
    const msg = `The following error${plural} occured:\n` + errors.map(e => "- " + e.message.replace(/\n/g, "\n  ")).join("\n") + "\n\n" + c.dim("Synopsis: ") + colorize(cmd.synopsis);
    super(msg, {
      errors
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
class NoActionError extends BaseError {
  constructor(cmd) {
    let message;
    if (cmd && !cmd.isProgramCommand()) {
      message = `Caporal Error: You haven't defined any action for command '${cmd.name}'.\nUse .action() to do so.`;
    } else {
      message = `Caporal Error: You haven't defined any action for program.\nUse .action() to do so.`;
    }
    super(message, {
      cmd
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
class OptionSynopsisSyntaxError extends BaseError {
  constructor(synopsis) {
    super(`Syntax error in option synopsis: ${synopsis}`, {
      synopsis
    });
  }
}

/**
 * Caporal-provided validator flags.
 */
var CaporalValidator;
(function (CaporalValidator) {
  /**
   * Number validator. Check that the value looks like a numeric one
   * and cast the provided value to a javascript `Number`.
   */
  CaporalValidator[CaporalValidator["NUMBER"] = 1] = "NUMBER";
  /**
   * Boolean validator. Check that the value looks like a boolean.
   * It accepts values like `true`, `false`, `yes`, `no`, `0`, and `1`
   * and will auto-cast those values to `true` or `false`.
   */
  CaporalValidator[CaporalValidator["BOOLEAN"] = 2] = "BOOLEAN";
  /**
   * String validator. Mainly used to make sure the value is a string,
   * and prevent Caporal auto-casting of numerics values and boolean
   * strings like `true` or `false`.
   */
  CaporalValidator[CaporalValidator["STRING"] = 4] = "STRING";
  /**
   * Array validator. Convert any provided value to an array. If a string is provided,
   * this validator will try to split it by commas.
   */
  CaporalValidator[CaporalValidator["ARRAY"] = 8] = "ARRAY";
})(CaporalValidator || (CaporalValidator = {}));
/**
 * Option possible value.
 *
 */
var OptionValueType;
(function (OptionValueType) {
  /**
   * Value is optional.
   */
  OptionValueType[OptionValueType["Optional"] = 0] = "Optional";
  /**
   * Value is required.
   */
  OptionValueType[OptionValueType["Required"] = 1] = "Required";
  /**
   * Option does not have any possible value
   */
  OptionValueType[OptionValueType["None"] = 2] = "None";
})(OptionValueType || (OptionValueType = {}));

const REG_SHORT_OPT = /^-[a-z]$/i;
const REG_LONG_OPT = /^--[a-z]{2,}/i;
const REG_OPT = /^(-[a-zA-Z]|--\D{1}[\w-]+)/;
function isShortOpt(flag) {
  return REG_SHORT_OPT.test(flag);
}
function isLongOpt(flag) {
  return REG_LONG_OPT.test(flag);
}
/**
 * Specific version of camelCase which does not lowercase short flags
 *
 * @param name Flag short or long name
 */
function camelCaseOpt(name) {
  return name.length === 1 ? name : camelCase(name);
}
function getCleanNameFromNotation(str, camelCased = true) {
  str = str.replace(/([[\]<>]+)/g, "").replace("...", "").replace(/^no-/, "");
  return camelCased ? camelCaseOpt(str) : str;
}
function getDashedOpt(name) {
  const l = Math.min(name.length, 2);
  return "-".repeat(l) + kebabCase(name);
}
function isNumeric(n) {
  return !isNaN(parseFloat(n)) && isFinite(Number(n));
}
function isOptionStr(str) {
  return str !== undefined && str !== "--" && REG_OPT.test(str);
}
function isConcatenatedOpt(str) {
  if (str.match(/^-([a-z]{2,})/i)) {
    return str.substr(1).split("");
  }
  return false;
}
function isNegativeOpt(opt) {
  return opt.substr(0, 5) === "--no-";
}
function isOptArray(flag) {
  return Array.isArray(flag);
}
function formatOptName(name) {
  return camelCaseOpt(name.replace(/^--?(no-)?/, ""));
}
/**
 * Parse a option synopsis
 *
 * @example
 * parseSynopsis("-f, --file <path>")
 * // Returns...
 * {
 *    longName: 'file',
 *    longNotation: '--file',
 *    shortNotation: '-f',
 *    shortName: 'f'
 *    valueType: 0, // 0 = optional, 1 = required, 2 = no value
 *    variadic: false
 *    name: 'file'
 *    notation: '--file' // either the long or short notation
 * }
 *
 * @param synopsis
 * @ignore
 */
function parseOptionSynopsis(synopsis) {
  // synopsis = synopsis.trim()
  const analysis = {
    variadic: false,
    valueType: OptionValueType.None,
    valueRequired: false,
    allNames: [],
    allNotations: [],
    name: "",
    notation: "",
    synopsis
  };
  const infos = synopsis.split(/[\s\t,]+/).reduce((acc, value) => {
    if (isLongOpt(value)) {
      acc.longNotation = value;
      acc.longName = getCleanNameFromNotation(value.substring(2));
      acc.allNames.push(acc.longName);
      acc.allNotations.push(value);
    } else if (isShortOpt(value)) {
      acc.shortNotation = value;
      acc.shortName = value.substring(1);
      acc.allNames.push(acc.shortName);
      acc.allNotations.push(value);
    } else if (value.substring(0, 1) === "[") {
      acc.valueType = OptionValueType.Optional;
      acc.valueRequired = false;
      acc.variadic = value.substr(-4, 3) === "...";
    } else if (value.substring(0, 1) === "<") {
      acc.valueType = OptionValueType.Required;
      acc.valueRequired = true;
      acc.variadic = value.substr(-4, 3) === "...";
    }
    return acc;
  }, analysis);
  if (infos.longName === undefined && infos.shortName === undefined) {
    throw new OptionSynopsisSyntaxError(synopsis);
  }
  infos.name = infos.longName || infos.shortName;
  infos.notation = infos.longNotation || infos.shortNotation;
  const fullSynopsis = _extends({}, infos);
  return fullSynopsis;
}

/**
 * @packageDocumentation
 * @internal
 */
function isCaporalValidator(validator) {
  if (typeof validator !== "number") {
    return false;
  }
  const mask = getCaporalValidatorsMask();
  const exist = (mask & validator) === validator;
  return exist;
}
function isNumericValidator(validator) {
  return isCaporalValidator(validator) && Boolean(validator & CaporalValidator.NUMBER);
}
function isStringValidator(validator) {
  return isCaporalValidator(validator) && Boolean(validator & CaporalValidator.STRING);
}
function isBoolValidator(validator) {
  return isCaporalValidator(validator) && Boolean(validator & CaporalValidator.BOOLEAN);
}
function isArrayValidator(validator) {
  return isCaporalValidator(validator) && Boolean(validator & CaporalValidator.ARRAY);
}
function getCaporalValidatorsMask() {
  return Object.values(CaporalValidator).filter(isNumber).reduce((a, b) => a | b, 0);
}
function checkCaporalValidator(validator) {
  if (!isCaporalValidator(validator)) {
    throw new InvalidValidatorError(validator);
  }
}
function checkUserDefinedValidator(validator) {
  if (typeof validator !== "function" && !(validator instanceof RegExp) && !Array.isArray(validator)) {
    throw new InvalidValidatorError(validator);
  }
}
function checkValidator(validator) {
  if (validator !== undefined) {
    typeof validator === "number" ? checkCaporalValidator(validator) : checkUserDefinedValidator(validator);
  }
}
function getTypeHint(obj) {
  let hint;
  if (isBoolValidator(obj.validator) || "boolean" in obj && obj.boolean && obj.default !== false) {
    hint = "boolean";
  } else if (isNumericValidator(obj.validator)) {
    hint = "number";
  } else if (Array.isArray(obj.validator)) {
    const stringified = JSON.stringify(obj.validator);
    if (stringified.length < 300) {
      hint = "one of " + stringified.substr(1, stringified.length - 2);
    }
  }
  return hint;
}

function buildTable(data, options = {}) {
  return table(data, _extends({
    border: getBorderCharacters(`void`),
    columnDefault: {
      paddingLeft: 0,
      paddingRight: 2
    },
    columns: {
      0: {
        paddingLeft: 4,
        width: 35
      },
      1: {
        width: 55,
        wrapWord: true,
        paddingRight: 0
      }
    },
    drawHorizontalLine: () => {
      return false;
    }
  }, options));
}
function getDefaultValueHint(obj) {
  return obj.default !== undefined && !("boolean" in obj && obj.boolean && obj.default === false) ? "default: " + JSON.stringify(obj.default) : undefined;
}
function getOptionSynopsisHelp(opt, {
  eol: crlf,
  chalk: c
}) {
  return opt.synopsis + (opt.required && opt.default === undefined ? crlf + c.dim("required") : "");
}
function getOptionsTable(options, ctx, title = "OPTIONS") {
  options = filter(options, "visible");
  if (!options.length) {
    return "";
  }
  const {
    chalk: c,
    eol: crlf,
    table,
    spaces
  } = ctx;
  const help = spaces + c.bold(title) + crlf + crlf;
  const rows = options.map(opt => {
    const def = getDefaultValueHint(opt);
    const more = [opt.typeHint, def].filter(d => d).join(", ");
    const syno = getOptionSynopsisHelp(opt, ctx);
    const desc = opt.description + (more.length ? crlf + c.dim(more) : "");
    return [syno, desc];
  });
  return help + table(rows);
}
function getArgumentsTable(args, ctx, title = "ARGUMENTS") {
  if (!args.length) {
    return "";
  }
  const {
    chalk: c,
    eol,
    eol2,
    table,
    spaces
  } = ctx;
  const help = spaces + c.bold(title) + eol2;
  const rows = args.map(a => {
    const def = getDefaultValueHint(a);
    const more = [a.typeHint, def].filter(d => d).join(", ");
    const desc = a.description + (more.length ? eol + c.dim(more) : "");
    return [a.synopsis, desc];
  });
  return help + table(rows);
}
function getCommandsTable(commands, ctx, title = "COMMANDS") {
  const {
    chalk,
    prog,
    eol2,
    table,
    spaces
  } = ctx;
  const cmdHint = `Type '${prog.getBin()} help <command>' to get some help about a command`;
  const help = spaces + chalk.bold(title) + ` ${chalk.dim("\u2014")} ` + chalk.dim(cmdHint) + eol2;
  const rows = commands.filter(c => c.visible).map(cmd => {
    return [chalk.keyword("orange")(cmd.name), cmd.description || ""];
  });
  return help + table(rows);
}

const command = async ctx => {
  const {
    cmd,
    globalOptions: globalFlags,
    eol,
    eol3,
    colorize,
    tpl
  } = ctx;
  const options = sortBy(cmd.options, "name"),
    globalOptions = Array.from(globalFlags.keys());
  const help = cmd.synopsis + eol3 + (await tpl("custom", ctx)) + getArgumentsTable(cmd.args, ctx) + eol + getOptionsTable(options, ctx) + eol + getOptionsTable(globalOptions, ctx, "GLOBAL OPTIONS");
  return colorize(help);
};

const header = ctx => {
  var _process$env;
  const {
    prog,
    chalk: c,
    spaces,
    eol,
    eol2
  } = ctx;
  const version = ((_process$env = process.env) == null ? void 0 : _process$env.NODE_ENV) === "test" ? "" : prog.getVersion();
  return eol + spaces + (prog.getName() || prog.getBin()) + " " + (version || "") + (prog.getDescription() ? " \u2014 " + c.dim(prog.getDescription()) : "") + eol2;
};

const program$1 = async ctx => {
  const {
    prog,
    globalOptions,
    eol,
    eol3,
    colorize,
    tpl
  } = ctx;
  const commands = await prog.getAllCommands();
  const options = Array.from(globalOptions.keys());
  const help = (await prog.getSynopsis()) + eol3 + (await tpl("custom", ctx)) + getCommandsTable(commands, ctx) + eol + getOptionsTable(options, ctx, "GLOBAL OPTIONS");
  return colorize(help);
};

const usage = async ctx => {
  var _cmd;
  const {
    tpl,
    prog,
    chalk: c,
    spaces,
    eol
  } = ctx;
  let {
    cmd
  } = ctx;
  // if help is asked without a `cmd` and that no command exists
  // within the program, override `cmd` with the program-command
  if (!cmd && !(await prog.hasCommands())) {
    ctx.cmd = cmd = prog.progCommand;
  }
  // usage
  const usage = `${spaces + c.bold("USAGE")} ${(_cmd = cmd) != null && _cmd.name ? "— " + c.dim(cmd.name) : ""}
  ${eol + spaces + spaces + c.dim("\u25B8")} `;
  const next = cmd ? await tpl("command", ctx) : await tpl("program", ctx);
  return usage + next;
};

const custom = ctx => {
  const {
    prog,
    cmd,
    eol2,
    eol3,
    chalk,
    colorize,
    customHelp,
    indent
  } = ctx;
  const data = customHelp.get(cmd || prog);
  if (data) {
    const txt = data.map(({
      text,
      options
    }) => {
      let str = "";
      if (options.sectionName) {
        str += chalk.bold(options.sectionName) + eol2;
      }
      const subtxt = options.colorize ? colorize(text) : text;
      str += options.sectionName ? indent(subtxt) : subtxt;
      return str + eol3;
    }).join("");
    return indent(txt) + eol3;
  }
  return "";
};

/**
 * @packageDocumentation
 * @internal
 */

var allTemplates = {
  __proto__: null,
  command: command,
  header: header,
  program: program$1,
  usage: usage,
  custom: custom
};

const templates = new Map(Object.entries(allTemplates));
const customHelpMap = new Map();
/**
 * Customize the help
 *
 * @param obj
 * @param text
 * @param options
 * @internal
 */
function customizeHelp(obj, text, options) {
  const opts = _extends({
    sectionName: "",
    colorize: true
  }, options);
  const data = customHelpMap.get(obj) || [];
  data.push({
    text,
    options: opts
  });
  customHelpMap.set(obj, data);
}
/**
 * Helper to be used to call templates from within templates
 *
 * @param name Template name
 * @param ctx Execution context
 * @internal
 */
async function tpl(name, ctx) {
  const template = templates.get(name);
  if (!template) {
    throw Error(`Caporal setup error: Unknown help template '${name}'`);
  }
  return template(ctx);
}
/**
 * @internal
 * @param program
 * @param command
 */
function getContext(program, command) {
  const spaces = " ".repeat(2);
  const ctx = {
    prog: program,
    cmd: command,
    chalk: c,
    colorize: colorize,
    customHelp: customHelpMap,
    tpl,
    globalOptions: getGlobalOptions(),
    table: buildTable,
    spaces,
    indent(str, sp = spaces) {
      return sp + replace(str.trim(), /(\r\n|\r|\n)/g, "\n" + sp);
    },
    eol: "\n",
    eol2: "\n\n",
    eol3: "\n\n\n"
  };
  return ctx;
}
/**
 * Return the help text
 *
 * @param program Program instance
 * @param command Command instance, if any
 * @internal
 */
async function getHelp(program, command) {
  const ctx = getContext(program, command);
  return [await tpl("header", ctx), await tpl("usage", ctx)].join("");
}

/**
 * Create an Option object
 *
 * @internal
 * @param synopsis
 * @param description
 * @param options
 */
function createOption(synopsis, description, options = {}) {
  // eslint-disable-next-line prefer-const
  let {
    validator,
    required,
    hidden
  } = options;
  // force casting
  required = Boolean(required);
  checkValidator(validator);
  const syno = parseOptionSynopsis(synopsis);
  let boolean = syno.valueType === OptionValueType.None || isBoolValidator(validator);
  if (validator && !isBoolValidator(validator)) {
    boolean = false;
  }
  const opt = _extends({
    kind: "option",
    default: boolean == true ? Boolean(options.default) : options.default,
    description,
    choices: Array.isArray(validator) ? validator : []
  }, syno, {
    required,
    visible: !hidden,
    boolean,
    validator
  });
  opt.typeHint = getTypeHint(opt);
  return opt;
}
/**
 * Display help. Return false to prevent further processing.
 *
 * @internal
 */
const showHelp = async ({
    program,
    command
  }) => {
    const help = await getHelp(program, command);
    // eslint-disable-next-line no-console
    console.log(help);
    program.emit("help", help);
    return false;
  },
  /**
   * Display program version. Return false to prevent further processing.
   *
   * @internal
   */
  showVersion = ({
    program
  }) => {
    // eslint-disable-next-line no-console
    console.log(program.getVersion());
    program.emit("version", program.getVersion());
    return false;
  },
  /**
   * Disable colors in output
   *
   * @internal
   */
  disableColors = ({
    logger
  }) => {
    logger.disableColors();
  },
  /**
   * Set verbosity to the maximum
   *
   * @internal
   */
  setVerbose = ({
    logger
  }) => {
    logger.level = "silly";
  },
  /**
   * Makes the program quiet, eg displaying logs with level >= warning
   */
  setQuiet = ({
    logger
  }) => {
    logger.level = "warn";
  },
  /**
   * Makes the program totally silent
   */
  setSilent = ({
    logger
  }) => {
    logger.silent = true;
  },
  /**
   * Install completion
   */
  installComp = ({
    program
  }) => {
    throw new Error("Completion not supported.");
  },
  /**
   * Uninstall completion
   */
  uninstallComp = ({
    program
  }) => {
    throw new Error("Completion not supported.");
  };
/**
 * Global options container
 *
 * @internal
 */
let globalOptions;
/**
 * Get the list of registered global flags
 *
 * @internal
 */
function getGlobalOptions() {
  if (globalOptions === undefined) {
    globalOptions = setupGlobalOptions();
  }
  return globalOptions;
}
/**
 * Set up the global flags
 *
 * @internal
 */
function setupGlobalOptions() {
  const help = createOption("-h, --help", "Display global help or command-related help."),
    verbose = createOption("-v, --verbose", "Verbose mode: will also output debug messages."),
    quiet = createOption("--quiet", "Quiet mode - only displays warn and error messages."),
    silent = createOption("--silent", "Silent mode: does not output anything, giving no indication of success or failure other than the exit code."),
    version = createOption("-V, --version", "Display version."),
    color = createOption("--no-color", "Disable use of colors in output."),
    installCompOpt = createOption("--install-completion", "Install completion for your shell.", {
      hidden: true
    }),
    uninstallCompOpt = createOption("--uninstall-completion", "Uninstall completion for your shell.", {
      hidden: true
    });
  return new Map([[help, showHelp], [version, showVersion], [color, disableColors], [verbose, setVerbose], [quiet, setQuiet], [silent, setSilent], [installCompOpt, installComp], [uninstallCompOpt, uninstallComp]]);
}
/**
 * Disable a global option
 *
 * @param name Can be the option short/long name or notation
 */
function disableGlobalOption(name) {
  const opts = getGlobalOptions();
  for (const [opt] of opts) {
    if (opt.allNames.includes(name) || opt.allNotations.includes(name)) {
      return opts.delete(opt);
    }
  }
  return false;
}
/**
 * Add a global option to the program.
 * A global option is available at the program level,
 * and associated with one given {@link Action}.
 *
 * @param a {@link Option} instance, for example created using {@link createOption()}
 */
function addGlobalOption(opt, action) {
  return getGlobalOptions().set(opt, action);
}
/**
 * Process global options, if any
 * @internal
 */
async function processGlobalOptions(parsed, program, command) {
  const {
    options
  } = parsed;
  const actionsParams = _extends({}, parsed, {
    logger,
    program,
    command
  });
  const promises = Object.entries(options).map(([opt]) => {
    const action = findGlobalOptAction(opt);
    if (action) {
      return action(actionsParams);
    }
  });
  const results = await Promise.all(promises);
  return results.some(r => r === false);
}
/**
 * Find a global Option action from the option name (short or long)
 *
 * @param name Short or long name
 * @internal
 */
function findGlobalOptAction(name) {
  for (const [opt, action] of getGlobalOptions()) {
    if (opt.allNames.includes(name)) {
      return action;
    }
  }
}
/**
 * Find a global Option by it's name (short or long)
 *
 * @param name Short or long name
 * @internal
 */
function findGlobalOption(name) {
  for (const [opt] of getGlobalOptions()) {
    if (opt.allNames.includes(name)) {
      return opt;
    }
  }
}

/**
 * @packageDocumentation
 * @internal
 */
function levenshtein(a, b) {
  if (a === b) {
    return 0;
  }
  if (!a.length || !b.length) {
    return a.length || b.length;
  }
  let cell = 0;
  let lcell = 0;
  let dcell = 0;
  const row = [...Array(b.length + 1).keys()];
  for (let i = 0; i < a.length; i++) {
    dcell = i;
    lcell = i + 1;
    for (let j = 0; j < b.length; j++) {
      cell = a[i] === b[j] ? dcell : Math.min(...[dcell, row[j + 1], lcell]) + 1;
      dcell = row[j + 1];
      row[j] = lcell;
      lcell = cell;
    }
    row[row.length - 1] = cell;
  }
  return cell;
}

/**
 * @packageDocumentation
 * @internal
 */
const MAX_DISTANCE = 2;
const sortByDistance = (a, b) => a.distance - b.distance;
const keepMeaningfulSuggestions = s => s.distance <= MAX_DISTANCE;
const possibilitesMapper = (input, p) => {
  return {
    suggestion: p,
    distance: levenshtein(input, p)
  };
};
/**
 * Get autocomplete suggestions
 *
 * @param {String} input - User input
 * @param {String[]} possibilities - Possibilities to retrieve suggestions from
 */
function getSuggestions(input, possibilities) {
  return possibilities.map(p => possibilitesMapper(input, p)).filter(keepMeaningfulSuggestions).sort(sortByDistance).map(p => p.suggestion);
}
/**
 * Make diff bolder in a string
 *
 * @param from original string
 * @param to target string
 */
function boldDiffString(from, to) {
  return [...to].map((char, index) => {
    if (char != from.charAt(index)) {
      return c.bold.greenBright(char);
    }
    return char;
  }).join("");
}

/**
 * @packageDocumentation
 * @internal
 */
/**
 * @todo Rewrite
 */
class UnknownOptionError extends BaseError {
  constructor(flag, command) {
    const longFlags = filter$1(f => f.name.length > 1),
      getFlagNames = map(f => f.name),
      possibilities = getFlagNames([...longFlags(command.options), ...getGlobalOptions().keys()]),
      suggestions = getSuggestions(flag, possibilities);
    let msg = `Unknown option ${c.bold.redBright(getDashedOpt(flag))}. `;
    if (suggestions.length) {
      msg += "Did you mean " + suggestions.map(s => boldDiffString(getDashedOpt(flag), getDashedOpt(s))).join(" or maybe ") + " ?";
    }
    super(msg, {
      flag,
      command
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
/**
 * @todo Rewrite
 */
class UnknownOrUnspecifiedCommandError extends BaseError {
  constructor(program, command) {
    const possibilities = filter(flatMap(program.getCommands(), c => [c.name, ...c.getAliases()]));
    let msg = "";
    if (command) {
      msg = `Unknown command ${c.bold(command)}.`;
      const suggestions = getSuggestions(command, possibilities);
      if (suggestions.length) {
        msg += " Did you mean " + suggestions.map(s => boldDiffString(command, s)).join(" or maybe ") + " ?";
      }
    } else {
      msg = "Unspecified command. Available commands are:\n" + wrap(possibilities.map(p => c.whiteBright(p)).join(", "), 60) + "." + `\n\nFor more help, type ${c.whiteBright(program.getBin() + " --help")}`;
    }
    super(msg, {
      command
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
function isOptionObject(obj) {
  return "allNotations" in obj;
}
class ValidationError extends BaseError {
  constructor({
    value,
    error,
    validator,
    context
  }) {
    let message = error instanceof Error ? error.message : error;
    const varName = isOptionObject(context) ? "option" : "argument";
    const name = isOptionObject(context) ? context.allNotations.join("|") : context.synopsis;
    if (isCaporalValidator(validator)) {
      switch (validator) {
        case CaporalValidator.NUMBER:
          message = format$1('Invalid value for %s %s.\nExpected a %s but got "%s".', varName, c.redBright(name), c.underline("number"), c.redBright(value));
          break;
        case CaporalValidator.BOOLEAN:
          message = format$1('Invalid value for %s %s.\nExpected a %s (true, false, 0, 1), but got "%s".', varName, c.redBright(name), c.underline("boolean"), c.redBright(value));
          break;
        case CaporalValidator.STRING:
          message = format$1('Invalid value for %s %s.\nExpected a %s, but got "%s".', varName, c.redBright(name), c.underline("string"), c.redBright(value));
          break;
      }
    } else if (Array.isArray(validator)) {
      message = format$1('Invalid value for %s %s.\nExpected one of %s, but got "%s".', varName, c.redBright(name), "'" + validator.join("', '") + "'", c.redBright(value));
    } else if (validator instanceof RegExp) {
      message = format$1('Invalid value for %s %s.\nExpected a value matching %s, but got "%s".', varName, c.redBright(name), c.whiteBright(validator.toString()), c.redBright(value));
    }
    super(message + "");
  }
}

/**
 * @packageDocumentation
 * @internal
 */
class TooManyArgumentsError extends BaseError {
  constructor(cmd, range, argsCount) {
    const expArgsStr = range.min === range.max ? `exactly ${range.min}.` : `between ${range.min} and ${range.max}.`;
    const cmdName = cmd.isProgramCommand() ? "" : `for command ${c.bold(cmd.name)}`;
    const message = format$1(`Too many argument(s) %s. Got %s, expected %s`, cmdName, c.bold.redBright(argsCount), c.bold.greenBright(expArgsStr));
    super(message, {
      command: cmd
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
/**
 * Validate using a RegExp
 *
 * @param validator
 * @param value
 * @ignore
 */
function validateWithRegExp(validator, value, context) {
  if (Array.isArray(value)) {
    return value.map(v => {
      return validateWithRegExp(validator, v, context);
    });
  }
  if (!validator.test(value + "")) {
    throw new ValidationError({
      validator: validator,
      value,
      context
    });
  }
  return value;
}

/**
 * @packageDocumentation
 * @internal
 */
/**
 * Validate using an array of valid values.
 *
 * @param validator
 * @param value
 * @ignore
 */
function validateWithArray(validator, value, context) {
  if (Array.isArray(value)) {
    value.forEach(v => validateWithArray(validator, v, context));
  } else if (validator.includes(value) === false) {
    throw new ValidationError({
      validator,
      value,
      context
    });
  }
  return value;
}

/**
 * @packageDocumentation
 * @internal
 */
async function validateWithFunction(validator, value, context) {
  if (Array.isArray(value)) {
    return Promise.all(value.map(v => {
      return validateWithFunction(validator, v, context);
    }));
  }
  try {
    return await validator(value);
  } catch (error) {
    throw new ValidationError({
      validator,
      value,
      error,
      context
    });
  }
}

/**
 * @packageDocumentation
 * @internal
 */
function validateWithCaporal(validator, value, context, skipArrayValidation = false) {
  if (!skipArrayValidation && isArrayValidator(validator)) {
    return validateArrayFlag(validator, value, context);
  } else if (Array.isArray(value)) {
    // should not happen!
    throw new ValidationError({
      error: "Expected a scalar value, got an array",
      value,
      validator,
      context
    });
  } else if (isNumericValidator(validator)) {
    return validateNumericFlag(validator, value, context);
  } else if (isStringValidator(validator)) {
    return validateStringFlag(value);
  } else if (isBoolValidator(validator)) {
    return validateBoolFlag(value, context);
  }
  return value;
}
/**
 * The string validator actually just cast the value to string
 *
 * @param value
 * @ignore
 */
function validateBoolFlag(value, context) {
  if (typeof value === "boolean") {
    return value;
  } else if (/^(true|false|yes|no|0|1)$/i.test(String(value)) === false) {
    throw new ValidationError({
      value,
      validator: CaporalValidator.BOOLEAN,
      context
    });
  }
  return /^0|no|false$/.test(String(value)) === false;
}
function validateNumericFlag(validator, value, context) {
  const str = value + "";
  if (Array.isArray(value) || !isNumeric(str)) {
    throw new ValidationError({
      value,
      validator,
      context
    });
  }
  return parseFloat(str);
}
function validateArrayFlag(validator, value, context) {
  const values = typeof value === "string" ? value.split(",") : !Array.isArray(value) ? [value] : value;
  return flatMap(values, el => validateWithCaporal(validator, el, context, true));
}
/**
 * The string validator actually just cast the value to string
 *
 * @param value
 * @ignore
 */
function validateStringFlag(value) {
  return value + "";
}

/**
 * @packageDocumentation
 * @internal
 */
function validate(value, validator, context) {
  if (typeof validator === "function") {
    return validateWithFunction(validator, value, context);
  } else if (validator instanceof RegExp) {
    return validateWithRegExp(validator, value, context);
  } else if (Array.isArray(validator)) {
    return validateWithArray(validator, value, context);
  }
  // Caporal flag validator
  else if (isCaporalValidator(validator)) {
    return validateWithCaporal(validator, value, context);
  }
  return value;
}

/**
 * @packageDocumentation
 * @internal
 */
function findArgument(cmd, name) {
  return cmd.args.find(a => a.name === name);
}

/**
 * @packageDocumentation
 * @internal
 */
/**
 * Get the number of required argument for a given command
 *
 * @param cmd
 */
function getRequiredArgsCount(cmd) {
  return cmd.args.filter(a => a.required).length;
}
function getArgsObjectFromArray(cmd, args) {
  const result = {};
  return cmd.args.reduce((acc, arg, index) => {
    if (args[index] !== undefined) {
      acc[arg.name] = args[index];
    } else if (arg.default !== undefined) {
      acc[arg.name] = arg.default;
    }
    return acc;
  }, result);
}
/**
 * Check if the given command has at leat one variadic argument
 *
 * @param cmd
 */
function hasVariadicArgument(cmd) {
  return cmd.args.some(a => a.variadic);
}
function getArgsRange(cmd) {
  const min = getRequiredArgsCount(cmd);
  const max = hasVariadicArgument(cmd) ? Infinity : cmd.args.length;
  return {
    min,
    max
  };
}
function checkRequiredArgs(cmd, args, parsedArgv) {
  const errors = cmd.args.reduce((acc, arg) => {
    if (args[arg.name] === undefined && arg.required) {
      acc.push(new MissingArgumentError(arg, cmd));
    }
    return acc;
  }, []);
  // Check if there is more args than specified
  if (cmd.strictArgsCount) {
    const numArgsError = checkNumberOfArgs(cmd, parsedArgv);
    if (numArgsError) {
      errors.push(numArgsError);
    }
  }
  return errors;
}
function checkNumberOfArgs(cmd, args) {
  const range = getArgsRange(cmd);
  const argsCount = Object.keys(args).length;
  if (range.max !== Infinity && range.max < Object.keys(args).length) {
    return new TooManyArgumentsError(cmd, range, argsCount);
  }
}
function removeCommandFromArgs(cmd, args) {
  const words = cmd.name.split(" ").length;
  return args.slice(words);
}
function validateArg(arg, value) {
  return arg.validator ? validate(value, arg.validator, arg) : value;
}
/**
 *
 * @param cmd
 * @param parsedArgv
 *
 * @todo Bugs:
 *
 *
 * ts-node examples/pizza/pizza.ts cancel my-order jhazd hazd
 *
 * -> result ok, should be too many arguments
 *
 */
async function validateArgs(cmd, parsedArgv) {
  // remove the command from the argv array
  const formatedArgs = cmd.isProgramCommand() ? parsedArgv : removeCommandFromArgs(cmd, parsedArgv);
  // transfrom args array to object, and set defaults for arguments not passed
  const argsObj = getArgsObjectFromArray(cmd, formatedArgs);
  const errors = [];
  const validations = reduce(argsObj, (acc, value, key) => {
    const arg = findArgument(cmd, key);
    try {
      /* istanbul ignore if -- should not happen */
      if (!arg) {
        throw new BaseError(`Unknown argumment ${key}`);
      }
      acc[key] = validateArg(arg, value);
    } catch (e) {
      errors.push(e);
    }
    return acc;
  }, {});
  const result = await reduce(validations, async (prevPromise, value, key) => {
    const collection = await prevPromise;
    collection[key] = await value;
    return collection;
  }, Promise.resolve({}));
  errors.push(...checkRequiredArgs(cmd, result, formatedArgs));
  return {
    args: result,
    errors
  };
}

/**
 * @packageDocumentation
 * @internal
 */
/**
 * Find an option from its name for a given command
 *
 * @param cmd Command object
 * @param name Option name, short or long, camel-cased
 */
function findOption(cmd, name) {
  return cmd.options.find(o => o.allNames.find(opt => opt === name));
}

/**
 * @packageDocumentation
 * @internal
 */
function validateOption(opt, value) {
  return opt.validator ? validate(value, opt.validator, opt) : value;
}
function checkRequiredOpts(cmd, opts) {
  return cmd.options.reduce((acc, opt) => {
    if (opts[opt.name] === undefined && opt.required) {
      acc.push(new MissingFlagError(opt, cmd));
    }
    return acc;
  }, []);
}
function applyDefaults(cmd, opts) {
  return cmd.options.reduce((acc, opt) => {
    if (acc[opt.name] === undefined && opt.default !== undefined) {
      acc[opt.name] = opt.default;
    }
    return acc;
  }, opts);
}
async function validateOptions(cmd, options) {
  options = applyDefaults(cmd, options);
  const errors = [];
  const validations = reduce(options, (...args) => {
    const [acc, value, name] = args;
    const opt = findGlobalOption(name) || findOption(cmd, name);
    try {
      if (opt) {
        acc[name] = validateOption(opt, value);
      } else if (cmd.strictOptions) {
        throw new UnknownOptionError(name, cmd);
      }
    } catch (e) {
      errors.push(e);
    }
    return acc;
  }, {});
  const result = await reduce(validations, async (prevPromise, value, key) => {
    const collection = await prevPromise;
    collection[key] = await value;
    return collection;
  }, Promise.resolve({}));
  errors.push(...checkRequiredOpts(cmd, result));
  return {
    options: result,
    errors
  };
}

async function validateCall(cmd, result) {
  const {
    args: parsedArgs,
    options: parsedFlags
  } = result;
  // check if there are some global flags to handle
  const {
    options: flags,
    errors: flagsErrors
  } = await validateOptions(cmd, parsedFlags);
  const {
    args,
    errors: argsErrors
  } = await validateArgs(cmd, parsedArgs);
  return _extends({}, result, {
    args,
    options: flags,
    errors: [...argsErrors, ...flagsErrors]
  });
}

/**
 * Check if the argument is explicitly required
 *
 * @ignore
 * @param synopsis
 */
function isRequired(synopsis) {
  return synopsis.substring(0, 1) === "<";
}
/**
 *
 * @param synopsis
 */
function isVariadic(synopsis) {
  return synopsis.substr(-4, 3) === "..." || synopsis.endsWith("...");
}
function parseArgumentSynopsis(synopsis) {
  synopsis = synopsis.trim();
  const rawName = getCleanNameFromNotation(synopsis, false);
  const name = getCleanNameFromNotation(synopsis);
  const required = isRequired(synopsis);
  const variadic = isVariadic(synopsis);
  const cleanSynopsis = required ? `<${rawName}${variadic ? "..." : ""}>` : `[${rawName}${variadic ? "..." : ""}]`;
  return {
    name,
    synopsis: cleanSynopsis,
    required,
    variadic
  };
}

/**
 *
 * @param synopsis - Argument synopsis
 * @param description - Argument description
 * @param [options] - Various argument options like validator and default value
 */
function createArgument(synopsis, description, options = {}) {
  const {
    validator,
    default: defaultValue
  } = options;
  checkValidator(validator);
  const syno = parseArgumentSynopsis(synopsis);
  const arg = _extends({
    kind: "argument",
    default: defaultValue,
    description,
    choices: Array.isArray(validator) ? validator : [],
    validator
  }, syno);
  arg.typeHint = getTypeHint(arg);
  return arg;
}

function getOptsMapping(cmd) {
  const names = map$1(cmd.options, "name");
  const aliases = map$1(cmd.options, o => o.shortName || o.longName);
  const result = zipObject(names, aliases);
  return pickBy(_extends({}, result, invert(result)));
}

function createConfigurator(defaults) {
  const _defaults = defaults;
  let config = defaults;
  return {
    reset() {
      config = _defaults;
      return config;
    },
    get(key) {
      return config[key];
    },
    getAll() {
      return config;
    },
    set(props) {
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      return Object.assign(config, props);
    }
  };
}

/**
 * @ignore
 */
const PROG_CMD = "__self_cmd";
/**
 * @ignore
 */
const HELP_CMD = "help";
/**
 * Command class
 *
 */
class Command {
  /**
   *
   * @param program
   * @param name
   * @param description
   * @internal
   */
  constructor(program, name, description, config = {}) {
    this.program = void 0;
    this._action = void 0;
    this._lastAddedArgOrOpt = void 0;
    this._aliases = [];
    this._name = void 0;
    this._config = void 0;
    /**
     * Command description
     *
     * @internal
     */
    this.description = void 0;
    /**
     * Command options array
     *
     * @internal
     */
    this.options = [];
    /**
     * Command arguments array
     *
     * @internal
     */
    this.args = [];
    this.program = program;
    this._name = name;
    this.description = description;
    this._config = createConfigurator(_extends({
      visible: true
    }, config));
  }
  /**
   * Add one or more aliases so the command can be called by different names.
   *
   * @param aliases Command aliases
   */
  alias(...aliases) {
    this._aliases.push(...aliases);
    return this;
  }
  /**
   * Name getter. Will return an empty string in the program-command context
   *
   * @internal
   */
  get name() {
    return this.isProgramCommand() ? "" : this._name;
  }
  /**
   * Add an argument to the command.
   * Synopsis is a string like `<my-argument>` or `[my-argument]`.
   * Angled brackets (e.g. `<item>`) indicate required input. Square brackets (e.g. `[env]`) indicate optional input.
   *
   * Returns the {@link Command} object to facilitate chaining of methods.
   *
   * @param synopsis Argument synopsis.
   * @param description - Argument description.
   * @param [options] - Optional parameters including validator and default value.
   */
  argument(synopsis, description, options = {}) {
    this._lastAddedArgOrOpt = createArgument(synopsis, description, options);
    this.args.push(this._lastAddedArgOrOpt);
    return this;
  }
  /**
   * Set the corresponding action to execute for this command
   *
   * @param action Action to execute
   */
  action(action) {
    this._action = action;
    return this;
  }
  /**
   * Allow chaining command() calls. See {@link Program.command}.
   *
   */
  command(name, description, config = {}) {
    return this.program.command(name, description, config);
  }
  /**
   * Makes the command the default one for the program.
   */
  default() {
    this.program.defaultCommand = this;
    return this;
  }
  /**
   * Checks if the command has the given alias registered.
   *
   * @param alias
   * @internal
   */
  hasAlias(alias) {
    return this._aliases.includes(alias);
  }
  /**
   * Get command aliases.
   * @internal
   */
  getAliases() {
    return this._aliases;
  }
  /**
   * @internal
   */
  isProgramCommand() {
    return this._name === PROG_CMD;
  }
  /**
   * @internal
   */
  isHelpCommand() {
    return this._name === HELP_CMD;
  }
  /**
   * Hide the command from help.
   * Shortcut to calling `.configure({ visible: false })`.
   */
  hide() {
    return this.configure({
      visible: false
    });
  }
  /**
   * Add an option to the current command.
   *
   * @param synopsis Option synopsis like '-f, --force', or '-f, --file \<file\>', or '--with-openssl [path]'
   * @param description Option description
   * @param options Additional parameters
   */
  option(synopsis, description, options = {}) {
    const opt = this._lastAddedArgOrOpt = createOption(synopsis, description, options);
    this.options.push(opt);
    return this;
  }
  /**
   * @internal
   */
  getParserConfig() {
    const defaults = {
      boolean: [],
      string: [],
      alias: getOptsMapping(this),
      autoCast: this.autoCast,
      variadic: [],
      ddash: false
    };
    let parserOpts = this.options.reduce((parserOpts, opt) => {
      if (opt.boolean) {
        parserOpts.boolean.push(opt.name);
      }
      if (isStringValidator(opt.validator)) {
        parserOpts.string.push(opt.name);
      }
      if (opt.variadic) {
        parserOpts.variadic.push(opt.name);
      }
      return parserOpts;
    }, defaults);
    parserOpts = this.args.reduce((parserOpts, arg, index) => {
      if (!this.isProgramCommand()) {
        index++;
      }
      if (isBoolValidator(arg.validator)) {
        parserOpts.boolean.push(index);
      }
      if (isStringValidator(arg.validator)) {
        parserOpts.string.push(index);
      }
      if (arg.variadic) {
        parserOpts.variadic.push(index);
      }
      return parserOpts;
    }, parserOpts);
    return parserOpts;
  }
  /**
   * Return a reformated synopsis string
   * @internal
   */
  get synopsis() {
    const opts = this.options.length ? this.options.some(f => f.required) ? "<OPTIONS...>" : "[OPTIONS...]" : "";
    const name = this._name !== PROG_CMD ? " " + this._name : "";
    return (this.program.getBin() + name + " " + this.args.map(a => a.synopsis).join(" ") + " " + opts).trim();
  }
  /**
   * Customize command help. Can be called multiple times to add more paragraphs and/or sections.
   *
   * @param text Help contents
   * @param options Display options
   */
  help(text, options = {}) {
    customizeHelp(this, text, options);
    return this;
  }
  /**
   * Configure some behavioral properties.
   *
   * @param props properties to set/update
   */
  configure(props) {
    this._config.set(props);
    return this;
  }
  /**
   * Get a configuration property value.
   *
   * @internal
   * @param key Property key to get value for. See {@link CommandConfig}.
   */
  getConfigProperty(key) {
    return this._config.get(key);
  }
  /**
   * Get the auto-casting flag.
   *
   * @internal
   */
  get autoCast() {
    var _this$getConfigProper;
    return (_this$getConfigProper = this.getConfigProperty("autoCast")) != null ? _this$getConfigProper : this.program.getConfigProperty("autoCast");
  }
  /**
   * Auto-complete
   */
  complete(completer) {
    throw new Error("Completion not supported.");
  }
  /**
   * Toggle strict mode.
   * Shortcut to calling: `.configure({ strictArgsCount: strict, strictOptions: strict }).
   * By default, strict settings are not defined for commands, and inherit from the
   * program settings. Calling `.strict(value)` on a command will override the program
   * settings.
   *
   * @param strict boolean enabled flag
   */
  strict(strict = true) {
    return this.configure({
      strictArgsCount: strict,
      strictOptions: strict
    });
  }
  /**
   * Computed strictOptions flag.
   *
   * @internal
   */
  get strictOptions() {
    var _this$getConfigProper2;
    return (_this$getConfigProper2 = this.getConfigProperty("strictOptions")) != null ? _this$getConfigProper2 : this.program.getConfigProperty("strictOptions");
  }
  /**
   * Computed strictArgsCount flag.
   *
   * @internal
   */
  get strictArgsCount() {
    var _this$getConfigProper3;
    return (_this$getConfigProper3 = this.getConfigProperty("strictArgsCount")) != null ? _this$getConfigProper3 : this.program.getConfigProperty("strictArgsCount");
  }
  /**
   * Enable or disable auto casting of arguments & options for the command.
   * This is basically a shortcut to calling `command.configure({ autoCast: enabled })`.
   * By default, auto-casting is inherited from the program configuration.
   * This method allows overriding what's been set on the program level.
   *
   * @param enabled
   */
  cast(enabled) {
    return this.configure({
      autoCast: enabled
    });
  }
  /**
   * Visible flag
   *
   * @internal
   */
  get visible() {
    return this.getConfigProperty("visible");
  }
  /**
   * Run the action associated with the command
   *
   * @internal
   */
  async run(parsed) {
    const data = _extends({
      args: [],
      options: {},
      line: "",
      rawOptions: {},
      rawArgv: [],
      ddash: []
    }, parsed);
    try {
      // Validate args and options, including global options
      const result = await validateCall(this, data);
      const {
        args,
        options,
        ddash,
        errors
      } = result;
      // Process any global options
      const shouldStop = await processGlobalOptions(result, this.program, this);
      if (shouldStop) {
        return -1;
      }
      if (errors.length) {
        throw new ValidationSummaryError(this, errors);
      }
      if (!this._action) {
        throw new NoActionError(this);
      }
      return await this._action({
        args,
        options,
        ddash,
        logger,
        program: this.program,
        command: this
      });
    } catch (err) {
      const ctor = Object.getPrototypeOf(err).constructor.name;
      throw err instanceof BaseError && ctor !== "Error" ? err : new ActionError(err);
    }
  }
}
/**
 * Create a new command
 *
 * @internal
 */
function createCommand(...args) {
  return new Command(...args);
}

async function importCommand(file) {
  var _mod$default;
  const {
    dir,
    name
  } = path.parse(file);
  const filename = path.join(dir, name);
  const mod = require(filename);
  return (_mod$default = mod.default) != null ? _mod$default : mod;
}

async function findCommand(program, argv) {
  const commands = program.getCommands();
  const findRegisteredCommand = search => commands.find(c => c.name === search || c.hasAlias(search));
  let foundCommand;
  let i;
  for (i = 0; i < argv.length; i++) {
    const cmd = argv.slice(0, i + 1).join(" ");
    // break as soon as possible
    if (argv[i].startsWith("-")) {
      break;
    }
    const potentialCmd = findRegisteredCommand(cmd) || (await discoverCommand(program, cmd));
    foundCommand = potentialCmd || foundCommand;
  }
  return foundCommand;
}
/**
 * Search for a command in discovery path
 */
async function discoverCommand(program, cmd) {
  if (program.discoveryPath === undefined) {
    return;
  }
  const filename = cmd.split(" ").join("/");
  try {
    const fullPath = path.join(program.discoveryPath, filename);
    const cmdBuilder = await importCommand(fullPath);
    const options = {
      createCommand: createCommand.bind(null, program, cmd),
      program
    };
    return cmdBuilder(options);
    // eslint-disable-next-line no-empty
  } catch (e) {}
}

/**
 * @packageDocumentation
 * @internal
 */
function readdir(dirPath, extensions = "js,ts") {
  return new Promise((resolve, reject) => {
    if (!fs.existsSync(dirPath)) {
      return reject(new Error(`'${dirPath}' does not exist!`));
    }
    glob(`**/*.{${extensions}}`, {
      cwd: dirPath
    }).then(resolve).catch(reject);
  });
}

/**
 * @packageDocumentation
 * @internal
 */
async function scanCommands(program, dirPath) {
  const files = await readdir(dirPath);
  const imp = await Promise.all(files.map(f => importCommand(path.join(dirPath, f))));
  const data = zipObject(files, imp);
  return map$1(data, (cmdBuilder, filename) => {
    const {
      dir,
      name
    } = path.parse(filename);
    const cmd = dir ? [...dir.split("/"), name].join(" ") : name;
    const options = {
      createCommand: createCommand.bind(null, program, cmd),
      program
    };
    return cmdBuilder(options);
  });
}

const DDASH = "--";
function isDdash(str) {
  return str === DDASH;
}
function castAsBool(value) {
  if (typeof value === "boolean") {
    return value;
  }
  return /^true|1|yes|on$/.test(value);
}
function castAsString(val) {
  return val + "";
}
function autoCast(val) {
  // auto-casting "true" & "false"
  if (/^true|false$/.test(val)) {
    return val === "true";
  }
  // auto-casting numbers
  return isNumeric(val) ? parseFloat(val) : val;
}
function cast(name, val, options) {
  const cleanName = formatOptName(name);
  // Force casting to string
  if (options.string.includes(cleanName)) {
    return castAsString(val);
  }
  // Force casting to bool
  if (options.boolean.includes(cleanName) || typeof val === "boolean") {
    return castAsBool(val);
  }
  return options.autoCast ? autoCast(val) : val;
}
/**
 * Parse a line
 *
 * @param line Line to be parsed
 * @param options Parser options
 * @internal
 */
function parseLine(line, options = {}) {
  return parseArgv(options, line.split(" "));
}
/**
 *
 * @param args Return the next option position unless there is some ddash before
 */
function getNextOptPosition(args) {
  const ddash = args.indexOf("--");
  const opt = findIndex(args, isOptionStr);
  return ddash < opt && ddash !== -1 ? -1 : opt;
}
class Tree {
  constructor(argv) {
    this.argv = void 0;
    this.cursor = void 0;
    this.ddashHandled = false;
    this.argv = argv;
    this.cursor = 0;
  }
  /* istanbul ignore next */
  toJSON() {
    return {
      cursor: this.cursor,
      ddashHandled: this.ddashHandled,
      current: this.current,
      argv: this.argv
    };
  }
  markDdashHandled() {
    this.ddashHandled = true;
    return this;
  }
  hasDdashHandled() {
    return this.ddashHandled;
  }
  next() {
    return this.argv[this.cursor + 1];
  }
  slice(start, end) {
    return this.argv.slice(start, end);
  }
  sliceFromHere(end) {
    return this.slice(this.cursor, end);
  }
  forward(by = 1) {
    if (by === -1) {
      return this.end();
    }
    this.cursor += by;
    return true;
  }
  end() {
    this.cursor = this.length;
    return true;
  }
  get current() {
    return this.argv[this.cursor];
  }
  get length() {
    return this.argv.length;
  }
}
class ArgumentParser {
  constructor(config, argv) {
    this.config = void 0;
    this.args = [];
    this.ddash = [];
    this.rawArgv = void 0;
    this.line = void 0;
    this.variadicId = void 0;
    this.key = "args";
    this.config = config;
    this.line = argv.join(" ");
    this.rawArgv = argv;
  }
  toJSON() {
    return {
      args: this.args,
      ddash: this.ddash,
      rawArgv: this.rawArgv,
      line: this.line
    };
  }
  inVariadicContext() {
    const argsLen = this[this.key].length;
    if (this.config.variadic.includes(argsLen)) {
      this.variadicId = argsLen;
    }
    if (this.variadicId !== undefined) {
      return true;
    }
  }
  markDdashHandled(tree) {
    if (this.config.ddash) {
      // if ddash enabled, update the key
      this.key = "ddash";
    }
    return tree.markDdashHandled().forward();
  }
  push(...values) {
    this[this.key].push(...values.map(this.config.autoCast ? autoCast : String));
    return true;
  }
  pushVariadic(tree) {
    const args = tree.sliceFromHere();
    const until = getNextOptPosition(args);
    this.variadicId = this.variadicId || 0;
    const variadic = this[this.key][this.variadicId] = this[this.key][this.variadicId] || [];
    variadic.push(...args.slice(0, until === -1 ? undefined : until).filter(s => !isDdash(s)).map(this.config.autoCast ? autoCast : String));
    return tree.forward(until);
  }
  visit(tree) {
    if (!tree.current || isOptionStr(tree.current) && !tree.hasDdashHandled()) {
      return false;
    }
    if (isDdash(tree.current)) {
      return this.markDdashHandled(tree);
    } else if (!this.inVariadicContext()) {
      this.push(tree.current);
      return tree.forward();
    }
    return this.pushVariadic(tree);
  }
}
class OptionParser {
  constructor(config) {
    this.config = void 0;
    this.options = {};
    this.rawOptions = {};
    this.config = config;
  }
  toJSON() {
    return {
      options: this.options,
      rawOptions: this.rawOptions
    };
  }
  handleOptWithoutValue(name, tree) {
    const next = tree.next();
    const nextIsOptOrUndef = isOptionStr(next) || isDdash(next) || next === undefined;
    this.compute(name, cast(name, nextIsOptOrUndef ? true : next, this.config));
    if (!nextIsOptOrUndef) {
      tree.forward();
    }
  }
  handleConcatenatedOpts(tree, names, val) {
    if (val === undefined) {
      val = true;
      const next = tree.next();
      const last = names[names.length - 1];
      const alias = this.config.alias[last];
      const shouldTakeNextAsVal = next && !isOptionStr(next) && !isDdash(next) && !this.isBoolean(last, alias);
      if (shouldTakeNextAsVal) {
        tree.forward();
        val = next;
      }
    }
    this.computeMulti(names, val);
  }
  visit(tree) {
    // only handle options
    /* istanbul ignore if */
    if (!tree.current || !isOptionStr(tree.current) || tree.hasDdashHandled()) {
      // this is never reached because the scan stops if
      // a visior returns true, and as the Argument visitor is the first in the
      // list, arguments objects never reach the Options visitor
      // keeping it here in case we change the order of visitors
      return false;
    }
    const [name, rawval] = tree.current.split("=", 2);
    const concatOpts = isConcatenatedOpt(name);
    if (concatOpts) {
      this.handleConcatenatedOpts(tree, concatOpts, rawval);
    } else if (rawval) {
      this.compute(name, cast(name, rawval, this.config));
    } else {
      this.handleOptWithoutValue(name, tree);
    }
    return tree.forward();
  }
  compute(name, val) {
    const no = isNegativeOpt(name);
    const cleanName = formatOptName(name);
    const alias = this.config.alias[cleanName];
    if (this.isVariadic(cleanName, alias)) {
      const prop = this.options[cleanName];
      this.rawOptions[name] = this.options[cleanName] = (isOptArray(prop) ? prop : [prop]).concat(val);
    } else {
      this.rawOptions[name] = this.options[cleanName] = no ? !val : val;
    }
    if (alias) {
      this.options[alias] = this.options[cleanName];
    }
  }
  // todo: handle variadic, even for compute multi
  // TIP: (maybe just split and redirect the last char to compute())
  computeMulti(multi, val) {
    const n = multi.length;
    multi.forEach((o, index) => {
      const alias = this.config.alias[o];
      this.options[o] = index + 1 === n ? cast(o, val, this.config) : true;
      this.rawOptions["-" + o] = this.options[o];
      if (alias) {
        this.options[alias] = this.options[o];
      }
    });
  }
  isVariadic(name, alias) {
    return name in this.options && (this.config.variadic.includes(name) || this.config.variadic.includes(alias));
  }
  isBoolean(name, alias) {
    return this.config.boolean.includes(name) || this.config.boolean.includes(alias);
  }
}
/**
 * Parse command line arguments
 *
 * @param options Parser options
 * @param argv command line arguments array (a.k.a. "argv")
 */
function parseArgv(options = {}, argv = process.argv.slice(2)) {
  const parseOpts = _extends({
    autoCast: true,
    ddash: false,
    alias: {},
    boolean: [],
    string: [],
    variadic: []
  }, options);
  parseOpts.alias = _extends({}, parseOpts.alias, invert(parseOpts.alias));
  const tree = new Tree(argv);
  const flagParser = new OptionParser(parseOpts);
  const argParser = new ArgumentParser(parseOpts, argv);
  const visitors = [argParser, flagParser];
  while (tree.current) {
    visitors.some(v => v.visit(tree));
  }
  return _extends({}, flagParser.toJSON(), argParser.toJSON());
}

/**
 * @packageDocumentation
 * @internal
 */
function detectVersion() {
  try {
    // eslint-disable-next-line @typescript-eslint/no-var-requires
    return require(path.join(__filename, "..", "..", "..", "package.json")).version;
    // eslint-disable-next-line no-empty
  } catch (e) {}
}

const LOG_LEVEL_ENV_VAR = "CAPORAL_LOG_LEVEL";
// const SUPPORTED_SHELL = ["bash", "zsh", "fish"]
/**
 * Program class
 *
 * @noInheritDoc
 */
class Program extends EventEmitter {
  /**
   * Program constructor.
   * - Detects the "bin" name from process argv
   * - Detects the version from package.json
   * - Set up the help command
   * @ignore
   */
  constructor() {
    super();
    this.commands = [];
    this._config = void 0;
    this._version = void 0;
    this._name = void 0;
    this._description = void 0;
    this._programmaticMode = false;
    /**
     * @internal
     */
    this.defaultCommand = void 0;
    this._progCommand = void 0;
    this._bin = void 0;
    this._discoveryPath = void 0;
    this._discoveredCommands = void 0;
    /**
     * Number validator. Check that the value looks like a numeric one
     * and cast the provided value to a javascript `Number`.
     */
    this.NUMBER = CaporalValidator.NUMBER;
    /**
     * String validator. Mainly used to make sure the value is a string,
     * and prevent Caporal auto-casting of numerical values and boolean
     * strings like `true` or `false`.
     */
    this.STRING = CaporalValidator.STRING;
    /**
     * Array validator. Convert any provided value to an array. If a string is provided,
     * this validator will try to split it by commas.
     */
    this.ARRAY = CaporalValidator.ARRAY;
    /**
     * Boolean validator. Check that the value looks like a boolean.
     * It accepts values like `true`, `false`, `yes`, `no`, `0`, and `1`
     * and will auto-cast those values to `true` or `false`.
     */
    this.BOOLEAN = CaporalValidator.BOOLEAN;
    this._bin = path.basename(process.argv[1]);
    this._version = detectVersion();
    this._config = createConfigurator({
      strictArgsCount: true,
      strictOptions: true,
      autoCast: true,
      logLevelEnvVar: LOG_LEVEL_ENV_VAR
    });
    this.setupHelpCommand();
    this.setupErrorHandlers();
  }
  /**
   * @internal
   */
  setupErrorHandlers() {
    process.once("unhandledRejection", err => {
      if (this._programmaticMode) {
        throw err;
      } else {
        this.emit("error", err);
      }
    });
    this.on("error", fatalError);
  }
  /**
   * The program-command is the command attached directly to the program,
   * meaning there is no command-keyword used to trigger it.
   * Mainly used for programs executing only one possible action.
   *
   * @internal
   */
  get progCommand() {
    if (this._progCommand === undefined) {
      this._progCommand = createCommand(this, PROG_CMD, "");
    }
    return this._progCommand;
  }
  /**
   * Setup the help command
   */
  setupHelpCommand() {
    var _this = this;
    return this.command(HELP_CMD, "Get help about a specific command").argument("[command]", "Command name to get help for. If the command does not exist, global help will be displayed.").action(async function (actionParams) {
      const {
        args
      } = actionParams;
      const command = args.command ? await findCommand(_this, [args.command]) : undefined;
      // eslint-disable-next-line no-console
      showHelp(_extends({}, actionParams, {
        command
      }));
      return -1;
    }).hide();
  }
  /**
   * Customize program help. Can be called multiple times to add more paragraphs and/or sections.
   *
   * @param text Help contents
   * @param options Display options
   */
  help(text, options = {}) {
    customizeHelp(this, text, options);
    return this;
  }
  /**
   * Toggle strict mode.
   * Shortcut to calling: `.configure({ strictArgsCount: strict, strictOptions: strict })`.
   * By default, the program is strict, so if you want to disable strict checking,
   * just call `.strict(false)`. This setting can be overridden at the command level.
   *
   * @param strict boolean enabled flag
   */
  strict(strict = true) {
    return this.configure({
      strictArgsCount: strict,
      strictOptions: strict
    });
  }
  /**
   * Configure some behavioral properties.
   *
   * @param props properties to set/update
   */
  configure(props) {
    this._config.set(props);
    return this;
  }
  /**
   * Get a configuration property value. {@link ProgramConfig Possible keys}.
   *
   * @param key Property
   * @internal
   */
  getConfigProperty(key) {
    return this._config.get(key);
  }
  /**
   * Return a reformatted synopsis string
   *
   * @internal
   */
  async getSynopsis() {
    return (this.getBin() + " " + ((await this.hasCommands()) ? "<command> " : "") + "[ARGUMENTS...] [OPTIONS...]").trim();
  }
  /**
   * Return the discovery path, if set
   *
   * @internal
   */
  get discoveryPath() {
    return this._discoveryPath;
  }
  /**
   * Return the program version
   *
   * @internal
   */
  getVersion() {
    return this._version;
  }
  /**
   * Set the version fo your program.
   * You won't likely use this method as Caporal tries to guess it from your package.json
   */
  version(ver) {
    this._version = ver;
    return this;
  }
  /**
   * Set the program name. If not set, the filename minus the extension will be used.
   */
  name(name) {
    this._name = name;
    return this;
  }
  /**
   * Return the program name.
   *
   * @internal
   */
  getName() {
    return this._name;
  }
  /**
   * Return the program description.
   *
   * @internal
   */
  getDescription() {
    return this._description;
  }
  /**
   * Set the program description displayed in help.
   */
  description(desc) {
    this._description = desc;
    return this;
  }
  /**
   * Get the bin name (the name of your executable).
   *
   * @internal
   */
  getBin() {
    return this._bin;
  }
  /**
   * Sets the executable name. By default, it's auto-detected from the filename of your program.
   *
   * @param name Executable name
   * @example
   * ```ts
   * program.bin('myprog')
   * ```
   */
  bin(name) {
    this._bin = name;
    return this;
  }
  /**
   * Set a custom logger for your program.
   * Your logger should implement the {@link Logger} interface.
   */
  logger(logger) {
    setLogger(logger);
    return this;
  }
  /**
   * Disable a global option. Will warn if the global option
   * does not exist of has already been disabled.
   *
   * @param name Name, short, or long notation of the option to disable.
   */
  disableGlobalOption(name) {
    const disabled = disableGlobalOption(name);
    if (!disabled) {
      logger.warn("Cannot disable global option %s. Either the global option does not exist or has already been disabled.");
    }
    return this;
  }
  /**
   * Returns the list of all commands registered
   * - By default, Caporal creates one: the "help" command
   * - When calling argument() or action() on the program instance,
   * Caporal also create what is called the "program command", which
   * is a command directly attach to the program, usually used
   * in mono-command programs.
   * @internal
   */
  getCommands() {
    return this.commands;
  }
  /**
   * Add a command to the program.
   *
   * @param name Command name
   * @param description Command description
   * @example
   * ```ts
   * program.command('order', 'Order some food')
   * ```
   */
  command(name, description, config = {}) {
    const cmd = createCommand(this, name, description, config);
    this.commands.push(cmd);
    return cmd;
  }
  /**
   * Check if the program has user-defined commands.
   *
   * @internal
   * @private
   */
  async hasCommands() {
    return (await this.getAllCommands()).length > 1;
  }
  /**
   * @internal
   */
  async getAllCommands() {
    const discoveredCommands = await this.scanCommands();
    return [...this.commands, ...discoveredCommands];
  }
  /**
   * Return the log level override, if any is provided using
   * the right environment variable.
   *
   * @internal
   * @private
   */
  getLogLevelOverride() {
    return process.env[this.getConfigProperty("logLevelEnvVar")];
  }
  /**
   * Enable or disable auto casting of arguments & options at the program level.
   *
   * @param enabled
   */
  cast(enabled) {
    return this.configure({
      autoCast: enabled
    });
  }
  /**
   * Sets a *unique* action for the *entire* program.
   *
   * @param {Function} action - Action to run
   */
  action(action) {
    this.progCommand.action(action);
    return this;
  }
  /**
   * Add an argument to the *unique* command of the program.
   */
  argument(synopsis, description, options = {}) {
    return this.progCommand.argument(synopsis, description, options);
  }
  /**
   * Add an option to the *unique* command of the program,
   * or add a global option to the program when `options.global`
   * is set to `true`.
   *
   * @param synopsis Option synopsis like '-f, --force', or '-f, --file \<file\>', or '--with-openssl [path]'
   * @param description Option description
   * @param options Additional parameters
   */
  option(synopsis, description, options = {}) {
    if (options.global) {
      const opt = createOption(synopsis, description, options);
      addGlobalOption(opt, options.action);
    } else {
      this.progCommand.option(synopsis, description, options);
    }
    return this;
  }
  /**
   * Discover commands from a specified path.
   *
   * Commands must be organized into files (one command per file) in a file tree like:
   *
   * ```sh
   * └── commands
   *     ├── config
   *     │   ├── set.ts
   *     │   └── unset.ts
   *     ├── create
   *     │   ├── job.ts
   *     │   └── service.ts
   *     ├── create.ts
   *     ├── describe.ts
   *     └── get.ts
   * ```
   *
   * The code above shows a short example of `kubectl` commands and subcommands.
   * In this case, Caporal will generate the following commands:
   *
   * - kubectl get [args...] [options...]
   * - kubectl config set [args...] [options...]
   * - kubectl config unset [args...] [options...]
   * - kubectl create [args...] [options...]
   * - kubectl create job [args...] [options...]
   * - kubectl create service [args...] [options...]
   * - kubectl describe [args...] [options...]
   * - kubectl get [args...] [options...]
   *
   * Notice how the `config` command has a mandatory subcommand associated,
   * hence cannot be called without a subcommand, contrary to the `create` command.
   * This is why there is no `config.ts` in the tree.
   *
   * @param path
   */
  discover(dirPath) {
    let stat;
    try {
      stat = fs.statSync(dirPath);
      // eslint-disable-next-line no-empty
    } catch (e) {}
    if (!stat || !stat.isDirectory()) {
      throw new Error("Caporal setup error: parameter `dirPath` of discover() should be a directory");
    }
    this._discoveryPath = dirPath;
    return this;
  }
  /**
   * Do a full scan of the discovery path to get all existing commands
   * This should only be used to generate the full list of command,
   * as for help rendering
   *
   * @private
   */
  async scanCommands() {
    if (this._discoveryPath === undefined) {
      return [];
    }
    if (this._discoveredCommands) {
      return this._discoveredCommands;
    }
    this._discoveredCommands = await scanCommands(this, this._discoveryPath);
    return this._discoveredCommands;
  }
  /* istanbul ignore next */
  /**
   * Reset all commands
   *
   * @internal
   */
  reset() {
    this.commands = [];
    this._progCommand = undefined;
    this.setupHelpCommand();
    return this;
  }
  /**
   * Run the program by parsing command line arguments.
   * Caporal will automatically detect command line arguments from `process.argv` values,
   * but it can be overridden by providing the `argv` parameter. It returns a Promise
   * of the value returned by the *Action* triggered.
   *
   * ::: warning Be careful
   * This method returns a `Promise`. You'll usually ignore the returned promise and call run() like this:
   *
   * ```ts
   * [...]
   * program.action(...)
   * program.run()
   * ```
   *
   * If you do add some `.catch()` handler to it, Caporal won't display any potential errors
   * that the promise could reject, and will let you the responsibility to do it.
   * :::
   *
   * @param argv Command line arguments to parse, default to `process.argv.slice(2)`.
   */
  async run(argv) {
    if (!argv) {
      // used on web playground
      if (process.env.CAPORAL_CMD_LINE) {
        argv = process.env.CAPORAL_CMD_LINE.split(" ").slice(1);
        // defaut value for common usage
      } else {
        argv = process.argv.slice(2);
      }
    }
    /*
      Search for the command from args, then, if a default command exists,
      take it, otherwise take the command attached to the program,
      and lastly the help command/
    */
    const cmd = await this.findCommand(argv);
    // parse command line args
    const result = parseArgv(cmd == null ? void 0 : cmd.getParserConfig(), argv);
    /*
      Run command with parsed args.
      We are forced to catch a potential error to prevent the rejected
      promise to propagate un in the stack.
    */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    return this._run(result, cmd); /*
                                   .catch((e) => e) */
  }
  /**
   * Try to find the executed command from argv
   * If command cannot be found from argv, return the default command if any,
   * then the program-command if any, or finally `undefined`.
   * If argv is empty, and there is no defaultCommand or progCommand
   * use the help command
   *
   * @param argv
   */
  async findCommand(argv) {
    return (await findCommand(this, argv)) || this.defaultCommand || this._progCommand;
  }
  /**
   * Run a command, providing parsed data
   *
   * @param result
   * @param cmd
   * @internal
   */
  async _run(result, cmd) {
    // Override logger level via ENV if needed
    const loggerLevel = this.getLogLevelOverride();
    if (loggerLevel && Object.keys(logger.levels).includes(loggerLevel)) {
      logger.level = loggerLevel;
    }
    // try to run the command
    // try {
    if (!cmd) {
      // we may not have any associated command, but some global options may have been passed
      // process them, if any
      // Process any global options
      const processedResult = _extends({}, result, {
        errors: [],
        args: {}
      });
      const shouldStop = await processGlobalOptions(processedResult, this);
      if (shouldStop) {
        this.emit("run");
        return -1;
      }
      // todo: use case: "git unknown-command some args" will display "unknown command 'git'"
      // but should display "unknown command 'git unknown-command'"
      throw new UnknownOrUnspecifiedCommandError(this, result.rawArgv[0]);
    }
    const ret = await cmd.run(result);
    this.emit("run", ret);
    return ret;
  }
  /**
   * Programmatic usage. Execute input command with given arguments & options
   *
   * Not ideal regarding type casting etc.
   *
   * @param args argv array
   * @param options options object
   * @param ddash double dash array
   * @public
   */
  async exec(args, options = {}, ddash = []) {
    this._programmaticMode = true;
    const cmd = await this.findCommand(args);
    options = mapKeys(options, (v, key) => kebabCase(key));
    return this._run({
      args,
      options,
      line: args.join(" "),
      rawOptions: options,
      rawArgv: args,
      ddash
    }, cmd);
  }
}

/**
 * Main Caporal module.
 *
 * ## program
 *
 * This represents your program. You don't have to instanciate the {@link Program} class,
 * it's already done for you.
 *
 * **Usage**
 *
 * ```ts
 * // The Program instance generated for you
 * import program from "@caporal/core"
 *
 * program
 *  .command(...)
 *  .action(...)
 * [...]
 * ```
 *
 *
 * ## parseArgv()
 *
 *  This is the command line parser internaly used by Caporal.
 *
 * ::: tip Advanced usage
 * Usually, **you won't need to use the parser** directly, but if you
 * just want to parse some args without all capabilities brought
 * by Caporal, feel free to play with it.
 * :::
 *
 * **Usage**
 *
 * ```ts
 * import { parseArgv } from "@caporal/core"
 *
 * const {args, options} = parseArgv({
 *  // ... options
 * })
 * ```
 *
 * Checkout `parseArgv()` [documentation here](/api/modules/parser.md).
 *
 *
 * ## chalk
 *
 * `chalk` npm module re-export
 *
 * **Usage**
 *
 * ```ts
 * import { program, chalk } from "caporal"
 *
 * program
 *  .command('pay')
 *  .argument('<amount>', 'Amount to pay', Validator.NUMBER)
 *  .action(({logger, args}) => {
 *    logger.info("You paid $%s", chalk.red(args.amount))
 *  })
 * [...]
 * ```
 *
 *
 * @packageDocumentation
 * @module @caporal/core
 */
/**
 * @ignore
 */
const program = new Program();

export { CaporalValidator, Command, OptionValueType, Program, parseArgv, parseLine, program };
//# sourceMappingURL=caporal.modern.js.map
