<?php
/**
 * Patch für OpenSSL 3.0 auf Windows
 * Überschreibt die createLocalKeyObject Methode der Encryption-Klasse
 */

// Lade die Encryption-Klasse
if (!class_exists('\Minishlink\WebPush\Encryption')) {
    return;
}

// Erstelle eine neue Klasse, die die Encryption-Klasse erweitert
class EncryptionPatched extends \Minishlink\WebPush\Encryption
{
    /**
     * Überschreibe createLocalKeyObject mit Windows-Workaround
     */
    private static function createLocalKeyObject(): array
    {
        // Versuche verschiedene Methoden
        $methods = [
            'method1_standard',
            'method2_with_config',
            'method3_with_full_path',
            'method4_openssl_command',
        ];
        
        foreach ($methods as $method) {
            try {
                $result = self::$method();
                if ($result !== null) {
                    return $result;
                }
            } catch (\Exception $e) {
                error_log("OpenSSL Method $method failed: " . $e->getMessage());
                continue;
            }
        }
        
        throw new \RuntimeException('Unable to create the local key. All methods failed.');
    }
    
    /**
     * Methode 1: Standard-Methode
     */
    private static function method1_standard(): ?array
    {
        $keyResource = @openssl_pkey_new([
            'curve_name'       => 'prime256v1',
            'private_key_type' => OPENSSL_KEYTYPE_EC,
        ]);
        
        if ($keyResource) {
            return self::extractKeyDetails($keyResource);
        }
        
        return null;
    }
    
    /**
     * Methode 2: Mit config-Option
     */
    private static function method2_with_config(): ?array
    {
        $configPath = self::findOpenSSLConfig();
        if (!$configPath) {
            return null;
        }
        
        $keyResource = @openssl_pkey_new([
            'config' => $configPath,
            'curve_name' => 'prime256v1',
            'private_key_type' => OPENSSL_KEYTYPE_EC,
        ]);
        
        if ($keyResource) {
            return self::extractKeyDetails($keyResource);
        }
        
        return null;
    }
    
    /**
     * Methode 3: Mit vollem Pfad und config
     */
    private static function method3_with_full_path(): ?array
    {
        $configPath = self::findOpenSSLConfig();
        if (!$configPath) {
            return null;
        }
        
        // Setze auch Umgebungsvariable
        putenv("OPENSSL_CONF=$configPath");
        
        $keyResource = @openssl_pkey_new([
            'config' => $configPath,
            'curve_name' => 'prime256v1',
            'private_key_type' => OPENSSL_KEYTYPE_EC,
        ]);
        
        if ($keyResource) {
            return self::extractKeyDetails($keyResource);
        }
        
        return null;
    }
    
    /**
     * Methode 4: Verwende openssl command line
     */
    private static function method4_openssl_command(): ?array
    {
        $tempKeyFile = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'temp_ec_key_' . uniqid() . '.pem';
        $configPath = self::findOpenSSLConfig();
        
        try {
            // Finde openssl.exe
            $opensslExe = self::findOpenSSLExecutable();
            if (!$opensslExe) {
                return null;
            }
            
            $command = sprintf(
                '"%s" ecparam -genkey -name prime256v1 -out %s %s 2>&1',
                $opensslExe,
                escapeshellarg($tempKeyFile),
                $configPath ? '-config ' . escapeshellarg($configPath) : ''
            );
            
            $output = [];
            $returnVar = 0;
            exec($command, $output, $returnVar);
            
            if ($returnVar === 0 && file_exists($tempKeyFile)) {
                $keyPem = file_get_contents($tempKeyFile);
                $keyResource = @openssl_pkey_get_private($keyPem);
                
                if ($keyResource) {
                    $result = self::extractKeyDetails($keyResource);
                    @unlink($tempKeyFile);
                    return $result;
                }
            }
            
            @unlink($tempKeyFile);
            return null;
            
        } catch (\Exception $e) {
            if (file_exists($tempKeyFile)) {
                @unlink($tempKeyFile);
            }
            return null;
        }
    }
    
    /**
     * Extrahiere Key-Details
     */
    private static function extractKeyDetails($keyResource): array
    {
        $details = openssl_pkey_get_details($keyResource);
        if (!$details || !isset($details['ec'])) {
            throw new \RuntimeException('Unable to get the local key details.');
        }
        
        return [
            new \Jose\Component\Core\JWK([
                'kty' => 'EC',
                'crv' => 'P-256',
                'x' => \Base64Url\Base64Url::encode(self::addNullPadding($details['ec']['x'])),
                'y' => \Base64Url\Base64Url::encode(self::addNullPadding($details['ec']['y'])),
                'd' => \Base64Url\Base64Url::encode(self::addNullPadding($details['ec']['d'])),
            ]),
        ];
    }
    
    /**
     * Finde openssl.cnf
     */
    private static function findOpenSSLConfig(): ?string
    {
        $paths = [
            'C:\\xampp\\apache\\conf\\openssl.cnf',
            'C:\\xampp\\php\\extras\\ssl\\openssl.cnf',
            'C:\\OpenSSL-Win64\\bin\\cnf\\openssl.cnf',
            'C:\\Program Files\\OpenSSL-Win64\\bin\\cnf\\openssl.cnf',
            'C:\\Program Files (x86)\\OpenSSL-Win32\\bin\\cnf\\openssl.cnf',
        ];
        
        foreach ($paths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }
        
        return null;
    }
    
    /**
     * Finde openssl.exe
     */
    private static function findOpenSSLExecutable(): ?string
    {
        $paths = [
            'C:\\xampp\\apache\\bin\\openssl.exe',
            'C:\\OpenSSL-Win64\\bin\\openssl.exe',
            'C:\\Program Files\\OpenSSL-Win64\\bin\\openssl.exe',
            'C:\\Program Files (x86)\\OpenSSL-Win32\\bin\\openssl.exe',
        ];
        
        foreach ($paths as $path) {
            if (file_exists($path)) {
                return $path;
            }
        }
        
        // Versuche openssl im PATH zu finden
        $output = [];
        exec('where openssl 2>&1', $output, $returnVar);
        if ($returnVar === 0 && !empty($output[0]) && file_exists($output[0])) {
            return $output[0];
        }
        
        return null;
    }
}

// Verwende Reflection, um die private Methode zu überschreiben
// Da createLocalKeyObject private ist, müssen wir einen anderen Ansatz verwenden
// Stattdessen patchen wir die Encryption-Klasse direkt mit einem Monkey-Patch

