<?php
/**
 * Feedback-Formular-Komponente
 * Wiederverwendbar für alle Seiten
 */

// Prüfe ob User eingeloggt ist
if (!isset($_SESSION['user_logged_in']) || !$_SESSION['user_logged_in']) {
    return; // Formular nur für eingeloggte Benutzer
}

// Prüfe ob User bereits Feedback gegeben hat
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/auth.php';

$user = getCurrentUser();
$has_feedback = false;

if ($user && isset($user['id'])) {
    try {
        $db = new Database();
        $conn = $db->getConnection();
        $stmt = $conn->prepare("SELECT id FROM feedback WHERE user_id = ? LIMIT 1");
        $stmt->execute([$user['id']]);
        $has_feedback = $stmt->rowCount() > 0;
    } catch (Exception $e) {
        // Fehler beim Prüfen - Formular trotzdem anzeigen
        error_log("Fehler beim Prüfen des Feedbacks: " . $e->getMessage());
    }
}

$current_page = basename($_SERVER['PHP_SELF']);
?>

<!-- Feedback-Formular -->
<div id="feedback-widget" class="feedback-widget">
    <?php if ($has_feedback): ?>
        <!-- Benutzer hat bereits Feedback gegeben -->
        <button id="feedback-toggle-btn" class="feedback-toggle-btn" style="background: #6c757d; cursor: not-allowed;" disabled title="Sie haben bereits Feedback abgegeben">
            <i class="fas fa-check-circle"></i>
            <span>Feedback bereits abgegeben</span>
        </button>
    <?php else: ?>
        <button id="feedback-toggle-btn" class="feedback-toggle-btn" aria-label="Feedback geben">
            <i class="fas fa-comment-dots"></i>
            <span>Feedback</span>
        </button>
    <?php endif; ?>
    
    <div id="feedback-panel" class="feedback-panel" style="display: none;">
        <div class="feedback-panel-header">
            <h3>
                <i class="fas fa-star"></i>
                Ihr Feedback ist uns wichtig
            </h3>
            <button id="feedback-close-btn" class="feedback-close-btn" aria-label="Schließen">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form id="feedback-form" class="feedback-form">
            <input type="hidden" name="page_url" value="<?php echo htmlspecialchars($current_page); ?>">
            <input type="hidden" name="feedback_type" value="general">
            
            <div class="feedback-rating">
                <label>Wie zufrieden sind Sie mit unserem Service?</label>
                <div class="rating-stars">
                    <input type="radio" name="rating" value="5" id="rating-5" required>
                    <label for="rating-5" class="star-label" data-rating="5">
                        <i class="fas fa-star"></i>
                    </label>
                    <input type="radio" name="rating" value="4" id="rating-4">
                    <label for="rating-4" class="star-label" data-rating="4">
                        <i class="fas fa-star"></i>
                    </label>
                    <input type="radio" name="rating" value="3" id="rating-3">
                    <label for="rating-3" class="star-label" data-rating="3">
                        <i class="fas fa-star"></i>
                    </label>
                    <input type="radio" name="rating" value="2" id="rating-2">
                    <label for="rating-2" class="star-label" data-rating="2">
                        <i class="fas fa-star"></i>
                    </label>
                    <input type="radio" name="rating" value="1" id="rating-1">
                    <label for="rating-1" class="star-label" data-rating="1">
                        <i class="fas fa-star"></i>
                    </label>
                </div>
            </div>
            
            <div class="feedback-textarea">
                <label for="feedback_text">Ihr Feedback *</label>
                <textarea 
                    id="feedback_text" 
                    name="feedback_text" 
                    rows="4" 
                    placeholder="Teilen Sie uns Ihre Erfahrungen mit unserem Online-Buchungssystem mit..."
                    required
                    minlength="10"
                ></textarea>
                <small>Mindestens 10 Zeichen</small>
            </div>
            
            <div class="feedback-options">
                <label class="checkbox-label">
                    <input type="checkbox" name="is_anonymous" value="1">
                    <span>Feedback anonym abgeben</span>
                </label>
            </div>
            
            <div id="feedback-message" class="feedback-message" style="display: none;"></div>
            
            <div class="feedback-actions">
                <button type="submit" class="btn btn-primary feedback-submit-btn">
                    <i class="fas fa-paper-plane"></i>
                    Feedback senden
                </button>
            </div>
        </form>
    </div>
</div>

<style>
/* Feedback Widget Styles */
/* Feedback Widget Styles */
.feedback-widget {
    position: fixed;
    bottom: 20px;
    right: 20px;
    z-index: 999; /* Unter WhatsApp-Widget (normalerweise z-index: 1000+) */
    font-family: 'Inter', sans-serif;
    /* Sicherstellen, dass Feedback-Widget nicht mit anderen Floating-Elementen kollidiert */
    margin-bottom: 0;
}

.feedback-toggle-btn {
    background: #0056b3;
    color: white;
    border: none;
    border-radius: 50px;
    padding: 1rem 1.5rem;
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    box-shadow: 0 4px 12px rgba(0, 86, 179, 0.3);
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
}

.feedback-toggle-btn:hover {
    background: #0066cc;
    box-shadow: 0 6px 16px rgba(0, 86, 179, 0.4);
    transform: translateY(-2px);
}

.feedback-toggle-btn i {
    font-size: 1.2rem;
}

.feedback-panel {
    position: absolute;
    bottom: 80px;
    right: 0;
    width: 400px;
    max-width: calc(100vw - 40px);
    background: white;
    border-radius: 12px;
    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
    overflow: hidden;
    animation: slideUp 0.3s ease;
}

@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.feedback-panel-header {
    background: linear-gradient(135deg, #0056b3 0%, #0066cc 100%);
    color: white;
    padding: 1.25rem 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.feedback-panel-header h3 {
    margin: 0;
    font-size: 1.1rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.feedback-close-btn {
    background: transparent;
    border: none;
    color: white;
    font-size: 1.2rem;
    cursor: pointer;
    padding: 0.25rem;
    transition: opacity 0.2s;
}

.feedback-close-btn:hover {
    opacity: 0.8;
}

.feedback-form {
    padding: 1.5rem;
}

.feedback-rating {
    margin-bottom: 1.5rem;
}

.feedback-rating label {
    display: block;
    margin-bottom: 0.75rem;
    font-weight: 500;
    color: #495057;
}

.rating-stars {
    display: flex;
    gap: 0.5rem;
    justify-content: center;
    flex-direction: row-reverse;
}

.rating-stars input[type="radio"] {
    display: none;
}

.star-label {
    font-size: 2rem;
    color: #dee2e6;
    cursor: pointer;
    transition: all 0.2s;
}

.star-label:hover,
.star-label:hover ~ .star-label {
    color: #ffc107;
}

.rating-stars input[type="radio"]:checked ~ .star-label,
.rating-stars input[type="radio"]:checked ~ .star-label ~ .star-label {
    color: #ffc107;
}

.rating-stars:has(input[type="radio"]:checked) .star-label {
    color: #ffc107;
}

.feedback-textarea {
    margin-bottom: 1rem;
}

.feedback-textarea label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 500;
    color: #495057;
}

.feedback-textarea textarea {
    width: 100%;
    padding: 0.75rem;
    border: 1px solid #dee2e6;
    border-radius: 8px;
    font-family: inherit;
    font-size: 0.95rem;
    resize: vertical;
    transition: border-color 0.2s;
}

.feedback-textarea textarea:focus {
    outline: none;
    border-color: #0056b3;
    box-shadow: 0 0 0 3px rgba(0, 86, 179, 0.1);
}

.feedback-textarea small {
    display: block;
    margin-top: 0.25rem;
    color: #6c757d;
    font-size: 0.85rem;
}

.feedback-options {
    margin-bottom: 1rem;
}

.checkbox-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
    font-size: 0.9rem;
    color: #6c757d;
}

.checkbox-label input[type="checkbox"] {
    cursor: pointer;
}

.feedback-actions {
    margin-top: 1.5rem;
}

.feedback-submit-btn {
    width: 100%;
    padding: 0.875rem;
    font-size: 1rem;
    font-weight: 600;
}

.feedback-message {
    padding: 0.75rem 1rem;
    border-radius: 8px;
    margin-bottom: 1rem;
    font-size: 0.9rem;
}

.feedback-message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.feedback-message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .feedback-panel {
        width: calc(100vw - 40px);
        max-width: 400px;
        right: 0;
    }
    
    .feedback-widget {
        bottom: 15px;
        right: 15px;
        left: auto;
    }
    
    .feedback-toggle-btn {
        padding: 0.875rem 1.25rem;
        font-size: 0.9rem;
    }
    
    .feedback-toggle-btn span {
        display: none;
    }
}

/* Anpassung für WhatsApp-Widget-Kompatibilität */
@media (max-width: 1024px) {
    /* Stelle sicher, dass Feedback-Widget nicht mit WhatsApp-Widget kollidiert */
    .feedback-widget {
        bottom: 90px; /* Mehr Abstand nach oben, falls WhatsApp-Widget größer ist */
    }
    
    .feedback-panel {
        bottom: 80px;
    }
}

/* Zusätzlicher Abstand wenn WhatsApp-Widget sichtbar ist */
body:has([data-widget-host="mateo"]) .feedback-widget {
    bottom: 90px;
}

/* Fallback für Browser ohne :has() Support */
@media (max-width: 768px) {
    .feedback-widget {
        bottom: 90px;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const toggleBtn = document.getElementById('feedback-toggle-btn');
    const closeBtn = document.getElementById('feedback-close-btn');
    const panel = document.getElementById('feedback-panel');
    const form = document.getElementById('feedback-form');
    const messageDiv = document.getElementById('feedback-message');
    
    // Panel öffnen/schließen
    toggleBtn.addEventListener('click', function() {
        panel.style.display = panel.style.display === 'none' ? 'block' : 'none';
    });
    
    closeBtn.addEventListener('click', function() {
        panel.style.display = 'none';
    });
    
    // Star-Rating Interaktivität
    const starLabels = document.querySelectorAll('.star-label');
    starLabels.forEach(label => {
        label.addEventListener('click', function() {
            const rating = this.getAttribute('data-rating');
            document.getElementById('rating-' + rating).checked = true;
            updateStarDisplay(rating);
        });
    });
    
    function updateStarDisplay(selectedRating) {
        starLabels.forEach((label, index) => {
            const rating = 5 - index;
            if (rating <= selectedRating) {
                label.style.color = '#ffc107';
            } else {
                label.style.color = '#dee2e6';
            }
        });
    }
    
    // Formular absenden
    form.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const formData = new FormData(form);
        const data = {
            rating: parseInt(formData.get('rating')),
            feedback_text: formData.get('feedback_text'),
            feedback_type: formData.get('feedback_type') || 'general',
            page_url: formData.get('page_url'),
            is_anonymous: formData.get('is_anonymous') === '1'
        };
        
        // Validierung
        if (!data.rating || data.rating < 1 || data.rating > 5) {
            showMessage('Bitte wählen Sie eine Bewertung aus', 'error');
            return;
        }
        
        if (!data.feedback_text || data.feedback_text.length < 10) {
            showMessage('Bitte geben Sie mindestens 10 Zeichen Feedback ein', 'error');
            return;
        }
        
        // Submit-Button deaktivieren
        const submitBtn = form.querySelector('.feedback-submit-btn');
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Wird gesendet...';
        
        try {
            // Verwende absoluten Pfad basierend auf dem aktuellen Verzeichnis
            const apiPath = window.location.pathname.includes('/admin/') 
                ? '../api/save_feedback.php' 
                : 'api/save_feedback.php';
            
            const response = await fetch(apiPath, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(data)
            });
            
            // Prüfe ob Response OK ist
            if (!response.ok) {
                const errorText = await response.text();
                let errorMessage = 'Fehler beim Senden des Feedbacks';
                
                try {
                    const errorJson = JSON.parse(errorText);
                    errorMessage = errorJson.error || errorMessage;
                } catch (e) {
                    // Response ist kein JSON
                    errorMessage = `Server-Fehler (${response.status}): ${errorText.substring(0, 100)}`;
                }
                
                showMessage(errorMessage, 'error');
                return;
            }
            
            const result = await response.json();
            
            if (result.success) {
                showMessage('Vielen Dank für ihr Feedback', 'success');
                form.reset();
                updateStarDisplay(0);
                
                // Panel nach 2 Sekunden schließen
                setTimeout(() => {
                    panel.style.display = 'none';
                }, 2000);
            } else {
                let errorMessage = result.error || 'Fehler beim Senden des Feedbacks';
                
                // Debug-Informationen anzeigen, falls vorhanden
                if (result.debug) {
                    console.error('Feedback Error Debug:', result.debug);
                    // In Entwicklungsumgebung mehr Details anzeigen
                    if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
                        errorMessage += ' (Details in Konsole)';
                    }
                }
                
                showMessage(errorMessage, 'error');
            }
        } catch (error) {
            console.error('Feedback Error:', error);
            showMessage('Fehler beim Senden des Feedbacks: ' + error.message, 'error');
        } finally {
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Feedback senden';
        }
    });
    
    function showMessage(text, type) {
        messageDiv.textContent = text;
        messageDiv.className = 'feedback-message ' + type;
        messageDiv.style.display = 'block';
        
        // Nach 5 Sekunden ausblenden
        setTimeout(() => {
            messageDiv.style.display = 'none';
        }, 5000);
    }
});
</script>

