param(
  [Parameter(Mandatory=$true)]
  [string]$GltfUrl,
  [string]$Out = ".\downloaded_model",
  [switch]$HdrOnly,
  # Optional: eigene HTTP-Header (z.B. @{ "Cookie"="__Host=..."; "User-Agent"="..." })
  [hashtable]$Headers
)

function Write-Info($msg){ Write-Host $msg -ForegroundColor Cyan }
function Write-Ok($msg){ Write-Host $msg -ForegroundColor Green }
function Write-Err($msg){ Write-Host $msg -ForegroundColor Red }

function Join-Uri([string]$base,[string]$rel){
  # korrekte Auflösung relativer Pfade (../ etc.)
  $baseUri = [System.Uri]$base
  $u = [System.Uri]::new($baseUri, $rel)
  return $u.AbsoluteUri
}

function Ensure-Dir([string]$path){
  $dir = Split-Path -Parent $path
  if (-not [string]::IsNullOrWhiteSpace($dir)) {
    New-Item -ItemType Directory -Force -Path $dir | Out-Null
  }
}

function Is-DataUri([string]$uri){
  return ($null -ne $uri -and $uri.StartsWith("data:"))
}

function Get-JsonDepthUris($obj, [System.Collections.Generic.List[string]]$list){
  if ($null -eq $obj) { return }
  if ($obj -is [System.Collections.IDictionary]) {
    foreach($k in $obj.Keys){
      $v = $obj[$k]
      if ($k -eq "uri" -and $v -is [string] -and -not (Is-DataUri $v)) {
        $list.Add($v)
      } else {
        Get-JsonDepthUris $v $list
      }
    }
  } elseif ($obj -is [System.Collections.IEnumerable] -and -not ($obj -is [string])) {
    foreach($it in $obj){ Get-JsonDepthUris $it $list }
  }
}

# --- Start ---
try {
  if (-not (Test-Path $Out)) { New-Item -ItemType Directory -Force -Path $Out | Out-Null }

  $baseUrl = $GltfUrl.Substring(0, $GltfUrl.LastIndexOf("/"))
  $gltfName = Split-Path -Leaf $GltfUrl
  $localGltfPath = Join-Path $Out $gltfName

  Write-Info "Lade GLTF:`n  $GltfUrl"
  if ($Headers) {
    Invoke-WebRequest -Uri $GltfUrl -OutFile $localGltfPath -Headers $Headers -UseBasicParsing
  } else {
    Invoke-WebRequest -Uri $GltfUrl -OutFile $localGltfPath -UseBasicParsing
  }
  Write-Ok "✓ GLTF gespeichert: $localGltfPath"

  $jsonRaw = Get-Content -LiteralPath $localGltfPath -Raw
  $gltf = $jsonRaw | ConvertFrom-Json

  $uris = New-Object System.Collections.Generic.List[string]

  # buffers[].uri (immer)
  if ($gltf.buffers) {
    foreach($b in $gltf.buffers){
      $u = $b.uri
      if ($u -and -not (Is-DataUri $u)) { $uris.Add($u) }
    }
  }

  # images[].uri (optional nur .hdr)
  if ($gltf.images) {
    foreach($img in $gltf.images){
      $u = $img.uri
      if ($u -and -not (Is-DataUri $u)) {
        if ($HdrOnly) {
          if ($u.ToLower().EndsWith(".hdr")) { $uris.Add($u) }
        } else {
          $uris.Add($u)
        }
      }
    }
  }

  # generisch: alle weiteren "uri" Felder (z.B. in extensions)
  Get-JsonDepthUris ($gltf.PSObject.Properties | ForEach-Object { @{($_.Name) = $_.Value} }) $uris

  # Deduplizieren & normalisieren (Windows Pfade beibehalten, URLs bleiben /)
  $all = $uris | Sort-Object -Unique

  Write-Info ("Gefundene URIs: {0}" -f $all.Count)
  $all | ForEach-Object { Write-Host " - $_" }

  foreach($rel in $all){
    try {
      $fileUrl = Join-Uri $baseUrl $rel
      $dst = Join-Path $Out ($rel -replace '[\\]+','\') # Unterordner mit übernehmen
      Ensure-Dir $dst
      Write-Host "↓ $fileUrl"
      if ($Headers) {
        Invoke-WebRequest -Uri $fileUrl -OutFile $dst -Headers $Headers -UseBasicParsing
      } else {
        Invoke-WebRequest -Uri $fileUrl -OutFile $dst -UseBasicParsing
      }
      Write-Ok "  ✓ $dst"
    } catch {
      Write-Err ("  ✗ Fehler bei {0} : {1}" -f $rel, $_.Exception.Message)
    }
  }

  Write-Host ""
  Write-Ok ("Fertig. Ordner: {0}" -f (Resolve-Path $Out))

  Write-Host ""
  Write-Info "Optional: Zu .glb konvertieren (Node-Tool 'gltf-pipeline' erforderlich):"
  Write-Host "  npm install -g gltf-pipeline"
  Write-Host ("  gltf-pipeline -i `"{0}`" -o `"{1}`"" -f $localGltfPath ($localGltfPath -replace '\.gltf$','.glb'))

} catch {
  Write-Err ("Abbruch: {0}" -f $_.Exception.Message)
  exit 1
}