<?php
/**
 * Cron-Job für automatische WhatsApp-Terminerinnerungen
 * Läuft stündlich und sendet Erinnerungen für Termine am nächsten Tag
 */

// Ermittle das Root-Verzeichnis des Projekts (unabhängig vom Ausführungsverzeichnis)
$rootDir = dirname(__DIR__);

// Lade benötigte Dateien mit absolutem Pfad
require_once $rootDir . '/config/database.php';
require_once $rootDir . '/api/autohaus_whatsapp_notifications.php';

// Log-Datei für Cron-Jobs
$logDir = $rootDir . '/logs';
$logFile = $logDir . '/whatsapp_reminders.log';

// Stelle sicher, dass das Log-Verzeichnis existiert
if (!is_dir($logDir)) {
    mkdir($logDir, 0755, true);
}

function logMessage($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] $message" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

try {
    logMessage("WhatsApp-Erinnerungen Cron-Job gestartet");
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Hole alle aktiven Autohäuser
    $stmt = $conn->prepare("SELECT id, name FROM autohaus WHERE is_active = 1");
    $stmt->execute();
    $autohauser = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $totalReminders = 0;
    $totalErrors = 0;
    
    foreach ($autohauser as $autohaus) {
        logMessage("Verarbeite Autohaus: " . $autohaus['name'] . " (ID: " . $autohaus['id'] . ")");
        
        try {
            // Erstelle WhatsApp-Instanz für dieses Autohaus
            $whatsapp = new AutohausWhatsAppNotifications($autohaus['id']);
            
            // Prüfe ob reminder_24h aktiviert ist
            $stmt = $conn->prepare("
                SELECT id, is_enabled FROM whatsapp_template_configs 
                WHERE autohaus_id = ? AND action_type = 'reminder_24h' AND is_active = 1
                LIMIT 1
            ");
            $stmt->execute([$autohaus['id']]);
            $config24h = $stmt->fetch(PDO::FETCH_ASSOC);
            $reminder24hEnabled = $config24h && ($config24h['is_enabled'] == 1 || $config24h['is_enabled'] === null);
            
            // Prüfe ob reminder_48h aktiviert ist
            $stmt = $conn->prepare("
                SELECT id, is_enabled FROM whatsapp_template_configs 
                WHERE autohaus_id = ? AND action_type = 'reminder_48h' AND is_active = 1
                LIMIT 1
            ");
            $stmt->execute([$autohaus['id']]);
            $config48h = $stmt->fetch(PDO::FETCH_ASSOC);
            $reminder48hEnabled = $config48h && ($config48h['is_enabled'] == 1 || $config48h['is_enabled'] === null);
            
            // 1. HOLE TERMINE FÜR ÜBERMORGEN (48h-Erinnerungen)
            if ($reminder48hEnabled) {
                $stmt = $conn->prepare("
                    SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                           u.phone_number as user_phone_number, u.whatsapp_opt_out
                    FROM appointments a 
                    LEFT JOIN events e ON a.event_id = e.id 
                    LEFT JOIN users u ON a.user_id = u.id
                    WHERE a.status IN ('approved', 'booked')
                    AND a.slot_date = DATE_ADD(CURDATE(), INTERVAL 2 DAY)
                    AND a.autohaus_id = ?
                    AND (a.whatsapp_reminder_48h_sent = 0 OR a.whatsapp_reminder_48h_sent IS NULL)
                    AND (
                        a.phone IS NOT NULL AND a.phone != ''
                        OR (a.user_id IS NOT NULL AND u.phone_number IS NOT NULL AND u.phone_number != '')
                    )
                    AND (u.whatsapp_opt_out IS NULL OR u.whatsapp_opt_out = 0)
                ");
                $stmt->execute([$autohaus['id']]);
                $appointments48h = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                logMessage("Gefundene Termine für übermorgen (48h-Erinnerung): " . count($appointments48h));
                
                foreach ($appointments48h as $appointment) {
                    try {
                        // Sende 48h-Erinnerung
                        $result = $whatsapp->sendAppointmentNotification($appointment['id'], 'reminder_48h');
                        
                        if ($result['success']) {
                            // Markiere 48h-Erinnerung als gesendet
                            // Prüfe ob Spalte existiert
                            $stmt = $conn->query("SHOW COLUMNS FROM appointments LIKE 'whatsapp_reminder_48h_sent'");
                            if ($stmt->rowCount() > 0) {
                                $updateStmt = $conn->prepare("
                                    UPDATE appointments 
                                    SET whatsapp_reminder_48h_sent = 1, whatsapp_reminder_48h_sent_at = NOW() 
                                    WHERE id = ?
                                ");
                            } else {
                                // Fallback: Verwende whatsapp_reminder_sent
                                $updateStmt = $conn->prepare("
                                    UPDATE appointments 
                                    SET whatsapp_reminder_sent = 1, whatsapp_reminder_sent_at = NOW() 
                                    WHERE id = ?
                                ");
                            }
                            $updateStmt->execute([$appointment['id']]);
                            
                            $totalReminders++;
                            logMessage("48h-Erinnerung gesendet für Termin ID " . $appointment['id'] . " (" . $appointment['customer_name'] . ")");
                        } else {
                            $totalErrors++;
                            logMessage("Fehler beim Senden der 48h-Erinnerung für Termin ID " . $appointment['id'] . ": " . $result['error']);
                        }
                        
                    } catch (Exception $e) {
                        $totalErrors++;
                        logMessage("Exception beim Senden der 48h-Erinnerung für Termin ID " . $appointment['id'] . ": " . $e->getMessage());
                    }
                }
            }
            
            // 2. HOLE TERMINE FÜR MORGEN (24h-Erinnerungen)
            if ($reminder24hEnabled) {
                $stmt = $conn->prepare("
                    SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                           u.phone_number as user_phone_number, u.whatsapp_opt_out
                    FROM appointments a 
                    LEFT JOIN events e ON a.event_id = e.id 
                    LEFT JOIN users u ON a.user_id = u.id
                    WHERE a.status IN ('approved', 'booked')
                    AND a.slot_date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)
                    AND a.autohaus_id = ?
                    AND (a.whatsapp_reminder_sent = 0 OR a.whatsapp_reminder_sent IS NULL)
                    AND (
                        a.phone IS NOT NULL AND a.phone != ''
                        OR (a.user_id IS NOT NULL AND u.phone_number IS NOT NULL AND u.phone_number != '')
                    )
                    AND (u.whatsapp_opt_out IS NULL OR u.whatsapp_opt_out = 0)
                ");
                $stmt->execute([$autohaus['id']]);
                $appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                logMessage("Gefundene Termine für morgen (24h-Erinnerung): " . count($appointments));
                
                foreach ($appointments as $appointment) {
                    try {
                        // Sende 24h-Erinnerung
                        $result = $whatsapp->sendAppointmentNotification($appointment['id'], 'reminder_24h');
                        
                        if ($result['success']) {
                            // Markiere Erinnerung als gesendet
                            $updateStmt = $conn->prepare("
                                UPDATE appointments 
                                SET whatsapp_reminder_sent = 1, whatsapp_reminder_sent_at = NOW() 
                                WHERE id = ?
                            ");
                            $updateStmt->execute([$appointment['id']]);
                            
                            $totalReminders++;
                            logMessage("24h-Erinnerung gesendet für Termin ID " . $appointment['id'] . " (" . $appointment['customer_name'] . ")");
                        } else {
                            $totalErrors++;
                            logMessage("Fehler beim Senden der 24h-Erinnerung für Termin ID " . $appointment['id'] . ": " . $result['error']);
                        }
                        
                    } catch (Exception $e) {
                        $totalErrors++;
                        logMessage("Exception beim Senden der 24h-Erinnerung für Termin ID " . $appointment['id'] . ": " . $e->getMessage());
                    }
                }
            }
            
            // 3. HOLE NEUE BUCHUNGEN FÜR BESTÄTIGUNGEN (wenn noch nicht gesendet)
            $stmt = $conn->prepare("
                SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                       u.phone_number as user_phone_number, u.whatsapp_opt_out
                FROM appointments a 
                LEFT JOIN events e ON a.event_id = e.id 
                LEFT JOIN users u ON a.user_id = u.id
                WHERE a.status IN ('approved', 'booked')
                AND a.autohaus_id = ?
                AND a.whatsapp_confirmation_sent = 0
                AND a.created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
                AND (
                    a.phone IS NOT NULL AND a.phone != ''
                    OR (a.user_id IS NOT NULL AND u.phone_number IS NOT NULL AND u.phone_number != '')
                )
                AND (u.whatsapp_opt_out IS NULL OR u.whatsapp_opt_out = 0)
            ");
            $stmt->execute([$autohaus['id']]);
            $newAppointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            logMessage("Gefundene neue Buchungen für Bestätigung: " . count($newAppointments));
            
            foreach ($newAppointments as $appointment) {
                try {
                    // Sende Bestätigung
                    $result = $whatsapp->sendAppointmentNotification($appointment['id'], 'confirmation');
                    
                    if ($result['success']) {
                        // Markiere Bestätigung als gesendet
                        $updateStmt = $conn->prepare("
                            UPDATE appointments 
                            SET whatsapp_confirmation_sent = 1, whatsapp_confirmation_sent_at = NOW() 
                            WHERE id = ?
                        ");
                        $updateStmt->execute([$appointment['id']]);
                        
                        $totalReminders++;
                        logMessage("Bestätigung gesendet für Termin ID " . $appointment['id'] . " (" . $appointment['customer_name'] . ")");
                    } else {
                        $totalErrors++;
                        logMessage("Fehler beim Senden der Bestätigung für Termin ID " . $appointment['id'] . ": " . $result['error']);
                    }
                    
                } catch (Exception $e) {
                    $totalErrors++;
                    logMessage("Exception beim Senden der Bestätigung für Termin ID " . $appointment['id'] . ": " . $e->getMessage());
                }
            }
            
            // 4. HOLE ABGESCHLOSSENE TERMINE FÜR BEWERTUNGSANFRAGE (24 Stunden nach Termin)
            // Prüfe ob review_request aktiviert ist
            $stmt = $conn->prepare("
                SELECT id, is_enabled FROM whatsapp_template_configs 
                WHERE autohaus_id = ? AND action_type = 'review_request' AND is_active = 1
                LIMIT 1
            ");
            $stmt->execute([$autohaus['id']]);
            $configReview = $stmt->fetch(PDO::FETCH_ASSOC);
            $reviewRequestEnabled = $configReview && ($configReview['is_enabled'] == 1 || $configReview['is_enabled'] === null);
            
            if ($reviewRequestEnabled) {
                // Prüfe ob Spalte existiert
                $stmt = $conn->query("SHOW COLUMNS FROM appointments LIKE 'whatsapp_review_request_sent'");
                $hasReviewColumn = $stmt->rowCount() > 0;
                
                if ($hasReviewColumn) {
                    $stmt = $conn->prepare("
                        SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                               u.phone_number as user_phone_number, u.whatsapp_opt_out
                        FROM appointments a 
                        LEFT JOIN events e ON a.event_id = e.id 
                        LEFT JOIN users u ON a.user_id = u.id
                        WHERE a.status IN ('completed', 'approved', 'booked')
                        AND a.autohaus_id = ?
                        AND (a.whatsapp_review_request_sent = 0 OR a.whatsapp_review_request_sent IS NULL)
                        AND DATE(a.slot_date) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)
                        AND (
                            a.phone IS NOT NULL AND a.phone != ''
                            OR (a.user_id IS NOT NULL AND u.phone_number IS NOT NULL AND u.phone_number != '')
                        )
                        AND (u.whatsapp_opt_out IS NULL OR u.whatsapp_opt_out = 0)
                    ");
                } else {
                    // Fallback: Verwende slot_date + 1 Tag für Termine, die gestern waren
                    $stmt = $conn->prepare("
                        SELECT a.*, e.name as event_name, e.event_date, e.start_time, e.end_time,
                               u.phone_number as user_phone_number, u.whatsapp_opt_out
                        FROM appointments a 
                        LEFT JOIN events e ON a.event_id = e.id 
                        LEFT JOIN users u ON a.user_id = u.id
                        WHERE a.status IN ('completed', 'approved', 'booked')
                        AND a.autohaus_id = ?
                        AND DATE(a.slot_date) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)
                        AND (
                            a.phone IS NOT NULL AND a.phone != ''
                            OR (a.user_id IS NOT NULL AND u.phone_number IS NOT NULL AND u.phone_number != '')
                        )
                        AND (u.whatsapp_opt_out IS NULL OR u.whatsapp_opt_out = 0)
                        AND NOT EXISTS (
                            SELECT 1 FROM whatsapp_logs wl 
                            WHERE wl.appointment_id = a.id 
                            AND wl.action_type = 'review_request' 
                            AND wl.status = 'success'
                        )
                    ");
                }
                $stmt->execute([$autohaus['id']]);
                $completedAppointments = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                logMessage("Gefundene abgeschlossene Termine für Bewertungsanfrage (24h nach Termin): " . count($completedAppointments));
                
                foreach ($completedAppointments as $appointment) {
                    try {
                        // Sende Bewertungsanfrage
                        $result = $whatsapp->sendAppointmentNotification($appointment['id'], 'review_request');
                        
                        if ($result['success']) {
                            // Markiere Bewertungsanfrage als gesendet
                            if ($hasReviewColumn) {
                                $updateStmt = $conn->prepare("
                                    UPDATE appointments 
                                    SET whatsapp_review_request_sent = 1, whatsapp_review_request_sent_at = NOW() 
                                    WHERE id = ?
                                ");
                            } else {
                                // Fallback: Erstelle Spalte falls nicht vorhanden
                                try {
                                    $conn->exec("ALTER TABLE appointments ADD COLUMN whatsapp_review_request_sent TINYINT(1) DEFAULT 0");
                                    $conn->exec("ALTER TABLE appointments ADD COLUMN whatsapp_review_request_sent_at DATETIME NULL");
                                    $hasReviewColumn = true;
                                    $updateStmt = $conn->prepare("
                                        UPDATE appointments 
                                        SET whatsapp_review_request_sent = 1, whatsapp_review_request_sent_at = NOW() 
                                        WHERE id = ?
                                    ");
                                } catch (Exception $e) {
                                    // Spalte existiert bereits oder Fehler - ignoriere
                                    logMessage("Hinweis: Spalte whatsapp_review_request_sent konnte nicht erstellt werden: " . $e->getMessage());
                                    continue;
                                }
                            }
                            $updateStmt->execute([$appointment['id']]);
                            
                            $totalReminders++;
                            logMessage("Bewertungsanfrage gesendet für Termin ID " . $appointment['id'] . " (" . $appointment['customer_name'] . ")");
                        } else {
                            $totalErrors++;
                            logMessage("Fehler beim Senden der Bewertungsanfrage für Termin ID " . $appointment['id'] . ": " . $result['error']);
                        }
                        
                    } catch (Exception $e) {
                        $totalErrors++;
                        logMessage("Exception beim Senden der Bewertungsanfrage für Termin ID " . $appointment['id'] . ": " . $e->getMessage());
                    }
                }
            }
            
        } catch (Exception $e) {
            $totalErrors++;
            logMessage("Fehler beim Verarbeiten von Autohaus " . $autohaus['name'] . ": " . $e->getMessage());
        }
    }
    
    logMessage("WhatsApp-Erinnerungen Cron-Job beendet. Gesendet: $totalReminders, Fehler: $totalErrors");
    
    // Optional: Sende Zusammenfassung per E-Mail an Admin
    if ($totalReminders > 0 || $totalErrors > 0) {
        $summaryMessage = "WhatsApp-Erinnerungen Cron-Job Zusammenfassung:\n\n";
        $summaryMessage .= "Gesendete Erinnerungen: $totalReminders\n";
        $summaryMessage .= "Fehler: $totalErrors\n";
        $summaryMessage .= "Zeit: " . date('Y-m-d H:i:s') . "\n";
        
        // Hier könnte eine E-Mail-Benachrichtigung implementiert werden
        logMessage("Zusammenfassung: " . $summaryMessage);
    }
    
} catch (Exception $e) {
    logMessage("Kritischer Fehler im WhatsApp-Erinnerungen Cron-Job: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    exit;
}

echo json_encode([
    'success' => true, 
    'reminders_sent' => $totalReminders,
    'errors' => $totalErrors,
    'message' => "WhatsApp-Erinnerungen verarbeitet"
]);
?>
