<?php
/**
 * Terminbuchungsseite
 */

session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';

// Prüfe ob User eingeloggt ist
if (!isLoggedIn()) {
    requireLoginRedirect();
    exit;
}

// Prüfe ob User verifiziert ist
$user = getCurrentUser();
if ($user) {
    // Prüfe ob User gesperrt ist - gesperrte User dürfen KEINE Termine buchen
    if (isset($user['status']) && $user['status'] === 'suspended') {
        header('Location: index.php?error=account_suspended');
        exit;
    }
    
    // Prüfe ob User inaktiv ist
    if (isset($user['is_active']) && $user['is_active'] == 0) {
        header('Location: index.php?error=account_suspended');
        exit;
    }
    
    // Prüfe ob User verifiziert ist
    if (!isset($user['verification_status']) || $user['verification_status'] !== 'approved') {
        header('Location: verification_pending.php');
        exit;
    }
}

$event_id = $_GET['event'] ?? null;

if (!$event_id) {
    header('Location: index.php');
    exit;
}

$db = new Database();
$conn = $db->getConnection();

// Event-Details mit Autohaus-Information laden
$stmt = $conn->prepare("
    SELECT e.*, a.name as autohaus_name, a.short_name as autohaus_short_name, a.id as autohaus_id
    FROM events e 
    LEFT JOIN autohaus a ON e.autohaus_id = a.id 
    WHERE e.id = ? AND e.is_active = 1
");
$stmt->execute([$event_id]);
$event = $stmt->fetch();

// Services für das Autohaus laden
$services = [];
$autohaus_pricing = null;
if ($event && $event['autohaus_id']) {
    $stmt = $conn->prepare("
        SELECT * FROM autohaus_services 
        WHERE autohaus_id = ? AND is_active = 1 
        ORDER BY sort_order, service_name
    ");
    $stmt->execute([$event['autohaus_id']]);
    $services = $stmt->fetchAll();
    
    // Autohaus-Preise laden
    $stmt = $conn->prepare("
        SELECT price_wheel_change, price_storage, price_spring_check, price_winter_check
        FROM autohaus 
        WHERE id = ? AND is_active = 1
    ");
    $stmt->execute([$event['autohaus_id']]);
    $autohaus_pricing = $stmt->fetch(PDO::FETCH_ASSOC);
}

if (!$event) {
    header('Location: index.php');
    exit;
}

// User-Daten laden für automatisches Ausfüllen
$user = getCurrentUser();
$customer_name = '';
$customer_email = '';
$customer_phone = '';
$customer_license_plate = '';

// Fahrzeuge des Users laden
$user_vehicles = [];
$default_vehicle = null;

if ($user) {
    $customer_name = $user['first_name'] . ' ' . $user['last_name'];
    $customer_email = $user['email'] ?? '';
    $customer_phone = $user['phone_number'] ?? '';
    
    // Lade Fahrzeuge aus user_vehicles Tabelle
    $stmt = $conn->prepare("
        SELECT * FROM user_vehicles 
        WHERE user_id = ? AND is_active = 1 
        ORDER BY is_default DESC, created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $user_vehicles = $stmt->fetchAll();
    
    // Finde Standard-Fahrzeug oder nimm das erste
    foreach ($user_vehicles as $vehicle) {
        if ($vehicle['is_default']) {
            $default_vehicle = $vehicle;
            break;
        }
    }
    
    if (!$default_vehicle && !empty($user_vehicles)) {
        $default_vehicle = $user_vehicles[0];
    }
    
    // Wenn Fahrzeuge existieren, Standard-Fahrzeug verwenden, sonst das alte Kennzeichen
    if ($default_vehicle) {
        $customer_license_plate = $default_vehicle['license_plate'];
    } else {
        $customer_license_plate = $user['license_plate'] ?? '';
    }
}

// Erfolgs-/Fehlermeldungen werden jetzt per JavaScript angezeigt
?>

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="<?php 
        require_once __DIR__ . '/includes/csrf.php';
        echo htmlspecialchars(csrf_token(), ENT_QUOTES, 'UTF-8'); 
    ?>">
    <meta name="robots" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="googlebot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    <meta name="bingbot" content="noindex, nofollow, noarchive, nosnippet, noimageindex">
    
    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#0056b3">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="default">
    <meta name="apple-mobile-web-app-title" content="B&E Boxenstop">
    <link rel="manifest" href="/manifest.json">
    
    <!-- Apple Touch Icons -->
    <!-- Apple Touch Icons - iOS (180x180 ist Standard für moderne iOS-Geräte) -->
    <link rel="apple-touch-icon" sizes="180x180" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon-precomposed" sizes="180x180" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="152x152" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="144x144" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="120x120" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="114x114" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="76x76" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="72x72" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="60x60" href="assets/images/new-pwa_icon.png">
    <link rel="apple-touch-icon" sizes="57x57" href="assets/images/new-pwa_icon.png">
    
    <!-- Favicons - Alle Browser -->
    <link rel="icon" type="image/png" sizes="512x512" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="384x384" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="192x192" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="152x152" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="144x144" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="128x128" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="96x96" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="72x72" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="48x48" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="32x32" href="assets/images/new-pwa_icon.png">
    <link rel="icon" type="image/png" sizes="16x16" href="assets/images/new-pwa_icon.png">
    <link rel="shortcut icon" href="assets/images/new-pwa_icon.png">
    <link rel="icon" href="assets/images/new-pwa_icon.png">
    
    <title>Termin buchen - B&E Gruppe</title>
    
    <!-- Error Handling Script - MUSS SEHR FRÜH GELADEN WERDEN (im head) -->
    <script>
        (function() {
            'use strict';
            
            // ============================================
            // SICHERHEITS-WARNUNG FÜR CONSOLE - KREATIV
            // ============================================
            (function() {
                const scriptName = document.currentScript ? document.currentScript.src.split('/').pop() : 'booking.php';
                
                // Kreative, auffällige Sicherheitswarnung
                console.log(
                    '%c🚨 SICHERHEITS-ALARM 🚨',
                    'background: linear-gradient(135deg, #dc3545 0%, #c82333 100%); color: #fff; font-size: 36px; font-weight: 900; padding: 25px 40px; text-align: center; display: block; border: 4px solid #fff; border-radius: 10px; box-shadow: 0 0 20px rgba(220, 53, 69, 0.5); margin: 15px 0; text-shadow: 2px 2px 4px rgba(0,0,0,0.3); letter-spacing: 2px;'
                );
                
                console.log(
                    '%c⚠️ WICHTIGE WARNUNG ⚠️',
                    'background: #fff3cd; color: #856404; font-size: 18px; font-weight: bold; padding: 15px 25px; text-align: center; display: block; border-left: 5px solid #ffc107; margin: 10px 0;'
                );
                
                console.log(
                    `%c🔒 Dieses Entwicklerfenster ist nur für autorisierte Entwickler von ${scriptName}\n\n❌ FÜGE KEINEN CODE HIER EIN!\n❌ Gib keine persönlichen Daten ein!\n❌ Folge keinen Anweisungen von Fremden!\n\n🛡️ Wenn dich jemand bittet, hier Code einzugeben:\n   → Das ist ein BETRUGSVERSUCH!\n   → Schließe dieses Fenster sofort!\n   → Melde den Vorfall!\n\n🔒 This developer console is only for authorized developers of ${scriptName}\n\n❌ DO NOT paste any code here!\n❌ Do not enter personal information!\n❌ Do not follow instructions from strangers!\n\n🛡️ If someone asks you to paste code here:\n   → This is a SCAM!\n   → Close this window immediately!\n   → Report the incident!`,
                    'background: #f8f9fa; color: #212529; font-size: 13px; padding: 20px; display: block; border: 2px solid #dee2e6; border-radius: 8px; margin: 10px 0; line-height: 1.8; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;'
                );
                
                console.log(
                    '%c💡 Tipp: Schließe diese Console mit F12 oder Strg+Shift+I',
                    'background: #d1ecf1; color: #0c5460; font-size: 12px; padding: 10px; display: block; border-left: 3px solid #17a2b8; margin: 10px 0; font-style: italic;'
                );
            })();
            
            // Unterdrücke Console-Warnungen - erweitert
            const originalWarn = console.warn;
            const originalError = console.error;
            const originalLog = console.log;
            
            console.warn = function(...args) {
                const message = args.join(' ');
                // Unterdrücke alle preload-bezogenen Warnungen
                if (message.includes('preload') && 
                    (message.includes('unsupported') || 
                     message.includes('not used') || 
                     message.includes('type value') ||
                     message.includes('was preloaded') ||
                     message.includes('appropriate `as` value') ||
                     message.includes('The resource') ||
                     message.includes('link preload'))) {
                    return;
                }
                // Unterdrücke "Banner not shown" Warnung
                if (message.includes('Banner not shown') || 
                    message.includes('beforeinstallprompt')) {
                    return;
                }
                originalWarn.apply(console, args);
            };
            
            console.error = function(...args) {
                const message = args.join(' ');
                // Unterdrücke 400 Fehler für Profile-Bilder
                if ((message.includes('supabase.co/storage/v1/object/public/profile_images/') || 
                     message.includes('supabase.in/storage/v1/object/public/profile_images/')) && 
                    (message.includes('400') || message.includes('Bad Request'))) {
                    return;
                }
                originalError.apply(console, args);
            };
            
            console.log = function(...args) {
                const message = args.join(' ');
                // Unterdrücke preload-bezogene Logs
                if (message.includes('preload') && 
                    (message.includes('unsupported') || 
                     message.includes('not used') || 
                     message.includes('type value') ||
                     message.includes('The resource') ||
                     message.includes('link preload'))) {
                    return;
                }
                // Unterdrücke PWA Debug-Ausgaben
                if (message.includes('PWA') && 
                    (message.includes('Script START') || 
                     message.includes('PWA_DEBUG') || 
                     message.includes('PWA DEBUG') ||
                     message.includes('Tippe PWA_DEBUG') ||
                     message.includes('verfügbar'))) {
                    return;
                }
                // Unterdrücke booking.js Debug-Ausgaben
                if (message.includes('Loading autohaus pricing') ||
                    message.includes('Autohaus pricing loaded') ||
                    message.includes('updatePricing called') ||
                    message.includes('Base wheel change') ||
                    message.includes('Wheels storage action') ||
                    message.includes('Final breakdown') ||
                    message.includes('Final total') ||
                    message.includes('API Response') ||
                    message.includes('Available Slots') ||
                    message.includes('Total Booked') ||
                    message.includes('DEBUG INFORMATION') ||
                    message.includes('renderOccupancy called') ||
                    message.includes('Calculated values')) {
                    return;
                }
                originalLog.apply(console, args);
            };
        })();
    </script>
    
    <link rel="stylesheet" href="assets/css/style.css?v=<?php echo filemtime(__DIR__ . '/assets/css/style.css'); ?>">
    <link rel="stylesheet" href="assets/css/volvo-theme.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Three.js -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/three.js/r128/three.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/three@0.128.0/examples/js/controls/OrbitControls.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/three@0.128.0/examples/js/loaders/GLTFLoader.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/three@0.128.0/examples/js/loaders/DRACOLoader.js"></script>
    
    <style>
        :root {
            /* Volvo Shadow Guidelines - 5 Elevation Levels */
            --shadow-1: 0 1px 2px rgba(0, 0, 0, 0.1); /* Content switcher, Tabs */
            --shadow-2: 0 2px 4px rgba(0, 0, 0, 0.12); /* Buttons, Slider, Tooltip */
            --shadow-4: 0 4px 8px rgba(0, 0, 0, 0.15); /* Cards, Dropdown, Search fields, Snackbar */
            --shadow-8: 0 8px 16px rgba(0, 0, 0, 0.2); /* Dialog, Date picker, Time picker */
            --shadow-16: 0 16px 32px rgba(0, 0, 0, 0.25); /* Modal dialog */
        }
        
        /* Button Styles - Volvo Default (4px) */
        .btn,
        button.btn,
        .btn-primary,
        .btn-secondary,
        .btn-outline-primary,
        .nav-link {
            border-radius: 4px !important;
        }
        
        /* Volvo Design System Button Styles - Volvo Guidelines */
        .btn,
        button.btn {
            padding: 0.625rem 1.25rem;
            border: none;
            border-radius: 4px;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            text-decoration: none;
            line-height: 1.5;
            font-family: inherit;
        }
        
        /* Button Style 4: Solid Blue (Primary) - Volvo Guidelines */
        .btn-primary {
            background: #0056b3;
            color: #ffffff;
            box-shadow: none;
        }
        
        .btn-primary:hover,
        .btn-primary:focus {
            background: #0066cc;
            color: #ffffff;
        }
        
        .btn-primary:active {
            background: #004c99;
            color: #ffffff;
        }
        
        .btn-primary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Button Style 2: Outlined Primary - Volvo Guidelines */
        .btn-outline-primary {
            background: #ffffff;
            color: #0056b3;
            border: 1px solid #0056b3;
            box-shadow: none;
        }
        
        .btn-outline-primary:hover,
        .btn-outline-primary:focus {
            background: #f8f9fa;
            color: #0056b3;
            border-color: #0066cc;
        }
        
        .btn-outline-primary:active {
            background: #e9ecef;
            color: #004c99;
            border-color: #004c99;
        }
        
        .btn-outline-primary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Button Style 2: Outlined (Secondary) - Volvo Guidelines */
        .btn-secondary {
            background: #ffffff;
            color: #1a1a1a;
            border: 1px solid #dee2e6;
            box-shadow: none;
        }
        
        .btn-secondary:hover,
        .btn-secondary:focus {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .btn-secondary:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .btn-secondary:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Disabled State - Volvo Guidelines */
        .btn:disabled,
        button.btn:disabled,
        .btn-primary:disabled,
        .btn-secondary:disabled,
        .btn-outline-primary:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
        }
        
        /* Einfache Input-Group Styles */
        .input-group {
            display: flex;
            gap: 0.5rem;
            align-items: stretch;
        }
        
        .input-group input {
            flex: 1;
        }
        
        .input-group button {
            white-space: nowrap;
            padding: 0.75rem 1rem;
        }
        
        @media (max-width: 768px) {
            .input-group {
                flex-direction: column;
            }
            
            /* Booking Form Mobile Optimierung */
            .booking-section {
                padding: 1rem;
            }
            
            .form-container {
                padding: 1.5rem;
            }
            
            .form-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .form-group label {
                font-size: 0.875rem;
            }
            
            .form-group input,
            .form-group select,
            .form-group textarea {
                padding: 0.875rem;
                font-size: 0.95rem;
            }
            
            /* Button Mobile Optimierung */
            .form-actions {
                flex-direction: column;
                gap: 1rem;
            }
            
            .form-actions button {
                width: 100%;
            }
            
            /* Time Slot Cards Mobile */
            .time-slots-grid {
                grid-template-columns: 1fr;
                gap: 0.75rem;
            }
            
            .slot-card {
                padding: 1rem;
            }
            
            .slot-card-header {
                flex-direction: column;
                gap: 0.5rem;
                align-items: flex-start;
            }
            
            .occupancy-display {
                padding: 0.5rem;
                font-size: 0.875rem;
            }
            
            /* Event Info Mobile */
            .event-banner {
                padding: 1rem;
            }
            
            .event-banner h2 {
                font-size: 1.25rem;
            }
            
            .event-banner p {
                font-size: 0.875rem;
            }
            
            .vehicle-selection {
                padding: 1rem;
            }
            
            .vehicle-card {
                padding: 1rem;
            }
            
            /* Autohaus Info Mobile */
            .autohaus-info-card {
                padding: 1rem;
            }
            
            .autohaus-info-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
        }
        
        @media (max-width: 480px) {
            .form-container {
                padding: 1rem;
            }
            
            .slot-card {
                padding: 0.875rem;
            }
            
            .occupancy-display {
                font-size: 0.75rem;
                padding: 0.375rem;
            }
            
            .form-actions button {
                font-size: 0.875rem;
                padding: 0.875rem 1rem;
            }
        }
        
        /* Verbessertes Navigation Design - Volvo Style */
        .nav-menu {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .nav-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.625rem 1.25rem;
            border-radius: 4px;
            text-decoration: none;
            font-weight: 500;
            font-size: 0.9rem;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            border: 1px solid #dee2e6;
            box-shadow: none;
        }
        
        .nav-link i {
            font-size: 0.9rem;
        }
        
        /* Navigation Links - Volvo Design System */
        .back-link {
            background: #ffffff;
            color: #1a1a1a;
            border-color: #dee2e6;
        }
        
        .back-link:hover,
        .back-link:focus {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .back-link:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .back-link:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .user-link {
            background: #0056b3;
            color: #ffffff;
            border-color: #0056b3;
        }
        
        .user-link:hover,
        .user-link:focus {
            background: #0066cc;
            color: #ffffff;
            border-color: #0066cc;
        }
        
        .user-link:active {
            background: #004c99;
            color: #ffffff;
            border-color: #004c99;
        }
        
        .user-link:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* User Profile Dropdown - Volvo Design System */
        .user-profile-dropdown {
            position: relative;
        }
        
        .user-profile-trigger {
            background: transparent;
            color: #1a1a1a;
            border: none;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.5rem 1rem;
            cursor: pointer;
            border-radius: 4px;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            font-family: inherit;
            font-size: 0.9rem;
            font-weight: 500;
        }
        
        .user-profile-trigger:hover,
        .user-profile-trigger:focus {
            background: #f8f9fa;
            color: #1a1a1a;
        }
        
        .user-profile-trigger:active {
            background: #e9ecef;
            color: #1a1a1a;
        }
        
        .user-profile-trigger:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .user-profile-trigger.active {
            background: #f8f9fa;
        }
        
        .user-dropdown-icon {
            font-size: 0.75rem;
            color: #6c757d;
            transition: transform 0.2s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .user-profile-trigger.active .user-dropdown-icon {
            transform: rotate(180deg);
        }
        
        .user-dropdown-menu {
            position: absolute;
            top: calc(100% + 0.5rem);
            right: 0;
            background: #ffffff;
            border-radius: 4px;
            box-shadow: var(--shadow-8);
            min-width: 200px;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            z-index: 1000;
            border: 1px solid #e9ecef;
            overflow: hidden;
        }
        
        .user-dropdown-menu.show {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }
        
        .dropdown-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 0.75rem 1rem;
            color: #1a1a1a;
            text-decoration: none;
            font-size: 0.9rem;
            font-weight: 500;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
            border: none;
            background: none;
            width: 100%;
            text-align: left;
        }
        
        .dropdown-item:hover {
            background: #f8f9fa;
            color: #1a1a1a;
        }
        
        .dropdown-item:active {
            background: #e9ecef;
        }
        
        .dropdown-item i {
            width: 18px;
            text-align: center;
            color: #6c757d;
        }
        
        .dropdown-item:hover i {
            color: #0056b3;
        }
        
        /* Logout Item - Rot */
        .logout-item {
            color: #dc3545 !important;
        }
        
        .logout-item i {
            color: #dc3545 !important;
        }
        
        .logout-item:hover {
            background-color: #f8f9fa;
            color: #c82333 !important;
        }
        
        .logout-item:hover i {
            color: #c82333 !important;
        }
        
        /* User Avatar in Navbar - Volvo Design System */
        .user-avatar-nav {
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
            overflow: hidden;
            background: linear-gradient(135deg, #0056b3 0%, #003057 100%);
            position: relative;
        }
        
        .user-avatar-img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 50%;
        }
        
        .user-avatar-initials {
            width: 100%;
            height: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #ffffff;
            font-weight: 600;
            font-size: 0.875rem;
            letter-spacing: 0.01em;
            background: linear-gradient(135deg, #0056b3 0%, #003057 100%);
        }
        
        .user-name-nav {
            font-weight: 500;
            font-size: 0.9rem;
            white-space: nowrap;
        }
        
        @media (max-width: 768px) {
            .user-name-nav {
                display: none;
            }
            
            .user-profile-trigger {
                padding: 0.5rem;
                min-width: 44px;
                justify-content: center;
            }
            
            .user-dropdown-menu {
                right: 0;
                left: auto;
            }
        }
        
        .logout-link {
            background: #1a1a1a;
            color: #ffffff;
            border-color: #1a1a1a;
        }
        
        .logout-link:hover,
        .logout-link:focus {
            background: #2d2d2d;
            color: #ffffff;
            border-color: #2d2d2d;
        }
        
        .logout-link:active {
            background: #0d0d0d;
            color: #ffffff;
            border-color: #0d0d0d;
        }
        
        .logout-link:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        .admin-link {
            background: #ffffff;
            color: #1a1a1a;
            border-color: #dee2e6;
        }
        
        .admin-link:hover,
        .admin-link:focus {
            background: #f8f9fa;
            color: #1a1a1a;
            border-color: #adb5bd;
        }
        
        .admin-link:active {
            background: #e9ecef;
            color: #1a1a1a;
            border-color: #6c757d;
        }
        
        .admin-link:focus {
            outline: 2px solid #0056b3;
            outline-offset: 2px;
        }
        
        /* Mobile Menu Button (Burger Menu) - ENTFERNT, nicht mehr verwendet */
        .mobile-menu-btn {
            display: none !important;
        }
        
        @media (max-width: 768px) {
            .nav-container {
                position: relative;
            }
            
            /* Nav-Menu ist immer sichtbar auf mobilen Geräten */
            .nav-menu {
                position: static;
                flex-direction: row;
                padding: 0;
                gap: 0.5rem;
                box-shadow: none;
                border: none;
                transform: none;
                opacity: 1;
                visibility: visible;
                order: 2;
            }
            
            /* User-Profile-Dropdown ist immer sichtbar */
            .user-profile-dropdown {
                display: flex;
            }
            
            /* User-Name kann auf sehr kleinen Bildschirmen ausgeblendet werden */
            .user-name-nav {
                display: none;
            }
            
            /* Nav-Links bleiben kompakt */
            .nav-link {
                padding: 0.5rem 0.75rem;
                font-size: 0.85rem;
            }
            
            .nav-link span {
                display: none;
            }
            
            /* User Avatar kleiner auf mobilen Geräten */
            .user-avatar-nav {
                width: 32px;
                height: 32px;
            }
            
            .user-profile-trigger {
                padding: 0.375rem 0.5rem;
                min-width: auto;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar">
        <div class="nav-container">
            <div class="nav-brand">
                <div class="logo-wrapper">
                    <img src="assets/images/volvo-logo.png" alt="Volvo" class="nav-logo">
                    <div class="brand-text">
                        <h1>B&E Gruppe</h1>
                        <span class="brand-subtitle">B&E Gruppe - Volvo Service</span>
                    </div>
                </div>
            </div>
            <div class="nav-menu">
                <a href="index.php" class="nav-link back-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Zurück</span>
                </a>
                <?php 
                    $current_user = getCurrentUser();
                    $user_profile_image = $current_user ? getUserProfileImage($current_user) : null;
                    $user_initials = $current_user ? getUserInitials($current_user) : '?';
                    $user_display_name = $current_user ? getUserDisplayName($current_user) : 'User';
                    
                    // Prüfe ob der eingeloggte User ein Admin ist
                    $is_admin = false;
                    if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] && isset($_SESSION['user_id'])) {
                        // Lade User-Daten
                        $stmt = $conn->prepare("SELECT id, email, admin_user_id FROM users WHERE id = ? AND is_active = 1");
                        $stmt->execute([$_SESSION['user_id']]);
                        $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($user_data) {
                            // Prüfe ob User direkt mit Admin verknüpft ist
                            if (!empty($user_data['admin_user_id'])) {
                                // Prüfe ob verknüpfter Admin aktiv ist
                                $stmt = $conn->prepare("SELECT id FROM admin_users WHERE id = ? AND is_active = 1");
                                $stmt->execute([$user_data['admin_user_id']]);
                                $admin_user = $stmt->fetch();
                                $is_admin = (bool)$admin_user;
                            } else {
                                // Fallback: Suche nach E-Mail in admin_users
                                $user_email = $user_data['email'] ?? '';
                                if (!empty($user_email)) {
                                    $stmt = $conn->prepare("SELECT id FROM admin_users WHERE email = ? AND is_active = 1");
                                    $stmt->execute([$user_email]);
                                    $admin_user = $stmt->fetch();
                                    $is_admin = (bool)$admin_user;
                                }
                            }
                        }
                    }
                    // Oder prüfe ob der User bereits als Admin eingeloggt ist
                    if (!$is_admin && isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in']) {
                        $is_admin = true;
                    }
                ?>
                <div class="user-profile-dropdown">
                    <button type="button" class="user-profile-trigger" onclick="toggleUserDropdown(event)">
                        <div class="user-avatar-nav">
                            <?php if ($user_profile_image): ?>
                                <img src="<?php echo htmlspecialchars($user_profile_image); ?>" 
                                     alt="<?php echo htmlspecialchars($user_display_name); ?>" 
                                     class="user-avatar-img"
                                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                                <div class="user-avatar-initials" style="display: none;">
                                    <?php echo htmlspecialchars($user_initials); ?>
                                </div>
                            <?php else: ?>
                                <div class="user-avatar-initials">
                                    <?php echo htmlspecialchars($user_initials); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <span class="user-name-nav"><?php echo htmlspecialchars($user_display_name); ?></span>
                        <i class="fas fa-chevron-down user-dropdown-icon"></i>
                    </button>
                    <div class="user-dropdown-menu" id="userDropdownMenu">
                        <a href="profile.php" class="dropdown-item">
                            <i class="fas fa-user"></i>
                            <span>Profil</span>
                        </a>
                        <?php if ($is_admin): ?>
                        <a href="admin/" class="dropdown-item">
                            <i class="fas fa-cog"></i>
                            <span>Admin</span>
                        </a>
                        <?php endif; ?>
                        <a href="api/logout.php" class="dropdown-item logout-item">
                            <i class="fas fa-sign-out-alt"></i>
                            <span>Abmelden</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </nav>

    <!-- Booking Section -->
    <section class="booking-section">
        <div class="section-container">
            <div class="booking-header">
                <h1 class="section-title">Termin buchen</h1>
                <p class="section-subtitle">Buchen Sie Ihren Reifenwechsel-Service</p>
                
                <div class="event-info-card">
                    <div class="event-info-header">
                        <h3><?php echo htmlspecialchars($event['name']); ?></h3>
                        <div class="event-info-badges">
                            <span class="info-badge">
                                <i class="fas fa-calendar"></i>
                                <?php echo date('d.m.Y', strtotime($event['event_date'])); ?>
                            </span>
                            <span class="info-badge">
                                <i class="fas fa-clock"></i>
                                <?php echo date('H:i', strtotime($event['start_time'])); ?> - <?php echo date('H:i', strtotime($event['end_time'])); ?>
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Alerts werden per JavaScript angezeigt -->

            <div class="booking-layout">
                <div class="booking-main">
                    <form id="booking-form" method="POST" class="booking-form">
                    <input type="hidden" name="event_id" id="event_id" value="<?php echo (int)$event_id; ?>">
                    <input type="hidden" name="autohaus_id" id="autohaus_id" value="<?php echo $event['autohaus_id'] ?? ''; ?>">
                    <input type="hidden" name="event_name" id="event_name" value="<?php echo htmlspecialchars($event['name']); ?>">
                    <div class="form-grid">
                        <!-- Zeitslot Auswahl -->
                        <div class="form-group">
                            <label for="booking_date">Wählen Sie Ihr Datum</label>
                            <input type="date" id="booking_date" name="slot_date" required 
                                   min="<?php echo htmlspecialchars($event['event_date']); ?>" 
                                   max="<?php echo htmlspecialchars($event['event_date']); ?>" 
                                   value="<?php echo htmlspecialchars($event['event_date']); ?>">
                        </div>

                        <div class="form-group full-width">
                            <label>Auslastung</label>
                            <div id="occupancy-calendar" class="occupancy-calendar"></div>
                        </div>

                        <div class="form-group">
                            <label>Wählen sie einen verfügbare Zeitslots</label>
                            <div id="time-slots" class="time-slots slot-grid">
                                <p class="loading">Lade verfügbare Zeitslots...</p>
                            </div>
                        </div>

                        <!-- Kundeninformationen -->
                        <?php if ($user): ?>
                        <div style="background: #d1ecf1; border: 1px solid #bee5eb; border-radius: 8px; padding: 1rem; margin-bottom: 1.5rem;">
                            <i class="fas fa-info-circle" style="color: #0c5460; margin-right: 0.5rem;"></i>
                            <span style="color: #0c5460; font-weight: 500;">Ihre Daten wurden aus Ihrem Profil geladen</span>
                        </div>
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label for="customer_name">Ihr Name *</label>
                            <input type="text" id="customer_name" name="customer_name" required 
                                   placeholder="Vor- und Nachname"
                                   value="<?php echo htmlspecialchars($customer_name); ?>">
                        </div>

                        <?php if (!empty($user_vehicles)): ?>
                        <div class="form-group">
                            <label for="selected_vehicle">Fahrzeug auswählen *</label>
                            <select id="selected_vehicle" name="selected_vehicle" required onchange="updateLicensePlate()">
                                <?php foreach ($user_vehicles as $vehicle): ?>
                                    <option value="<?php echo $vehicle['id']; ?>" 
                                        data-license="<?php echo htmlspecialchars($vehicle['license_plate']); ?>"
                                        data-vehicle-model="<?php echo htmlspecialchars($vehicle['vehicle_model'] ?? ''); ?>"
                                        data-vehicle-brand="<?php echo htmlspecialchars($vehicle['vehicle_brand'] ?? ''); ?>"
                                        <?php echo $vehicle['is_default'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($vehicle['license_plate']); ?>
                                        <?php if ($vehicle['vehicle_brand']): ?>
                                            - <?php echo htmlspecialchars($vehicle['vehicle_brand']); ?>
                                        <?php endif; ?>
                                        <?php if ($vehicle['vehicle_model']): ?>
                                            <?php echo htmlspecialchars($vehicle['vehicle_model']); ?>
                                        <?php endif; ?>
                                        <?php if ($vehicle['is_default']): ?>
                                            (Standard)
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">
                                Oder <a href="profile.php" style="color: #0056b3; text-decoration: underline;">Fahrzeug hinzufügen</a>
                            </small>
                        </div>
                        
                        <input type="hidden" id="license_plate" name="license_plate" value="<?php echo htmlspecialchars($customer_license_plate); ?>">
                        <input type="hidden" id="selected_vehicle_model" value="<?php echo htmlspecialchars($default_vehicle['vehicle_model'] ?? ''); ?>">
                        <?php else: ?>
                        <div class="form-group">
                            <label for="license_plate">Ihr Kennzeichen *</label>
                            <input type="text" id="license_plate" name="license_plate" required 
                                   placeholder="z. B. M-AB 123"
                                   value="<?php echo htmlspecialchars($customer_license_plate); ?>">
                            <small class="text-muted">
                                <a href="profile.php" style="color: #0056b3; text-decoration: underline;">Mehrere Fahrzeuge verwalten</a>
                            </small>
                        </div>
                        <?php endif; ?>

                        <div class="form-group">
                            <label for="phone">Ihre Telefonnummer *</label>
                            <input type="tel" id="phone" name="phone" required 
                                   placeholder="z. B. +49 123 456 789"
                                   value="<?php echo htmlspecialchars($customer_phone); ?>">
                            <small class="text-muted">Für WhatsApp-Benachrichtigungen und Bestätigungen</small>
                        </div>

                        <div class="form-group">
                            <label for="email">Ihre E-Mail-Adresse *</label>
                            <input type="email" id="email" name="email" required 
                                   placeholder="z. B. kunde@beispiel.de"
                                   value="<?php echo htmlspecialchars($customer_email); ?>">
                            <small class="text-muted">Für Bestätigungen und wichtige Informationen</small>
                        </div>

                        <!-- Reifen-Informationen -->
                        <div class="form-group">
                            <h4 class="question-title">Bringen Sie die Räder mit?</h4>
                            <div class="radio-group">
                                <label class="radio-label">
                                    <input type="radio" name="brings_wheels" value="1" checked>
                                    <span class="radio-text">Ja, ich bringe die Räder mit</span>
                                </label>
                                <label class="radio-label">
                                    <input type="radio" name="brings_wheels" value="0">
                                    <span class="radio-text">Nein, meine Räder sind eingelagert</span>
                                </label>
                            </div>
                            
                            <!-- Einlagerungsnummer - wird dynamisch angezeigt/versteckt -->
                            <div class="form-group" id="storage_number_group" style="display: none; margin-top: 1rem;">
                                <label for="storage_number">Einlagerungsnummer *</label>
                                <div class="input-group">
                                    <input type="text" id="storage_number" name="storage_number" 
                                           placeholder="z. B. 1.033.91" required>
                                    <button type="button" id="check_tire_status_btn" class="btn btn-outline-primary">
                                        <i class="fas fa-search"></i>
                                        <span>Reifenzustand prüfen</span>
                                    </button>
                                </div>
                                <small class="text-muted">Ihre Einlagerungsnummer finden Sie auf der letzten Rechnung vom letzten Räderwechsel.</small>
                                
                                <!-- Reifenzustand-Anzeige -->
                                <div id="tire_status_display" class="tire-status-display" style="display: none;">
                                    <div class="tire-status-header">
                                        <h4><i class="fas fa-tire"></i> Reifenzustand</h4>
                                    </div>

                                    <div id="tire_status_content" class="tire-status-content">
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Reifenstatus-Felder wurden entfernt -->
                        </div>

                        <!-- Tire Type Question - Will be toggled by JavaScript -->
                        <div class="form-group" id="tire_type_group">
                            <h4 class="question-title">Welche Art von Reifen/Räder bringen Sie?</h4>
                            <div class="radio-group">
                                <label class="radio-label">
                                    <input type="radio" name="tire_type" value="set" checked>
                                    <span class="radio-text">Kompletträder / Radsatz</span>
                                </label>
                                <label class="radio-label">
                                    <input type="radio" name="tire_type" value="loose">
                                    <span class="radio-text">Lose Reifen (nicht montiert)</span>
                                </label>
                            </div>
                            <small class="text-muted">Lose Reifen können am Tag des Services nicht gewechselt werden.</small>
                        </div>

                        <div class="form-group">
                            <label for="wheels_storage_action">Was sollen wir mit den abmontierten Rädern machen?</label>
                            <select id="wheels_storage_action" name="wheels_storage_action">
                                <option value="store">Einlagern</option>
                                <option value="vehicle">In Fahrzeug legen</option>
                            </select>
                        </div>

                        <?php if (!empty($services)): ?>
                        <div class="form-group">
                            <h4 class="question-title">Zusätzliche Services</h4>
                            <div class="checkbox-group">
                                <?php foreach ($services as $service): ?>
                                    <label class="checkbox-label">
                                        <input type="checkbox" name="service_<?php echo $service['id']; ?>" value="1" data-service-id="<?php echo $service['id']; ?>" data-service-name="<?php echo htmlspecialchars($service['service_name']); ?>" data-service-price="<?php echo $service['price'] ?? 0; ?>" data-service-show-price="<?php echo $service['show_price'] ? 'true' : 'false'; ?>">
                                        <span class="checkbox-text">
                                            <?php echo htmlspecialchars($service['service_name']); ?>
                                            <?php if ($service['show_price'] && $service['price']): ?>
                                                <span class="service-price">(<?php echo number_format($service['price'], 2); ?> €)</span>
                                            <?php else: ?>
                                                <span class="service-price">(auf Anfrage)</span>
                                            <?php endif; ?>
                                        </span>
                                    </label>
                                    <?php if ($service['service_description']): ?>
                                        <small class="text-muted"><?php echo htmlspecialchars($service['service_description']); ?></small>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <div class="form-group">
                            <label for="seasonal_check">Jahreszeiten-Check</label>
                            <select id="seasonal_check" name="seasonal_check">
                                <option value="none">Kein Check</option>
                                <option value="spring">Frühjahres-Check</option>
                                <option value="winter">Winter-Check</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <h4 class="question-title">Haben Sie ein Radmutterschloss?</h4>
                            <div class="radio-group">
                                <label class="radio-label">
                                    <input type="radio" name="has_wheel_lock" value="1">
                                    <span class="radio-text">Ja</span>
                                </label>
                                <label class="radio-label">
                                    <input type="radio" name="has_wheel_lock" value="0" checked>
                                    <span class="radio-text">Nein</span>
                                </label>
                            </div>
                            
                            <!-- Info-Text für Radmutterschloss -->
                            <div id="wheel_lock_info" class="wheel-lock-info" style="display: none; margin-top: 1rem;">
                                <div class="info-box">
                                    <i class="fas fa-info-circle"></i>
                                    <div class="info-content">
                                        <strong>Wichtiger Hinweis:</strong>
                                        <p>Bitte legen Sie das Radmutterschloss gut sichtbar auf den Beifahrersitz oder auf das Armaturenbrett, damit unsere Mitarbeiter es beim Service schnell finden können.</p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group full-width">
                            <label for="info_text">Zusätzliche Informationen</label>
                            <textarea id="info_text" name="info_text" rows="3" 
                                      placeholder="Besondere Wünsche oder Hinweise (optional)"></textarea>
                        </div>
                    </div>

                        <div class="form-actions">
                            <button type="button" class="btn btn-secondary" onclick="history.back()">
                                <i class="fas fa-arrow-left"></i>
                                <span>Abbrechen</span>
                            </button>
                            <button type="submit" class="btn btn-primary" id="submit-btn" disabled>
                                <i class="fas fa-paper-plane"></i>
                                <span>Termin buchen</span>
                            </button>
                        </div>
                    </form>
                </div>

                <aside class="booking-sidebar">
                    <div class="summary-card">
                        <h3>Zusammenfassung</h3>
                        <div class="summary-content">
                            <div class="summary-row">
                                <span class="summary-label">Event</span>
                                <span class="summary-value"><?php echo htmlspecialchars($event['name']); ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">Datum</span>
                                <span class="summary-value"><?php echo date('d.m.Y', strtotime($event['event_date'])); ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">Zeit</span>
                                <span class="summary-value"><?php echo date('H:i', strtotime($event['start_time'])); ?> - <?php echo date('H:i', strtotime($event['end_time'])); ?></span>
                            </div>
                            <div class="summary-row">
                                <span class="summary-label">Gewählter Slot</span>
                                <span class="summary-value" id="summary-slot">Bitte Slot wählen</span>
                            </div>
                            <div class="summary-row" id="summary-services" style="display: none;">
                                <span class="summary-label">Zusätzliche Services</span>
                                <span class="summary-value" id="summary-wiper-blades">-</span>
                            </div>
                            <div class="summary-row" id="summary-pricing" style="display: none;">
                                <span class="summary-label">Preisübersicht</span>
                                <div class="summary-value" id="price-breakdown"></div>
                            </div>
                            <div class="summary-row summary-total" id="summary-total" style="display: none;">
                                <span class="summary-label"><strong>Gesamtpreis</strong></span>
                                <span class="summary-value" id="total-price">-</span>
                            </div>
                        </div>
                        <div class="summary-footer">
                            <div class="approval-info">
                                <i class="fas fa-info-circle"></i>
                                <span>Den Termin können Sie nach Buchung in Ihrem Profil sehen.</span>
                            </div>
                        </div>
                    </div>
                </aside>
            </div>
        </div>
    </section>

    <script>
        // Autohaus-spezifische Services an JavaScript übergeben
        window.autohausServices = <?php echo json_encode($services); ?>;
        
        // Autohaus-spezifische Preise an JavaScript übergeben
        <?php if ($autohaus_pricing): ?>
        window.autohausPricing = {
            price_wheel_change: <?php echo (!empty($autohaus_pricing['price_wheel_change']) && $autohaus_pricing['price_wheel_change'] !== null) ? json_encode((float)$autohaus_pricing['price_wheel_change']) : json_encode('auf_anfrage'); ?>,
            price_storage: <?php echo (!empty($autohaus_pricing['price_storage']) && $autohaus_pricing['price_storage'] !== null) ? json_encode((float)$autohaus_pricing['price_storage']) : json_encode('auf_anfrage'); ?>,
            price_spring_check: <?php echo (!empty($autohaus_pricing['price_spring_check']) && $autohaus_pricing['price_spring_check'] !== null) ? json_encode((float)$autohaus_pricing['price_spring_check']) : json_encode('auf_anfrage'); ?>,
            price_winter_check: <?php echo (!empty($autohaus_pricing['price_winter_check']) && $autohaus_pricing['price_winter_check'] !== null) ? json_encode((float)$autohaus_pricing['price_winter_check']) : json_encode('auf_anfrage'); ?>
        };
        <?php else: ?>
        window.autohausPricing = {
            price_wheel_change: 'auf_anfrage',
            price_storage: 'auf_anfrage',
            price_spring_check: 'auf_anfrage',
            price_winter_check: 'auf_anfrage'
        };
        <?php endif; ?>
        
        // Reifenzustand prüfen mit neuer RESY-API - GLOBAL VERFÜGBAR (SOFORT VERFÜGBAR)
        async function checkTireStatus() {
            const storageNumber = document.getElementById('storage_number')?.value.trim();
            const button = document.getElementById('check_tire_status_btn');
            const display = document.getElementById('tire_status_display');
            const content = document.getElementById('tire_status_content');
            const autohausId = document.getElementById('autohaus_id')?.value || 1;
            
            if (!storageNumber) {
                alert('Bitte geben Sie zuerst eine Einlagerungsnummer ein.');
                return;
            }
            
            if (!button || !display || !content) {
                return;
            }
            
            // Button deaktivieren und Loading anzeigen
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> <span>Prüfe...</span>';
            
            // Display anzeigen
            display.style.display = 'block';
            
            // Zeige Loading-Spinner
            content.innerHTML = `
                <div class="loading-spinner" style="text-align: center; padding: 2rem;">
                    <i class="fas fa-spinner fa-spin" style="font-size: 2rem; color: #1976d2;"></i>
                    <p style="margin-top: 1rem;">Reifenzustand wird abgerufen...</p>
                </div>
            `;
            
            try {
                // Verwende die neue RESY-API
                const cleanStorageNumber = storageNumber.replace(/\./g, '');
                const response = await fetch(`api/resy_api_routes.php?action=get_wheelset_details&wheelset_number=${encodeURIComponent(cleanStorageNumber)}&autohaus_id=${autohausId}`);
                
                const data = await response.json();
                
                if (data.success && data.wheelset) {
                    // Transformiere RESY-Daten in unser Format
                    const tireData = transformResyWheelsetData(data.wheelset);
                    const vehicleModel = document.getElementById('selected_vehicle_model')?.value || '';
                    
                    // Entferne Loading-Spinner und zeige Reifendaten
                    const spinner = content.querySelector('.loading-spinner');
                    if (spinner) {
                        spinner.remove();
                    }
                    
                    // Zeige Reifendaten an
                    displayTireDataSimple(tireData, content);
                } else {
                    // Benutzerfreundliche Fehlermeldungen
                    let errorMessage = 'Die Einlagerungsnummer wurde nicht gefunden.';
                    let errorDetails = 'Bitte überprüfen Sie die Nummer auf Ihrer letzten Rechnung.';
                    
                    const errorText = (data.error || data.message || '').toLowerCase();
                    
                    if (errorText.includes('nicht gefunden') || errorText.includes('not found')) {
                        errorMessage = 'Einlagerungsnummer nicht gefunden';
                        errorDetails = 'Bitte überprüfen Sie die Nummer auf Ihrer letzten Rechnung vom Räderwechsel.';
                    } else if (errorText.includes('ungültig') || errorText.includes('invalid')) {
                        errorMessage = 'Ungültige Einlagerungsnummer';
                        errorDetails = 'Die eingegebene Nummer ist nicht korrekt. Bitte überprüfen Sie die Formatierung (z.B. 1.033.91).';
                    } else if (errorText.includes('fehler') || errorText.includes('error')) {
                        errorMessage = 'Fehler beim Abrufen der Daten';
                        errorDetails = 'Bitte versuchen Sie es in ein paar Momenten erneut.';
                    }
                    
                    content.innerHTML = `
                        <div class="tire-error-message">
                            <div class="tire-error-icon">
                                <i class="fas fa-exclamation-circle"></i>
                            </div>
                            <div class="tire-error-content">
                                <h4>${errorMessage}</h4>
                                <p>${errorDetails}</p>
                                <div class="tire-error-help">
                                    <i class="fas fa-info-circle"></i>
                                    <span>Ihre Einlagerungsnummer finden Sie auf der letzten Rechnung vom letzten Räderwechsel.</span>
                                </div>
                            </div>
                        </div>
                    `;
                }
            } catch (error) {
                content.innerHTML = `
                    <div class="tire-error-message">
                        <div class="tire-error-icon">
                            <i class="fas fa-wifi"></i>
                        </div>
                        <div class="tire-error-content">
                            <h4>Verbindungsproblem</h4>
                            <p>Die Verbindung zum Server konnte nicht hergestellt werden.</p>
                            <div class="tire-error-help">
                                <i class="fas fa-lightbulb"></i>
                                <span>Bitte überprüfen Sie Ihre Internetverbindung und versuchen Sie es erneut.</span>
                            </div>
                        </div>
                    </div>
                `;
            } finally {
                // Button wieder aktivieren
                button.disabled = false;
                button.innerHTML = '<i class="fas fa-search"></i> <span>Reifenzustand prüfen</span>';
            }
        }
        
        // Mache Funktion global verfügbar
        window.checkTireStatus = checkTireStatus;
        
        // Event-Listener für Button hinzufügen
        function attachCheckTireStatusListener() {
            const btn = document.getElementById('check_tire_status_btn');
            if (btn && btn.parentNode) {
                // Prüfe ob bereits ein Listener vorhanden ist
                if (!btn.hasAttribute('data-listener-attached')) {
                    // Füge Listener hinzu
                    btn.addEventListener('click', async function(e) {
                        e.preventDefault();
                        e.stopPropagation();
                        
                        // Bestehende Funktion aufrufen
                        checkTireStatus();
                    });
                    // Markiere als Listener hinzugefügt
                    btn.setAttribute('data-listener-attached', 'true');
                }
            }
        }
        
        // Versuche sofort
        attachCheckTireStatusListener();
        
        // Versuche auch nach DOMContentLoaded
        document.addEventListener('DOMContentLoaded', function() {
            attachCheckTireStatusListener();
        });
        
        // Versuche auch nach kurzer Verzögerung (falls Button dynamisch erstellt wird)
        setTimeout(function() {
            attachCheckTireStatusListener();
        }, 1000);
        
        // Fahrzeug-Auswahl-Update
        function updateLicensePlate() {
            const select = document.getElementById('selected_vehicle');
            const hiddenInput = document.getElementById('license_plate');
            const vehicleModelInput = document.getElementById('selected_vehicle_model');
            
            if (select && hiddenInput) {
                const selectedOption = select.options[select.selectedIndex];
                const licensePlate = selectedOption.getAttribute('data-license');
                const vehicleModel = selectedOption.getAttribute('data-vehicle-model') || '';
                hiddenInput.value = licensePlate;
                
                if (vehicleModelInput) {
                    vehicleModelInput.value = vehicleModel;
                }
                
            }
        }
        
        // Handle brings_wheels radio button change
        document.addEventListener('DOMContentLoaded', function() {
            const bringsWheelsRadios = document.querySelectorAll('input[name="brings_wheels"]');
            const storageNumberGroup = document.getElementById('storage_number_group');
            const tireTypeGroup = document.getElementById('tire_type_group');
            
            // Prüfe ob alle notwendigen Elemente existieren
            if (!bringsWheelsRadios || bringsWheelsRadios.length === 0) {
                return;
            }
            
            function updateFormFields() {
                const checkedRadio = document.querySelector('input[name="brings_wheels"]:checked');
                
                // Prüfe ob ein Radio-Button ausgewählt ist
                if (!checkedRadio) {
                    return;
                }
                
                const bringsWheels = checkedRadio.value;
                
                if (bringsWheels === '0') {
                    // If "Nein, meine Räder sind eingelagert" is selected
                    if (storageNumberGroup) {
                    storageNumberGroup.style.display = 'block';
                    }
                    const storageNumberInput = document.getElementById('storage_number');
                    if (storageNumberInput) {
                        storageNumberInput.required = true;
                    }
                    if (tireTypeGroup) {
                    tireTypeGroup.style.display = 'none';
                    }
                    const tireTypeSetRadio = document.querySelector('input[name="tire_type"][value="set"]');
                    if (tireTypeSetRadio) {
                        tireTypeSetRadio.checked = true;
                    }
                } else {
                    // If "Ja, ich bringe die Räder mit" is selected
                    if (storageNumberGroup) {
                    storageNumberGroup.style.display = 'none';
                    }
                    const storageNumberInput = document.getElementById('storage_number');
                    if (storageNumberInput) {
                        storageNumberInput.required = false;
                    }
                    if (tireTypeGroup) {
                    tireTypeGroup.style.display = 'block';
                    }
                }
            }
            
            // Add event listeners
            if (bringsWheelsRadios && bringsWheelsRadios.length > 0) {
                bringsWheelsRadios.forEach(radio => {
                    if (radio) {
                        radio.addEventListener('change', updateFormFields);
                    }
                });
            }
            
            // Initialize form fields (nur wenn Elemente existieren)
            if (storageNumberGroup && tireTypeGroup) {
            updateFormFields();
            }
            
            // Handle Radmutterschloss radio button change
            const wheelLockRadios = document.querySelectorAll('input[name="has_wheel_lock"]');
            const wheelLockInfo = document.getElementById('wheel_lock_info');
            
            function updateWheelLockInfo() {
                const checkedRadio = document.querySelector('input[name="has_wheel_lock"]:checked');
                
                if (wheelLockInfo && checkedRadio) {
                    if (checkedRadio.value === '1') {
                        // "Ja" ausgewählt - Info anzeigen
                        wheelLockInfo.style.display = 'block';
                        // Sanfte Animation
                        setTimeout(() => {
                            wheelLockInfo.style.opacity = '1';
                            wheelLockInfo.style.transform = 'translateY(0)';
                        }, 10);
                    } else {
                        // "Nein" ausgewählt - Info verstecken
                        wheelLockInfo.style.opacity = '0';
                        wheelLockInfo.style.transform = 'translateY(-10px)';
                        setTimeout(() => {
                            wheelLockInfo.style.display = 'none';
                        }, 300);
                    }
                }
            }
            
            // Add event listeners für Radmutterschloss
            if (wheelLockRadios && wheelLockRadios.length > 0) {
                wheelLockRadios.forEach(radio => {
                    if (radio) {
                        radio.addEventListener('change', updateWheelLockInfo);
                    }
                });
                
                // Initiale Prüfung
                updateWheelLockInfo();
            }
        });
        
        // Transformiere RESY Wheelset-Daten in unser Format
        function transformResyWheelsetData(resyWheelset) {
            const tireData = {};
            
            if (resyWheelset.wheels && Array.isArray(resyWheelset.wheels)) {
                resyWheelset.wheels.forEach((wheel, index) => {
                    // RESY verwendet Positionen: FRONT_LEFT, FRONT_RIGHT, REAR_LEFT, REAR_RIGHT
                    let position = '';
                    const wheelPosition = (wheel.position || '').toUpperCase();
                    
                    if (wheelPosition.includes('FRONT') && wheelPosition.includes('LEFT')) {
                        position = 'VL';
                    } else if (wheelPosition.includes('FRONT') && wheelPosition.includes('RIGHT')) {
                        position = 'VR';
                    } else if (wheelPosition.includes('REAR') && wheelPosition.includes('LEFT')) {
                        position = 'HL';
                    } else if (wheelPosition.includes('REAR') && wheelPosition.includes('RIGHT')) {
                        position = 'HR';
                    } else {
                        // Fallback: Verwende Index
                        const positions = ['VL', 'VR', 'HL', 'HR'];
                        position = positions[index] || positions[index % 4];
                    }
                    
                    // Berechne Profiltiefe (falls vorhanden)
                    const profileDepth = wheel.profileDepth || wheel.treadDepth || wheel.profile_depth || null;
                    
                    // Bestimme Zustand/Empfehlung basierend auf Profiltiefe
                    let condition = 'good';
                    let recommendation = 'In Ordnung';
                    
                    if (profileDepth !== null && profileDepth !== undefined) {
                        const depth = parseFloat(profileDepth);
                        if (depth < 1.6) {
                            condition = 'bad';
                            recommendation = 'Austausch dringend empfohlen';
                        } else if (depth < 3.0) {
                            condition = 'warning';
                            recommendation = 'Bald prüfen lassen';
                        } else {
                            condition = 'good';
                            recommendation = 'In Ordnung';
                        }
                    }
                    
                    tireData[position] = {
                        brand: wheel.brand || wheel.tireBrand || '-',
                        model: wheel.model || wheel.tireModel || '-',
                        size: wheel.size || (wheel.width && wheel.diameter ? `${wheel.width}/${wheel.ratio || ''}R${wheel.diameter}` : '-'),
                        tire_type: wheel.tireType || wheel.tire_type || '-',
                        profile_depth: profileDepth !== null ? `${profileDepth} mm` : '-',
                        recommendation: recommendation,
                        condition: condition
                    };
                });
            }
            
            return tireData;
        }
        
        function displayTireDataSimple(tireData, container) {
            const positions = ['VL', 'VR', 'HL', 'HR'];
            const positionNames = {
                'VL': 'Vorderrad Links',
                'VR': 'Vorderrad Rechts',
                'HL': 'Hinterrad Links',
                'HR': 'Hinterrad Rechts'
            };
            
            let html = '<div class="tire-status-grid">';
            
            positions.forEach((pos, index) => {
                const data = tireData[pos];
                if (data) {
                    // Bestimme Farben und Icons basierend auf Zustand
                    const conditionConfig = {
                        'good': {
                            color: '#28a745',
                            bgGradient: 'linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%)',
                            borderColor: '#28a745',
                            icon: 'fa-check-circle',
                            pulseColor: 'rgba(40, 167, 69, 0.3)'
                        },
                        'warning': {
                            color: '#ffc107',
                            bgGradient: 'linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%)',
                            borderColor: '#ffc107',
                            icon: 'fa-exclamation-triangle',
                            pulseColor: 'rgba(255, 193, 7, 0.3)'
                        },
                        'bad': {
                            color: '#dc3545',
                            bgGradient: 'linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%)',
                            borderColor: '#dc3545',
                            icon: 'fa-times-circle',
                            pulseColor: 'rgba(220, 53, 69, 0.3)'
                        }
                    };
                    
                    const config = conditionConfig[data.condition] || conditionConfig['good'];
                    
                    html += `
                        <div class="tire-card tire-card-${data.condition}" style="animation-delay: ${index * 0.1}s;">
                            <div class="tire-card-header">
                                <div class="tire-position-badge" style="background: ${config.color};">
                                    <span class="tire-position-short">${pos}</span>
                                    <span class="tire-position-full">${positionNames[pos]}</span>
                                </div>
                                <div class="tire-status-indicator tire-status-${data.condition}" style="--pulse-color: ${config.pulseColor};">
                                    <i class="fas ${config.icon}"></i>
                                    <span class="tire-status-text">${data.recommendation || 'In Ordnung'}</span>
                                </div>
                            </div>
                            
                            <div class="tire-card-body">
                                <div class="tire-info-grid">
                                    ${data.brand && data.brand !== '-' ? `
                                        <div class="tire-info-item">
                                            <i class="fas fa-tag"></i>
                                            <span class="tire-info-label">Marke:</span>
                                            <span class="tire-info-value">${data.brand}</span>
                                        </div>
                                    ` : ''}
                                    
                                    ${data.model && data.model !== '-' ? `
                                        <div class="tire-info-item">
                                            <i class="fas fa-cog"></i>
                                            <span class="tire-info-label">Modell:</span>
                                            <span class="tire-info-value">${data.model}</span>
                                        </div>
                                    ` : ''}
                                    
                                    ${data.size && data.size !== '-' ? `
                                        <div class="tire-info-item">
                                            <i class="fas fa-ruler-combined"></i>
                                            <span class="tire-info-label">Dimension:</span>
                                            <span class="tire-info-value">${data.size}</span>
                                        </div>
                                    ` : ''}
                                    
                                    ${data.profile_depth && data.profile_depth !== '-' ? `
                                        <div class="tire-info-item tire-profile-depth">
                                            <i class="fas fa-layer-group"></i>
                                            <span class="tire-info-label">Profiltiefe:</span>
                                            <span class="tire-info-value tire-depth-value" style="color: ${config.color}; font-weight: 700;">${data.profile_depth}</span>
                                        </div>
                                    ` : ''}
                                    
                                    ${data.tire_type && data.tire_type !== '-' ? `
                                        <div class="tire-info-item">
                                            <i class="fas fa-snowflake"></i>
                                            <span class="tire-info-label">Typ:</span>
                                            <span class="tire-info-value">${data.tire_type}</span>
                                        </div>
                                    ` : ''}
                                </div>
                            </div>
                            
                            <div class="tire-card-footer" style="border-top-color: ${config.borderColor};">
                                <div class="tire-condition-badge tire-condition-${data.condition}" style="background: ${config.color};">
                                    <i class="fas ${config.icon}"></i>
                                    <span>${data.condition === 'good' ? 'Gut' : data.condition === 'warning' ? 'Warnung' : 'Kritisch'}</span>
                                </div>
                            </div>
                        </div>
                    `;
                }
            });
            
            html += '</div>';
            
            // Fade-in Animation
            container.style.opacity = '0';
            container.innerHTML = html;
            
            // Trigger animation
            setTimeout(() => {
                container.style.transition = 'opacity 0.5s ease-in';
                container.style.opacity = '1';
            }, 50);
        }
    </script>
    
    <script>
        // User Dropdown Toggle Funktion
        function toggleUserDropdown(event) {
            event.stopPropagation();
            const menu = document.getElementById('userDropdownMenu');
            const trigger = event.currentTarget;
            
            if (menu && trigger) {
                const isOpen = menu.classList.contains('show');
                
                // Schließe alle anderen Dropdowns
                document.querySelectorAll('.user-dropdown-menu').forEach(m => {
                    m.classList.remove('show');
                });
                document.querySelectorAll('.user-profile-trigger').forEach(t => {
                    t.classList.remove('active');
                });
                
                // Toggle aktuelles Dropdown
                if (!isOpen) {
                    menu.classList.add('show');
                    trigger.classList.add('active');
                } else {
                    menu.classList.remove('show');
                    trigger.classList.remove('active');
                }
            }
        }
        
        // Schließe Dropdown beim Klick außerhalb
        document.addEventListener('click', function(event) {
            if (!event.target.closest('.user-profile-dropdown')) {
                document.querySelectorAll('.user-dropdown-menu').forEach(menu => {
                    menu.classList.remove('show');
                });
                document.querySelectorAll('.user-profile-trigger').forEach(trigger => {
                    trigger.classList.remove('active');
                });
            }
        });
    </script>
      
    <script src="assets/js/booking.js?v=<?php echo filemtime(__DIR__ . '/assets/js/booking.js'); ?>"></script>
    
    <style>
        .approval-info {
            background: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 8px;
            padding: 1rem;
            margin-top: 1rem;
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
        }
        
        .approval-info i {
            color: #1976d2;
            margin-top: 0.125rem;
            flex-shrink: 0;
        }
        
        .approval-info span {
            color: #1565c0;
            font-size: 0.9rem;
            line-height: 1.4;
        }
        
        .service-price {
            color: #28a745;
            font-weight: 600;
            margin-left: 0.5rem;
        }
        
        /* Radmutterschloss Info-Box */
        .wheel-lock-info {
            opacity: 0;
            transform: translateY(-10px);
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .wheel-lock-info .info-box {
            display: flex;
            align-items: flex-start;
            gap: 0.875rem;
            padding: 1rem 1.25rem;
            background: #e3f2fd;
            border: 1px solid #90caf9;
            border-left: 4px solid #1976d2;
            border-radius: 6px;
            box-shadow: var(--shadow-1);
        }
        
        .wheel-lock-info .info-box i {
            color: #1976d2;
            font-size: 1.25rem;
            margin-top: 0.125rem;
            flex-shrink: 0;
        }
        
        .wheel-lock-info .info-content {
            flex: 1;
        }
        
        .wheel-lock-info .info-content strong {
            display: block;
            color: #1565c0;
            font-size: 0.9375rem;
            font-weight: 600;
            margin-bottom: 0.5rem;
        }
        
        .wheel-lock-info .info-content p {
            margin: 0;
            color: #0d47a1;
            font-size: 0.875rem;
            line-height: 1.5;
        }
        
        @media (max-width: 768px) {
            .wheel-lock-info .info-box {
                padding: 0.875rem 1rem;
                gap: 0.75rem;
            }
            
            .wheel-lock-info .info-box i {
                font-size: 1.125rem;
            }
            
            .wheel-lock-info .info-content strong {
                font-size: 0.875rem;
                margin-bottom: 0.375rem;
            }
            
            .wheel-lock-info .info-content p {
                font-size: 0.8125rem;
            }
        }
        
        /* Reifenzustand-Anzeige Styles */
        .tire-status-display {
            margin-top: 1.5rem;
            margin-left: 0;
            margin-right: 0;
            background: #ffffff;
            border-radius: 8px;
            border: none;
            overflow: hidden;
            box-shadow: var(--shadow-2);
            width: 100%;
            max-width: 100%;
        }
        
        .tire-status-header {
            background: #1a1a1a;
            color: #ffffff;
            padding: 1.25rem 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .tire-status-header h4 {
            margin: 0;
            font-size: 1.1rem;
            font-weight: 600;
            color: #ffffff;
        }
        
        .tire-status-header i {
            font-size: 1.2rem;
            color: #ffffff;
        }
        
        .tire-status-content {
            padding: 1rem;
        }
        
        @media (max-width: 768px) {
            .tire-status-content {
                padding: 0.75rem;
            }
        }
        
        /* Neue verbesserte Reifenzustand-Grid - Kompakt und dezent */
        .tire-status-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 0.875rem;
            padding: 0;
        }
        
        .tire-card {
            background: #ffffff;
            border-radius: 8px;
            border: 1px solid #dee2e6;
            box-shadow: var(--shadow-1);
            overflow: hidden;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            animation: tireCardSlideIn 0.4s ease-out backwards;
            position: relative;
        }
        
        @keyframes tireCardSlideIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .tire-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-2);
            border-color: #adb5bd;
        }
        
        .tire-card-good {
            border-left: 3px solid #28a745;
        }
        
        .tire-card-warning {
            border-left: 3px solid #ffc107;
        }
        
        .tire-card-bad {
            border-left: 3px solid #dc3545;
        }
        
        .tire-card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.875rem 1rem;
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
        }
        
        .tire-position-badge {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            width: 44px;
            height: 44px;
            border-radius: 6px;
            color: #ffffff;
            font-weight: 600;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            transition: all 0.2s ease;
        }
        
        .tire-card:hover .tire-position-badge {
            transform: scale(1.05);
            box-shadow: 0 3px 8px rgba(0, 0, 0, 0.15);
        }
        
        .tire-position-short {
            font-size: 1rem;
            line-height: 1.2;
        }
        
        .tire-position-full {
            font-size: 0.6rem;
            opacity: 0.9;
            text-align: center;
            line-height: 1.1;
            margin-top: 1px;
        }
        
        .tire-status-indicator {
            display: flex;
            flex-direction: column;
            align-items: flex-end;
            gap: 0.125rem;
            position: relative;
        }
        
        .tire-status-indicator i {
            font-size: 1.125rem;
        }
        
        .tire-status-good i {
            color: #28a745;
        }
        
        .tire-status-warning i {
            color: #ffc107;
        }
        
        .tire-status-bad i {
            color: #dc3545;
        }
        
        .tire-status-text {
            font-size: 0.6875rem;
            font-weight: 500;
            color: #6c757d;
            text-align: right;
            white-space: nowrap;
        }
        
        .tire-card-body {
            padding: 0.875rem 1rem;
        }
        
        .tire-info-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 0.5rem;
        }
        
        .tire-info-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.5rem 0.625rem;
            background: #f8f9fa;
            border-radius: 4px;
            transition: all 0.2s ease;
        }
        
        .tire-info-item:hover {
            background: #e9ecef;
        }
        
        .tire-info-item i {
            color: #0056b3;
            width: 16px;
            text-align: center;
            font-size: 0.8125rem;
            flex-shrink: 0;
        }
        
        .tire-info-label {
            font-weight: 500;
            color: #495057;
            font-size: 0.8125rem;
            min-width: 70px;
        }
        
        .tire-info-value {
            color: #1a1a1a;
            font-weight: 500;
            font-size: 0.8125rem;
            flex: 1;
        }
        
        .tire-profile-depth {
            background: #e3f2fd !important;
            border-left: 2px solid #1976d2;
        }
        
        .tire-profile-depth:hover {
            background: #bbdefb !important;
        }
        
        .tire-depth-value {
            font-size: 0.9375rem !important;
            font-weight: 600 !important;
        }
        
        .tire-card-footer {
            padding: 0.625rem 1rem;
            background: #f8f9fa;
            border-top: 1px solid #e9ecef;
            display: flex;
            justify-content: center;
        }
        
        .tire-condition-badge {
            display: inline-flex;
            align-items: center;
            gap: 0.375rem;
            padding: 0.375rem 0.75rem;
            border-radius: 4px;
            color: #ffffff;
            font-weight: 500;
            font-size: 0.75rem;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }
        
        /* Fehlermeldungen - Verbessert */
        .tire-error-message {
            display: flex;
            gap: 1.5rem;
            padding: 2rem;
            background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
            border: 2px solid #ffc107;
            border-radius: 12px;
            animation: tireErrorSlideIn 0.5s ease-out;
        }
        
        @keyframes tireErrorSlideIn {
            from {
                opacity: 0;
                transform: translateX(-20px);
            }
            to {
                opacity: 1;
                transform: translateX(0);
            }
        }
        
        .tire-error-icon {
            flex-shrink: 0;
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #ffc107;
            border-radius: 50%;
            color: #ffffff;
            font-size: 1.75rem;
            box-shadow: 0 4px 12px rgba(255, 193, 7, 0.3);
        }
        
        .tire-error-content {
            flex: 1;
        }
        
        .tire-error-content h4 {
            margin: 0 0 0.75rem 0;
            color: #856404;
            font-size: 1.25rem;
            font-weight: 700;
        }
        
        .tire-error-content p {
            margin: 0 0 1rem 0;
            color: #856404;
            font-size: 1rem;
            line-height: 1.5;
        }
        
        .tire-error-help {
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
            padding: 1rem;
            background: rgba(255, 255, 255, 0.7);
            border-radius: 8px;
            border-left: 3px solid #ffc107;
        }
        
        .tire-error-help i {
            color: #ffc107;
            font-size: 1.1rem;
            margin-top: 0.125rem;
            flex-shrink: 0;
        }
        
        .tire-error-help span {
            color: #856404;
            font-size: 0.9rem;
            line-height: 1.5;
        }
        
        /* Legacy Styles für Kompatibilität */
        .tire-position-card {
            background: white;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1rem;
            border-left: 4px solid #dee2e6;
            box-shadow: var(--shadow-1);
            transition: all 0.3s ease;
        }
        
        .tire-position-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-4);
        }
        
        .tire-position-card.good {
            border-left-color: #28a745;
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
        }
        
        .tire-position-card.warning {
            border-left-color: #ffc107;
            background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
        }
        
        .tire-position-card.bad {
            border-left-color: #dc3545;
            background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
        }
        
        .tire-position-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.75rem;
        }
        
        .tire-position-name {
            font-weight: 600;
            font-size: 1rem;
            color: #495057;
        }
        
        .tire-status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 4px;
            font-size: 0.85rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .tire-status-badge.good {
            background: #28a745;
            color: white;
        }
        
        .tire-status-badge.warning {
            background: #ffc107;
            color: #212529;
        }
        
        .tire-status-badge.bad {
            background: #dc3545;
            color: white;
        }
        
        .tire-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 0.75rem;
            font-size: 0.9rem;
            color: #6c757d;
        }
        
        .tire-detail {
            display: flex;
            justify-content: space-between;
        }
        
        .tire-detail strong {
            color: #495057;
        }
        
        .loading-spinner {
            text-align: center;
            padding: 2rem;
            color: #6c757d;
        }
        
        .loading-spinner i {
            font-size: 2rem;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .error-message {
            background: #f8d7da;
            color: #721c24;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid #f5c6cb;
            text-align: center;
        }
        
        .error-message div {
            text-align: left;
            line-height: 1.5;
        }
        
        .success-message {
            background: #d4edda;
            color: #155724;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid #c3e6cb;
            text-align: center;
        }
        
        .info-message {
            background: #d1ecf1;
            color: #0c5460;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid #bee5eb;
            text-align: center;
        }
        
        @media (max-width: 768px) {
            .tire-details {
                grid-template-columns: 1fr;
            }
            
            .tire-position-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 0.5rem;
            }
            
            /* Mobile Styles für neue Reifenzustand-Karten - Kompakt */
            .tire-status-grid {
                grid-template-columns: 1fr;
                gap: 0.75rem;
            }
            
            .tire-card {
                border-radius: 6px;
            }
            
            .tire-card-header {
                padding: 0.75rem;
                flex-direction: row;
                gap: 0.75rem;
            }
            
            .tire-position-badge {
                width: 40px;
                height: 40px;
            }
            
            .tire-position-short {
                font-size: 0.9375rem;
            }
            
            .tire-position-full {
                font-size: 0.5625rem;
            }
            
            .tire-status-indicator {
                align-items: flex-end;
            }
            
            .tire-status-indicator i {
                font-size: 1rem;
            }
            
            .tire-status-text {
                font-size: 0.625rem;
            }
            
            .tire-card-body {
                padding: 0.75rem;
            }
            
            .tire-info-grid {
                gap: 0.375rem;
            }
            
            .tire-info-item {
                padding: 0.4375rem 0.5rem;
                gap: 0.4375rem;
            }
            
            .tire-info-item i {
                width: 14px;
                font-size: 0.75rem;
            }
            
            .tire-info-label {
                min-width: 60px;
                font-size: 0.75rem;
            }
            
            .tire-info-value {
                font-size: 0.75rem;
            }
            
            .tire-card-footer {
                padding: 0.5rem 0.75rem;
            }
            
            .tire-condition-badge {
                padding: 0.3125rem 0.625rem;
                font-size: 0.6875rem;
                gap: 0.3125rem;
            }
            
            .tire-error-message {
                flex-direction: column;
                padding: 1rem;
                gap: 0.75rem;
            }
            
            .tire-error-icon {
                width: 44px;
                height: 44px;
                font-size: 1.25rem;
            }
            
            .tire-error-content h4 {
                font-size: 1rem;
            }
            
            .tire-error-content p {
                font-size: 0.875rem;
            }
        }
    
        /* Z-Index Hierarchie - Zentral definiert
         * 9999999: Alert/Notification System (höchste Priorität)
         * 9999998: Loading Text
         * 9999997: Modal Content
         * 9999996: Modal Container
         * 9999995: Modal Backdrop
         * 1000: User Dropdown, Navbar
         * 20: Tabelle thead (innerhalb Modal)
         * 2: Such-Icon (innerhalb Modal)
         * 1: Standard Elemente
         */
        
        /* Modal Styles - Verbessert */
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            z-index: 9999996 !important;
        }
        
        .modal-backdrop {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.6);
            z-index: 9999995 !important;
            backdrop-filter: blur(4px);
        }
        
        .modal-content {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            border-radius: 12px;
            box-shadow: 0 16px 48px rgba(0, 0, 0, 0.3);
            z-index: 9999997 !important;
            max-width: 1700px;
            width: 95vw;
            max-height: 90vh;
            display: flex;
            flex-direction: column;
            overflow: hidden;
            animation: modalSlideIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translate(-50%, -48%);
            }
            to {
                opacity: 1;
                transform: translate(-50%, -50%);
            }
        }
        
        .modal-header {
            padding: 1.5rem 2rem;
            border-bottom: 2px solid #e9ecef;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
            flex-shrink: 0;
        }
        
        .modal-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #003057;
            margin: 0;
        }
        
        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #6c757d;
            cursor: pointer;
            padding: 0.5rem;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: all 0.2s;
        }
        
        .modal-close:hover {
            background: #e9ecef;
            color: #003057;
        }
        
        .modal-body {
            flex: 1;
            overflow: hidden;
            display: flex;
            flex-direction: column;
            padding: 0;
        }
        
        .modal-footer {
            padding: 1.5rem 2rem;
            border-top: 2px solid #e9ecef;
            background: #f8f9fa;
            display: flex;
            justify-content: flex-end;
            gap: 1rem;
            flex-shrink: 0;
        }
        
        /* Tire Table Styles */
        #tireSelectionTable {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.875rem;
        }
        
        #tireSelectionTable thead {
            position: sticky;
            top: 0;
            background: #f8f9fa;
            z-index: 20 !important; /* Innerhalb Modal, aber über anderen Elementen */
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        #tireSelectionTable th {
            padding: 0.75rem;
            text-align: left;
            border-bottom: 2px solid #e9ecef;
            font-weight: 600;
            color: #003057;
            background: #f8f9fa;
        }
        
        #tireSelectionTable td {
            padding: 0.75rem;
            border-bottom: 1px solid #e9ecef;
        }
        
        .tire-row {
            transition: background-color 0.15s ease;
            cursor: pointer;
        }
        
        .tire-row:hover {
            background-color: #f8f9fa !important;
        }
        
        .tire-row:active {
            background-color: rgba(0, 86, 179, 0.1) !important;
        }
        
        #tireTableContainer {
            scroll-behavior: smooth;
        }
        
        #tireTableContainer::-webkit-scrollbar {
            width: 8px;
        }
        
        #tireTableContainer::-webkit-scrollbar-track {
            background: #f8f9fa;
        }
        
        #tireTableContainer::-webkit-scrollbar-thumb {
            background: #dee2e6;
            border-radius: 4px;
        }
        
        #tireTableContainer::-webkit-scrollbar-thumb:hover {
            background: #adb5bd;
        }
        
        @media (max-width: 768px) {
            .modal-content {
                width: 98vw;
                max-height: 95vh;
            }
            
            .modal-header {
                padding: 1rem 1.5rem;
            }
            
            .modal-title {
                font-size: 1.25rem;
            }
        }
        
        /* Einlagerungs-Überzeugungs-Modal Styles */
        .storage-conviction-content {
            width: 100%;
        }
        
        .conviction-intro {
            margin-bottom: 1.5rem;
        }
        
        .conviction-benefits {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            margin-bottom: 0.75rem;
        }
        
        .benefit-item {
            display: flex;
            align-items: flex-start;
            gap: 0.875rem;
            padding: 0.875rem;
            background: #f8f9fa;
            border-radius: 4px;
            border-left: 3px solid #0056b3;
            transition: all 0.2s ease;
        }
        
        .benefit-item:hover {
            background: #e9ecef;
        }
        
        .benefit-icon {
            flex-shrink: 0;
            width: 36px;
            height: 36px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #0056b3;
            border-radius: 4px;
            color: #ffffff;
            font-size: 1rem;
        }
        
        .benefit-content {
            flex: 1;
        }
        
        .benefit-content h4 {
            margin: 0 0 0.375rem 0;
            color: #003057;
            font-size: 0.9375rem;
            font-weight: 600;
        }
        
        .benefit-content p {
            margin: 0;
            color: #495057;
            font-size: 0.8125rem;
            line-height: 1.4;
        }
        
        .conviction-note {
            display: flex;
            align-items: flex-start;
            gap: 0.75rem;
        }
        
        @media (max-width: 768px) {
            .benefit-item {
                padding: 0.75rem;
                gap: 0.625rem;
            }
            
            .benefit-icon {
                width: 32px;
                height: 32px;
                font-size: 0.9375rem;
            }
            
            .benefit-content h4 {
                font-size: 0.875rem;
            }
            
            .benefit-content p {
                font-size: 0.75rem;
            }
            
            .modal-content {
                max-width: 95vw !important;
            }
        }
    
    <?php if (isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in']): ?>
        <?php include 'includes/feedback_form.php'; ?>
    <?php endif; ?>
    
    <!-- Reifenauswahl Modal - Verbessert -->
    <div id="tireSelectionModal" class="modal" style="display: none; max-width: 1700px;">
        <div class="modal-backdrop" onclick="closeModal('tireSelectionModal')"></div>
        <div class="modal-content" style="max-width: 1700px;">
            <div class="modal-header" style="padding: 1.5rem 2rem; border-bottom: 2px solid #e9ecef; display: flex; justify-content: space-between; align-items: center; background: #f8f9fa; flex-shrink: 0;">
                <div class="modal-title" style="font-size: 1.5rem; font-weight: 700; color: #003057; margin: 0;">Reifen auswählen</div>
                <button class="modal-close" onclick="closeModal('tireSelectionModal')" style="background: none; border: none; font-size: 1.5rem; color: #6c757d; cursor: pointer; padding: 0.5rem; width: 40px; height: 40px; display: flex; align-items: center; justify-content: center; border-radius: 4px; transition: all 0.2s;" onmouseover="this.style.background='#e9ecef'; this.style.color='#003057'" onmouseout="this.style.background='none'; this.style.color='#6c757d'">×</button>
            </div>
            <div class="modal-body" style="flex: 1; overflow: hidden; display: flex; flex-direction: column; padding: 0;">
                <div style="display: flex; flex-direction: column; height: 100%; max-height: calc(90vh - 80px);">
                    <!-- Header mit Suche -->
                    <div style="padding: 1.5rem 2rem; border-bottom: 2px solid #e9ecef; background: #f8f9fa; flex-shrink: 0;">
                        <div style="margin-bottom: 1rem; padding: 1rem; background: rgba(0, 86, 179, 0.1); border-left: 4px solid #0056b3; border-radius: 6px;">
                            <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                                <i class="fas fa-info-circle" style="color: #0056b3;"></i>
                                <strong style="font-size: 0.875rem; color: #003057;">Analysierter Reifensatz:</strong>
                            </div>
                            <div style="display: flex; flex-wrap: wrap; gap: 0.5rem; font-size: 0.8125rem;" id="analyzedTireSetInfo">
                                <!-- Wird dynamisch gefüllt -->
                            </div>
                        </div>
                        <div style="display: flex; gap: 1rem; align-items: center; margin-bottom: 1rem; flex-wrap: wrap;">
                            <div style="flex: 1; position: relative; min-width: 250px;">
                                <i class="fas fa-search" style="position: absolute; left: 1rem; top: 50%; transform: translateY(-50%); color: #6c757d; z-index: 2;"></i>
                                <input type="text" id="tireSearchInput" placeholder="Suche nach Dimension, Hersteller, Modell..." class="form-control" style="padding-left: 2.5rem; font-size: 0.9375rem; height: 2.5rem; width: 100%; border: 1px solid #dee2e6; border-radius: 4px;">
                            </div>
                            <button type="button" class="btn btn-secondary" onclick="clearTireFilters()" style="white-space: nowrap; padding: 0.625rem 1.25rem;">
                                <i class="fas fa-eraser"></i> Filter zurücksetzen
                            </button>
                            <button type="button" class="btn btn-primary" onclick="applySelectedTires()" style="white-space: nowrap; padding: 0.625rem 1.25rem;">
                                <i class="fas fa-check"></i> Ausgewählte übernehmen (<span id="selectedTiresCount">0</span>)
                            </button>
                        </div>
                        <div style="display: flex; gap: 0.75rem; flex-wrap: wrap; align-items: center;">
                            <input type="text" id="tireDimensionFilter" placeholder="Dimension (z.B. 195/65 R 15)" class="form-control" style="width: 200px; font-size: 0.875rem; height: 2rem; border: 1px solid #dee2e6; border-radius: 4px; padding: 0.5rem;">
                            <input type="text" id="tireKeywordFilter" placeholder="Stichwort" class="form-control" style="width: 200px; font-size: 0.875rem; height: 2rem; border: 1px solid #dee2e6; border-radius: 4px; padding: 0.5rem;">
                            <label style="display: flex; align-items: center; gap: 0.5rem; cursor: pointer; font-size: 0.875rem;">
                                <input type="checkbox" id="tireStockFilter" style="cursor: pointer;">
                                <span>Nur mit Lagerbestand</span>
                            </label>
                        </div>
                    </div>
                    <div style="display: flex; flex: 1; overflow: hidden;">
                        <!-- Sidebar mit Marken-Filter -->
                        <div style="width: 280px; border-right: 2px solid #e9ecef; background: #f8f9fa; padding: 1.5rem; overflow-y: auto; flex-shrink: 0;">
                            <div style="margin-bottom: 1.5rem;">
                                <h3 style="font-size: 1rem; font-weight: 600; margin-bottom: 1rem; color: #003057;">
                                    <i class="fas fa-filter"></i> Marken
                                </h3>
                                <div style="max-height: 300px; overflow-y: auto;">
                                    <div id="manufacturerFilters" style="display: grid; grid-template-columns: 1fr; gap: 0.5rem;">
                                        <!-- Wird dynamisch gefüllt -->
                                    </div>
                                </div>
                            </div>
                            <div style="margin-bottom: 1.5rem; padding-top: 1.5rem; border-top: 1px solid #e9ecef;">
                                <h3 style="font-size: 1rem; font-weight: 600; margin-bottom: 1rem; color: #003057;">
                                    <i class="fas fa-tags"></i> Produkt-Segment
                                </h3>
                                <div style="display: flex; flex-direction: column; gap: 0.5rem;">
                                    <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                        <input type="checkbox" class="segment-filter" value="Sommer" style="cursor: pointer;">
                                        <span style="font-size: 0.875rem;">🌞 Sommer</span>
                                    </label>
                                    <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                        <input type="checkbox" class="segment-filter" value="Winter" style="cursor: pointer;">
                                        <span style="font-size: 0.875rem;">❄️ Winter</span>
                                    </label>
                                    <label style="display: flex; align-items: center; gap: 0.75rem; padding: 0.5rem; border-radius: 6px; cursor: pointer; transition: background 0.2s;">
                                        <input type="checkbox" class="segment-filter" value="All-Season" style="cursor: pointer;">
                                        <span style="font-size: 0.875rem;">🌍 Ganzjahres</span>
                                    </label>
                                </div>
                            </div>
                            <div style="padding-top: 1.5rem; border-top: 1px solid #e9ecef;">
                                <div style="font-size: 0.875rem; color: #6c757d;">
                                    <strong id="filteredTireCount">0</strong> von <strong id="totalTireCount">0</strong> Reifen
                                </div>
                            </div>
                        </div>
                        <!-- Hauptbereich mit Tabelle -->
                        <div style="flex: 1; overflow: hidden; display: flex; flex-direction: column;">
                            <div id="tireTableContainer" style="flex: 1; overflow-y: auto; padding: 1rem;">
                                <table id="tireSelectionTable" style="width: 100%; border-collapse: collapse; font-size: 0.875rem;">
                                    <thead style="position: sticky; top: 0; background: #f8f9fa; z-index: 20; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                                        <tr>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid #e9ecef; width: 40px;">
                                                <input type="checkbox" id="selectAllTires" style="cursor: pointer;">
                                            </th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid #e9ecef;">Dimension</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid #e9ecef;">EU-Label</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid #e9ecef;">Hersteller - Profil</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid #e9ecef;">Ausführung</th>
                                            <th style="padding: 0.75rem; text-align: left; border-bottom: 2px solid #e9ecef;">Einsatzgebiet</th>
                                            <th style="padding: 0.75rem; text-align: center; border-bottom: 2px solid #e9ecef;">Bestand</th>
                                            <th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid #e9ecef;">VK brutto</th>
                                            <th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid #e9ecef;">VK netto</th>
                                            <th style="padding: 0.75rem; text-align: right; border-bottom: 2px solid #e9ecef;">OT-EK</th>
                                        </tr>
                                    </thead>
                                    <tbody id="tireSelectionTableBody">
                                        <!-- Wird dynamisch gefüllt -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer" style="padding: 1.5rem 2rem; border-top: 2px solid #e9ecef; background: #f8f9fa; display: flex; justify-content: flex-end; gap: 1rem; flex-shrink: 0;">
                <button class="btn btn-secondary" onclick="closeModal('tireSelectionModal')" style="padding: 0.625rem 1.25rem;">Abbrechen</button>
            </div>
        </div>
    </div>
    
    <!-- Einlagerungs-Überzeugungs-Modal -->
    <div id="storageConvictionModal" class="modal" style="display: none;">
        <div class="modal-backdrop" onclick="closeStorageConvictionModal()"></div>
        <div class="modal-content" style="max-width: 600px;">
            <div class="modal-header" style="padding: 1.25rem 1.5rem;">
                <div class="modal-title" style="font-size: 1.125rem; font-weight: 600;">
                    Räder professionell einlagern?
                </div>
                <button class="modal-close" onclick="closeStorageConvictionModal()">×</button>
            </div>
            <div class="modal-body" style="padding: 1.5rem;">
                <div class="storage-conviction-content">
                    <div class="conviction-intro">
                        <p style="font-size: 0.9375rem; color: #495057; margin-bottom: 1.25rem; line-height: 1.5;">
                            Wir empfehlen Ihnen, Ihre Räder bei uns professionell einlagern zu lassen:
                        </p>
                    </div>
                    
                    <div class="conviction-benefits">
                        <div class="benefit-item">
                            <div class="benefit-icon">
                                <i class="fas fa-warehouse"></i>
                            </div>
                            <div class="benefit-content">
                                <h4>Professionelle Lagerung</h4>
                                <p>Ihre Räder werden in unserem klimakontrollierten Lager sicher und fachgerecht aufbewahrt.</p>
                            </div>
                        </div>
                        
                        <div class="benefit-item">
                            <div class="benefit-icon">
                                <i class="fas fa-spray-can"></i>
                            </div>
                            <div class="benefit-content">
                                <h4>Reinigung & Pflege</h4>
                                <p>Wir reinigen Ihre Räder gründlich und pflegen sie, damit sie beim nächsten Wechsel optimal vorbereitet sind.</p>
                            </div>
                        </div>
                        
                        <div class="benefit-item">
                            <div class="benefit-icon">
                                <i class="fas fa-search"></i>
                            </div>
                            <div class="benefit-content">
                                <h4>Profiltiefen-Prüfung</h4>
                                <p>Wir prüfen die Profiltiefe Ihrer Reifen und informieren Sie rechtzeitig, wenn ein Wechsel ansteht.</p>
                            </div>
                        </div>
                    </div>
                    
                    <div class="conviction-note" style="margin-top: 1.25rem; padding: 0.875rem; background: #f8f9fa; border-left: 3px solid #6c757d; border-radius: 4px;">
                        <i class="fas fa-info-circle" style="color: #6c757d; margin-right: 0.5rem; font-size: 0.875rem;"></i>
                        <span style="color: #495057; font-size: 0.8125rem; line-height: 1.4;">
                            Wenn Sie die Räder ins Fahrzeug legen, können wir diese Services nicht durchführen.
                        </span>
                    </div>
                </div>
            </div>
            <div class="modal-footer" style="padding: 1.25rem 1.5rem;">
                <button type="button" class="btn btn-secondary" onclick="closeStorageConvictionModal()" style="padding: 0.625rem 1.25rem; font-size: 0.875rem;">
                    <i class="fas fa-times"></i>
                    <span>Abbrechen</span>
                </button>
                <button type="button" class="btn btn-secondary" onclick="proceedWithVehicleStorage()" style="padding: 0.625rem 1.25rem; font-size: 0.875rem;">
                    <i class="fas fa-car"></i>
                    <span>Trotzdem in Fahrzeug legen</span>
                </button>
                <button type="button" class="btn btn-primary" onclick="switchToStorage()" style="padding: 0.625rem 1.25rem; font-size: 0.875rem;">
                    <i class="fas fa-check"></i>
                    <span>Doch einlagern</span>
                </button>
            </div>
        </div>
    </div>
    
    <script>
        // Modal Funktionen
        function openModal(modalId) {
            const modal = document.getElementById(modalId);
            if (modal) {
                modal.style.display = 'block';
                document.body.style.overflow = 'hidden';
            }
        }
        
        function closeModal(modalId) {
            const modal = document.getElementById(modalId);
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = '';
            }
        }
        
        // Einlagerungs-Überzeugungs-Modal Funktionen
        function openStorageConvictionModal() {
            const modal = document.getElementById('storageConvictionModal');
            if (modal) {
                modal.style.display = 'block';
                document.body.style.overflow = 'hidden';
            }
        }
        
        function closeStorageConvictionModal() {
            const modal = document.getElementById('storageConvictionModal');
            if (modal) {
                modal.style.display = 'none';
                document.body.style.overflow = '';
            }
        }
        
        function switchToStorage() {
            // Ändere Auswahl zu "Einlagern"
            const storageSelect = document.querySelector('select[name="wheels_storage_action"]');
            if (storageSelect) {
                storageSelect.value = 'store';
                // Trigger change event für Pricing-Update
                storageSelect.dispatchEvent(new Event('change'));
            }
            closeStorageConvictionModal();
            // Submit Form über globale Funktion oder direkt
            setTimeout(() => {
                if (window.proceedWithFormSubmission) {
                    window.proceedWithFormSubmission();
                } else {
                    // Fallback: Direktes Submit
                    const form = document.getElementById('booking-form');
                    if (form) {
                        const submitEvent = new Event('submit', { cancelable: true, bubbles: true });
                        form.dispatchEvent(submitEvent);
                    }
                }
            }, 300);
        }
        
        function proceedWithVehicleStorage() {
            // Behalte "In Fahrzeug legen" und submit trotzdem
            closeStorageConvictionModal();
            // Submit Form über globale Funktion oder direkt
            setTimeout(() => {
                if (window.proceedWithFormSubmission) {
                    window.proceedWithFormSubmission();
                } else {
                    // Fallback: Direktes Submit
                    const form = document.getElementById('booking-form');
                    if (form) {
                        const submitEvent = new Event('submit', { cancelable: true, bubbles: true });
                        form.dispatchEvent(submitEvent);
                    }
                }
            }, 300);
        }
        
        // Mache Modal-Funktionen global verfügbar - SOFORT verfügbar machen
        window.openStorageConvictionModal = openStorageConvictionModal;
        window.closeStorageConvictionModal = closeStorageConvictionModal;
        window.switchToStorage = switchToStorage;
        window.proceedWithVehicleStorage = proceedWithVehicleStorage;
        
        // ESC-Taste zum Schließen
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                const modals = document.querySelectorAll('.modal[style*="display: block"]');
                modals.forEach(modal => {
                    closeModal(modal.id);
                });
            }
        });
    </script>
    <!-- PWA Service Worker Registration -->
    <script>
        // CSRF-Token in Session Storage speichern für Fallback
        document.addEventListener('DOMContentLoaded', function() {
            const csrfMeta = document.querySelector('meta[name="csrf-token"]');
            if (csrfMeta && csrfMeta.content && csrfMeta.content.trim() !== '') {
                sessionStorage.setItem('csrf_token', csrfMeta.content.trim());
            }
        });
        
        if ('serviceWorker' in navigator) {
            window.addEventListener('load', () => {
                navigator.serviceWorker.register('/service-worker.js')
                    .then((registration) => {
                        // Service Worker registriert
                    })
                    .catch((error) => {
                        // Stille Fehlerbehandlung
                    });
            });
        }
    </script>
    
    <!-- PWA Install Prompt -->
    <script src="assets/js/pwa-install.js"></script>
</body>
</html>
